#!/usr/local/bin/perl -w
#
# testPerlCDFsi
#
#   This program tests the PerlCDF standard interface.  PerlCDF must be installed.
#   See README.
#
#  Usage:
#     perl testPerlCDFsi.pl
#
# Written By:
#
#    Version 1.0
#    Modified by: Michael Liu  - 08 August, 1998
#    Version 2.0
#    Modified by: Michael Liu  - 07 January, 2005
#
#############################################################################
#
#  Translated from qst2c.c v. 1.10 by J. Love
#
#  Differences include:
#     maxiumum records written is a variable
#     Arrays are all 1 Dimensional.  2 dimensionality handled within CDF.
#     Added section confirming rVARs recCount, etc.
#     Added a few informational messages
#     Tests Epoch routines
# Note: As Perl uses double, a variable of data type CDF_REAL4/FLOAT will 
#	cause some minor value deviation due to different floating-point type 
#	representations. If the value is returned from the CDF's C routine 
#	and compared with Perl value, an epsilon is used for checkng their
#	equality. 10**(-9) is used for double and 10**(-5) for 
#	float/real. No problem for CDF_REAL8 data type.
#     
#############################################################################

use strict;

BEGIN { unshift @INC,'/home/cdf/PerlCDF31/blib/arch',
                     '/home/cdf/PerlCDF31/blib/lib'; }
use CDF;

my $N_DIMS = 2;
my $DIM_0_SIZE = 2;
my $DIM_1_SIZE = 3;

my $encoding = NETWORK_ENCODING;
my $actual_encoding = NETWORK_ENCODING;
my $majority = ROW_MAJOR;
my $numDims = $N_DIMS;
my @dimSizes = ( $DIM_0_SIZE, $DIM_1_SIZE );
my $varDataType = CDF_INT2;
my $varNumElements = 1 ;
my @varValues = (1,2,3,4,5,6);
my $recNum = 0 ;
my $recStart = 0 ;
my $recCount = 1 ;
my $recInterval = 1 ;
my @counts = ( $DIM_0_SIZE, $DIM_1_SIZE );
my @intervals = ( 1, 1 );
my $entryNum = 2 ;
my $attrScope = GLOBAL_SCOPE ;
my $entryDataType = CDF_INT2 ;
my $entryDataTypeNew = CDF_UINT2 ;
my $entryNumElems = 1 ;
my $entry1NumElems = 6 ;
my $entryValue = 1 ;
my @entry1Values = (1,2,3,4,5,6);
my $varRecVariance = VARY ;
my @varDimVariances = ( VARY, VARY );
my $varName = "VAR1a";
my $new_varName = "VAR1b";
my $attrName = "ATTR1";
my $attr2Name = "ATTR2";
my $new_attrName = "ATTR1a";
my $rEntryValue =  4 ;

#############################################################################
# Display title.
#############################################################################

print "\nTesting Perl-CDF Standard/C interface\n\n";

#############################################################################
# Create CDF.
#############################################################################
# Uncomment the following line if you want to create a V2.7 CDF file.
# CDF::CDFsetFileBackward(BACKWARDFILEon);

my $id; my $status;
$status = CDF::CDFcreate("YEST",$numDims,\@dimSizes,$encoding,$majority,\$id);
if ($status < CDF_OK) {
  if ($status == CDF_EXISTS) {
    $status = CDF::CDFopen("YEST", \$id);
    QuitCDF("1.0", $status) if ($status < CDF_OK) ;

    $status = CDF::CDFdelete($id);
    QuitCDF ("1.1", $status) if ($status < CDF_OK) ;
    $status = CDF::CDFcreate("YEST",$numDims,\@dimSizes,$encoding,$majority,
                                \$id);
    QuitCDF ("1.2", $status) if ($status < CDF_OK) ;
  }
  else {
    QuitCDF ("1.3", $status);
    }
}

#############################################################################
# Create variables and set/confirm cache sizes, etc.
#############################################################################

my $varNum_out;
$status = CDF::CDFvarCreate($id,$varName,$varDataType,$varNumElements,
			      $varRecVariance,\@varDimVariances,
			      \$varNum_out); 
QuitCDF ("2.0a", $status) if ($status < CDF_OK) ;

#############################################################################
# PUT to variables.
#############################################################################
my $var_num = CDF::CDFvarNum($id,$varName);
my ($x0, $x1, $i, @indices, $temp1);
for ($x0 = 0; $x0 < $DIM_0_SIZE; $x0++) {
   for ($x1 = 0; $x1 < $DIM_1_SIZE; $x1++) {
      $indices[0] = $x0;
      $indices[1] = $x1;
      $temp1 = $varValues[$x0*$DIM_1_SIZE + $x1];
      $status = CDF::CDFvarPut($id,$var_num,$recNum,\@indices,\$temp1);
      QuitCDF ("8.1", $status) if ($status < CDF_OK) ;

   }
}

#############################################################################
# GET from the variables.
#############################################################################

my $varValue_out;
for ($x0 = 0; $x0 < $DIM_0_SIZE; $x0++) {
   for ($x1 = 0; $x1 < $DIM_1_SIZE; $x1++) {
      $indices[0] = $x0;
      $indices[1] = $x1;
      $status = CDF::CDFvarGet($id,$var_num,$recNum,\@indices,
			         \$varValue_out);
      QuitCDF ("9.0", $status) if ($status < CDF_OK) ;

      QuitCDF ("9.1", $status) if ($varValue_out != $varValues[$x0*$DIM_1_SIZE + $x1]) ;
   }
 }

#############################################################################
# HyperPUT to the variables.
#############################################################################

for ($x0 = 0; $x0 < $DIM_0_SIZE; $x0++) {
   for ($x1 = 0; $x1 < $DIM_1_SIZE; $x1++) {
      $varValues[$x0*$DIM_1_SIZE + $x1] = -$varValues[$x0*$DIM_1_SIZE + $x1];
   }
 }

$indices[0] = 0;
$indices[1] = 0;
$status = CDF::CDFvHpPut($id,$var_num,$recStart,$recCount,$recInterval,
				\@indices,\@counts,\@intervals,\@varValues);
QuitCDF ("10.0", $status) if ($status < CDF_OK) ;

############################################################################g
# HyperGET from variables.
#############################################################################

my @varBuffer_out;
$status = CDF::CDFvHpGet($id,$var_num,$recStart,$recCount,$recInterval,
                                \@indices,\@counts,\@intervals,\@varBuffer_out);
QuitCDF ("11.0", $status) if ($status < CDF_OK) ;

for ($x0 = 0; $x0 < $DIM_0_SIZE; $x0++) {
   for ($x1 = 0; $x1 < $DIM_1_SIZE; $x1++) {
   	QuitCDF ("11.1", $status) if ($varBuffer_out[$x0*$DIM_1_SIZE + $x1] != $varValues[$x0*$DIM_1_SIZE + $x1]);
   }
  }
  
#############################################################################
# Create attributes.
#############################################################################

my $attrNum_out;
$status = CDF::CDFattrCreate($id,$attrName, $attrScope, \$attrNum_out);
QuitCDF ("12.0", $status) if ($status < CDF_OK) ;

$status = CDF::CDFattrCreate($id,$attr2Name, $attrScope, \$attrNum_out);
QuitCDF ("12.1", $status) if ($status < CDF_OK) ;

#############################################################################
# PUT to attributes.
#############################################################################
my $attr_num;
$attr_num = CDF::CDFattrNum($id,$attrName);
$status = CDF::CDFattrPut($id,$attr_num,$entryNum,$entryDataType,
                            $entryNumElems,\$entryValue);
QuitCDF ("13.0", $status) if ($status < CDF_OK) ;

$attr_num = CDF::CDFattrNum($id,$attr2Name);
$status = CDF::CDFattrPut($id,$attr_num,$entryNum,$entryDataType,
                            $entry1NumElems,\@entry1Values);
QuitCDF ("13.5", $status) if ($status < CDF_OK) ;

#############################################################################
# GET from attributes.
#############################################################################

my ($entryValue_out, @entry_out_Values);
$attr_num = CDF::CDFattrNum($id,$attrName);
$status = CDF::CDFattrGet($id,$attr_num,$entryNum,\$entryValue_out);

QuitCDF ("14.0", $status) if ($status < CDF_OK) ;

QuitCDF ("14.1", $status) if ($entryValue != $entryValue_out) ;
$attr_num = CDF::CDFattrNum($id,$attr2Name);
$status = CDF::CDFattrGet($id,$attr_num,$entryNum,\@entry_out_Values);
for ($i=0; $i<$entry1NumElems; $i++) {
  QuitCDF ("13.5a", $status) if ($entry1Values[$i] != $entry_out_Values[$i]);
}

#############################################################################
# Get CDF documentation.
#############################################################################

my ($version_out, $release_out, $CopyRightText);
$status = CDF::CDFdoc ($id,\$version_out,\$release_out,\$CopyRightText);
QuitCDF ("15.0", $status) if ($status < CDF_OK) ;
print "CDF Library Version $version_out, Release $release_out
Copyright: $CopyRightText\n";

#############################################################################
# Inquire CDF.
#############################################################################

my ($formatOut, $numDims_out, @dimSizes_out, $encoding_out, $majority_out);
my ($maxRec_out, $numRvars, $numZvars, $numAttrs_out);
$status = CDF::CDFinquire($id,\$numDims_out,\@dimSizes_out,\$encoding_out,
			    \$majority_out,\$maxRec_out,\$numRvars,
                            \$numAttrs_out);
QuitCDF ("16.0", $status) if ($status < CDF_OK) ;

QuitCDF ("16.1", $status) if ($numDims_out != $numDims) ;

my $x;
for ($x = 0; $x < $N_DIMS; $x++) {
   QuitCDF ("16.2", $status) if ($dimSizes_out[$x] != $dimSizes[$x]) ;
   }
QuitCDF ("16.3", $status) if ($encoding_out != $actual_encoding) ;
QuitCDF ("16.4", $status) if ($majority_out != $majority) ;
QuitCDF ("16.5", $status) if ($maxRec_out != 0) ;
QuitCDF ("16.6", $status) if ($numRvars != 1) ;
QuitCDF ("16.7", $status) if ($numAttrs_out != 2) ;

#############################################################################
# Rename variables.
#############################################################################

$status = CDF::CDFvarRename($id,$var_num,$new_varName);
QuitCDF ("17.0a", $status) if ($status < CDF_OK) ;

#
#############################################################################
# Inquire variables.
#############################################################################
$var_num = CDF::CDFvarNum($id,$new_varName);
my ($varName_out, $varDataType_out, $varNumElements_out, $varRecVariance_out);
my (@varDimVariances_out);
$status = CDF::CDFvarInquire($id,$var_num,\$varName_out,
			       \$varDataType_out,\$varNumElements_out,
			       \$varRecVariance_out,\@varDimVariances_out);
QuitCDF ("18.0", $status) if ($status < CDF_OK) ;

QuitCDF ("18.1", $status) if ($varName_out ne $new_varName) ;
QuitCDF ("18.2", $status) if ($varDataType_out != $varDataType) ;
B
QuitCDF ("18.3", $status) if ($varNumElements_out != $varNumElements) ;
QuitCDF ("18.4", $status) if ($varRecVariance_out != $varRecVariance) ;

my $dim_n;
for ($dim_n = 0; $dim_n < $numDims; $dim_n++) {
   if ($varDimVariances_out[$dim_n] != $varDimVariances[$dim_n]) {
     QuitCDF ("18.5", $status);
   }
}

#############################################################################
# Close variable
#############################################################################
$status = CDF::CDFvarClose ($id, $var_num);
QuitCDF ("19.0", $status) if ($status < CDF_OK);

#############################################################################
# Rename attribute.
#############################################################################
$attr_num = CDF::CDFattrNum($id,$attrName);
$status = CDF::CDFattrRename($id,$attr_num,$new_attrName);
QuitCDF ("20.0", $status) if ($status < CDF_OK) ;

#############################################################################
# Inquire attribute.
#############################################################################

my ($attrName_out, $attrScope_out, $maxEntry_out);
$attr_num = CDF::CDFattrNum($id,$new_attrName);
$status = CDF::CDFattrInquire($id,$attr_num,\$attrName_out,\$attrScope_out,
			        \$maxEntry_out);
QuitCDF ("22.0", $status) if ($status < CDF_OK) ;

QuitCDF ("22.1", $status) if ($attrName_out ne $new_attrName) ;
QuitCDF ("22.2", $status) if ($attrScope_out != $attrScope) ;
QuitCDF ("22.3", $status) if ($maxEntry_out != $entryNum) ;
QuitCDF ("22.4", $status) if ($attrNum_out != 1) ;

#############################################################################
# Inquire attribute entries.
#############################################################################

my ($entryDataType_out, $entryNumElems_out);
$status = CDF::CDFattrEntryInquire($id,$attr_num,$entryNum,
			             \$entryDataType_out,\$entryNumElems_out);
QuitCDF ("23.0", $status) if ($status < CDF_OK) ;

QuitCDF ("23.1", $status) if ($entryDataType_out != $entryDataType) ;
QuitCDF ("23.2", $status) if ($entryNumElems_out != $entryNumElems) ;

#############################################################################
# Get error text.
#############################################################################

my $errorText;
$status = CDF::CDFerror(CDF_OK,\$errorText);
		 
QuitCDF ("24.0", $status) if ($status < CDF_OK) ;

#############################################################################
# Close CDF.
#############################################################################

$status = CDF::CDFclose($id);
QuitCDF ("25.0", $status) if ($status < CDF_OK) ;

#############################################################################
# Test Epoch Routines
#############################################################################
print "Test Epoch...\n";

my $year = 1994;
my $month = 10;
my $day = 13;
my $hour = 1;
my $minute = 2;
my $second = 3;
my $msec = 987;
my $epoch = CDF::computeEPOCH ($year, $month, $day, $hour, $minute, $second, $msec);

my $epStringTrue = '13-Oct-1994 01:02:03.987';
my $epString1True = '19941013.0431017';
my $epString2True = '19941013010203';
my $epString3True = '1994-10-13T01:02:03.987Z';

my ($epString, $epString1, $epString2, $epString3);
CDF::encodeEPOCH ($epoch, $epString);
QuitEPOCH ("30.0") if ($epString ne $epStringTrue);

CDF::encodeEPOCH1 ($epoch, $epString1);
QuitEPOCH ("30.1") if ($epString1 ne $epString1True);

CDF::encodeEPOCH2 ($epoch, $epString2);
QuitEPOCH ("30.2") if ($epString2 ne $epString2True);

CDF::encodeEPOCH3 ($epoch, $epString3);
QuitEPOCH ("30.3") if ($epString3 ne $epString3True);

my $epochOut = CDF::parseEPOCH ($epString);
QuitEPOCH ("31.0") if ($epochOut != $epoch) ;

my ($yearOut, $monthOut, $dayOut, $hourOut, $minuteOut, $secondOut, $msecOut);
CDF::EPOCHbreakdown ($epoch, $yearOut, $monthOut, $dayOut, $hourOut, $minuteOut,
		$secondOut, $msecOut);
QuitEPOCH ("32.1") if ($yearOut != $year) ;
QuitEPOCH ("32.2") if ($monthOut != $month) ;
QuitEPOCH ("32.3") if ($dayOut != $day) ;
QuitEPOCH ("32.4") if ($hourOut != $hour) ;
QuitEPOCH ("32.5") if ($minuteOut != $minute) ;
QuitEPOCH ("32.6") if ($secondOut != $second) ;
QuitEPOCH ("32.7") if ($msecOut != $msec) ;

#############################################################################
# Test Epoch16 Routines
#############################################################################
print "Test Epoch16...\n";

my $usec = 876;
my $nsec = 765;
my $psec = 654;
my @epoch16;
my $retco = CDF::computeEPOCH16 ($year, $month, $day, $hour, $minute, $second,
                                 $msec, $usec, $nsec, $psec, \@epoch16);

my $ep16StringTrue = '13-Oct-1994 01:02:03.987.876.765.654';
my $ep16String1True = '19941013.043101711536640';
my $ep16String2True = '19941013010203';
my $ep16String3True = '1994-10-13T01:02:03.987.876.765.654Z';

my ($ep16String, $ep16String1, $ep16String2, $ep16String3);

CDF::encodeEPOCH16 (\@epoch16, $ep16String);
QuitEPOCH ("40.0") if ($ep16String ne $ep16StringTrue);

CDF::encodeEPOCH16_1 (\@epoch16, $ep16String1);
#print "encodeEPOCH16_1:$ep16String1";
QuitEPOCH ("40.1") if ($ep16String1 ne $ep16String1True);

CDF::encodeEPOCH16_2 (\@epoch16, $ep16String2);
QuitEPOCH ("40.2") if ($ep16String2 ne $ep16String2True);

CDF::encodeEPOCH16_3 (\@epoch16, $ep16String3);
QuitEPOCH ("40.3") if ($ep16String3 ne $ep16String3True);

my @epoch16Out;
$retco = CDF::parseEPOCH16 ($ep16String, \@epoch16Out);
QuitEPOCH ("41.0") if ($epoch16Out[0] != $epoch16[0]) ;
QuitEPOCH ("41.1") if ($epoch16Out[1] != $epoch16[1]) ;

my ($year16Out, $month16Out, $day16Out, $hour16Out, $minute16Out,
    $second16Out, $msec16Out, $usec16Out, $nsec16Out, $psec16Out);

CDF::EPOCH16breakdown (\@epoch16, $year16Out, $month16Out, $day16Out,
                       $hour16Out, $minute16Out, $second16Out, $msec16Out,
                       $usec16Out, $nsec16Out, $psec16Out);

QuitEPOCH ("42.0") if ($year16Out != $year) ;
QuitEPOCH ("42.1") if ($month16Out != $month) ;
QuitEPOCH ("42.2") if ($day16Out != $day) ;
QuitEPOCH ("42.3") if ($hour16Out != $hour) ;
QuitEPOCH ("42.4") if ($minute16Out != $minute) ;
QuitEPOCH ("42.5") if ($second16Out != $second) ;
QuitEPOCH ("42.6") if ($msec16Out != $msec) ;
QuitEPOCH ("42.7") if ($usec16Out != $usec) ;
QuitEPOCH ("42.8") if ($nsec16Out != $nsec) ;
QuitEPOCH ("42.9") if ($psec16Out != $psec) ;

#############################################################################
# Successful completion.
#############################################################################
print "All tests completed successfully\n";
exit;



#############################################################################
# QuitCDF.
#############################################################################
sub QuitCDF {
	my ($where, $status)=@_;

  print "Aborting at $where ...\n";
  if ($status < CDF_OK) {
    my $text;
    CDF::CDFlib (SELECT_, CDF_STATUS_, $status,
		   GET_, STATUS_TEXT_, \$text,
		   NULL_);
    print $text;
  }
  CDF::CDFlib (CLOSE_, CDF_,
	  NULL_);
  print "...test aborted.\n";
  exit;
	
}#endsub QuitCDF

#############################################################################
#  QuitEPOCH
#############################################################################
sub QuitEPOCH {
	my ($where)=@_;
  print "Aborting at $where...test aborted.\n";
  exit;

}#endsub QuitEPOCH
