/******************************************************************************
 *
 *  NSSDC/CDF				CDF Header file for C# interface.
 *
 *  Version 1.0, 22-Jan-10, GSFC/SPDF
 *
 *  Modification history:
 *
 *   V1.0  22-Jan-10, M Liu	Original version (for CDF C#).
 ******************************************************************************/

#include "cdflib.h"
#include "cdflib64.h"
#include "cdfrev.h"

CDFstatus flipDataElements (CDFid id, int zVar, long varNum, long recCount,
                            long* dimCounts, char* buffer, int direction);

/* Function: flipDataElements
 *
 * Purpose:  Shuffle data elements in records from row to column or column
 *           to row for a multi-dimensional variable in a column-major CDF.
 *
 * Parameters:
 *    buffer - buffer pointer for data from a full or partial record(s)
 *    direction - indicator, 1 for column-to-row, 2 for row-to-column
 *
 */
CDFstatus flipDataElements (CDFid id, int zVar, long varNum, long recCount,
                            long* dimCounts, char* buffer, int direction) 
{
  long majority, numDims, dimSizes[CDF_MAX_DIMS];
  int ix, toflip;
  CDFstatus status;
  long NvalueBytes, numElems, dataType, dimVarys[CDF_MAX_DIMS];
  long nValuesPerRecord = 1L, nBytesPerRecord;
  char *tmpbuf;
  if (recCount < 1) return CDF_OK;
  status = CDFlib (SELECT_, CDF_, id,
                            BOO(zVar,zVAR_,rVAR_), varNum,
                   GET_, CDF_MAJORITY_, &majority,
                         BOO(zVar,zVAR_NUMDIMS_,rVARs_NUMDIMS_), &numDims,
                         BOO(zVar,zVAR_DIMSIZES_,rVARs_DIMSIZES_), dimSizes,
                   NULL_);
  if (status != CDF_OK) return status;
  if (majority == ROW_MAJOR || numDims < 2) return CDF_OK;
  for (ix = 0; ix < (int) numDims; ix++) {
	  if (dimCounts[ix] > dimSizes[ix]) dimCounts[ix] = dimSizes[ix];
/*    if (dimSizes[ix] != dimCounts[ix]) return CDF_OK; */
  }
  status = CDFlib (GET_, BOO(zVar,zVAR_DATATYPE_,rVAR_DATATYPE_), &dataType,
                         BOO(zVar,zVAR_NUMELEMS_,rVAR_NUMELEMS_), &numElems,
                         BOO(zVar,zVAR_DIMVARYS_,rVAR_DIMVARYS_), dimVarys,
                   NULL_);
  if (status != CDF_OK) return status;

  for (ix = 0; ix < (int) numDims; ix++) {
    if (dimVarys[ix] && dimCounts[ix] > 0)
      nValuesPerRecord *= dimCounts[ix];
  }
  NvalueBytes = numElems * CDFelemSize(dataType);
  nBytesPerRecord = nValuesPerRecord * NvalueBytes;
  tmpbuf = (char *) malloc ((size_t) nBytesPerRecord * recCount);
  for (ix = 0; ix < (int) recCount; ix++) {
	 size_t offset = (size_t) (ix * nBytesPerRecord);
     if (direction == 1)
       COLtoROW (buffer + offset, tmpbuf + offset,
                 numDims, dimCounts, NvalueBytes);
     else
       ROWtoCOL (buffer + offset, tmpbuf + offset,
                 numDims, dimCounts, NvalueBytes);
  }
  memmove (buffer, tmpbuf, (size_t) recCount * nBytesPerRecord);
  free (tmpbuf);
  return CDF_OK;
}                  /* flipDataElements */

/******************************************************************************
* CDF standard interface.
******************************************************************************/

/*
 * CLOSE_  *
 *         */

VISIBLE_PREFIX CDFstatus CDFclose(id)
CDFid id;
{
return
CDFlib (SELECT_, CDF_, id,
        CLOSE_, CDF_,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFcloseVar(id,zVar,varNum)
CDFid id;
Logical zVar;
long varNum;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        CLOSE_, (zVar? zVAR_: rVAR_),
        NULL_);
}

/*
 * CONFIRM_  *
 *           */

VISIBLE_PREFIX CDFstatus CDFconfirmAttrExistence(id,attrName)
CDFid id;
char *attrName;
{
return
CDFlib (SELECT_, CDF_, id,
        CONFIRM_, ATTR_EXISTENCE_, attrName,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFconfirmgEntryExistence(id,attrNum,entryNum)
CDFid id;
long attrNum;
long entryNum;
{
return
CDFlib (SELECT_, CDF_, id,
                 ATTR_, attrNum,
        CONFIRM_, gENTRY_EXISTENCE_, entryNum,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFconfirmrEntryExistence(id,attrNum,entryNum)
CDFid id;
long attrNum;
long entryNum;
{
return
CDFlib (SELECT_, CDF_, id,
                 ATTR_, attrNum,
        CONFIRM_, rENTRY_EXISTENCE_, entryNum,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFconfirmVarExistence(id,zVar,varName)
CDFid id;
Logical zVar;
char *varName;
{
return
CDFlib (SELECT_, CDF_, id,
        CONFIRM_, (zVar?zVAR_EXISTENCE_:rVAR_EXISTENCE_), varName,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFconfirmVarPadValueExistence(id,zVar,varNum)
CDFid id;
Logical zVar;
long varNum;
{
return
CDFlib (SELECT_, CDF_, id,
		 (zVar?zVAR_:rVAR_), varNum,
        CONFIRM_, (zVar?zVAR_PADVALUE_:rVAR_PADVALUE_),
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFconfirmzEntryExistence(id,attrNum,entryNum)
CDFid id;
long attrNum;
long entryNum;
{
return
CDFlib (SELECT_, CDF_, id,
                 ATTR_, attrNum,
        CONFIRM_, zENTRY_EXISTENCE_, entryNum,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFconfirmChecksum(id)
CDFid id;
{
return
CDFlib (SELECT_, CDF_, id,
        CONFIRM_, CDF_CHECKSUM_,
        NULL_);
}

/*
 * CREATE_ *
 *         */

VISIBLE_PREFIX CDFstatus CDFcreate(CDFname,numDims,dimSizes,encoding,majority,id)
char *CDFname;
long numDims;
long *dimSizes;
long encoding;
long majority;
CDFid *id;
{
return
CDFlib (CREATE_, CDF_, CDFname, numDims, dimSizes, id,
        PUT_, CDF_ENCODING_, encoding,
              CDF_MAJORITY_, majority,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFcreateAttr(id,attrName,attrScope,attrNum)
CDFid id;
char *attrName;
long attrScope;
long *attrNum;
{
return
CDFlib (SELECT_, CDF_, id,
        CREATE_, ATTR_, attrName, attrScope, attrNum,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFcreaterVar(id,varName,dataType,numElements,recVary,dimVarys,varNum)
CDFid id;
char *varName;
long dataType;
long numElements;
long recVary;
long *dimVarys;
long *varNum;
{
return
CDFlib (SELECT_, CDF_, id,
        CREATE_, rVAR_, varName, dataType, numElements,
                        recVary, dimVarys, varNum,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFcreatezVar(id,varName,dataType,numElements,numDims,dimSizes,recVary,dimVarys,varNum)
CDFid id;
char *varName;
long dataType;
long numElements;
long numDims;
long *dimSizes;
long recVary;
long *dimVarys;
long *varNum;
{
return
CDFlib (SELECT_, CDF_, id,
        CREATE_, zVAR_, varName, dataType, numElements,
                        numDims,dimSizes, recVary, dimVarys, varNum,
        NULL_);
}

/*
 * DELETE_ *
 *         */

VISIBLE_PREFIX CDFstatus CDFdelete(id)
CDFid id;
{
return
CDFlib (SELECT_, CDF_, id,
        DELETE_, CDF_,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFdeleteAttr(id,attrNum)
CDFid id;
long attrNum;
{
return
CDFlib (SELECT_, CDF_, id,
                 ATTR_, attrNum,
        DELETE_, ATTR_,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFdeleteVar(id,zVar,varNum)
CDFid id;
Logical zVar;
long varNum;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar? zVAR_: rVAR_), varNum,
        DELETE_, (zVar? zVAR_: rVAR_),
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFdeleteVarRecords(id,zVar,varNum,firstRec,lastRec)
CDFid id;
Logical zVar;
long varNum;
long firstRec;
long lastRec;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar? zVAR_: rVAR_), varNum,
        DELETE_, (zVar? zVAR_RECORDS_: rVAR_RECORDS_), firstRec, lastRec,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFdeleteVarRecordsRenumber(id,zVar,varNum,firstRec,lastRec)
CDFid id;
Logical zVar;
long varNum;
long firstRec;
long lastRec;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar? zVAR_: rVAR_), varNum,
        DELETE_, (zVar? zVAR_RECORDS_RENUMBER_: rVAR_RECORDS_RENUMBER_),
               	 firstRec, lastRec,
        NULL_);
}

/*
 * GET_ *
 *      */

VISIBLE_PREFIX CDFstatus CDFgetCacheSize(id,numBuffers)
CDFid id;
long *numBuffers;
{
return
CDFlib (SELECT_, CDF_, id,
        CONFIRM_, CDF_CACHESIZE_, numBuffers,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarCacheSize(id,zVar,varNum,numBuffers)
CDFid id;
Logical zVar;
long varNum;
long *numBuffers;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        CONFIRM_, (zVar?zVAR_CACHESIZE_:rVAR_CACHESIZE_), numBuffers,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetDecoding(id,decoding)
CDFid id;
long *decoding;
{
return
CDFlib (SELECT_, CDF_, id,
        CONFIRM_, CDF_DECODING_, decoding,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetAttrName(id,attrNum,attrName)
CDFid id;
long attrNum;
char *attrName;
{
return
CDFlib (SELECT_, CDF_, id,
                 ATTR_, attrNum,
        GET_, ATTR_NAME_, attrName,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetName(id,cdfName)
CDFid id;
char *cdfName;
{
return
CDFlib (SELECT_, CDF_, id,
        CONFIRM_, CDF_NAME_, cdfName,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetNegtoPosfp0Mode(id,negtoPosfp0)
CDFid id;
long *negtoPosfp0;
{
return
CDFlib (SELECT_, CDF_, id,
        CONFIRM_, CDF_NEGtoPOSfp0_MODE_, negtoPosfp0,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetReadOnlyMode(id,readOnlyMode)
CDFid id;
long *readOnlyMode;
{
return
CDFlib (SELECT_, CDF_, id,
        CONFIRM_, CDF_READONLY_MODE_, readOnlyMode,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetzMode(id,zMode)
CDFid id;
long *zMode;
{
return
CDFlib (SELECT_, CDF_, id,
        CONFIRM_, CDF_zMODE_, zMode,
        NULL_);
}


VISIBLE_PREFIX CDFstatus CDFgetCompressionCacheSize(id,numBuffers)
CDFid id;
long *numBuffers;
{
return
CDFlib (SELECT_, CDF_, id,
        CONFIRM_, COMPRESS_CACHESIZE_, numBuffers,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetStageCacheSize(id,numBuffers)
CDFid id;
long *numBuffers;
{
return
CDFlib (SELECT_, CDF_, id,
        CONFIRM_, STAGE_CACHESIZE_, numBuffers,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarReservePercent(id,zVar,varNum,percent)
CDFid id;
Logical zVar;
long varNum;
long *percent;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        CONFIRM_, (zVar?zVAR_RESERVEPERCENT_:rVAR_RESERVEPERCENT_), percent,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarSeqPos(id,zVar,varNum,recNum,indices)
CDFid id;
Logical zVar;
long varNum;
long *recNum;
long *indices;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        CONFIRM_, (zVar?zVAR_SEQPOS_:rVAR_SEQPOS_), recNum, indices,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetAttrScope(id,attrNum,attrScope)
CDFid id;
long attrNum;
long *attrScope;
{
return
CDFlib (SELECT_, CDF_, id,
                 ATTR_, attrNum,
        GET_, ATTR_SCOPE_, attrScope,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetCompression(id,cType, cParms, cPercent)
CDFid id;
long *cType;
long *cParms;
long *cPercent;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, CDF_COMPRESSION_, cType, cParms, cPercent,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetCopyright(id,copyright)
CDFid id;
char *copyright;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, CDF_COPYRIGHT_, copyright,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetEncoding(id,encoding)
CDFid id;
long *encoding;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, CDF_ENCODING_, encoding,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetFormat(id,format)
CDFid id;
long *format;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, CDF_FORMAT_, format,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetCompressionInfo(name,cType,cParms,cSize,uSize)
char *name;
long *cType;
long *cParms;
OFF_T *cSize;
OFF_T *uSize;
{
return
CDFlib (GET_, CDF_INFO_, name, cType, cParms, cSize, uSize,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetMajority(id,majority)
CDFid id;
long *majority;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, CDF_MAJORITY_, majority,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetNumAttributes(id,numAttrs)
CDFid id;
long *numAttrs;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, CDF_NUMATTRS_, numAttrs,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetNumgAttributes(id,numgAttrs)
CDFid id;
long *numgAttrs;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, CDF_NUMgATTRS_, numgAttrs,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetNumVars(id,zVar,numVars)
CDFid id;
Logical zVar;
long *numVars;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, (zVar?CDF_NUMzVARS_:CDF_NUMrVARS_), numVars,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetNumvAttributes(id,numvAttrs)
CDFid id;
long *numvAttrs;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, CDF_NUMvATTRS_, numvAttrs,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFdoc(id,version,release,copyright)
CDFid id;
long *version;
long *release;
char *copyright;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, CDF_VERSION_, version,
              CDF_RELEASE_, release,
              CDF_COPYRIGHT_, copyright,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetDataTypeSize(dataType,numBytes)
long dataType;
long *numBytes;
{
return
CDFlib (GET_, DATATYPE_SIZE_, dataType, numBytes,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetLibraryCopyright(copyright)
char *copyright;
{
return
CDFlib (GET_, LIB_COPYRIGHT_, copyright,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetLibraryVersion(version,release,increment,subincrement)
long *version;
long *release;
long *increment;
char *subincrement;
{
return
CDFlib (GET_, LIB_VERSION_, version,
              LIB_RELEASE_, release,
              LIB_INCREMENT_, increment,
              LIB_subINCREMENT_, subincrement,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVersion(id,version,release,increment)
CDFid id;
long *version;
long *release;
long *increment;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, CDF_VERSION_, version,
              CDF_RELEASE_, release,
              CDF_INCREMENT_, increment,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarBlockingFactor(id,zVar,varNum,bf)
CDFid id;
Logical zVar;
long varNum;
long *bf;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_BLOCKINGFACTOR_:rVAR_BLOCKINGFACTOR_), bf,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarCompression(id,zVar,varNum,cType,cParms,cPct)
CDFid id;
Logical zVar;
long varNum;
long *cType;
long *cParms;
long *cPct;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_COMPRESSION_:rVAR_COMPRESSION_), cType, cParms, cPct,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetzVarData(id,varNum,recNum,indices,value)
CDFid id;
long varNum;
long recNum;
long *indices;
void *value;
{
return
CDFgetVarData (id, 1, varNum, recNum, indices, value);
}

VISIBLE_PREFIX CDFstatus CDFgetVarData(id,zVar,varNum,recNum,indices,value)
CDFid id;
Logical zVar;
long varNum;
long recNum;
long *indices;
void *value;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
                 (zVar?zVAR_RECNUMBER_:rVARs_RECNUMBER_), recNum,
                 (zVar?zVAR_DIMINDICES_:rVARs_DIMINDICES_), indices,
        GET_, (zVar?zVAR_DATA_:rVAR_DATA_), value,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarDataType(id,zVar,varNum,dataType) 
CDFid id;
Logical zVar;
long varNum;
long *dataType;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_DATATYPE_:rVAR_DATATYPE_), dataType,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarDimVariances(id,zVar,varNum,dimVarys)
CDFid id;
Logical zVar;
long varNum;
long *dimVarys;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_DIMVARYS_:rVAR_DIMVARYS_), dimVarys,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarMaxAllocRecNum(id,zVar,varNum,maxAllocRec)
CDFid id;
Logical zVar;
long varNum;
long *maxAllocRec;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_MAXallocREC_:rVAR_MAXallocREC_), maxAllocRec,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarMaxWrittenRecNum(id,zVar,varNum,maxRec)
CDFid id;
Logical zVar;
long varNum;
long *maxRec;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_MAXREC_:rVAR_MAXREC_), maxRec,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarAllocRecords(id,zVar,varNum,numAllocRecs)
CDFid id;
Logical zVar;
long varNum;
long *numAllocRecs;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_NUMallocRECS_:rVAR_NUMallocRECS_), numAllocRecs,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarNumElements(id,zVar,varNum,numElements)
CDFid id;
Logical zVar;
long varNum;
long *numElements;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_NUMELEMS_:rVAR_NUMELEMS_), numElements,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarNumRecsWritten(id,zVar,varNum,numRecs)
CDFid id;
Logical zVar;
long varNum;
long *numRecs;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_NUMRECS_:rVAR_NUMRECS_), numRecs,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarPadValue(id,zVar,varNum,padValue)
CDFid id;
Logical zVar;
long varNum;
void *padValue;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_PADVALUE_:rVAR_PADVALUE_), padValue,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarRecVariance(id,zVar,varNum,recVary)
CDFid id;
Logical zVar;
long varNum;
long *recVary;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_RECVARY_:rVAR_RECVARY_), recVary,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarSeqData(id,zVar,varNum,seqData)
CDFid id;
Logical zVar;
long varNum;
void *seqData;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_SEQDATA_:rVAR_SEQDATA_), seqData,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarsRecordDatabyNumbers(id,zVar,numVars,varNums,recNum,buffer)
CDFid id;
Logical zVar;
long numVars;
long *varNums;
long recNum;
void *buffer;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVARs_RECNUMBER_:rVARs_RECNUMBER_), recNum,
        GET_, (zVar?zVARs_RECDATA_:rVARs_RECDATA_), numVars, varNums, buffer,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarSparseRecords(id,zVar,varNum,sparseRecs)
CDFid id;
Logical zVar;
long varNum;
long *sparseRecs;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_SPARSERECORDS_:rVAR_SPARSERECORDS_), sparseRecs,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetrVarsDimSizes(id,dimSizes)
CDFid id;
long *dimSizes;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, rVARs_DIMSIZES_, dimSizes,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarInfo(id,zVar,varNum,dataType,numElems,numDims,dimSizes)
CDFid id;
Logical zVar;
long varNum;
long *dataType;
long *numElems;
long *numDims;
long *dimSizes;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_DATATYPE_:rVAR_DATATYPE_), dataType,
              (zVar?zVAR_NUMELEMS_:rVAR_NUMELEMS_), numElems,
              (zVar?zVAR_DIMSIZES_:rVARs_DIMSIZES_), dimSizes,
              (zVar?zVAR_NUMDIMS_:rVARs_NUMDIMS_), numDims,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetzVarDataType(id,varNum,dataType)
CDFid id;
long varNum;
long *dataType;
{
return
CDFlib (SELECT_, CDF_, id,
                 zVAR_, varNum,
        GET_, zVAR_DATATYPE_, dataType,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetzVarDimSizes(id,varNum,dimSizes)
CDFid id;
long varNum;
long *dimSizes;
{
return
CDFlib (SELECT_, CDF_, id,
                 zVAR_, varNum,
        GET_, zVAR_DIMSIZES_, dimSizes,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetzVarDimVariances(id,varNum,dimVarys)
CDFid id;
long varNum;
long *dimVarys;
{
return
CDFlib (SELECT_, CDF_, id,
                 zVAR_, varNum,
        GET_, zVAR_DIMVARYS_, dimVarys,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetzVarRecVariance(id,varNum,recVary)
CDFid id;
long varNum;
long *recVary;
{
return
CDFlib (SELECT_, CDF_, id,
                 zVAR_, varNum,
        GET_, zVAR_RECVARY_, recVary,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarName(id,zVar,varNum,varName)
CDFid id;
Logical zVar;
long varNum;
char *varName;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        GET_, (zVar?zVAR_NAME_:rVAR_NAME_), varName,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetzVarNumDims(id,varNum,numDims)
CDFid id;
long varNum;
long *numDims;
{
return
CDFlib (SELECT_, CDF_, id,
                 zVAR_, varNum,
        GET_, zVAR_NUMDIMS_, numDims,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetzVarNumElements(id,varNum,numElems)
CDFid id;
long varNum;
long *numElems;
{
return
CDFlib (SELECT_, CDF_, id,
                 zVAR_, varNum,
        GET_, zVAR_NUMELEMS_, numElems,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetzVarPadValue(id,varNum,value)
CDFid id;
long varNum;
void *value;
{
return
CDFlib (SELECT_, CDF_, id,
                 zVAR_, varNum,
        GET_, zVAR_PADVALUE_, value,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetrVarsNumDims(id,numDims)
CDFid id;
long *numDims;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, rVARs_NUMDIMS_, numDims,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetStatusText(status,text)
CDFstatus status;
char *text;
{
return
CDFlib (SELECT_, CDF_STATUS_, status,
        GET_, STATUS_TEXT_, text,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFhyperGetVarData(id,zVar,varN,recS,recC,recI,indices,counts,intervals,buff)
CDFid id;
Logical zVar;
long varN;
long recS;
long recC;
long recI;
long *indices;
long *counts;
long *intervals;
void *buff;
{
  CDFstatus status;
  status = CDFlib (SELECT_, CDF_, id,
                            (zVar?zVAR_:rVAR_), varN,
                            (zVar?zVAR_RECNUMBER_:rVARs_RECNUMBER_), recS,
                            (zVar?zVAR_RECCOUNT_:rVARs_RECCOUNT_), recC,
                            (zVar?zVAR_RECINTERVAL_:rVARs_RECINTERVAL_), recI,
                            (zVar?zVAR_DIMINDICES_:rVARs_DIMINDICES_), indices,
                            (zVar?zVAR_DIMCOUNTS_:rVARs_DIMCOUNTS_), counts,
                            (zVar?zVAR_DIMINTERVALS_:rVARs_DIMINTERVALS_), intervals,
                   GET_, (zVar?zVAR_HYPERDATA_:rVAR_HYPERDATA_), buff,
				   NULL_);
  if (status < CDF_OK) return status;
  return flipDataElements (id, zVar, varN, recC, counts, (char*) buff, 1);

}

VISIBLE_PREFIX CDFstatus CDFgetMaxWrittenRecNums(id,maxRecrVars,maxReczVars)
CDFid id;
long *maxRecrVars;
long *maxReczVars;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, rVARs_MAXREC_, maxRecrVars,
              zVARs_MAXREC_, maxReczVars,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetVarsMaxWrittenRecNum(id,zVar,maxRecVar)
CDFid id;
Logical zVar;
long *maxRecVar;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, (zVar?zVARs_MAXREC_:rVARs_MAXREC_), maxRecVar,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFinquireVar(id,zVar,varN,varName,dataType,numElements,numDims,dimSizes,recVary,dimVarys)
CDFid id;
Logical zVar;
long varN;
char *varName;
long *dataType;
long *numElements;
long *numDims;
long *dimSizes;
long *recVary;
long *dimVarys;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varN,
        GET_, (zVar?zVAR_NAME_:rVAR_NAME_), varName,
              (zVar?zVAR_DATATYPE_:rVAR_DATATYPE_), dataType,
              (zVar?zVAR_NUMELEMS_:rVAR_NUMELEMS_), numElements,
              (zVar?zVAR_NUMDIMS_:rVARs_NUMDIMS_), numDims,
              (zVar?zVAR_DIMSIZES_:rVARs_DIMSIZES_), dimSizes,
              (zVar?zVAR_RECVARY_:rVAR_RECVARY_), recVary,
              (zVar?zVAR_DIMVARYS_:rVAR_DIMVARYS_), dimVarys,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFvarInquire(id,varN,varName,dataType,numElements,recVary,dimVarys)
CDFid id;
long varN;
char *varName;
long *dataType;
long *numElements;
long *recVary;
long *dimVarys;
{
return
CDFlib (SELECT_, CDF_, id,
                 rVAR_, varN,
        GET_, rVAR_NAME_, varName,
              rVAR_DATATYPE_, dataType,
              rVAR_NUMELEMS_, numElements,
              rVAR_RECVARY_, recVary,
              rVAR_DIMVARYS_, dimVarys,
        NULL_);
}
VISIBLE_PREFIX CDFstatus CDFinquire(id,numDims,dimSizes,encoding,majority,maxRec,nVars,nAttrs)
CDFid id;
long *numDims;
long *dimSizes;
long *encoding;
long *majority;
long *maxRec;
long *nVars;
long *nAttrs;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, rVARs_NUMDIMS_, numDims,
              rVARs_DIMSIZES_, dimSizes,
              CDF_ENCODING_, encoding,
              CDF_MAJORITY_, majority,
              rVARs_MAXREC_, maxRec,
              CDF_NUMrVARS_, nVars,
              CDF_NUMATTRS_, nAttrs,
        NULL_);
}
VISIBLE_PREFIX CDFstatus CDFinquireCDF(id,numDims,dimSizes,encoding,majority,maxrRec,nrVars,maxzRec,nzVars,nAttrs)
CDFid id;
long *numDims;
long *dimSizes;
long *encoding;
long *majority;
long *maxrRec;
long *nrVars;
long *maxzRec;
long *nzVars;
long *nAttrs;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, rVARs_NUMDIMS_, numDims,
              rVARs_DIMSIZES_, dimSizes,
              CDF_ENCODING_, encoding,
              CDF_MAJORITY_, majority,
              rVARs_MAXREC_, maxrRec,
              CDF_NUMrVARS_, nrVars,
              zVARs_MAXREC_, maxzRec,
              CDF_NUMzVARS_, nzVars,
              CDF_NUMATTRS_, nAttrs,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFgetChecksum(id,checksum)
CDFid id;
long *checksum;
{
return
CDFlib (SELECT_, CDF_, id,
        GET_, CDF_CHECKSUM_, checksum,
        NULL_);
}

/*
 * OPEN_ *
 *       */

VISIBLE_PREFIX CDFstatus CDFopen(CDFname,id)
char *CDFname;
CDFid *id;
{
return
CDFlib (OPEN_, CDF_, CDFname, id,
        NULL_);
}

/*
 * PUT_ *
 *      */

VISIBLE_PREFIX CDFstatus CDFsetAttrScope(id,attrNum,attrScope)
CDFid id;
long attrNum;
long attrScope;
{
return
CDFlib (SELECT_, CDF_, id,
                 ATTR_, attrNum,
        PUT_, ATTR_SCOPE_, attrScope,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetCompression(id,cType, cParms)
CDFid id;
long cType;
long *cParms;
{
return
CDFlib (SELECT_, CDF_, id,
        PUT_, CDF_COMPRESSION_, cType, cParms,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetEncoding(id,encoding)
CDFid id;
long encoding;
{
return
CDFlib (SELECT_, CDF_, id,
        PUT_, CDF_ENCODING_, encoding,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetFormat(id,format)
CDFid id;
long format;
{
return
CDFlib (SELECT_, CDF_, id,
        PUT_, CDF_FORMAT_, format,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetMajority(id,majority)
CDFid id;
long majority;
{
return
CDFlib (SELECT_, CDF_, id,
        PUT_, CDF_MAJORITY_, majority,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFrenameAttr(id,attrNum,attrName)
CDFid id;
long attrNum;
char *attrName;
{
return
CDFlib (SELECT_, CDF_, id,
		 ATTR_, attrNum,
	PUT_, ATTR_NAME_, attrName,
	NULL_);
}

VISIBLE_PREFIX CDFstatus CDFrenameVar(id,zVar,varNum,varName)
CDFid id;
Logical zVar;
long varNum;
char *varName;
{
return
CDFlib (SELECT_, CDF_, id,
		 (zVar?zVAR_:rVAR_), varNum,
	PUT_, (zVar?zVAR_NAME_:rVAR_NAME_), varName,
	NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarAllocRecords(id,zVar,varNum,allocRecs)
CDFid id;
Logical zVar;
long varNum;
long allocRecs;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        PUT_, (zVar?zVAR_ALLOCATERECS_:rVAR_ALLOCATERECS_), allocRecs,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarAllocBlockRecords(id,zVar,varNum,firstRec, lastRec)
CDFid id;
Logical zVar;
long varNum;
long firstRec;
long lastRec;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        PUT_, (zVar?zVAR_ALLOCATEBLOCK_:rVAR_ALLOCATEBLOCK_), firstRec,
                                                              lastRec,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarBlockingFactor(id,zVar,varNum,bf)
CDFid id;
Logical zVar;
long varNum;
long bf;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        PUT_, (zVar?zVAR_BLOCKINGFACTOR_:rVAR_BLOCKINGFACTOR_), bf,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarCompression(id,zVar,varNum,cType,cParms)
CDFid id;
Logical zVar;
long varNum;
long cType;
long *cParms;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        PUT_, (zVar?zVAR_COMPRESSION_:rVAR_COMPRESSION_), cType, cParms,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarDataSpec(id,zVar,varNum,dataType,numElems)
CDFid id;
Logical zVar;
long varNum;
long dataType;
long numElems;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        PUT_, (zVar?zVAR_DATASPEC_:rVAR_DATASPEC_), dataType, numElems,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarDimVariances(id,zVar,varNum,dimVarys)
CDFid id;
Logical zVar;
long varNum;
long *dimVarys;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        PUT_, (zVar?zVAR_DIMVARYS_:rVAR_DIMVARYS_), dimVarys,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarInitialRecs(id,zVar,varNum,numRecs)
CDFid id;
Logical zVar;
long varNum;
long numRecs;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        PUT_, (zVar?zVAR_INITIALRECS_:rVAR_INITIALRECS_), numRecs,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarPadValue(id,zVar,varNum,padValue)
CDFid id;
Logical zVar;
long varNum;
void *padValue;
{
return
CDFlib (SELECT_, CDF_, id,
                (zVar?zVAR_:rVAR_), varNum,
        PUT_, (zVar?zVAR_PADVALUE_:rVAR_PADVALUE_), padValue,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarRecVariance(id,zVar,varNum,recVary)
CDFid id;
Logical zVar;
long varNum;
long recVary;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        PUT_, (zVar?zVAR_RECVARY_:rVAR_RECVARY_), recVary,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFputVarSeqData(id,zVar,varNum,seqData)
CDFid id;
Logical zVar;
long varNum;
void *seqData;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        PUT_, (zVar?zVAR_SEQDATA_:rVAR_SEQDATA_), seqData,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarSparseRecords(id,zVar,varNum,sparseRecs)
CDFid id;
Logical zVar;
long varNum;
long sparseRecs;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
        PUT_, (zVar?zVAR_SPARSERECORDS_:rVAR_SPARSERECORDS_), sparseRecs,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFputVarData(id,zVar,varNum,recNum,indices,value)
CDFid id;
Logical zVar;
long varNum;
long recNum;
long *indices;
void *value;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
                 (zVar?zVAR_RECNUMBER_:rVARs_RECNUMBER_), recNum,
                 (zVar?zVAR_DIMINDICES_:rVARs_DIMINDICES_), indices,
        PUT_, (zVar?zVAR_DATA_:rVAR_DATA_), value,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFputVarsRecordDatabyNumbers(id,zVar,numVars,varNums,recNum,buffer)
CDFid id;
Logical zVar;
long numVars;
long *varNums;
long recNum;
void *buffer;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVARs_RECNUMBER_:rVARs_RECNUMBER_), recNum,
        PUT_, (zVar?zVARs_RECDATA_:rVARs_RECDATA_), numVars, varNums, buffer,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFhyperPutVarData(id,zVar,varN,recS,recC,recI,indices,counts,intervals,buff)
CDFid id;
Logical zVar;
long varN;
long recS;
long recC;
long recI;
long *indices;
long *counts;
long *intervals;
void *buff;
{
  CDFstatus status;
  status = flipDataElements (id, zVar, varN, recC, counts, (char*) buff, 2);
  if (status < CDF_OK) return status;
return
CDFlib (SELECT_, CDF_, id,
		 (zVar?zVAR_:rVAR_), varN,
		 (zVar?zVAR_RECNUMBER_:rVARs_RECNUMBER_), recS,
		 (zVar?zVAR_RECCOUNT_:rVARs_RECCOUNT_), recC,
		 (zVar?zVAR_RECINTERVAL_:rVARs_RECINTERVAL_), recI,
		 (zVar?zVAR_DIMINDICES_:rVARs_DIMINDICES_), indices,
		 (zVar?zVAR_DIMCOUNTS_:rVARs_DIMCOUNTS_), counts,
		 (zVar?zVAR_DIMINTERVALS_:rVARs_DIMINTERVALS_), intervals,
	PUT_, (zVar?zVAR_HYPERDATA_:rVAR_HYPERDATA_), buff,
	NULL_);

}

VISIBLE_PREFIX CDFstatus CDFsetChecksum(id,checksum)
CDFid id;
long checksum;
{
return
CDFlib (SELECT_, CDF_, id,
        PUT_, CDF_CHECKSUM_, checksum,
        NULL_);
}

/*
 * SELECT_ *
 *         */

VISIBLE_PREFIX CDFstatus CDFselect(id)
CDFid id;
{
return
CDFlib (SELECT_, CDF_, id,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetDecoding(id,decoding)
CDFid id;
long decoding;
{
return
CDFlib (SELECT_, CDF_, id,
                 CDF_DECODING_, decoding,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetCacheSize(id,numBuffers)
CDFid id;
long numBuffers;
{
return
CDFlib (SELECT_, CDF_, id,
                 CDF_CACHESIZE_, numBuffers,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarCacheSize(id,zVar,varNum,numBuffers)
CDFid id;
Logical zVar;
long varNum;
long numBuffers;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
                 (zVar?zVAR_CACHESIZE_:rVAR_CACHESIZE_), numBuffers,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarsCacheSize(id,zVar,numBuffers)
CDFid id;
Logical zVar;
long numBuffers;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVARs_CACHESIZE_:rVARs_CACHESIZE_), numBuffers,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarSeqPos(id,zVar,varNum,recNum,indices)
CDFid id;
Logical zVar;
long varNum;
long recNum;
long *indices;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
                 (zVar?zVAR_SEQPOS_:rVAR_SEQPOS_), recNum, indices,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetNegtoPosfp0Mode(id,negtoPosfp0)
CDFid id;
long negtoPosfp0;
{
return
CDFlib (SELECT_, CDF_, id,
                 CDF_NEGtoPOSfp0_MODE_, negtoPosfp0,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetReadOnlyMode(id,readOnlyMode)
CDFid id;
long readOnlyMode;
{
return
CDFlib (SELECT_, CDF_, id,
                 CDF_READONLY_MODE_, readOnlyMode,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetVarReservePercent(id,zVar,varNum,percent)
CDFid id;
Logical zVar;
long varNum;
long percent;
{
return
CDFlib (SELECT_, CDF_, id,
                 (zVar?zVAR_:rVAR_), varNum,
                 (zVar?zVAR_RESERVEPERCENT_:rVAR_RESERVEPERCENT_), percent,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetCompressionCacheSize(id,numBuffers)
CDFid id;
long numBuffers;
{
return
CDFlib (SELECT_, CDF_, id,
                 COMPRESS_CACHESIZE_, numBuffers,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetStageCacheSize(id,numBuffers)
CDFid id;
long numBuffers;
{
return
CDFlib (SELECT_, CDF_, id,
                 STAGE_CACHESIZE_, numBuffers,
        NULL_);
}

VISIBLE_PREFIX CDFstatus CDFsetzMode(id,zMode)
CDFid id;
long zMode;
{
return
CDFlib (SELECT_, CDF_, id,
                 CDF_zMODE_, zMode,
        NULL_);
}

/******************************************************************************
* CDFgetLeapSecondLastUpdatedCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetLeapSecondLastUpdatedCS (id, lastUpdated)
CDFid   id;            /* In -- CDF id. */
long    *lastUpdated;  /* In -- lastUpdated. */
{
  return CDFlib (SELECT_, CDF_, id,
		 GET_, CDF_LEAPSECONDLASTUPDATED_, lastUpdated,
	         NULL_);
}

/******************************************************************************
* CDFgetLeapSecondLastUpdated.
******************************************************************************/

VISIBLE_PREFIX CDFstatus CDFgetLeapSecondLastUpdated (id, lastUpdated)
CDFid   id;            /* In -- CDF id. */
long    *lastUpdated;  /* In -- lastUpdated. */
{
  return CDFgetLeapSecondLastUpdatedCS(id, lastUpdated);
}

/******************************************************************************
* CDFsetLeapSecondLastUpdatedCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetLeapSecondLastUpdatedCS (id, ymd)
CDFid   id;            /* In -- CDF id. */
long    ymd;           /* In -- last updated date. */
{
  return CDFlib (SELECT_, CDF_, id,
		 PUT_, CDF_LEAPSECONDLASTUPDATED_, ymd,
	         NULL_);
}

/******************************************************************************
* CDFsetLeapSecondLastUpdated.
******************************************************************************/

VISIBLE_PREFIX CDFstatus CDFsetLeapSecondLastUpdated (id, ymd)
CDFid   id;            /* In -- CDF id. */
long    ymd;           /* In -- last updated date. */
{
  return CDFsetLeapSecondLastUpdatedCS (id, ymd);
}

/******************************************************************************
* CDFcreateCDFCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFcreateCDFCS (cdfName, id)
char    *cdfName;	/* In -- CDF name. */
CDFid   *id;            /* Out -- CDF id. */
{
  return CDFcreateCDF (cdfName, id);
}

/******************************************************************************
* CDFattrInquireCS.
* Can't implement with macro because the attribute's scope determines which
* item(s) to use. 
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFattrInquireCS(id, attrNum, attrName,
	                                           attrScope, maxEntry)
CDFid   id;             /* In -- CDF id. */
long    attrNum;        /* In -- Attribute number. */
char    *attrName;      /* Out -- Attribute name. */
long    *attrScope;     /* Out -- Attribute scope. */
long    *maxEntry;      /* Out -- Maximum gEntry/rEntry number used. */
{
  return CDFattrInquire(id, attrNum, attrName, attrScope, maxEntry);
}

/******************************************************************************
* CDFinquireAttrCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFinquireAttrCS(id, attrNum, attrName,
	                                           scope, maxgEntry,
						   maxrEntry, maxzEntry)
CDFid   id;             /* In -- CDF id. */
long    attrNum;        /* In -- Attribute number. */
char    *attrName;      /* Out -- Attribute name. */
long    *scope;         /* Out -- Attribute scope. */
long    *maxgEntry;     /* Out -- Maximum gEntry number used for gAttribute. */
long    *maxrEntry;     /* Out -- Maximum rEntry number used for vAttribute. */
long    *maxzEntry;     /* Out -- Maximum zEntry number used for vAttribute. */
{
  return CDFinquireAttr(id, attrNum, attrName, scope,
                        maxgEntry, maxrEntry, maxzEntry);
}

/******************************************************************************
* CDFinquireAttrEntryCS.
******************************************************************************/
  
VISIBLE_PREFIX CDFstatus STD_CALL CDFinquireAttrEntryCS(id, grzEntry, attrNum, 
                                                        entryNum, dataType,
						       	numElems)
CDFid   id;             /* In -- CDF id. */
int     grzEntry;       /* In -- Flag for g/r/zEntry. */
long    attrNum;        /* In -- Attribute number. */
long    entryNum;       /* In -- gEntry/rEntry/zEntry number. */
long    *dataType;      /* Out -- gEntry/rEntry/zEntry data type. */
long    *numElems;      /* Out -- gEntry/rEntry/zEntry number of elements. */
{
  return CDFinquireAttrEntry(id, grzEntry, attrNum,
                             entryNum, dataType, numElems);
}

/******************************************************************************
* CDFputAttrEntryCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFputAttrEntryCS(id, grzEntry, attrNum,
	                                            entryNum, dataType,
						    numElems, value)
CDFid	id;		/* In -- CDF id. */
int     grzEntry;       /* In -- Flag for g/r/zEntry. */
long	attrNum;	/* In -- Attribute number. */
long	entryNum;	/* In -- gEntry/rEntry/zEntry number. */
long	dataType;	/* In -- gEntry/rEntry/zEntry data type. */
long	numElems;	/* In -- gEntry/rEntry/zEntry number of elements. */
void	*value;		/* In -- Value. */
{
  return CDFputAttrEntry(id, grzEntry, attrNum, entryNum,
                         dataType, numElems, value);
}

/******************************************************************************
* CDFgetAttrEntryNumStringsCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetAttrEntryNumStringsCS(id, rzEntry,
	                                                      attrNum,
							      entryNum,
							      numStrings)
CDFid   id;             /* In -- CDF id. */
int     rzEntry;        /* In -- Flag for r/zEntry. */
long    attrNum;        /* In -- Attribute number. */
long    entryNum;       /* In -- rEntry/zEntry number. */
long    *numStrings;    /* Out -- Number of strings. */
{
  return CDFgetAttrEntryNumStrings(id, rzEntry, attrNum, entryNum, numStrings);
}

/******************************************************************************
* CDFgetAttrEntryStringsCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetAttrEntryStringsCS(id, rzEntry,
	                                                   attrNum, entryNum,
							   numStrings, strings)
CDFid   id;             /* In -- CDF id. */
int     rzEntry;        /* In -- Flag for r/zEntry. */
long    attrNum;        /* In -- Attribute number. */
long    entryNum;       /* In -- gEntry/rEntry/zEntry number. */
long    *numStrings;    /* Out -- Number of strings. */
char    ***strings;     /* Out -- Strings. */
{
  return CDFgetAttrEntryStrings(id, rzEntry, attrNum, entryNum, numStrings,
		                strings);
}

/******************************************************************************
* CDFputAttrEntryStringsCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFputAttrEntryStringsCS(id, rzEntry,
	                                                   attrNum, entryNum,
							   numStrings, strings)
CDFid   id;             /* In -- CDF id. */
int     rzEntry;        /* In -- Flag for r/zEntry. */
long    attrNum;        /* In -- Attribute number. */
long    entryNum;       /* In -- gEntry/rEntry/zEntry number. */
long    numStrings;     /* In -- Number of strings. */
char    **strings;      /* In -- Strings. */
{
  return CDFputAttrEntryStrings(id, rzEntry, attrNum, entryNum, numStrings,
		                strings);
}

/******************************************************************************
* CDFgetAttrEntryCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetAttrEntryCS(id, grzEntry, attrNum,
	           				    entryNum, value)
CDFid   id;             /* In -- CDF id. */
int     grzEntry;       /* In -- Flag for g/r/zEntry. */
long    attrNum;        /* In -- Attribute number. */
long    entryNum;       /* In -- gEntry/rEntry/zEntry number. */
void    *value;         /* Out -- Value. */
{
  return CDFgetAttrEntry(id, grzEntry, attrNum, entryNum, value);
}

/******************************************************************************
* CDFreadgAttrEntryCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFreadgAttrEntryCS(id, attrNum, entryNum,
	       					      dataType, numElems,
						      value)
CDFid   id;             /* In -- CDF id. */
long    attrNum;        /* In -- Attribute number. */
long    entryNum;       /* In -- gEntry number. */
long    *dataType;      /* Out -- Entry's data type. */
long    *numElems;      /* Out -- Entry's number of elements. */
CDFdata *value;         /* Out -- Value. */
{
  return CDFreadAttrEntry(id, 0, attrNum, entryNum, dataType, numElems,
		          value);
}

/******************************************************************************
* CDFreadzAttrEntryCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFreadgAttzEntryCS(id, attrNum, entryNum,
	       					      dataType, numElems,
						      value)
CDFid   id;             /* In -- CDF id. */
long    attrNum;        /* In -- Attribute number. */
long    entryNum;       /* In -- zEntry number. */
long    *dataType;      /* Out -- Entry's data type. */
long    *numElems;      /* Out -- Entry's number of elements. */
CDFdata *value;         /* Out -- Value. */
{
  return CDFreadAttrEntry(id, 2, attrNum, entryNum, dataType, numElems,
		          value);
}

/******************************************************************************
* CDFreadAttrEntryCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFreadAttrEntryCS(id, grzEntry, attrNum,
	           				     entryNum, dataType,
						     numElems, value)
CDFid   id;             /* In -- CDF id. */
int     grzEntry;       /* In -- Flag for g/r/zEntry. */
long    attrNum;        /* In -- Attribute number. */
long    entryNum;       /* In -- gEntry/rEntry/zEntry number. */
long    *dataType;      /* Out -- Entry's data type. */
long    *numElems;      /* Out -- Entry's number of elements. */
CDFdata *value;         /* Out -- Value. */
{
  return CDFreadAttrEntry(id, grzEntry, attrNum, entryNum, dataType, numElems,
		          value);
}

/******************************************************************************
* CDFreadzVarPadValueCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFreadzVarPadValueCS(id, varNum, dataType,
						        numElems, value)
CDFid   id;             /* In -- CDF id. */
long    varNum;         /* In -- zVariable number. */
long    *dataType;      /* Out -- Variable pad value's data type. */
long    *numElems;      /* Out -- Variable pad value's number of elements. */
CDFdata *value;         /* Out -- Value. */
{
  return CDFreadzVarPadValue(id, varNum, dataType, numElems, value);
}

/******************************************************************************
* CDFreadzVarAllByVarIDCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFreadzVarAllByVarIDCS(id, varNum, numRecs,
							  dataType, numElems,
							  numDims, dimSizes,
							  recVary, dimsVary,
							  values)
CDFid   id;             /* In -- CDF id. */
long    varNum;         /* In -- zVariable number. */
long    *numRecs;       /* Out -- Variable's written records. */
long    *dataType;      /* Out -- Variable's data type. */
long    *numElems;      /* Out -- Variable's number of elements. */
long    *numDims;       /* Out -- Variable's number of dimensions. */
long    *dimSizes;      /* Out -- Variable's dimensional sizes. */
long    *recVary;       /* Out -- Variable's record varying. */
long    *dimsVary;      /* Out -- Variable's dimensional varying. */
CDFdata *values;        /* Out -- Values. */
{
  return CDFreadzVarAllByVarID(id, varNum, numRecs, dataType, numElems,
			       numDims, dimSizes, recVary, dimsVary, values);
}

/******************************************************************************
* CDFreadzVarDataByVarIDCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFreadzVarDataByVarIDCS(id, varNum, numRecs,
							   values)
CDFid   id;             /* In -- CDF id. */
long    varNum;         /* In -- zVariable number. */
long    *numRecs;       /* Out -- Variable's written records. */
CDFdata *values;        /* Out -- Values. */
{
  return CDFreadzVarDataByVarID(id, varNum, numRecs, values);
}

/******************************************************************************
* CDFreadzVarRangeDataByVarIDCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFreadzVarRangeDataByVarIDCS(id, varNum,
	       							startNum,
								endNum,
								values)
CDFid   id;             /* In -- CDF id. */
long    varNum;         /* In -- zVariable number. */
long    startNum;       /* In -- Starting record number to read. */
long    endNum;         /* In -- Ending record number to read. */
CDFdata *values;        /* Out -- Values. */
{
  return CDFreadzVarRangeDataByVarID(id, varNum, startNum, endNum, values);
}

/******************************************************************************
* CDFreadzVarAllByVarNameCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFreadzVarAllByVarNameCS(id, varName,
	       						    numRecs, dataType,
							    numElems, numDims,
							    dimSizes, recVary,
							    dimsVary, values)
CDFid   id;             /* In -- CDF id. */
char    *varName;       /* In -- zVariable name. */
long    *numRecs;       /* Out -- Variable's written records. */
long    *dataType;      /* Out -- Variable's data type. */
long    *numElems;      /* Out -- Variable's number of elements. */
long    *numDims;       /* Out -- Variable's number of dimensions. */
long    *dimSizes;      /* Out -- Variable's dimensional sizes. */
long    *recVary;       /* Out -- Variable's record varying. */
long    *dimsVary;      /* Out -- Variable's dimensional varying. */
CDFdata *values;        /* Out -- Values. */
{
  return CDFreadzVarAllByVarName(id, varName, numRecs, dataType, numElems,
		 		 numDims, dimSizes, recVary, dimsVary, values);
}

/******************************************************************************
* CDFreadzVarDataByVarNameCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFreadzVarDataByVarNameCS(id, varName,
	       						     numRecs, values)
CDFid   id;             /* In -- CDF id. */
char    *varName;       /* In -- zVariable name. */
long    *numRecs;       /* Out -- Variable's written records. */
CDFdata *values;        /* Out -- Values. */
{
  return CDFreadzVarDataByVarName(id, varName, numRecs, values);
}

/******************************************************************************
* CDFreadzVarRangeDataByVarNameCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFreadzVarRangeDataByVarNameCS(id, varName,
	       							  startNum,
								  endNum,
								  values)
CDFid   id;             /* In -- CDF id. */
char    *varName;       /* In -- zVariable name. */
long    startNum;       /* In -- Starting record number to read. */
long    endNum;         /* In -- Ending record number to read. */
CDFdata *values;        /* Out -- Values. */
{
  return CDFreadzVarRangeDataByVarName(id, varName, startNum, endNum, values);
}

/******************************************************************************
* CDFdeleteAttrEntryCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFdeleteAttrEntryCS(id, grzEntry, attrNum,
	       					       entryNum)
CDFid   id;             /* In -- CDF id. */
int     grzEntry;       /* In -- Flag for g/r/zEntry. */
long    attrNum;        /* In -- Attribute number. */
long    entryNum;       /* In -- gEntry/rEntry/zEntry number. */
{
  return CDFdeleteAttrEntry(id, grzEntry, attrNum, entryNum);
}

/******************************************************************************
* CDFsetAttrEntryDataSpecCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetAttrEntryDataSpecCS(id, grzEntry,
	       						    attrNum, entryNum,
							    dataType, numElems)
CDFid   id;             /* In -- CDF id. */
int     grzEntry;       /* In -- Flag for g/r/zEntry. */
long    attrNum;        /* In -- Attribute number. */
long    entryNum;       /* In -- gEntry/rEntry/zEntry number. */
long    dataType;       /* In -- Data type. */
long    numElems;       /* In -- Number of elements. */
{
  return CDFsetAttrEntryDataSpec(id, grzEntry, attrNum,
                                 entryNum, dataType, numElems);
}

/******************************************************************************
* CDFgetAttrNumCS.
******************************************************************************/

VISIBLE_PREFIX long STD_CALL CDFgetAttrNumCS(id,attrName)
CDFid	id;		/* In -- CDF id. */
char	*attrName;	/* In -- attribute name. */
{
    return CDFgetAttrNum(id,attrName);
}

/******************************************************************************
* CDFgetVarNumCS.
******************************************************************************/

VISIBLE_PREFIX long STD_CALL CDFgetVarNumCS(id,varName)
CDFid	id;		/* In -- CDF id. */
char	*varName;	/* In -- variable name. */
{
  return CDFgetVarNum(id,varName);
}

/******************************************************************************
* CDFgetNumAttrEntriesCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetNumAttrEntriesCS(id, grzEntry, attrNum, 
                                                         numEntries)
CDFid   id;             /* In -- CDF id. */
int     grzEntry;       /* In -- Flag for g/r/zEntry. */
long    attrNum;        /* In -- Attribute number. */
long    *numEntries;    /* Out -- number of gEntries/rEntries/zEntries. */
{
  return CDFgetNumAttrEntries(id, grzEntry, attrNum, numEntries);
}

/******************************************************************************
* CDFgetAttrMaxEntryCS.  
******************************************************************************/
  
VISIBLE_PREFIX CDFstatus STD_CALL CDFgetAttrMaxEntryCS(id, grzEntry, attrNum,
	                			       maxEntry)
CDFid   id;             /* In -- CDF id. */
int     grzEntry;       /* In -- Flag for g/r/zEntry. */
long    attrNum;        /* In -- Attribute number. */
long    *maxEntry;      /* Out -- Max. number of gEntry/rEntry/zEntry. */
{                
  return CDFgetAttrMaxEntry(id, grzEntry, attrNum, maxEntry);
}

/******************************************************************************
* CDFgetAttrEntryDataTypeCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetAttrEntryDataTypeCS(id, grzEntry,
	       					            attrNum, entryNum,
						            dataType)
CDFid   id;             /* In -- CDF id. */
int     grzEntry;       /* In -- Flag for g/r/zEntry. */
long    attrNum;        /* In -- Attribute number. */
long    entryNum;       /* In -- gEntry/rEntry/zEntry number. */
long    *dataType;      /* Out -- gEntry/rEntry/zEntry dataType. */
{
  return CDFgetAttrEntryDataType(id, grzEntry, attrNum,
                                 entryNum, dataType);
}

/******************************************************************************
* CDFgetAttrEntryNumElementsCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetAttrEntryNumElementsCS(id, grzEntry,
	       				 		       attrNum, 
                                                               entryNum,
							       numElements)
CDFid   id;             /* In -- CDF id. */
int     grzEntry;       /* Flag for g/r/zEntry. */
long    attrNum;        /* In -- Attribute number. */
long    entryNum;       /* In -- gEntry/rEntry/zEntry number. */
long    *numElements;   /* Out -- gEntry/rEntry/zEntry numElements. */
{                
  return CDFgetAttrEntryNumElements(id, grzEntry, attrNum,
                                    entryNum, numElements);         
}                
                 
/******************************************************************************
* CDFgetVarRecordDataCS.  
******************************************************************************/
  
VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarRecordDataCS(id, zVar, varNum,
	                                         	recNum, buffer)
CDFid   id;             /* In -- CDF id. */
int     zVar;           /* In -- Flag for zVariable. */
long    varNum;         /* In -- Variable number. */
long    recNum;         /* In -- Record number to read. */
void    *buffer;        /* Out -- Buffer for holding data. */

{
  return CDFgetVarRecordData(id, zVar, varNum, recNum, buffer);
}

/******************************************************************************
* CDFputVarRecordDataCS.
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFputVarRecordDataCS(id, zVar, varNum,
	                                        	recNum, buffer)
CDFid   id;             /* In -- CDF id. */
int     zVar;           /* In -- Flag for zVariable. */
long    varNum;         /* In -- Variable number. */
long    recNum;         /* In -- Record number to read. */
void    *buffer;        /* In -- Buffer for holding data. */

{
  return CDFputVarRecordData(id, zVar, varNum, recNum, buffer);
}

/******************************************************************************
* CDFgetVarsRecordDatabyNamesCS. 
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarsRecordDatabyNamesCS(id, zVar,
	                                                	numVars, 
                                                                varNames,
							       	recNum, 
                                                                buffptr)
CDFid   id;             /* In -- CDF id. */
int     zVar;           /* In -- Flag for zVariable. */
long    numVars;        /* In -- Number of variables. */
char    *varNames[];    /* In -- Array of variable names. */
long    recNum;         /* In -- Record number to read. */ 
void    *buffptr[];     /* Out -- Array of buffer pointers for holding data. */

{ 
  return CDFgetVarsRecordDatabyNames(id, zVar, numVars, varNames, recNum,
                                     buffptr);

}

/******************************************************************************
* CDFputVarsRecordDatabyNamesCS.  
******************************************************************************/
                   
VISIBLE_PREFIX CDFstatus STD_CALL CDFputVarsRecordDatabyNamesCS(id, zVar,
	                                                 	numVars, 
                                                                varNames,
							       	recNum, 
                                                                buffptr)
CDFid   id;             /* In -- CDF id. */
int     zVar;           /* In -- Flag for zVariable. */
long    numVars;        /* In -- Number of variables. */
char    *varNames[];    /* In -- Array of variable names. */
long    recNum;         /* In -- Record number to read. */
void    *buffptr[];     /* In -- Array of buffer pointers for holding data. */
    
{   
  return CDFputVarsRecordDatabyNames(id, zVar, numVars, varNames, recNum,
                                     buffptr);

}

/******************************************************************************
* CDFinquireAttrInfoCS. (Holding for backward version -- should not use)
******************************************************************************/

VISIBLE_PREFIX CDFstatus STD_CALL CDFinquireAttrInfoCS(id, zEntry, attrNum,
	                                               attrName, scope,
						       maxEntry)
CDFid   id;             /* In -- CDF id. */
int     zEntry;         /* In -- Flag for zEntry. */
long    attrNum;        /* In -- Attribute number. */
char    *attrName;      /* Out -- Attribute name. */
long    *scope;         /* Out -- Attribute scope. */
long    *maxEntry;      /* Out -- Maximum g/r/zEntry number used. */
{
  return CDFinquireAttrInfo(id, zEntry, attrNum, attrName, scope, maxEntry);
}

/******************************************************************************
* CDFsetFileBackwardCS. 
******************************************************************************/
  
VISIBLE_PREFIX void STD_CALL CDFsetFileBackwardCS (flag)
int flag;         
{                 
  CDFsetFileBackward (flag);
} 

/******************************************************************************
* CDFsetChecksumModeCS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL CDFsetChecksumModeCS (flag)
long flag;
{
  CDFsetChecksumMode (flag);
}

/******************************************************************************
* CDFgetFileBackwardCS.
******************************************************************************/

VISIBLE_PREFIX int STD_CALL CDFgetFileBackwardCS ()
{
  return CDFgetFileBackward ();
}

/******************************************************************************
* CDFgetChecksumModeCS.
******************************************************************************/

VISIBLE_PREFIX long STD_CALL CDFgetChecksumModeCS ()
{
  return CDFgetChecksumMode ();
}

/******************************************************************************
* CDFsetFileBackwardFlagCS. (Holding for backward version -- should not use.)
******************************************************************************/

VISIBLE_PREFIX void STD_CALL CDFsetFileBackwardFlagCS (flag)
int flag;
{
  CDFsetFileBackwardFlag (flag);
}

/******************************************************************************
* CDFgetFileBackwardFlagCS. (Holding for backward version -- should not use.)
******************************************************************************/

VISIBLE_PREFIX int STD_CALL CDFgetFileBackwardFlagCS ()
{
  return CDFgetFileBackwardFlag ();
}

/*
 * CLOSE_  *
 *         */

VISIBLE_PREFIX CDFstatus STD_CALL CDFcloseCS(id)
CDFid id;
{
  return CDFclose(id);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFcloseVarCS(id,zVar,varNum)
CDFid id;
Logical zVar;
long varNum;
{
  return CDFcloseVar(id,zVar,varNum);
}

/*
 * CONFIRM_  *
 *           */

VISIBLE_PREFIX CDFstatus STD_CALL CDFconfirmAttrExistenceCS(id,attrName)
CDFid id;
char *attrName;
{
  return CDFconfirmAttrExistence(id,attrName);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFconfirmgEntryExistenceCS(id,attrNum,entryNum)
CDFid id;
long attrNum;
long entryNum;
{
  return CDFconfirmgEntryExistence(id,attrNum,entryNum);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFconfirmrEntryExistenceCS(id,attrNum,entryNum)
CDFid id;
long attrNum;
long entryNum;
{
  return CDFconfirmrEntryExistence(id,attrNum,entryNum);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFconfirmVarExistenceCS(id,zVar,varName)
CDFid id;
Logical zVar;
char *varName;
{
  return CDFconfirmVarExistence(id,zVar,varName);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFconfirmVarPadValueExistenceCS(id,zVar,varNum)
CDFid id;
Logical zVar;
long varNum;
{
  return CDFconfirmVarPadValueExistence(id,zVar,varNum);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFconfirmzEntryExistenceCS(id,attrNum,entryNum)
CDFid id;
long attrNum;
long entryNum;
{
  return CDFconfirmzEntryExistence(id,attrNum,entryNum);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFconfirmChecksumCS(id)
CDFid id;
{
  return CDFconfirmChecksum(id);
}

/*
 * CREATE_ *
 *         */

VISIBLE_PREFIX CDFstatus STD_CALL CDFcreateCS(CDFname,numDims,dimSizes,encoding,majority,id)
char *CDFname;
long numDims;
long *dimSizes;
long encoding;
long majority;
CDFid *id;
{
  return CDFcreate(CDFname,numDims,dimSizes,encoding, majority,id);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFcreateAttrCS(id,attrName,attrScope,attrNum)
CDFid id;
char *attrName;
long attrScope;
long *attrNum;
{
  return CDFcreateAttr(id,attrName,attrScope,attrNum);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFcreaterVarCS(id,varName,dataType,numElements,recVary,dimVarys,varNum)
CDFid id;
char *varName;
long dataType;
long numElements;
long recVary;
long *dimVarys;
long *varNum;
{
  return CDFcreaterVar(id,varName,dataType,numElements,recVary,dimVarys,varNum);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFcreatezVarCS(id,varName,dataType,numElements,numDims,dimSizes,recVary,dimVarys,varNum)
CDFid id;
char *varName;
long dataType;
long numElements;
long numDims;
long *dimSizes;
long recVary;
long *dimVarys;
long *varNum;
{
  return CDFcreatezVar(id,varName,dataType,numElements,numDims,dimSizes,recVary,dimVarys,varNum);
}

/*
 * DELETE_ *
 *         */

VISIBLE_PREFIX CDFstatus STD_CALL CDFdeleteCS(id)
CDFid id;
{
  return CDFdelete(id);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFdeleteAttrCS(id,attrNum)
CDFid id;
long attrNum;
{
  return CDFdeleteAttr(id,attrNum);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFdeleteVarCS(id,zVar,varNum)
CDFid id;
Logical zVar;
long varNum;
{
  return CDFdeleteVar(id,zVar,varNum);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFdeleteVarRecordsCS(id,zVar,varNum,firstRec,lastRec)
CDFid id;
Logical zVar;
long varNum;
long firstRec;
long lastRec;
{
  return CDFdeleteVarRecords(id,zVar,varNum,firstRec, lastRec);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFdeleteVarRecordsRenumberCS(id,zVar,varNum,firstRec,lastRec)
CDFid id;
Logical zVar;
long varNum;
long firstRec;
long lastRec;
{
  return CDFdeleteVarRecordsRenumber(id,zVar,varNum,firstRec, lastRec);
}

/*
 * GET_ *
 *      */

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetCacheSizeCS(id,numBuffers)
CDFid id;
long *numBuffers;
{
  return CDFgetCacheSize(id,numBuffers);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarCacheSizeCS(id,zVar,varNum,numBuffers)
CDFid id;
Logical zVar;
long varNum;
long *numBuffers;
{
  return CDFgetVarCacheSize(id,zVar,varNum,numBuffers);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetDecodingCS(id,decoding)
CDFid id;
long *decoding;
{
  return CDFgetDecoding(id,decoding);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetAttrNameCS(id,attrNum,attrName)
CDFid id;
long attrNum;
char *attrName;
{
  return CDFgetAttrName(id,attrNum,attrName);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetNameCS(id,cdfName)
CDFid id;
char *cdfName;
{
  return CDFgetName(id,cdfName);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetNegtoPosfp0ModeCS(id,negtoPosfp0)
CDFid id;
long *negtoPosfp0;
{
  return CDFgetNegtoPosfp0Mode(id,negtoPosfp0);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetReadOnlyModeCS(id,readOnlyMode)
CDFid id;
long *readOnlyMode;
{
  return CDFgetReadOnlyMode(id,readOnlyMode);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetzModeCS(id,zMode)
CDFid id;
long *zMode;
{
  return CDFgetzMode(id,zMode);
}


VISIBLE_PREFIX CDFstatus STD_CALL CDFgetCompressionCacheSizeCS(id,numBuffers)
CDFid id;
long *numBuffers;
{
  return CDFgetCompressionCacheSize(id,numBuffers);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetStageCacheSizeCS(id,numBuffers)
CDFid id;
long *numBuffers;
{
  return CDFgetStageCacheSize(id,numBuffers);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarReservePercentCS(id,zVar,varNum,percent)
CDFid id;
Logical zVar;
long varNum;
long *percent;
{
  return CDFgetVarReservePercent(id,zVar,varNum,percent);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarSeqPosCS(id,zVar,varNum,recNum,indices)
CDFid id;
Logical zVar;
long varNum;
long *recNum;
long *indices;
{
  return CDFgetVarSeqPos(id,zVar,varNum,recNum,indices);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetAttrScopeCS(id,attrNum,attrScope)
CDFid id;
long attrNum;
long *attrScope;
{
  return CDFgetAttrScope(id,attrNum,attrScope);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetCompressionCS(id,cType, cParms, cPercent)
CDFid id;
long *cType;
long *cParms;
long *cPercent;
{
  return CDFgetCompression(id,cType, cParms, cPercent);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetCopyrightCS(id,copyright)
CDFid id;
char *copyright;
{
  return CDFgetCopyright(id,copyright);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetEncodingCS(id,encoding)
CDFid id;
long *encoding;
{
  return CDFgetEncoding(id,encoding);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetFormatCS(id,format)
CDFid id;
long *format;
{
  return CDFgetFormat(id,format);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetCompressionInfoCS(name,cType,cParms,cSize,uSize)
char *name;
long *cType;
long *cParms;
OFF_T *cSize;
OFF_T *uSize;
{
  return CDFgetCompressionInfo(name,cType,cParms,cSize, uSize);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetMajorityCS(id,majority)
CDFid id;
long *majority;
{
  return CDFgetMajority(id,majority);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetNumAttributesCS(id,numAttrs)
CDFid id;
long *numAttrs;
{
  return CDFgetNumAttributes(id,numAttrs);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetNumgAttributesCS(id,numgAttrs)
CDFid id;
long *numgAttrs;
{
  return CDFgetNumgAttributes(id,numgAttrs);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetNumVarsCS(id,zVar,numVars)
CDFid id;
Logical zVar;
long *numVars;
{
  return CDFgetNumVars(id,zVar,numVars);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetNumvAttributesCS(id,numvAttrs)
CDFid id;
long *numvAttrs;
{
  return CDFgetNumvAttributes(id,numvAttrs);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFdocCS(id,version,release,copyright)
CDFid id;
long *version;
long *release;
char *copyright;
{
  return CDFdoc(id,version,release,copyright);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetDataTypeSizeCS(dataType,numBytes)
long dataType;
long *numBytes;
{
  return CDFgetDataTypeSize(dataType,numBytes);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetLibraryCopyrightCS(copyright)
char *copyright;
{
  return CDFgetLibraryCopyright(copyright);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetLibraryVersionCS(version,release,increment,subincrement)
long *version;
long *release;
long *increment;
char *subincrement;
{
  return CDFgetLibraryVersion(version,release,increment,subincrement);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVersionCS(id,version,release,increment)
CDFid id;
long *version;
long *release;
long *increment;
{
  return CDFgetVersion(id,version,release,increment);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarBlockingFactorCS(id,zVar,varNum,bf)
CDFid id;
Logical zVar;
long varNum;
long *bf;
{
  return CDFgetVarBlockingFactor(id,zVar,varNum,bf);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarCompressionCS(id,zVar,varNum,cType,cParms,cPct)
CDFid id;
Logical zVar;
long varNum;
long *cType;
long *cParms;
long *cPct;
{
  return CDFgetVarCompression(id,zVar,varNum,cType,cParms,cPct);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarDataCS(id,zVar,varNum,recNum,indices,value)
CDFid id;
Logical zVar;
long varNum;
long recNum;
long *indices;
void *value;
{
  return CDFgetVarData(id,zVar,varNum,recNum,indices,value);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarDataTypeCS(id,zVar,varNum,dataType) 
CDFid id;
Logical zVar;
long varNum;
long *dataType;
{
  return CDFgetVarDataType(id,zVar,varNum,dataType);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarDimVariancesCS(id,zVar,varNum,dimVarys)
CDFid id;
Logical zVar;
long varNum;
long *dimVarys;
{
  return CDFgetVarDimVariances(id,zVar,varNum,dimVarys);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarMaxAllocRecNumCS(id,zVar,varNum,maxAllocRec)
CDFid id;
Logical zVar;
long varNum;
long *maxAllocRec;
{
  return CDFgetVarMaxAllocRecNum(id,zVar,varNum,maxAllocRec);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarMaxWrittenRecNumCS(id,zVar,varNum,maxRec)
CDFid id;
Logical zVar;
long varNum;
long *maxRec;
{
  return CDFgetVarMaxWrittenRecNum(id,zVar,varNum,maxRec);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarAllocRecordsCS(id,zVar,varNum,numAllocRecs)
CDFid id;
Logical zVar;
long varNum;
long *numAllocRecs;
{
  return CDFgetVarAllocRecords(id,zVar,varNum,numAllocRecs);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarNumElementsCS(id,zVar,varNum,numElements)
CDFid id;
Logical zVar;
long varNum;
long *numElements;
{
  return CDFgetVarNumElements(id,zVar,varNum,numElements);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarNumRecsWrittenCS(id,zVar,varNum,numRecs)
CDFid id;
Logical zVar;
long varNum;
long *numRecs;
{
  return CDFgetVarNumRecsWritten(id,zVar,varNum,numRecs);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarPadValueCS(id,zVar,varNum,padValue)
CDFid id;
Logical zVar;
long varNum;
void *padValue;
{
  return CDFgetVarPadValue(id,zVar,varNum,padValue);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarRecVarianceCS(id,zVar,varNum,recVary)
CDFid id;
Logical zVar;
long varNum;
long *recVary;
{
  return CDFgetVarRecVariance(id,zVar,varNum,recVary);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarSeqDataCS(id,zVar,varNum,seqData)
CDFid id;
Logical zVar;
long varNum;
void *seqData;
{
  return CDFgetVarSeqData(id,zVar,varNum,seqData);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarsRecordDatabyNumbersCS(id,zVar,numVars,varNums,recNum,buffer)
CDFid id;
Logical zVar;
long numVars;
long *varNums;
long recNum;
void *buffer;
{
  return CDFgetVarsRecordDatabyNumbers(id,zVar,numVars,varNums,recNum,buffer);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarSparseRecordsCS(id,zVar,varNum,sparseRecs)
CDFid id;
Logical zVar;
long varNum;
long *sparseRecs;
{
  return CDFgetVarSparseRecords(id,zVar,varNum,sparseRecs);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetrVarsDimSizesCS(id,dimSizes)
CDFid id;
long *dimSizes;
{
  return CDFgetrVarsDimSizes(id,dimSizes);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetzVarDimSizesCS(id,varNum,dimSizes)
CDFid id;
long varNum;
long *dimSizes;
{
  return CDFgetzVarDimSizes(id,varNum,dimSizes);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarNameCS(id,zVar,varNum,varName)
CDFid id;
Logical zVar;
long varNum;
char *varName;
{
  return CDFgetVarName(id,zVar,varNum,varName);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetzVarNumDimsCS(id,varNum,numDims)
CDFid id;
long varNum;
long *numDims;
{
  return CDFgetzVarNumDims(id,varNum,numDims);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetrVarsNumDimsCS(id,numDims)
CDFid id;
long *numDims;
{
  return CDFgetrVarsNumDims(id,numDims);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetStatusTextCS(status,text)
CDFstatus status;
char *text;
{
  return CDFgetStatusText(status,text);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFhyperGetVarDataCS(id,zVar,varN,recS,recC,recI,indices,counts,intervals,buff)
CDFid id;
Logical zVar;
long varN;
long recS;
long recC;
long recI;
long *indices;
long *counts;
long *intervals;
void *buff;
{
  return CDFhyperGetVarData(id,zVar,varN,recS,recC,recI,indices,counts,intervals,buff);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetMaxWrittenRecNumsCS(id,maxRecrVars,maxReczVars)
CDFid id;
long *maxRecrVars;
long *maxReczVars;
{
  return CDFgetMaxWrittenRecNums(id,maxRecrVars,maxReczVars);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarsMaxWrittenRecNumCS(id,zVar,maxRecVar)
CDFid id;
Logical zVar;
long *maxRecVar;
{
  return CDFgetVarsMaxWrittenRecNum(id,zVar,maxRecVar);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFinquireVarCS(id,zVar,varN,varName,dataType,numElements,numDims,dimSizes,recVary,dimVarys)
CDFid id;
Logical zVar;
long varN;
char *varName;
long *dataType;
long *numElements;
long *numDims;
long *dimSizes;
long *recVary;
long *dimVarys;
{
  return CDFinquireVar(id,zVar,varN,varName,dataType,numElements,numDims,dimSizes,recVary,dimVarys);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFvarInquireCS(id,varN,varName,dataType,numElements,recVary,dimVarys)
CDFid id;
long varN;
char *varName;
long *dataType;
long *numElements;
long *recVary;
long *dimVarys;
{
  return CDFvarInquire(id,varN,varName,dataType,numElements,recVary,dimVarys);
}
VISIBLE_PREFIX CDFstatus STD_CALL CDFinquireCS(id,numDims,dimSizes,encoding,majority,maxRec,nVars,nAttrs)
CDFid id;
long *numDims;
long *dimSizes;
long *encoding;
long *majority;
long *maxRec;
long *nVars;
long *nAttrs;
{
  return CDFinquire(id,numDims,dimSizes,encoding,majority,maxRec,nVars,nAttrs);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFinquireCDFCS(id,numDims,dimSizes,encoding,majority,maxrRec,nrVars,maxzRec,nzVars,nAttrs)
CDFid id;
long *numDims;
long *dimSizes;
long *encoding;
long *majority;
long *maxrRec;
long *nrVars;
long *maxzRec;
long *nzVars;
long *nAttrs;
{
  return CDFinquireCDF(id,numDims,dimSizes,encoding,majority,maxrRec,nrVars,maxzRec,nzVars,nAttrs);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetChecksumCS(id,checksum)
CDFid id;
long *checksum;
{
  return CDFgetChecksum(id,checksum);
}

/*
 * OPEN_ *
 *       */

VISIBLE_PREFIX CDFstatus STD_CALL CDFopenCS(CDFname,id)
char *CDFname;
CDFid *id;
{
  return CDFopen(CDFname,id);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFgetVarInfoCS(id,zVar,varNum,dataType,numElems,numDims,
dimSizes)
CDFid id;
Logical zVar;
long varNum;
long *dataType;
long *numElems;
long *numDims;
long *dimSizes;
{
  return CDFgetVarInfo(id,zVar,varNum,dataType,numElems,numDims, dimSizes);
}

/*
 * PUT_ *
 *      */

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetAttrScopeCS(id,attrNum,attrScope)
CDFid id;
long attrNum;
long attrScope;
{
  return CDFsetAttrScope(id,attrNum,attrScope);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetCompressionCS(id,cType, cParms)
CDFid id;
long cType;
long *cParms;
{
  return CDFsetCompression(id,cType, cParms);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetEncodingCS(id,encoding)
CDFid id;
long encoding;
{
  return CDFsetEncoding(id,encoding);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetFormatCS(id,format)
CDFid id;
long format;
{
  return CDFsetFormat(id,format);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetMajorityCS(id,majority)
CDFid id;
long majority;
{
  return CDFsetMajority(id,majority);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFrenameAttrCS(id,attrNum,attrName)
CDFid id;
long attrNum;
char *attrName;
{
  return CDFrenameAttr(id,attrNum,attrName);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFrenameVarCS(id,zVar,varNum,varName)
CDFid id;
Logical zVar;
long varNum;
char *varName;
{
  return CDFrenameVar(id,zVar,varNum,varName);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarAllocRecordsCS(id,zVar,varNum,allocRecs)
CDFid id;
Logical zVar;
long varNum;
long allocRecs;
{
  return CDFsetVarAllocRecords(id,zVar,varNum,allocRecs);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarAllocBlockRecordsCS(id,zVar,varNum,firstRec, lastRec)
CDFid id;
Logical zVar;
long varNum;
long firstRec;
long lastRec;
{
  return CDFsetVarAllocBlockRecords(id,zVar,varNum,firstRec, lastRec);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarBlockingFactorCS(id,zVar,varNum,bf)
CDFid id;
Logical zVar;
long varNum;
long bf;
{
  return CDFsetVarBlockingFactor(id,zVar,varNum,bf);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarCompressionCS(id,zVar,varNum,cType,cParms)
CDFid id;
Logical zVar;
long varNum;
long cType;
long *cParms;
{
  return CDFsetVarCompression(id,zVar,varNum,cType,cParms);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarDataSpecCS(id,zVar,varNum,dataType,numElems)
CDFid id;
Logical zVar;
long varNum;
long dataType;
long numElems;
{
  return CDFsetVarDataSpec(id,zVar,varNum,dataType,numElems);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarDimVariancesCS(id,zVar,varNum,dimVarys)
CDFid id;
Logical zVar;
long varNum;
long *dimVarys;
{
  return CDFsetVarDimVariances(id,zVar,varNum,dimVarys);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarInitialRecsCS(id,zVar,varNum,numRecs)
CDFid id;
Logical zVar;
long varNum;
long numRecs;
{
  return CDFsetVarInitialRecs(id,zVar,varNum,numRecs);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarPadValueCS(id,zVar,varNum,padValue)
CDFid id;
Logical zVar;
long varNum;
void *padValue;
{
  return CDFsetVarPadValue(id,zVar,varNum,padValue);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarRecVarianceCS(id,zVar,varNum,recVary)
CDFid id;
Logical zVar;
long varNum;
long recVary;
{
  return CDFsetVarRecVariance(id,zVar,varNum,recVary);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFputVarSeqDataCS(id,zVar,varNum,seqData)
CDFid id;
Logical zVar;
long varNum;
void *seqData;
{
  return CDFputVarSeqData(id,zVar,varNum,seqData);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarSparseRecordsCS(id,zVar,varNum,sparseRecs)
CDFid id;
Logical zVar;
long varNum;
long sparseRecs;
{
  return CDFsetVarSparseRecords(id,zVar,varNum,sparseRecs);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFputVarDataCS(id,zVar,varNum,recNum,indices,value)
CDFid id;
Logical zVar;
long varNum;
long recNum;
long *indices;
void *value;
{
  return CDFputVarData(id,zVar,varNum,recNum,indices, value);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFputVarsRecordDatabyNumbersCS(id,zVar,numVars,varNums,recNum,buffer)
CDFid id;
Logical zVar;
long numVars;
long *varNums;
long recNum;
void *buffer;
{
  return CDFputVarsRecordDatabyNumbers(id,zVar,numVars,varNums,recNum,buffer);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFhyperPutVarDataCS(id,zVar,varN,recS,recC,recI,indices,counts,intervals,buff)
CDFid id;
Logical zVar;
long varN;
long recS;
long recC;
long recI;
long *indices;
long *counts;
long *intervals;
void *buff;
{
  return CDFhyperPutVarData(id,zVar,varN,recS,recC,recI,indices,counts,intervals,buff);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetChecksumCS(id,checksum)
CDFid id;
long checksum;
{
  return CDFsetChecksum(id,checksum);
}

/*
 * SELECT_ *
 *         */

VISIBLE_PREFIX CDFstatus STD_CALL CDFselectCS(id)
CDFid id;
{
  return CDFselect(id);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetDecodingCS(id,decoding)
CDFid id;
long decoding;
{
  return CDFsetDecoding(id,decoding);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetCacheSizeCS(id,numBuffers)
CDFid id;
long numBuffers;
{
  return CDFsetCacheSize(id,numBuffers);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarCacheSizeCS(id,zVar,varNum,numBuffers)
CDFid id;
Logical zVar;
long varNum;
long numBuffers;
{
  return CDFsetVarCacheSize(id,zVar,varNum,numBuffers);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarsCacheSizeCS(id,zVar,numBuffers)
CDFid id;
Logical zVar;
long numBuffers;
{
  return CDFsetVarsCacheSize(id,zVar,numBuffers);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarSeqPosCS(id,zVar,varNum,recNum,indices)
CDFid id;
Logical zVar;
long varNum;
long recNum;
long *indices;
{
  return CDFsetVarSeqPos(id,zVar,varNum,recNum,indices);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetNegtoPosfp0ModeCS(id,negtoPosfp0)
CDFid id;
long negtoPosfp0;
{
  return CDFsetNegtoPosfp0Mode(id,negtoPosfp0);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetReadOnlyModeCS(id,readOnlyMode)
CDFid id;
long readOnlyMode;
{
  return CDFsetReadOnlyMode(id,readOnlyMode);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetVarReservePercentCS(id,zVar,varNum,percent)
CDFid id;
Logical zVar;
long varNum;
long percent;
{
  return CDFsetVarReservePercent(id,zVar,varNum,percent);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetCompressionCacheSizeCS(id,numBuffers)
CDFid id;
long numBuffers;
{
  return CDFsetCompressionCacheSize(id,numBuffers);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetStageCacheSizeCS(id,numBuffers)
CDFid id;
long numBuffers;
{
  return CDFsetStageCacheSize(id,numBuffers);
}

VISIBLE_PREFIX CDFstatus STD_CALL CDFsetzModeCS(id,zMode)
CDFid id;
long zMode;
{
  return CDFsetzMode(id,zMode);
}

/******************************************************************************
* toParseEPOCHCS.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL toParseEPOCHCS (inString)
void *inString;
{
  return toParseEPOCH ((char *)inString);
}  

/******************************************************************************
* parseEPOCHCS.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL parseEPOCHCS (inString)
void *inString;
{
  return parseEPOCH ((char *)inString);
}  

/******************************************************************************
* toParseEPOCH16CS.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL toParseEPOCH16CS (inString, epoch)
void *inString;
double *epoch;
{
  return toParseEPOCH16 ((char *)inString, epoch);
}

/******************************************************************************
* parseEPOCH16CS.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL parseEPOCH16CS (inString, epoch)
void *inString;
double *epoch;
{
  return parseEPOCH16 ((char *)inString, epoch);
}

/******************************************************************************
* parseTT2000CS.
******************************************************************************/

VISIBLE_PREFIX long long STD_CALL parseTT2000CS (inString)
void *inString;
{
  return CDF_TT2000_from_UTC_string ((char *)inString);
}

/******************************************************************************
* toParseTT2000CS.
******************************************************************************/

VISIBLE_PREFIX long long STD_CALL toParseTT2000CS (inString)
void *inString;
{
  return toParseTT2000 ((char *)inString);
}

/******************************************************************************
* parseEPOCH1CS.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL parseEPOCH1CS (inString)
void *inString;
{
  return parseEPOCH1 ((char *)inString);
}

/******************************************************************************
* parseEPOCH16_1CS.     
******************************************************************************/

VISIBLE_PREFIX double STD_CALL parseEPOCH16_1CS (inString, epoch)
void *inString;
double *epoch;
{
  return parseEPOCH16_1 ((char *)inString, epoch);
}

/******************************************************************************
* parseEPOCH2CS.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL parseEPOCH2CS (inString)
void *inString;
{
  return parseEPOCH2 ((char *)inString);
}

/******************************************************************************
* parseEPOCH16_2CS.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL parseEPOCH16_2CS (inString, epoch)
void *inString;
double *epoch;
{
  return parseEPOCH16_2 ((char *)inString, epoch);
}

/******************************************************************************
* parseEPOCH3CS.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL parseEPOCH3CS (inString)
void *inString;
{
  return parseEPOCH3 ((char *)inString);
}

/******************************************************************************
* parseEPOCH16_3CS.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL parseEPOCH16_3CS (inString, epoch)
void *inString;
double *epoch;
{
  return parseEPOCH16_3 ((char *)inString, epoch);
}

/******************************************************************************
* parseEPOCH4CS.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL parseEPOCH4CS (inString)
void *inString;
{
  return parseEPOCH4 ((char *)inString);
}

/******************************************************************************
* parseEPOCH16_4CS.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL parseEPOCH16_4CS (inString, epoch)
void *inString;
double *epoch;
{
  return parseEPOCH16_4 ((char *)inString, epoch);
}

/******************************************************************************
* EPOCHtoUnixTimeCS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL EPOCHtoUnixTimeCS (epoch, unixTime, numTimes)
double *epoch;
double *unixTime;
int    numTimes;
{
  EPOCHtoUnixTime (epoch, unixTime, numTimes);
}

/******************************************************************************
* UnixTimetoEPOCHCS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL UnixTimetoEPOCHCS (unixTime, epoch, numTimes)
double *unixTime;
double *epoch;
int    numTimes;
{ 
  UnixTimetoEPOCH (unixTime, epoch, numTimes);
}

/******************************************************************************
* EPOCH16toUnixTimeCS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL EPOCH16toUnixTimeCS (epoch, unixTime, numTimes)
double *epoch;
double *unixTime;
int    numTimes;
{ 
  EPOCH16toUnixTime (epoch, unixTime, numTimes);
}

/******************************************************************************
* UnixTimetoEPOCH16CS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL UnixTimetoEPOCH16CS (unixTime, epoch16, numTimes)
double *unixTime;
double *epoch16;
int    numTimes;
{ 
  UnixTimetoEPOCH16 (unixTime, epoch16, numTimes);
}

/******************************************************************************
* TT2000toUnixTimeCS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL TT2000toUnixTimeCS (epoch, unixTime, numTimes)
long long *epoch;
double *unixTime;
int    numTimes;
{
  TT2000toUnixTime (epoch, unixTime, numTimes);
}

/******************************************************************************
* UnixTimetoTT2000CS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL UnixTimetoTT2000CS (unixTime, epoch, numTimes)
double *unixTime;
long long *epoch;
int    numTimes;
{ 
  UnixTimetoTT2000 (unixTime, epoch, numTimes);
}

/******************************************************************************
* encodeEPOCHCS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL encodeEPOCHCS (epoch, epString)
double epoch;
void *epString;
{ 
  encodeEPOCH (epoch, (char *)epString);
  return;
}

/******************************************************************************
* encodeEPOCH16CS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL encodeEPOCH16CS (epoch, epString)
double *epoch;
void *epString;
{
  encodeEPOCH16 (epoch, (char *)epString);
  return;
}

/******************************************************************************
* encodeEPOCH1CS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL encodeEPOCH1CS (epoch, epString)
double epoch;
void *epString;
{
  encodeEPOCH1 (epoch, (char *)epString);
  return;
}

/******************************************************************************   
* encodeEPOCH16_1CS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL encodeEPOCH16_1CS (epoch, epString)
double *epoch;
void *epString;
{
  encodeEPOCH16_1 (epoch, (char *)epString);
  return;
}

/******************************************************************************
* encodeEPOCH2CS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL encodeEPOCH2CS (epoch, epString)
double epoch;
void *epString;
{
  encodeEPOCH2 (epoch, (char *)epString);
  return;
}

/******************************************************************************   
* encodeEPOCH16_2CS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL encodeEPOCH16_2CS (epoch, epString)
double *epoch;
void *epString;
{ 
  encodeEPOCH16_2 (epoch, (char *)epString);
  return;
}

/******************************************************************************
* encodeEPOCH3CS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL encodeEPOCH3CS (epoch, epString)
double epoch;
void *epString;
{
  encodeEPOCH3 (epoch, (char *)epString);
  return;
}

/******************************************************************************   
* encodeEPOCH16_3CS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL encodeEPOCH16_3CS (epoch, epString)
double *epoch;
void *epString;
{
  encodeEPOCH16_3 (epoch, (char *)epString);
  return;
}

/******************************************************************************
* encodeEPOCH4CS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL encodeEPOCH4CS (epoch, epString)
double epoch;
void *epString;
{
  encodeEPOCH4 (epoch, (char *)epString);
  return;
}

/****************************************************************************** 
* encodeEPOCH16_4CS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL encodeEPOCH16_4CS (epoch, epString)
double *epoch;
void *epString;
{
  encodeEPOCH16_4 (epoch, (char *)epString);
  return;
}

/****************************************************************************** 
* encodeTT2000CS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL encodeTT2000CS (epoch, epString, style)
long long epoch;
void *epString;
int  style;
{
  CDF_TT2000_to_UTC_string (epoch, (char *)epString, style);
  return;
}

/******************************************************************************
* encodeEPOCHxCS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL encodeEPOCHxCS (epoch, format, encoded)
double epoch;
void *format;
void *encoded;
{
  encodeEPOCHx (epoch, (char *)format, (char *)encoded);
  return;
}

/******************************************************************************
* encodeEPOCH16_xCS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL encodeEPOCH16_xCS (epoch, format, encoded)
double *epoch;
void *format;
void *encoded;
{
  encodeEPOCH16_x (epoch, (char *)format, (char *)encoded);
  return;
}

/******************************************************************************
* computeEPOCHCS.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL computeEPOCHCS (year, month, day, hour, minute,
			 		       second, msec)
long year, month, day, hour, minute, second, msec;
{
  return computeEPOCH (year, month, day, hour, minute, second, msec);
}

/******************************************************************************
* computeEPOCH16CS.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL computeEPOCH16CS (year, month, day, hour, minute, 
                                                 second, msec, usec, nsec, psec, 
				                 epoch)
long year, month, day, hour, minute, second, msec, usec, nsec, psec;
double *epoch;
{
  return computeEPOCH16 (year, month, day, hour, minute, 
                         second, msec, usec, nsec, psec, epoch);
}

/******************************************************************************
* computeTT2000CS.
******************************************************************************/

VISIBLE_PREFIX long long STD_CALL computeTT2000CS (year, month, day, hour,
	       				           minute, second, millisec,
	       					   microsec, nanosec)
double year, month, day, hour, minute, second, millisec, microsec, nanosec;
{
  return CDF_TT2000_from_UTC_parts (year, month, day, hour, minute, second,
		 		    millisec, microsec, nanosec);
}

/******************************************************************************
* EPOCHbreakdownCS.
* Breaks an EPOCH value down into its component parts.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL EPOCHbreakdownCS (epoch, year, month, day, hour, minute,
				    second, msec)
double epoch;
long *year, *month, *day, *hour, *minute, *second, *msec;
{
  EPOCHbreakdown (epoch, year, month, day, hour, minute, second, msec);
  return;
}

/******************************************************************************
* EPOCH16breakdownCS. 
******************************************************************************/

VISIBLE_PREFIX void STD_CALL EPOCH16breakdownCS (epoch, year, month, day, hour, 
                                      minute, second, msec, usec, nsec, psec)
double *epoch;
long *year, *month, *day, *hour, *minute, *second;
long *msec, *usec, *nsec, *psec;
{
   EPOCH16breakdown (epoch, year, month, day, hour,
                     minute, second, msec, usec, nsec, psec);
   return;
}

/******************************************************************************
* TT2000breakdownCS.
* Breaks a TT2000 value down into its component parts.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL TT2000breakdownCS (epoch, year, month, day, hour,
	       					minute, second, millisec,
					       	microsec, nanosec)
long long epoch;
double *year, *month, *day, *hour, *minute, *second, *millisec, *microsec,
     *nanosec;
{
  CDF_TT2000_to_UTC_parts (epoch, year, month, day, hour, minute, second, 
	                   millisec, microsec, nanosec);
  return;
}

/******************************************************************************
* TT2000toEPOCHCS.
* Convert a TT2000 value to CDF EPOCH value.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL TT2000toEPOCHCS (epoch)
long long epoch;
{
  return CDF_TT2000_to_UTC_EPOCH (epoch);
}

/******************************************************************************
* TT2000toEPOCH16CS.
* Convert a TT2000 value to CDF EPOCH16 value.
******************************************************************************/

VISIBLE_PREFIX double STD_CALL TT2000toEPOCH16CS (tt2000, epoch16)
long long tt2000;
double *epoch16;
{
  return CDF_TT2000_to_UTC_EPOCH16 (tt2000, epoch16);
}

/******************************************************************************
* TT2000fromEPOCHCS.
* Convert a TT2000 value from CDF EPOCH value.
******************************************************************************/

VISIBLE_PREFIX long long STD_CALL TT2000fromEPOCHCS (epoch)
double epoch;
{
  return CDF_TT2000_from_UTC_EPOCH (epoch);
}

/******************************************************************************
* TT2000fromEPOCH16CS.
* Convert a TT2000 value from CDF EPOCH16 value.
******************************************************************************/

VISIBLE_PREFIX long long STD_CALL TT2000fromEPOCH16CS (epoch16)
double *epoch16;
{
  return CDF_TT2000_from_UTC_EPOCH16 (epoch16);
}

/******************************************************************************
* CDFgetLeapSecondsTableEnvVarCS.
* Get the date, year, month, day, in the last entry of the leap seconds table.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL CDFgetLeapSecondsTableEnvVarCS (table)
char *table;
{
  char *tmp;  
  tmp = (char *) CDFgetLeapSecondsTableEnvVar ();
  if (tmp != NULL) 
	 strncpy(table, tmp, (size_t) MINIMUM((int)strlen(tmp),
	                                      CDF_PATHNAME_LEN));
  else table = NULL;
}

/******************************************************************************
* CDFgetLastinLeapSecondsTableCS.
* Get the date, year, month, day, in the last entry of the leap seconds table.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL CDFgetLastDateinLeapSecondsTableCS (year, month, day)
long *year;
long *month;
long *day;
{
  CDFgetLastDateinLeapSecondsTable (year, month, day);
}

/******************************************************************************
* CDFgetLeapSecondsTableStatusCS.
* Get the status of leap seconds table being read from an environment variable
* defined file.
******************************************************************************/

VISIBLE_PREFIX int STD_CALL CDFgetLeapSecondsTableStatusCS ()
{
  return CDFgetLeapSecondsTableStatus ();
}

/******************************************************************************
* CDFgetRowsinLeapSecondsTableCS.
* Get the row count in the leap seconds table.
******************************************************************************/

VISIBLE_PREFIX int STD_CALL CDFgetRowsinLeapSecondsTableCS ()
{
  return CDFgetRowsinLeapSecondsTable ();
}

/******************************************************************************
* CDFgetLeapSecondsTableCS.
* Get the data contents form the leap seconds table.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL CDFgetLeapSecondsTableCS (double* table)
{
  int rows = CDFgetRowsinLeapSecondsTable();
  int i, j, ix;
  double** tmp;
  tmp = (double **) malloc (sizeof (double *) * rows);
  for (i = 0; i < rows; ++i)
	tmp[i] = (double *) malloc (sizeof (double) * 6);
  CDFfillLeapSecondsTable (tmp);
  ix = 0;
  for (i = 0; i < rows; ++i)
	for (j = 0; j < 6; ++j) {
      *(((double *) table)+ix) = tmp[i][j];
      ++ix;
	}
  for (i = 0; i < rows; ++i) free (tmp[i]);
  free (tmp);
}

/******************************************************************************
* CDFgetFileBackwardEnvVarCS.
******************************************************************************/

VISIBLE_PREFIX int STD_CALL CDFgetFileBackwardEnvVarCS ()
{
  return CDFgetFileBackwardEnvVar ();
}

/******************************************************************************
* CDFgetChecksumEnvVarCS.
******************************************************************************/

VISIBLE_PREFIX int STD_CALL CDFgetChecksumEnvVarCS ()
{
  return CDFgetChecksumEnvVar ();
}

/******************************************************************************
* CDFChecksumMethodCS.
******************************************************************************/

VISIBLE_PREFIX long STD_CALL CDFChecksumMethodCS (int flags)
{
  return CDFChecksumMethod ((Int32)flags);
}

/******************************************************************************
* CDFsetValidateCS.
******************************************************************************/

VISIBLE_PREFIX void STD_CALL CDFsetValidateCS (long validate)
{
  CDFsetValidate (validate);
  return;
}

/******************************************************************************
* CDFgetValidateCS.
******************************************************************************/

VISIBLE_PREFIX int STD_CALL CDFgetValidateCS ()
{
  return CDFgetValidate ();
}

/******************************************************************************
* CDFgetValidateDebugCS.
******************************************************************************/

VISIBLE_PREFIX int STD_CALL CDFgetValidateDebugCS ()
{
  return CDFgetValidateDebug ();
}

