#!/usr/bin/perl -w

#
# testPerlCDFii
#
#   This program tests the PerlCDF internal interface.  PerlCDF must be installed.
#   See README.
#
#  Usage:
#     perl testPerlCDFii.pl
#
# Written By:
#
#    Emily A. Greene
#    NSSDC / Hughes STX
#
#    17 October, 1995
#
#    Version 1.0
#    Modified by: Michael Liu  - 20 March, 1998
#    Version 2.0
#    Modified by: Michael Liu  - 07 January, 2005
#
#############################################################################
#
#  Translated from qst2ic.c v. 1.10 by J. Love
#
#  Differences include:
#     z Variable is 5 strings each 8 characters long not a 40 character buffer
#          tests changed accordingly
#     maxiumum records written is a variable
#     Arrays are all 1 Dimensional.  2 dimensionality handled within CDF.
#     Added section confirming rVARs recCount, etc.
#     Added a few informational messages
#     Tests Epoch routines
# Note: As Perl uses double, a variable of data type CDF_REAL4/FLOAT will 
#	cause some minor value deviation due to different floating-point type 
#	representations. If the value is returned from the CDF's C routine 
#	and compared with Perl value, an epsilon is used for checkng their
#	equality. 10**(-9) is used for double and 10**(-5) for 
#	float/real. No problem for CDF_REAL8 data type.
#     
#############################################################################

use strict;
use Math::BigInt;

BEGIN { unshift @INC,$ENV{"HOME"}.'/PerlCDF39_0/blib/arch',
                     $ENV{"HOME"}.'/PerlCDF39_0/blib/lib'; }
use CDF;

#############################################################################
# Test Epoch Routines
#############################################################################
print "Test Epoch...\n";
my $year = 1994;
my $month = 10;
my $day = 13;
my $hour = 1;
my $minute = 2;
my $second = 3;
my $msec = 987;

my $epoch = CDF::computeEPOCH ($year, $month, $day, $hour, $minute,  
                               $second, $msec);

my $epStringTrue = '13-Oct-1994 01:02:03.987';
my $epString1True = '19941013.0431017';
my $epString2True = '19941013010203';
my $epString3True = '1994-10-13T01:02:03.987Z';
my $epString4True = '1994-10-13T01:02:03.987';

my ($epString, $epString1, $epString2, $epString3, $epString4);

CDF::toEncodeEPOCH ($epoch, 0, $epString);
QuitEPOCH ("30.0") if ($epString ne $epStringTrue);
my (@epochd, @epochs);
$epochd[0] = $epoch;
$epochd[1] = $epoch;
CDF::toEncodeEPOCH (\@epochd, 4, \@epochs);
QuitEPOCH ("30.1") if ($epochs[0] ne $epString4True);
QuitEPOCH ("30.2") if ($epochs[1] ne $epString4True);

my $epochOut;
$epochOut = CDF::toParseEPOCH ($epString);
QuitEPOCH ("31.0") if ($epochOut != $epoch);
$epochs[0] = $epString;
$epochs[1] = $epString;
@epochd = CDF::toParseEPOCH (\@epochs);
QuitEPOCH ("31.1") if ($epochd[0] != $epoch);
QuitEPOCH ("31.2") if ($epochd[1] != $epoch);

#############################################################################
# Test Epoch16 Routines
#############################################################################
print "Test Epoch16...\n";

my $usec = 876;
my $nsec = 765;
my $psec = 654;
my @epoch16;
my $retco = CDF::computeEPOCH16 ($year, $month, $day, $hour, $minute, $second, 
                                 $msec, $usec, $nsec, $psec, \@epoch16);

my $ep16StringTrue = '13-Oct-1994 01:02:03.987.876.765.654';
my $ep16StringTruea = '13-Oct-1994 01:02:03.987.876.765.655';
my $ep16StringTruex = '13-Oct-1994 01:02:03.987';
my $ep16StringTruey = '13-Oct-1994 01:02:03.987Z';
my $ep16StringTruez = '13-Oct-1994 01:02:03.987.876.765.654Z';
my $ep16String1True = '19941013.043101711536640';
my $ep16String2True = '19941013010203';
my $ep16String3True = '1994-10-13T01:02:03.987.876.765.654Z';
my $ep16String4True = '1994-10-13T01:02:03.987876765654';

my ($ep16String, $ep16String1, $ep16String2, $ep16String3, $ep16String4);

my @epoch16o;
CDF::toEncodeEPOCH16 (\@epoch16, 0, $ep16String);
QuitEPOCH ("40.0") if ($ep16String ne $ep16StringTrue);

$epoch=CDF::toParseEPOCH16 ($ep16StringTrue, \@epoch16o);
CDF::toEncodeEPOCH16 (\@epoch16o, 0, $ep16String);
QuitEPOCH ("41.0") if ($epoch16[0] != $epoch16o[0]);
QuitEPOCH ("41.1") if ($epoch16[1] != $epoch16o[1]);

@epoch16o = ();
$epoch=CDF::toParseEPOCH16 ($ep16StringTruea, \@epoch16o);
CDF::toEncodeEPOCH16 (\@epoch16o, 0, $ep16String);
print "a: $ep16StringTruea => $ep16String\n";

@epoch16o = ();
$epoch=CDF::toParseEPOCH16 ($ep16StringTruex, \@epoch16o);
CDF::toEncodeEPOCH16 (\@epoch16o, 0, $ep16String);
print "x: $ep16StringTruex => $ep16String\n";

@epoch16o = ();
$epoch=CDF::toParseEPOCH16 ($ep16StringTruey, \@epoch16o);
CDF::toEncodeEPOCH16 (\@epoch16o, 0, $ep16String);
print "y: $ep16StringTruey => $ep16String\n";

@epoch16o = ();
$epoch=CDF::toParseEPOCH16 ($ep16StringTruez, \@epoch16o);
CDF::toEncodeEPOCH16 (\@epoch16o, 0, $ep16String);
print "z: $ep16StringTruez => $ep16String\n";

@epoch16o = ();
$epoch=CDF::toParseEPOCH16 ($ep16String3True, \@epoch16o);
QuitEPOCH ("41.3") if ($epoch16[0] != $epoch16o[0]) ;
QuitEPOCH ("41.4") if ($epoch16[1] != $epoch16o[1]) ;
CDF::toEncodeEPOCH16 (\@epoch16, 3, $ep16String);
QuitEPOCH ("41.5") if ($ep16String ne $ep16String3True);

@epoch16o = ();
$epoch=CDF::toParseEPOCH16 ($ep16String4True, \@epoch16o);
QuitEPOCH ("42.1") if ($epoch16[0] != $epoch16o[0]) ;
QuitEPOCH ("42.2") if ($epoch16[1] != $epoch16o[1]) ;
CDF::toEncodeEPOCH16 (\@epoch16, 4, $ep16String);
QuitEPOCH ("42.3") if ($ep16String ne $ep16String4True);
print "\n\n";
#############################################################################
# Test TT2000 Routines
#############################################################################
print "Test TT2000...\n";
$year = 2010;

my $tt2000=CDF::computeTT2000 ($year, $month, $day, $hour, $minute,  
                               $second, $msec, $usec, $nsec);
my $ttString0True = '13-Oct-2010 01:02:03.987876765';
my $ttString0Truex = '13-Oct-2010 01:02:03.987';
my $ttString0Truey= '13-Oct-2010 01:02:03.987Z';
my $ttString3True = '2010-10-13T01:02:03.987876765';
my $ttString3Truex = '2010-10-13T01:02:03.987Z';
my $ttString3Truey = '2010-10-13T01:02:03.987';
my $ttString4True = '2010-10-13T01:02:03.987876765Z';
my ($ttString0, $ttString3, $ttString4);

CDF::toEncodeTT2000 ($tt2000, 0, $ttString0);
QuitEPOCH ("50.0") if ($ttString0 ne $ttString0True);
my (@tt2000d, @tt2000s);
$tt2000d[0] = $tt2000;
$tt2000d[1] = $tt2000;
CDF::toEncodeTT2000 (\@tt2000d, 3, \@tt2000s);
QuitEPOCH ("50.1") if ($tt2000s[0] ne $ttString3True);
QuitEPOCH ("50.2") if ($tt2000s[1] ne $ttString3True);

my $tt2000Out;
my $tt2000n;

$tt2000Out=CDF::toParseTT2000 ($ttString3True);
QuitEPOCH ("51.0") if ($tt2000Out != $tt2000);

$tt2000s[0] = $ttString3True;
$tt2000s[1] = $ttString3True;
@tt2000d=CDF::toParseTT2000 (\@tt2000s);
QuitEPOCH ("51.3") if ($tt2000d[0] != $tt2000) ;
QuitEPOCH ("51.4") if ($tt2000d[1] != $tt2000) ;

my $tststr;
CDF::toEncodeTT2000 (CDF::toParseTT2000 ($ttString0Truex), 3, $tststr);
print "$ttString0Truex => $tststr\n";

$tt2000Out=CDF::toParseTT2000 ($ttString0Truex);
CDF::toEncodeTT2000 ($tt2000Out, 0, $tt2000n);
print "$ttString0Truex => $tt2000n\n";

$tt2000Out=CDF::toParseTT2000 ($ttString0Truey);
CDF::toEncodeTT2000 ($tt2000Out, 0, $tt2000n);
print "$ttString0Truey => $tt2000n\n";

$tt2000Out=CDF::toParseTT2000 ($ttString3Truex);
CDF::toEncodeTT2000 ($tt2000Out, 3, $tt2000n);
print "$ttString3Truex => $tt2000n\n";

$tt2000Out=CDF::toParseTT2000 ($ttString3Truey);
CDF::toEncodeTT2000 ($tt2000Out, 3, $tt2000n);
print "$ttString3Truey => $tt2000n\n";

#############################################################################
# Successful completion.
#############################################################################
print "All tests completed successfully\n";
exit;

#############################################################################
# QuitCDF.
#############################################################################
sub QuitCDF {
	my $where=@_;

  print "Aborting at $where ...\n";
  exit;
	
}#endsub QuitCDF

#############################################################################
#  QuitEPOCH
#############################################################################
sub QuitEPOCH {
  my ($where) = @_;
  print "Aborting at $where...test aborted.\n";
  exit;

}#endsub QuitEPOCH
