// $Id: CDFTools.java,v 1.2 2021/10/01 14:51:33 mhliu Exp $
/*
 * Copyright 1996-2014 United States Government as represented by the
 * Administrator of the National Aeronautics and Space Administration.
 * All Rights Reserved.
 */

package gsfc.nssdc.cdf;

import java.lang.*;
import java.util.*;
import java.io.*;
import gsfc.nssdc.cdf.util.*;

/**
 * CDFTools.java
 *
 *
 * Created: Tue Nov 24 16:14:50 1998
 *
 * @author Phil Williams
 * @version $Id: CDFTools.java,v 1.2 2021/10/01 14:51:33 mhliu Exp $
 */

public class CDFTools implements CDFConstants {
    
    private static String cdfbase   = System.getProperty("cdf.base");
    private static String separator = System.getProperty("file.separator");
    private static String cmdBase   = cdfbase+separator+"bin"+separator;

    // Where to send messages
    private static java.io.OutputStream err = System.err;

    public static final int NO_VALUES     = 0;
    public static final int NRV_VALUES    = 1;
    public static final int RV_VALUES     = 2;
    public static final int ALL_VALUES    = 3;
    public static final int NAMED_VALUES   = 4;

    public static final int NO_REPORTS    = 0;
    public static final int REPORT_ERRORS = 1;
    public static final int REPORT_WARNINGS = 2;
    public static final int REPORT_INFORMATION = 4;

    /**
     * skeletonTable produces a skeleton table from a CDF.  A
     * skeleton table is a text file which can be read by the
     * SkeletonCDF program to build a skeleton CDF.
     *
     * @param skeletonName is the pathname of the skeleton table
     *            to be created.  (Do not enter an extension because ".skt"
     *            is appended automatically).  If <B>null</B> is specified, the
     *            skeleton table is named &lt;cdfName&gt;.skt in the current
     *            directory<BR><BR>
     *
     * @param cdfName The pathname of the CDF from which the skeleton table
     *                will be created.  Do not enter an extension.<BR><BR>
     *               
     * @param log   Specifies whether or not messages are displayed as the
     *             program executes.<BR><BR>
     *            
     * @param format Specifies whether or not the FORMAT attribute is used
     *            when writing variable values (if the FORMAT attribute
     *            exists and an entry exists for the variable).<BR><BR>
     *
     * @param neg2posfp0
     *            Specifies whether or not -0.0 is converted to 0.0 by the CDF
     *            library when read from a CDF.  -0.0 is an illegal floating
     *            point value on VAXes and DEC Alphas running OpenVMS.<BR><BR>
     *
     * @param statistics Specifies whether or not caching statistics
     *            are displayed at the end of each CDF.<BR><BR>
     *                 
     * @param screen Specifies whether or not the skeleton table is displayed
     *            on the terminal screen (written to the "standard output").
     *            If not, the skeleton table is written to a file.<BR><BR>
     *                       
     * @param page If the skeleton table is being displayed on the terminal
     *            screen, specifies whether or not the output is displayed
     *            one page (screen) at a time.<BR><BR>
     *                       
     * @param values Specifies which variable values are to
     *           be put in the skeleton table.  It may be one of
     *           the following...<BR><BR>
     *           <DL>
     *              <DT>CDFTools.NO_VALUES
     *                 <DD>Ignore all NRV data values.
     *              <DT>CDFTools.NRV_VALUES
     *                 <DD>Put NRV data values in the skeleton table.
     *              <DT>CDFTools.RV_VALUES
     *                 <DD>Put RV variable values in the skeleton table.
     *               <DT>CDFTools.ALL_VALUES
     *                 <DD>Put all variable values in the skeleton table.
     *               <DT>CDFTools.NAMED_VALUES
     *                  <DD>Put named variables values in the skeleton table.
     *                      This requires that valueList be non-null
     *           </DL><BR><BR>
     *
     * @param valueList the named variables to list values.<BR><BR>
     *
     * @param zMode Specifies which zMode should be used.  May be one
     *            of the following...<BR><BR>
     *
     *               <DL>
     *               <DT>0
     *                 <DD>Indicates that zMode is disabled.
     *               <DT>1
     *                  <DD>Indicates that zMode/1 should be used (the
     *                       dimension variances of rVariables will be
     *                       preserved).
     *               <DT>2
     *                  <DD>Indicates that zMode/2 should be used (the
     *                       dimensions of rVariables having a variance
     *                       of NOVARY (false) are hidden.
     *               </DL><BR><BR>
     *
     * @param reportType Specifies the types of return status codes from the CDF
     *            library which should be reported/displayed.  <tt>report</tt>
     *            is a bit mask made up from the following
     *            CDFTools.NO_REPORTS, CDFTools.REPORT_ERRORS, 
     *            CDFTools.REPORT_WARNINGS and CDFTools.REPORT_INFORMATION<BR><BR>
     *
     * @param cacheSize The number of buffers to be used for the CDF's dotCDF
     *            file. Pass 0 to use the default. <BR> <BR>
     *
     * @throws java.io.IOException An exception is thrown if the file is bad
     *                             or non-existent
     * @throws InterruptedException An exception is thrown if an interrupt is
     *                              issued
     *
     * <B>Note:</B><BR>
     *    When running an application that use this static method, make sure
     *    this <b>cdf.base</b> system property is defined (to the cdf installation).
     *    For example, <BR>
     *    java -Dcdf.base=/home/cdf/cdf34_1-dist MyApp <BR>
     */
    public static void skeletonTable(String skeletonName,
				     String cdfName,
				     boolean log,
				     boolean format,
				     boolean neg2posfp0,
				     boolean statistics,
				     boolean screen,
				     boolean page,
				     int values,
				     String [] valueList,
				     int zMode,
				     int reportType,
				     int cacheSize)
	throws java.io.IOException, InterruptedException
    {
        if (cacheSize <= 0)
           CDFTools.skeletonTable(skeletonName, cdfName, log, format,
                                  neg2posfp0, statistics, screen, page, values,
                                  valueList, zMode, reportType, null);
        else {
          String cSize = new StringBuffer(""+cacheSize+"d").toString();
          CDFTools.skeletonTable(skeletonName, cdfName, log, format, 
                                 neg2posfp0, statistics, screen, page, values,
                                 valueList, zMode, reportType, cSize);
        } 
    }

    /**
     * skeletonTable produces a skeleton table from a CDF.  A
     * skeleton table is a text file which can be read by the
     * SkeletonCDF program to build a skeleton CDF.
     *
     * @param skeletonName is the pathname of the skeleton table
     *            to be created.  (Do not enter an extension because ".skt"
     *            is appended automatically).  If <B>null</B> is specified, the
     *            skeleton table is named &lt;cdfName&gt;.skt in the current
     *            directory<BR><BR>
     *
     * @param cdfName The pathname of the CDF from which the skeleton table
     *                will be created.  Do not enter an extension.<BR><BR>
     *               
     * @param log   Specifies whether or not messages are displayed as the
     *             program executes.<BR><BR>
     *            
     * @param format Specifies whether or not the FORMAT attribute is used
     *            when writing variable values (if the FORMAT attribute
     *            exists and an entry exists for the variable).<BR><BR>
     *
     * @param neg2posfp0
     *            Specifies whether or not -0.0 is converted to 0.0 by the CDF
     *            library when read from a CDF.  -0.0 is an illegal floating
     *            point value on VAXes and DEC Alphas running OpenVMS.<BR><BR>
     *
     * @param statistics Specifies whether or not caching statistics
     *            are displayed at the end of each CDF.<BR><BR>
     *                 
     * @param screen Specifies whether or not the skeleton table is displayed
     *            on the terminal screen (written to the "standard output").
     *            If not, the skeleton table is written to a file.<BR><BR>
     *                       
     * @param page If the skeleton table is being displayed on the terminal
     *            screen, specifies whether or not the output is displayed
     *            one page (screen) at a time.<BR><BR>
     *                       
     * @param values Specifies which variable values are to
     *           be put in the skeleton table.  It may be one of
     *           the following...<BR><BR>
     *           <DL>
     *              <DT>CDFTools.NO_VALUES
     *                 <DD>Ignore all NRV data values.
     *              <DT>CDFTools.NRV_VALUES
     *                 <DD>Put NRV data values in the skeleton table.
     *              <DT>CDFTools.RV_VALUES
     *                 <DD>Put RV variable values in the skeleton table.
     *               <DT>CDFTools.ALL_VALUES
     *                 <DD>Put all variable values in the skeleton table.
     *               <DT>CDFTools.NAMED_VALUES
     *                  <DD>Put named variables values in the skeleton table.
     *                      This requires that valueList be non-null
     *           </DL><BR><BR>
     *
     * @param valueList the named variables to list values.<BR><BR>
     *
     * @param zMode Specifies which zMode should be used.  May be one
     *            of the following...<BR><BR>
     *
     *               <DL>
     *               <DT>0
     *                 <DD>Indicates that zMode is disabled.
     *               <DT>1
     *                  <DD>Indicates that zMode/1 should be used (the
     *                       dimension variances of rVariables will be
     *                       preserved).
     *               <DT>2
     *                  <DD>Indicates that zMode/2 should be used (the
     *                       dimensions of rVariables having a variance
     *                       of NOVARY (false) are hidden.
     *               </DL><BR><BR>
     *
     * @param reportType Specifies the types of return status codes from the CDF
     *            library which should be reported/displayed.  <tt>report</tt>
     *            is a bit mask made up from the following
     *            CDFTools.NO_REPORTS, CDFTools.REPORT_ERRORS, 
     *            CDFTools.REPORT_WARNINGS and CDFTools.REPORT_INFORMATION<BR><BR>
     *
     * @param cacheSizeD The number of buffers to be used for the CDF's dotCDF
     *            file. Pass 0 to use the default. <BR> <BR>
     *
     * @param cacheSizeS The number of buffers to be used for the CDF's staging
     *            file. Pass 0 to use the default. <BR> <BR>
     *
     * @throws java.io.IOException An exception is thrown if the file is bad
     *                             or non-existent
     * @throws InterruptedException An exception is thrown if an interrupt is
     *                              issued
     *
     * <B>Note:</B><BR>
     *    When running an application that use this static method, make sure
     *    this <b>cdf.base</b> system property is defined (to the cdf installation).
     *    For example, <BR>
     *    java -Dcdf.base=/home/cdf/cdf34_1-dist MyApp <BR>
     */
    public static void skeletonTable(String skeletonName,
				     String cdfName,
				     boolean log,
				     boolean format,
				     boolean neg2posfp0,
				     boolean statistics,
				     boolean screen,
				     boolean page,
				     int values,
				     String [] valueList,
				     int zMode,
				     int reportType,
				     int cacheSizeD,
				     int cacheSizeS)
	throws java.io.IOException, InterruptedException
    {
        if (cacheSizeD <= 0 && cacheSizeS <= 0)
           CDFTools.skeletonTable(skeletonName, cdfName, log, format,
                                  neg2posfp0, statistics, screen, page, values,
                                  valueList, zMode, reportType, null);
        else {
          StringBuffer cs = new StringBuffer();
          if (cacheSizeD > 0) cs = cs.append(""+cacheSizeD+"d");
          if (cacheSizeS > 0) {
            if (cacheSizeD > 0) cs = cs.append(","+cacheSizeS+"s");
            else cs = cs.append(""+cacheSizeS+"s");
          }
          CDFTools.skeletonTable(skeletonName, cdfName, log, format, 
                                 neg2posfp0, statistics, screen, page, values,
                                 valueList, zMode, reportType, cs.toString());
        } 
    }

    /**
     * skeletonTable produces a skeleton table from a CDF.  A
     * skeleton table is a text file which can be read by the
     * SkeletonCDF program to build a skeleton CDF.
     *
     * @param skeletonName is the pathname of the skeleton table
     *            to be created.  (Do not enter an extension because ".skt"
     *            is appended automatically).  If <B>null</B> is specified, the
     *            skeleton table is named &lt;cdfName&gt;.skt in the current
     *            directory<BR><BR>
     *
     * @param cdfName The pathname of the CDF from which the skeleton table
     *                will be created.  Do not enter an extension.<BR><BR>
     *               
     * @param log   Specifies whether or not messages are displayed as the
     *             program executes.<BR><BR>
     *            
     * @param format Specifies whether or not the FORMAT attribute is used
     *            when writing variable values (if the FORMAT attribute
     *            exists and an entry exists for the variable).<BR><BR>
     *
     * @param neg2posfp0
     *            Specifies whether or not -0.0 is converted to 0.0 by the CDF
     *            library when read from a CDF.  -0.0 is an illegal floating
     *            point value on VAXes and DEC Alphas running OpenVMS.<BR><BR>
     *
     * @param statistics Specifies whether or not caching statistics
     *            are displayed at the end of each CDF.<BR><BR>
     *                 
     * @param screen Specifies whether or not the skeleton table is displayed
     *            on the terminal screen (written to the "standard output").
     *            If not, the skeleton table is written to a file.<BR><BR>
     *                       
     * @param page If the skeleton table is being displayed on the terminal
     *            screen, specifies whether or not the output is displayed
     *            one page (screen) at a time.<BR><BR>
     *                       
     * @param values Specifies which variable values are to
     *           be put in the skeleton table.  It may be one of
     *           the following...<BR><BR>
     *           <DL>
     *              <DT>CDFTools.NO_VALUES
     *                 <DD>Ignore all NRV data values.
     *              <DT>CDFTools.NRV_VALUES
     *                 <DD>Put NRV data values in the skeleton table.
     *              <DT>CDFTools.RV_VALUES
     *                 <DD>Put RV variable values in the skeleton table.
     *               <DT>CDFTools.ALL_VALUES
     *                 <DD>Put all variable values in the skeleton table.
     *               <DT>CDFTools.NAMED_VALUES
     *                  <DD>Put named variables values in the skeleton table.
     *                      This requires that valueList be non-null
     *           </DL><BR><BR>
     *
     * @param valueList the named variables to list values.<BR><BR>
     *
     * @param zMode Specifies which zMode should be used.  May be one
     *            of the following...<BR><BR>
     *
     *               <DL>
     *               <DT>0
     *                 <DD>Indicates that zMode is disabled.
     *               <DT>1
     *                  <DD>Indicates that zMode/1 should be used (the
     *                       dimension variances of rVariables will be
     *                       preserved).
     *               <DT>2
     *                  <DD>Indicates that zMode/2 should be used (the
     *                       dimensions of rVariables having a variance
     *                       of NOVARY (false) are hidden.
     *               </DL><BR><BR>
     *
     * @param reportType Specifies the types of return status codes from the CDF
     *            library which should be reported/displayed.  <tt>report</tt>
     *            is a bit mask made up from the following
     *            CDFTools.NO_REPORTS, CDFTools.REPORT_ERRORS, 
     *            CDFTools.REPORT_WARNINGS and CDFTools.REPORT_INFORMATION<BR><BR>
     *
     * @param cacheSizeD The number of buffers to be used for the CDF's dotCDF
     *            file. Pass 0 to use the default. <BR> <BR>
     *
     * @param cacheSizeS The number of buffers to be used for the CDF's staging
     *            file. Pass 0 to use the default. <BR> <BR>
     *
     * @param cacheSizeC The number of buffers to be used for the CDF's
     *            compression scratch file. Pass 0 to use the default. <BR> <BR>
     *
     * @throws java.io.IOException An exception is thrown if the file is bad
     *                             or non-existent
     * @throws InterruptedException An exception is thrown if an interrupt is
     *                              issued
     *
     * <B>Note:</B><BR>
     *    When running an application that use this static method, make sure
     *    this <b>cdf.base</b> system property is defined (to the cdf installation).
     *    For example, <BR>
     *    java -Dcdf.base=/home/cdf/cdf34_1-dist MyApp <BR>
     */
    public static void skeletonTable(String skeletonName,
				     String cdfName,
				     boolean log,
				     boolean format,
				     boolean neg2posfp0,
				     boolean statistics,
				     boolean screen,
				     boolean page,
				     int values,
				     String [] valueList,
				     int zMode,
				     int reportType,
				     int cacheSizeD,
				     int cacheSizeS,
				     int cacheSizeC)
	throws java.io.IOException, InterruptedException
    {
        if (cacheSizeD <= 0 && cacheSizeS <= 0 && cacheSizeC <= 0)
           CDFTools.skeletonTable(skeletonName, cdfName, log, format,
                                  neg2posfp0, statistics, screen, page, values,
                                  valueList, zMode, reportType, null);
        else {
          StringBuffer cs = new StringBuffer();
          if (cacheSizeD > 0) cs = cs.append(""+cacheSizeD+"d");
          if (cacheSizeS > 0) {
            if (cacheSizeD > 0) cs = cs.append(","+cacheSizeS+"s");
            else cs = cs.append(""+cacheSizeS+"s");
          }
          if (cacheSizeC > 0) {
            if (cacheSizeD > 0 || cacheSizeS > 0)
              cs = cs.append(","+cacheSizeC+"c");
            else cs = cs.append(""+cacheSizeC+"c");
          }
          CDFTools.skeletonTable(skeletonName, cdfName, log, format, 
                                 neg2posfp0, statistics, screen, page, values,
                                 valueList, zMode, reportType, cs.toString());
        } 
    }

    /**
     * skeletonTable produces a skeleton table from a CDF.  A
     * skeleton table is a text file which can be read by the
     * SkeletonCDF program to build a skeleton CDF.
     *
     * @param skeletonName is the pathname of the skeleton table
     *            to be created.  (Do not enter an extension because ".skt"
     *            is appended automatically).  If <B>null</B> is specified, the
     *            skeleton table is named &lt;cdfName&gt;.skt in the current
     *            directory<BR><BR>
     *
     * @param cdfName The pathname of the CDF from which the skeleton table
     *                will be created.  Do not enter an extension.<BR><BR>
     *               
     * @param log   Specifies whether or not messages are displayed as the
     *             program executes.<BR><BR>
     *            
     * @param format Specifies whether or not the FORMAT attribute is used
     *            when writing variable values (if the FORMAT attribute
     *            exists and an entry exists for the variable).<BR><BR>
     *
     * @param neg2posfp0
     *            Specifies whether or not -0.0 is converted to 0.0 by the CDF
     *            library when read from a CDF.  -0.0 is an illegal floating
     *            point value on VAXes and DEC Alphas running OpenVMS.<BR><BR>
     *
     * @param statistics Specifies whether or not caching statistics
     *            are displayed at the end of each CDF.<BR><BR>
     *                 
     * @param screen Specifies whether or not the skeleton table is displayed
     *            on the terminal screen (written to the "standard output").
     *            If not, the skeleton table is written to a file.<BR><BR>
     *                       
     * @param page If the skeleton table is being displayed on the terminal
     *            screen, specifies whether or not the output is displayed
     *            one page (screen) at a time.<BR><BR>
     *                       
     * @param values Specifies which variable values are to
     *           be put in the skeleton table.  It may be one of
     *           the following...<BR><BR>
     *           <DL>
     *              <DT>CDFTools.NO_VALUES
     *                 <DD>Ignore all NRV data values.
     *              <DT>CDFTools.NRV_VALUES
     *                 <DD>Put NRV data values in the skeleton table.
     *              <DT>CDFTools.RV_VALUES
     *                 <DD>Put RV variable values in the skeleton table.
     *               <DT>CDFTools.ALL_VALUES
     *                 <DD>Put all variable values in the skeleton table.
     *               <DT>CDFTools.NAMED_VALUES
     *                  <DD>Put named variables values in the skeleton table.
     *                      This requires that valueList be non-null
     *           </DL><BR><BR>
     *
     * @param valueList the named variables to list values.<BR><BR>
     *
     * @param zMode Specifies which zMode should be used.  May be one
     *            of the following...<BR><BR>
     *
     *               <DL>
     *               <DT>0
     *                 <DD>Indicates that zMode is disabled.
     *               <DT>1
     *                  <DD>Indicates that zMode/1 should be used (the
     *                       dimension variances of rVariables will be
     *                       preserved).
     *               <DT>2
     *                  <DD>Indicates that zMode/2 should be used (the
     *                       dimensions of rVariables having a variance
     *                       of NOVARY (false) are hidden.
     *               </DL><BR><BR>
     *
     * @param reportType Specifies the types of return status codes from the CDF
     *            library which should be reported/displayed.  <tt>report</tt>
     *            is a bit mask made up from the following
     *            CDFTools.NO_REPORTS, CDFTools.REPORT_ERRORS, 
     *            CDFTools.REPORT_WARNINGS and CDFTools.REPORT_INFORMATION<BR><BR>
     *
     * @param cacheSize The number of buffers to be used for the CDF's dotCDF
     *            file, staging file, and compression scratch file. This field
     *            used only in the CDF access process will not change anything
     *            to the contents of the skeleton table. Large number(s) will
     *            likely reduce physical I/Os if variable data are involved. 
     *            If this field is null, the default cache sizes chosen by
     *            the CDF library are used.  The cache sizes are specified
     *            with a comma-separated list of &lt;number&gt;&lt;type&gt; pairs where
     *            &lt;number&gt; is the number of cache buffers and &lt;type&gt; is the
     *            type of file.  The file &lt;type&gt;'s are as follows: `d' for
     *            the dotCDF file, `s' for the staging file, and `c' for the
     *            compression scratch file.  For example, `200d,100s'
     *            specifies 200 cache buffers for the dotCDF file and 100
     *            cache buffers for the staging file.  The dotCDF file cache
     *            size can also be specified without the `d' &lt;type&gt; for
     *            compatibility with older CDF releases (eg. `200,100s').
     *            Note that not all of the file types must be specified.
     *            Those not specified will receive a default cache size.<BR><BR>
     *
     * @throws java.io.IOException An exception is thrown if the file is bad
     *                             or non-existent
     * @throws InterruptedException An exception is thrown if an interrupt is
     *                              issued
     *
     * <B>Note:</B><BR>
     *    When running an application that use this static method, make sure
     *    this <b>cdf.base</b> system property is defined (to the cdf installation).
     *    For example, <BR>
     *    java -Dcdf.base=/home/cdf/cdf34_1-dist MyApp <BR>
     */
    public static void skeletonTable(String skeletonName,
				     String cdfName,
				     boolean log,
				     boolean format,
				     boolean neg2posfp0,
				     boolean statistics,
				     boolean screen,
				     boolean page,
				     int values,
				     String [] valueList,
				     int zMode,
				     int reportType,
				     String cacheSize)
	throws java.io.IOException, InterruptedException
    {

        CDFTools.skeletonTable(skeletonName, cdfName, log, format,
                               neg2posfp0, statistics, screen, page, values,
                               valueList, zMode, reportType, cacheSize, null);
    }

    /**
     * skeletonTable produces a skeleton table from a CDF.  A
     * skeleton table is a text file which can be read by the
     * SkeletonCDF program to build a skeleton CDF.
     *
     * @param skeletonName is the pathname of the skeleton table
     *            to be created.  (Do not enter an extension because ".skt"
     *            is appended automatically).  If <B>null</B> is specified, the
     *            skeleton table is named &lt;cdfName&gt;.skt in the current
     *            directory<BR><BR>
     *
     * @param cdfName The pathname of the CDF from which the skeleton table
     *                will be created.  Do not enter an extension.<BR><BR>
     *               
     * @param log   Specifies whether or not messages are displayed as the
     *             program executes.<BR><BR>
     *            
     * @param format Specifies whether or not the FORMAT attribute is used
     *            when writing variable values (if the FORMAT attribute
     *            exists and an entry exists for the variable). Metadata
     *            always uses noformat form.<BR><BR>
     *
     * @param neg2posfp0
     *            Specifies whether or not -0.0 is converted to 0.0 by the CDF
     *            library when read from a CDF.  -0.0 is an illegal floating
     *            point value on VAXes and DEC Alphas running OpenVMS.<BR><BR>
     *
     * @param statistics Specifies whether or not caching statistics
     *            are displayed at the end of each CDF.<BR><BR>
     *                 
     * @param screen Specifies whether or not the skeleton table is displayed
     *            on the terminal screen (written to the "standard output").
     *            If not, the skeleton table is written to a file.<BR><BR>
     *                       
     * @param page If the skeleton table is being displayed on the terminal
     *            screen, specifies whether or not the output is displayed
     *            one page (screen) at a time.<BR><BR>
     *                       
     * @param values Specifies which variable values are to
     *           be put in the skeleton table.  It may be one of
     *           the following...<BR><BR>
     *           <DL>
     *              <DT>CDFTools.NO_VALUES
     *                 <DD>Ignore all NRV data values.
     *              <DT>CDFTools.NRV_VALUES
     *                 <DD>Put NRV data values in the skeleton table.
     *              <DT>CDFTools.RV_VALUES
     *                 <DD>Put RV variable values in the skeleton table.
     *               <DT>CDFTools.ALL_VALUES
     *                 <DD>Put all variable values in the skeleton table.
     *               <DT>CDFTools.NAMED_VALUES
     *                  <DD>Put named variables values in the skeleton table.
     *                      This requires that valueList be non-null
     *           </DL><BR><BR>
     *
     * @param valueList the named variables to list values.<BR><BR>
     *
     * @param zMode Specifies which zMode should be used.  May be one
     *            of the following...<BR><BR>
     *
     *               <DL>
     *               <DT>0
     *                 <DD>Indicates that zMode is disabled.
     *               <DT>1
     *                  <DD>Indicates that zMode/1 should be used (the
     *                       dimension variances of rVariables will be
     *                       preserved).
     *               <DT>2
     *                  <DD>Indicates that zMode/2 should be used (the
     *                       dimensions of rVariables having a variance
     *                       of NOVARY (false) are hidden.
     *               </DL><BR><BR>
     *
     * @param reportType Specifies the types of return status codes from the CDF
     *            library which should be reported/displayed.  <tt>report</tt>
     *            is a bit mask made up from the following
     *            CDFTools.NO_REPORTS, CDFTools.REPORT_ERRORS, 
     *            CDFTools.REPORT_WARNINGS and CDFTools.REPORT_INFORMATION<BR><BR>
     *
     * @param cacheSize The number of buffers to be used for the CDF's dotCDF
     *            file, staging file, and compression scratch file. This field
     *            used only in the CDF access process will not change anything
     *            to the contents of the skeleton table. Large number(s) will
     *            likely reduce physical I/Os if variable data are involved. 
     *            If this field is null, the default cache sizes chosen by
     *            the CDF library are used.  The cache sizes are specified
     *            with a comma-separated list of &lt;number&gt;&lt;type&gt; pairs where
     *            &lt;number&gt; is the number of cache buffers and &lt;type&gt; is the
     *            type of file.  The file &lt;type&gt;'s are as follows: `d' for
     *            the dotCDF file, `s' for the staging file, and `c' for the
     *            compression scratch file.  For example, `200d,100s'
     *            specifies 200 cache buffers for the dotCDF file and 100
     *            cache buffers for the staging file.  The dotCDF file cache
     *            size can also be specified without the `d' &lt;type&gt; for
     *            compatibility with older CDF releases (eg. `200,100s').
     *            Note that not all of the file types must be specified.
     *            Those not specified will receive a default cache size.<BR><BR>
     * @param advFormat This new argument supersedes the prior format argument.
     *            It accepts one of "best", "no", "metaonly", "dataonly" or "all"
     *            options. "best" option all use the best approach to encode
     *            floating point data with a higher precision, even larger than
     *            the FORMAT attribute for both metadata and data. "no" option
     *            will not use and FORMAT attribute, "metaonly" will use
     *            FORMAT attribute for metdata only. "dataonly" will use
     *            FORMAT attribute for data only. "all" option will use
     *            FORMAT attribute for both metadata and data. "best" option
     *            is the default. If null is passed in, which is the old 
     *            method calling this method, it is changed to "best" option 
     *            that works as C based tool would do.
     *
     *
     * @throws java.io.IOException An exception is thrown if the file is bad
     *                             or non-existent
     * @throws InterruptedException An exception is thrown if an interrupt is
     *                              issued
     *
     * <B>Note:</B><BR>
     *    When running an application that use this static method, make sure
     *    this <b>cdf.base</b> system property is defined (to the cdf installation).
     *    For example, <BR>
     *    java -Dcdf.base=/home/cdf/cdf34_1-dist MyApp <BR>
     */
    public static void skeletonTable(String skeletonName,
				     String cdfName,
				     boolean log,
				     boolean format,
				     boolean neg2posfp0,
				     boolean statistics,
				     boolean screen,
				     boolean page,
				     int values,
				     String [] valueList,
				     int zMode,
				     int reportType,
				     String cacheSize,
				     String advFormat)
	throws java.io.IOException, InterruptedException
    {

	// StringBuffers are faster than using String and +
	StringBuffer cmd = new StringBuffer();  
	cmd.append(cmdBase).append("skeletontable ");
	
	if (skeletonName != null) {
	    cmd.append("-skeleton ");
	    cmd.append(skeletonName+" ");
	}

	switch (values) {
	case NO_VALUES:
	    break;
	case NRV_VALUES:
	    cmd.append("-values nrv ");
	    break;
	case RV_VALUES:
	    cmd.append("-values rv ");
	    break;
	case ALL_VALUES:
	    cmd.append("-values all");
	    break;
	case NAMED_VALUES:
	    cmd.append("-values ");
	    for (int i=0; i < valueList.length; i++)
		cmd.append(valueList[i]+",");
	    cmd.setLength(cmd.length()-1);
	    cmd.append(" ");
	    break;
	}

	cmd.append("-");
	if (!log)
	    cmd.append("no");
	cmd.append("log ");
	    
	cmd.append("-zmode "+zMode+" ");

	cmd.append("-");
	if (!format)
	    cmd.append("no");
	cmd.append("format ");

        if (advFormat != null) {
            if (advFormat.toLowerCase().equals("best"))
              cmd.append("-advformat \"best \"");
            else if (advFormat.toLowerCase().equals("no"))
              cmd.append("-advformat \"no \"");
            else if (advFormat.toLowerCase().equals("metaonly"))
              cmd.append("-advformat \"metaonly \"");
            else if (advFormat.toLowerCase().equals("dataonly"))
              cmd.append("-advformat \"dataonly \"");
            else if (advFormat.toLowerCase().equals("all"))
              cmd.append("-advformat \"all \"");
        } else
          cmd.append("-advformat \"best \"");

	cmd.append("-");
	if (!neg2posfp0)
	    cmd.append("no");
	cmd.append("neg2posfp0 ");

	if (reportType != 0) {
	    cmd.append("-report \"");
	    if ((reportType & REPORT_ERRORS) != 0)
		cmd.append("errors,");
	    if ((reportType & REPORT_WARNINGS) != 0)
		cmd.append("warnings,");
	    if ((reportType & REPORT_INFORMATION) != 0)
		cmd.append("informationals,");
	    cmd.setLength(cmd.length()-1); // chop off the last 2 chars
	    cmd.append("\" ");
	}

	if (cacheSize != null) {
	    cmd.append("-cache ");
	    cmd.append("\""+cacheSize+"\" ");
	}

	cmd.append("-");
	if (!statistics)
	    cmd.append("no");
	cmd.append("statistics ");

	cmd.append("-");
	if (!screen)
	    cmd.append("no");
	cmd.append("screen ");

	cmd.append("-");
	if (!page)
	    cmd.append("no");
	cmd.append("page ");
	
	cmd.append(cdfName);

	Process p = Runtime.getRuntime().exec(cmd.toString());
	int retcode = p.waitFor();

	// Echo any errors
	BufferedReader inStream = 
	    new BufferedReader(new InputStreamReader(p.getInputStream(), "UTF-8"));
	String line;
	while ((line = inStream.readLine()) != null) {
	    if (err instanceof PrintStream)
		((PrintStream)err).println(line);
	}
    }

    /**
     * skeletonCDF produces a CDF file from a skeleton table.  A
     * skeleton table is a text file which can be made through a text
     * editor, or by SkeletonCDF program from an existng CDF.
     *
     * @param skeletonName is the pathname of the skeleton table to be used
     *            to create a CDF.  (Do not enter an extension because ".skt"
     *            is appended automatically).  If <B>null</B> is specified, the
     *            skeleton table is named &lt;cdfName&gt;.skt in the current
     *            directory<BR><BR>
     *
     * @param cdfName The pathname of the created CDF from the skeleton table.
     *                Do not enter an extension.<BR><BR>
     *
     * @param delete specifies whether or not the CDF should be deleted
     *               if it already exists.
     *
     * @param log   Specifies whether or not messages are displayed as the
     *             program executes.<BR><BR>
     *
     * @param neg2posfp0
     *            Specifies whether or not -0.0 is converted to 0.0 by the CDF
     *            library when read from a CDF.  -0.0 is an illegal floating
     *            point value on VAXes and DEC Alphas running OpenVMS.<BR><BR>
     *
     * @param statistics Specifies whether or not caching statistics
     *            are displayed at the end of each CDF.<BR><BR>
     *
     * @param zMode Specifies which zMode should be used.  May be one
     *            of the following...<BR><BR>
     *
     *               <DL>
     *               <DT>0
     *                 <DD>Indicates that zMode is disabled.
     *               <DT>1
     *                  <DD>Indicates that zMode/1 should be used (the
     *                       dimension variances of rVariables will be
     *                       preserved).
     *               <DT>2
     *                  <DD>Indicates that zMode/2 should be used (the
     *                       dimensions of rVariables having a variance
     *                       of NOVARY (false) are hidden.
     *               </DL><BR><BR>
     *
     * @param reportType Specifies the types of return status codes from the CDF
     *            library which should be reported/displayed.  <tt>report</tt>
     *            is a bit mask made up from the following
     *            CDFTools.NO_REPORTS, CDFTools.REPORT_ERRORS, 
     *            CDFTools.REPORT_WARNINGS and CDFTools.REPORT_INFORMATION<BR><BR>
     *
     * @param cacheSizeD The number of buffers to be used for the CDF's
     *            dotCDF file. Pass 0 to use the default. <BR> <BR>
     *
     * @throws java.io.IOException An exception is thrown if the file is bad
     *                             or non-existent
     * @throws InterruptedException An exception is thrown if an interrupt is
     *                              issued
     *
     * <B>Note:</B><BR>
     *    When running an application that use this static method, make sure
     *    this <b>cdf.base</b> system property is defined (to the cdf installation).
     *    For example, <BR>
     *    java -Dcdf.base=/home/cdf/cdf34_1-dist MyApp  <BR>
     *
     */
    public static void skeletonCDF(String skeletonName,
				   String cdfName,
				   boolean delete,
				   boolean log,
				   boolean neg2posfp0,
				   boolean statistics,
				   int zMode,
				   int reportType,
				   int cacheSizeD)
	throws java.io.IOException, InterruptedException
    {
        if (cacheSizeD <= 0)
           CDFTools.skeletonCDF(skeletonName, cdfName, delete, log,
                                neg2posfp0, statistics, zMode, reportType,
                                null);
        else {
          StringBuffer sb = new StringBuffer(""+cacheSizeD+"d");
          CDFTools.skeletonCDF(skeletonName, cdfName, delete, log,
                               neg2posfp0, statistics, zMode, reportType,
                               sb.toString());
        }
    }

    /**
     * skeletonCDF produces a CDF file from a skeleton table.  A
     * skeleton table is a text file which can be made through a text
     * editor, or by SkeletonCDF program from an existng CDF.
     *
     * @param skeletonName is the pathname of the skeleton table to be used
     *            to create a CDF.  (Do not enter an extension because ".skt"
     *            is appended automatically).  If <B>null</B> is specified, the
     *            skeleton table is named &lt;cdfName&gt;.skt in the current
     *            directory<BR><BR>
     *
     * @param cdfName The pathname of the created CDF from the skeleton table.
     *                Do not enter an extension.<BR><BR>
     *
     * @param delete specifies whether or not the CDF should be deleted
     *               if it already exists.
     *
     * @param log   Specifies whether or not messages are displayed as the
     *             program executes.<BR><BR>
     *
     * @param neg2posfp0
     *            Specifies whether or not -0.0 is converted to 0.0 by the CDF
     *            library when read from a CDF.  -0.0 is an illegal floating
     *            point value on VAXes and DEC Alphas running OpenVMS.<BR><BR>
     *
     * @param statistics Specifies whether or not caching statistics
     *            are displayed at the end of each CDF.<BR><BR>
     *
     * @param zMode Specifies which zMode should be used.  May be one
     *            of the following...<BR><BR>
     *
     *               <DL>
     *               <DT>0
     *                 <DD>Indicates that zMode is disabled.
     *               <DT>1
     *                  <DD>Indicates that zMode/1 should be used (the
     *                       dimension variances of rVariables will be
     *                       preserved).
     *               <DT>2
     *                  <DD>Indicates that zMode/2 should be used (the
     *                       dimensions of rVariables having a variance
     *                       of NOVARY (false) are hidden.
     *               </DL><BR><BR>
     *
     * @param reportType Specifies the types of return status codes from the CDF
     *            library which should be reported/displayed.  <tt>report</tt>
     *            is a bit mask made up from the following
     *            CDFTools.NO_REPORTS, CDFTools.REPORT_ERRORS, 
     *            CDFTools.REPORT_WARNINGS and CDFTools.REPORT_INFORMATION<BR><BR>
     *
     * @param cacheSizeD The number of buffers to be used for the CDF's
     *            dotCDF file. Pass 0 to use the default. <BR> <BR>
     *
     * @param cacheSizeS The number of buffers to be used for the CDF's
     *            staging file. Pass 0 to use the default. <BR> <BR>
     *
     * @throws java.io.IOException An exception is thrown if the file is bad
     *                             or non-existent
     * @throws InterruptedException An exception is thrown if an interrupt is
     *                              issued
     *
     * <B>Note:</B><BR>
     *    When running an application that use this static method, make sure
     *    this <b>cdf.base</b> system property is defined (to the cdf installation).
     *    For example, <BR>
     *    java -Dcdf.base=/home/cdf/cdf34_1-dist MyApp  <BR>
     *
     */
    public static void skeletonCDF(String skeletonName,
				   String cdfName,
				   boolean delete,
				   boolean log,
				   boolean neg2posfp0,
				   boolean statistics,
				   int zMode,
				   int reportType,
				   int cacheSizeD,
				   int cacheSizeS)
	throws java.io.IOException, InterruptedException
    {
        if (cacheSizeD <= 0 && cacheSizeS <= 0)
           CDFTools.skeletonCDF(skeletonName, cdfName, delete, log,
                                neg2posfp0, statistics, zMode, reportType,
                                null);
        else {
          StringBuffer sb = new StringBuffer();
          if (cacheSizeD > 0) sb = sb.append(""+cacheSizeD+"d");
          if (cacheSizeS > 0) {
            if (cacheSizeD > 0) sb = sb.append(","+cacheSizeS+"s");
            else sb = sb.append(""+cacheSizeS+"s");
          }
          CDFTools.skeletonCDF(skeletonName, cdfName, delete, log,
                               neg2posfp0, statistics, zMode, reportType,
                               sb.toString());
        }
    }

    /**
     * skeletonCDF produces a CDF file from a skeleton table.  A
     * skeleton table is a text file which can be made through a text
     * editor, or by SkeletonCDF program from an existng CDF.
     *
     * @param skeletonName is the pathname of the skeleton table to be used
     *            to create a CDF.  (Do not enter an extension because ".skt"
     *            is appended automatically).  If <B>null</B> is specified, the
     *            skeleton table is named &lt;cdfName&gt;.skt in the current
     *            directory<BR><BR>
     *
     * @param cdfName The pathname of the created CDF from the skeleton table.
     *                Do not enter an extension.<BR><BR>
     *
     * @param delete specifies whether or not the CDF should be deleted
     *               if it already exists.
     *
     * @param log   Specifies whether or not messages are displayed as the
     *             program executes.<BR><BR>
     *
     * @param neg2posfp0
     *            Specifies whether or not -0.0 is converted to 0.0 by the CDF
     *            library when read from a CDF.  -0.0 is an illegal floating
     *            point value on VAXes and DEC Alphas running OpenVMS.<BR><BR>
     *
     * @param statistics Specifies whether or not caching statistics
     *            are displayed at the end of each CDF.<BR><BR>
     *
     * @param zMode Specifies which zMode should be used.  May be one
     *            of the following...<BR><BR>
     *
     *               <DL>
     *               <DT>0
     *                 <DD>Indicates that zMode is disabled.
     *               <DT>1
     *                  <DD>Indicates that zMode/1 should be used (the
     *                       dimension variances of rVariables will be
     *                       preserved).
     *               <DT>2
     *                  <DD>Indicates that zMode/2 should be used (the
     *                       dimensions of rVariables having a variance
     *                       of NOVARY (false) are hidden.
     *               </DL><BR><BR>
     *
     * @param reportType Specifies the types of return status codes from the CDF
     *            library which should be reported/displayed.  <tt>report</tt>
     *            is a bit mask made up from the following
     *            CDFTools.NO_REPORTS, CDFTools.REPORT_ERRORS, 
     *            CDFTools.REPORT_WARNINGS and CDFTools.REPORT_INFORMATION<BR><BR>
     *
     * @param cacheSizeD The number of buffers to be used for the CDF's
     *            dotCDF file. Pass 0 to use the default. <BR> <BR>
     *
     * @param cacheSizeS The number of buffers to be used for the CDF's
     *            staging file. Pass 0 to use the default. <BR> <BR>
     *
     * @param cacheSizeC The number of buffers to be used for the CDF's
     *            compression scratch file. Pass 0 to use the default. <BR> <BR>
     * @throws java.io.IOException An exception is thrown if the file is bad
     *                             or non-existent
     * @throws InterruptedException An exception is thrown if an interrupt is
     *                              issued
     *
     * <B>Note:</B><BR>
     *    When running an application that use this static method, make sure
     *    this <b>cdf.base</b> system property is defined (to the cdf installation).
     *    For example, <BR>
     *    java -Dcdf.base=/home/cdf/cdf34_1-dist MyApp  <BR>
     *
     */
    public static void skeletonCDF(String skeletonName,
				   String cdfName,
				   boolean delete,
				   boolean log,
				   boolean neg2posfp0,
				   boolean statistics,
				   int zMode,
				   int reportType,
				   int cacheSizeD,
				   int cacheSizeS,
				   int cacheSizeC)
	throws java.io.IOException, InterruptedException
    {
        if (cacheSizeD <= 0 && cacheSizeS <= 0 && cacheSizeC <= 0)
           CDFTools.skeletonCDF(skeletonName, cdfName, delete, log,
                                neg2posfp0, statistics, zMode, reportType,
                                null);
        else {
          StringBuffer sb = new StringBuffer();
          if (cacheSizeD > 0) sb = sb.append(""+cacheSizeD+"d");
          if (cacheSizeS > 0) {
            if (cacheSizeD > 0) sb = sb.append(","+cacheSizeS+"s");
            else sb = sb.append(""+cacheSizeS+"s");
          }
          if (cacheSizeC > 0) {
            if (cacheSizeD > 0 || cacheSizeS > 0)
              sb = sb.append(","+cacheSizeC+"c");
            else sb = sb.append(""+cacheSizeC+"c");
          }
          CDFTools.skeletonCDF(skeletonName, cdfName, delete, log,
                               neg2posfp0, statistics, zMode, reportType,
                               sb.toString());
        }
    }


    /**
     * skeletonCDF produces a CDF file from a skeleton table.  A
     * skeleton table is a text file which can be made through a text
     * editor, or by SkeletonCDF program from an existng CDF.
     *
     * @param skeletonName is the pathname of the skeleton table to be used
     *            to create a CDF.  (Do not enter an extension because ".skt"
     *            is appended automatically).  If <B>null</B> is specified, the
     *            skeleton table is named &lt;cdfName&gt;.skt in the current
     *            directory<BR><BR>
     *
     * @param cdfName The pathname of the created CDF from the skeleton table.
     *                Do not enter an extension.<BR><BR>
     *
     * @param delete specifies whether or not the CDF should be deleted
     *               if it already exists.
     *
     * @param log   Specifies whether or not messages are displayed as the
     *             program executes.<BR><BR>
     *
     * @param neg2posfp0
     *            Specifies whether or not -0.0 is converted to 0.0 by the CDF
     *            library when read from a CDF.  -0.0 is an illegal floating
     *            point value on VAXes and DEC Alphas running OpenVMS.<BR><BR>
     *
     * @param statistics Specifies whether or not caching statistics
     *            are displayed at the end of each CDF.<BR><BR>
     *
     * @param zMode Specifies which zMode should be used.  May be one
     *            of the following...<BR><BR>
     *
     *               <DL>
     *               <DT>0
     *                 <DD>Indicates that zMode is disabled.
     *               <DT>1
     *                  <DD>Indicates that zMode/1 should be used (the
     *                       dimension variances of rVariables will be
     *                       preserved).
     *               <DT>2
     *                  <DD>Indicates that zMode/2 should be used (the
     *                       dimensions of rVariables having a variance
     *                       of NOVARY (false) are hidden.
     *               </DL><BR><BR>
     *
     * @param reportType Specifies the types of return status codes from the CDF
     *            library which should be reported/displayed.  <tt>report</tt>
     *            is a bit mask made up from the following
     *            CDFTools.NO_REPORTS, CDFTools.REPORT_ERRORS, 
     *            CDFTools.REPORT_WARNINGS and CDFTools.REPORT_INFORMATION<BR><BR>
     *
     * @param cacheSize The number of buffers to be used for the CDF's
     *            dotCDF file, staging file, and compression scratch file.
     *            This field used only in the CDF access process will not
     *            change the contents of the CDF. Large number(s) will likely
     *            reduce physical I/Os if variable data are involved.
     *            If this field is null, default cache sizes chosen by
     *            the CDF library are used.  The cache sizes are specified
     *            with a comma-separated list of &lt;number&gt;&lt;type&gt; pairs where
     *            &lt;number&gt; is the number of cache buffers and &lt;type&gt; is the
     *            type of file.  The file &lt;type&gt;'s are as follows: `d' for
     *            the dotCDF file, `s' for the staging file, and `c' for the
     *            compression scratch file.  For example, `200d,100s'
     *            specifies 200 cache buffers for the dotCDF file and 100
     *            cache buffers for the staging file.  The dotCDF file cache
     *            size can also be specified without the `d' &lt;type&gt; for
     *            compatibility with older CDF releases (eg. `200,100s').
     *            Note that not all of the file types must be specified.
     *            Those not specified will receive a default cache size.<BR><BR>
     * @throws java.io.IOException An exception is thrown if the file is bad
     *                             or non-existent
     * @throws InterruptedException An exception is thrown if an interrupt is
     *                              issued
     *
     * <B>Note:</B><BR>
     *    When running an application that use this static method, make sure
     *    this <b>cdf.base</b> system property is defined (to the cdf installation).
     *    For example, <BR>
     *    java -Dcdf.base=/home/cdf/cdf34_1-dist MyApp  <BR>
     *
     */
    public static void skeletonCDF(String skeletonName,
				   String cdfName,
				   boolean delete,
				   boolean log,
				   boolean neg2posfp0,
				   boolean statistics,
				   int zMode,
				   int reportType,
				   String cacheSize)
	throws java.io.IOException, InterruptedException
    {

	// StringBuffers are faster than using String and +
	StringBuffer cmd = new StringBuffer();  
	cmd.append(cmdBase).append("skeletoncdf ");
	
	if (cdfName != null) {
	    cmd.append("-cdf ");
	    cmd.append(cdfName+" ");
	}

	cmd.append("-");
	if (!delete)
	    cmd.append("no");
	cmd.append("delete ");
	    
	cmd.append("-");
	if (!log)
	    cmd.append("no");
	cmd.append("log ");
	    
	cmd.append("-zmode "+zMode+" ");

	cmd.append("-");
	if (!neg2posfp0)
	    cmd.append("no");
	cmd.append("neg2posfp0 ");

	if (reportType != 0) {
	    cmd.append("-report \"");
	    if ((reportType & REPORT_ERRORS) != 0)
		cmd.append("errors,");
	    if ((reportType & REPORT_WARNINGS) != 0)
		cmd.append("warnings,");
	    if ((reportType & REPORT_INFORMATION) != 0)
		cmd.append("informationals,");
	    cmd.setLength(cmd.length()-1); // chop off the last 2 chars
	    cmd.append("\" ");
	}

	if (cacheSize != null) {
	    cmd.append("-cache ");
	    cmd.append("\""+cacheSize+"\" ");
	}

	cmd.append("-");
	if (!statistics)
	    cmd.append("no");
	cmd.append("statistics ");
	
	cmd.append(skeletonName);

	Process p = Runtime.getRuntime().exec(cmd.toString());
	int returnCode = p.waitFor();

	// Echo any errors
	BufferedReader inStream = 
	    new BufferedReader(new InputStreamReader(p.getInputStream(), "UTF-8"));
	String line;
	while ((line = inStream.readLine()) != null) {
	    if (err instanceof PrintStream)
		((PrintStream)err).println(line);
	}

    }

    /**
     * cdfElemSize returns the byte size of the given CDF data type.
     *
     * @param dataType the given CDF data type.
     * @return byte count of the given data type.
     *         if the data type is invalid, zero (0) is returned. 
     */

    static int cdfElemSize (long dataType)
    {
      switch ((int)dataType) {
        case (int) CDF_BYTE: return 1;
        case (int) CDF_INT1: return 1;
        case (int) CDF_INT2: return 2;
        case (int) CDF_INT4: return 4;
        case (int) CDF_INT8: return 8;
        case (int) CDF_UINT1: return 1;
        case (int) CDF_UINT2: return 2;
        case (int) CDF_UINT4: return 4;
        case (int) CDF_REAL4: return 4;
        case (int) CDF_REAL8: return 8;
        case (int) CDF_FLOAT: return 4;
        case (int) CDF_DOUBLE: return 8;
        case (int) CDF_EPOCH: return 8;
        case (int) CDF_EPOCH16: return 16;
        case (int) CDF_TIME_TT2000: return 8;
        case (int) CDF_CHAR: return 1;
        case (int) CDF_UCHAR: return 1;
      }
      return 0;

    }

} // CDFTools
