//$Id: EpochNative.java,v 1.1.1.1 2018/08/29 19:16:17 mhliu Exp $
/*
 * Copyright 1996-2014 United States Government as represented by the
 * Administrator of the National Aeronautics and Space Administration.
 * All Rights Reserved.
 */

package gsfc.nssdc.cdf.util;

import java.util.*;

/**
 * The Epoch class is a Java wrapper to the CDF epoch handling routines.  See
 * Chapter 8 of the CDF C Reference Manual Version 2.6 for details
 *
 * <B>Example:</B>
 * <PRE> 
 * // Get the milliseconds to Aug 5, 1990 at 5:00
 * double ep = Epoch.compute(1990, 8, 5, 5, 0, 0, 0);
 * //Get the year, month, day, hour, minutes, seconds, milliseconds for ep
 * long times[] = Epoch.breakdown(ep);
 * for (int i=0;i&lt;times.length;i++)
 *     System.out.print(times[i]+" ");
 * System.out.println();
 * // Printout the epoch in various formats
 * System.out.println(Epoch.encode(ep));
 * System.out.println(Epoch.encode1(ep));
 * System.out.println(Epoch.encode2(ep));
 * System.out.println(Epoch.encode3(ep));
 * // Print out the date using format
 * String format = "&lt;month&gt; &lt;dom.02&gt;, &lt;year&gt; at &lt;hour&gt;:&lt;min&gt;";
 * System.out.println(Epoch.encodex(ep,format));
 * </PRE>
 */
public class EpochNative {

    /**
     * Mirrors computeEPOCH from the CDF library. See Section 8.1 of the 
     * CDF C Reference Manual Version 2.6 for details
     *
     * @param year the year componment
     * @param month the month componment
     * @param day the day componment
     * @param hour the hour componment
     * @param minute the minute componment
     * @param second the second componment
     * @param msec the millisecond componment
     *
     * @return a epoch value
     */
    public native static double compute(long year, 
					long month, 
					long day, 
					long hour,
					long minute,
					long second,
					long msec);

    /**
     * Mirrors EPOCHbreakdown from the CDF library. See Section 8.2 of the
     * CDF C Reference Manual Version 2.6 for details
     * @param epoch the epoch, milliseconds since 0AD
     *
     * @return the day-time componments
     */
    public native static long[] breakdown(double epoch);

    /**
     * Mirrors encodeEPOCH from the CDF library. See Section 8.3 of the
     * CDF C Reference Manual Version 2.6 for details
     * @param epoch the epoch, milliseconds since 0AD
     *
     * @return the epoch in UTC string
     */
    public native static String encode(double epoch);

    /**
     * Mirrors encodeEPOCH1 from the CDF library. See Section 8.4 of the
     * CDF C Reference Manual Version 2.6 for details
     * @param epoch the epoch, milliseconds since 0AD
     *
     * @return the epoch in UTC string

     */
    public native static String encode1(double epoch);

    /**
     * Mirrors encodeEPOCH2 from the CDF library. See Section 8.5 of the
     * CDF C Reference Manual Version 2.6 for details
     * @param epoch the epoch, milliseconds since 0AD
     *
     * @return the epoch in UTC string

     */
    public native static String encode2(double epoch);

    /**
     * Mirrors encodeEPOCH3 from the CDF library. See Section 8.6 of the
     * CDF C Reference Manual Version 2.6 for details
     * @param epoch the epoch, milliseconds since 0AD
     *
     * @return the epoch in UTC string

     */

    public native static String encode3(double epoch);
    /**
     * Mirrors encodeEPOCH4 from the CDF library. See Section 8.6 of the
     * CDF C Reference Manual Version 2.6 for details
     * @param epoch the epoch, milliseconds since 0AD
     *
     * @return the epoch in UTC string 

     */
    public native static String encode4(double epoch);
    
    /**
     * Mirrors encodeEPOCHx from the CDF library. See Section 8.7 of the
     * CDF C Reference Manual Version 2.6 for details
     * @param epoch the epoch, milliseconds since 0AD
     * @param format the format to encode the string
     *
     * @return the epoch in UTC string

     */
    public native static String encodex(double epoch, String format);

    /**
     * Mirrors parseEPOCH from CDF library.  See Section 8.8 of the
     * CDF C Reference Manual Version 2.6 for details
     * @param sEpoch the epoch string
     *
     * @return the epoch

     */
    public native static double parse(String sEpoch);

    /**
     * Mirrors parseEPOCH from CDF library.  See Section 8.9 of the
     * CDF C Reference Manual Version 2.6 for details
     * @param sEpoch the epoch string
     *
     * @return the epoch

     */
    public native static double parse1(String sEpoch);

    /**
     * Mirrors parseEPOCH from CDF library.  See Section 8.10 of the
     * CDF C Reference Manual Version 2.6 for details
     * @param sEpoch the epoch string
     *
     * @return the epoch

     */
    public native static double parse2(String sEpoch);

    /**
     * Mirrors parseEPOCH from CDF library.  See Section 8.11 of the
     * CDF C Reference Manual Version 2.6 for details
     * @param sEpoch the epoch string
     *
     * @return the epoch

     */
    public native static double parse3(String sEpoch);

    /**
     * Mirrors parseEPOCH from CDF library.  See Section 8.11 of the
     * CDF C Reference Manual Version 2.6 for details
     * @param sEpoch the epoch string
     *
     * @return the epoch

     */
    public native static double parse4(String sEpoch);

    static {
	System.loadLibrary("cdfNativeLibrary");
    }
}
