/******************************************************************************
* Copyright 1996-2013 United States Government as represented by the
* Administrator of the National Aeronautics and Space Administration.
* All Rights Reserved.
******************************************************************************/

import java.io.*;
import java.lang.*;
import java.lang.reflect.*;
import java.util.*;
import gsfc.nssdc.cdf.*;
import gsfc.nssdc.cdf.util.*;

public class GetMyCDFData implements CDFConstants {

    /**
     * The Variable to which the data belongs.
     */
    private Variable _myVar;

    /**
     * The data returned from the CDFlib call.  This will always be 
     * a 1D array of the correct primative or String data
     */
    private Object _data;

    /**
     * The data after _dataArray has been unwrapped to the proper
     * array dimensions.
     */
    private Object _dataArray;


    /**
     * Variable id. 
     */
    private long id;

    /**
     * How many data values a data get/put action involves.  Note that for
     * CDF_CHAR and CDF_UCHAR, this is the total number of characters.
     */
    private long numValues;

    /**
     * Mirror _myVar.getNumElements(). numElements = 1 or the string length
     * for CDF_CHAR and CDF_UCHAR data.  
     */
    private long numElements;

     /**
     * Used in the JNI. This is the "real" signature of _data.
     */
    private String dataSignature;

    private long dataType;

    /**
     * The dimensionality of the actual passed data or retrieved data.
     */
    private int nDims = 0;

    /**
     * The dimension sizes of this variable. 
     */
    private int [] dimSizes;


    /**
     * At which record does this hyper get/put function starts.
     */
    private long recStart;
    
    
    /**
     * The number of records a hyper get/put function involves.
     */
    private long recCount;
    
    
    /**
     * What is the spacing between records for a hyper get/put function.
     */
    private long recInterval;
    
    
    /**
     * What are the starting dimension indices within a record for a 
     * hyper get/put function.
     */
    private long [] dimIndices;
    
    /**
     * Gets the starting dimension index within a record for a hyper 
     * get/put function.  Dimension index indicates where the data search 
     * started from within a record.  
     * Let's say a record is comprised of a 2x5 two-dimensional array 
     * (2 rows and 5 columns).  If the index returned from this method has  
     * a value of {1,0}, then the data search was performed starting at the 
     * first element of the second row.  Similarly, the value of {0,0} 
     * represents that the data search search was performed starting at the
     * first element of the first record. 
     *
     * @return the dimension index for this variable 
     */
    public long [] getDimIndices() {
         return dimIndices;
    }
    
    /**
     * How many values in each dimension within a record for a 
     * hyper get/put function.
     */
    private long [] dimCounts;
    
    /**
     * The dimension intervals. 
     */
    private long [] dimIntervals;
    
    
   /**
    * Constructor for GetMyCDFData.
    *
    * @param v the variable from which this GetMyCDFData object is created <P>
    */
    private GetMyCDFData (Object dataArray, 
                          long   datatype, 
                          long   numDims, 
                          long[] dimensionSizes,
                          long   nElements) {
	_dataArray = dataArray;
	dataType   = datatype; 
	// nDims   = (int) numDims;
        numElements = nElements; 
        dimSizes = new int[(int) numDims];

        nDims = 0;
        for (int i=0; i < numDims; i++)
             if (dimensionSizes[i] > 1) 
                 dimSizes[nDims++] = (int) dimensionSizes[i];
    }


    public int getDims () { return nDims; }


    public static GetMyCDFData create(Object dataArray,
                                      long   datatype,
                                      long   numDims,
                                      long[] dimensionSizes,
                                      long   nElements) 
    {
        GetMyCDFData theData = new GetMyCDFData (dataArray, datatype, numDims,
                                                 dimensionSizes, nElements);
        return theData;
    }
    

    // used to determine the boundaries
    private long product(int [] array, int start, int stop) {
	long product = 1;
	for (int i=start; i<stop; i++)
	    if (array[i] > 1) product *= array[i];

	// return product*numElements;
	return product;
    }


    public static void outputIndentation(PrintWriter outWriter,
                                         int indent,
                                         String indentation) {
        for (int i = 0; i < indent; i++) {
             outWriter.print(indentation);
        }
    }


    /**
     * Dumps variable data, one row at a time per record.  This is a generic
     * utility for dumping data to a screen.  Data can be scalar or 
     * 1-dimensional or multi-dimensional array of any data type.<P>
     *
     * The following example retrieves the first record, comprised of
     * 3x5 (3 rows and 5 columns) array, into a generic object and dumps 
     * its contents to screen one row at a time.  In this case three rows
     *  will be displayed on a screen, each row containing 5 elements. 
     * <PRE>
     *     GetMyCDFData data;
     *     long[] dimIndices   = {0,0};
     *     long[] dimIntervals = {3,5};
     *     long[] dimSizes     = {1,1}; <P>
     *     data = var.getHyperDataObject(0L,         // record start
     *                                   1,          // record counts 
     *                                   1,          // record interval
     *                                   dimIndices,
     *                                   dimSizes,
     *                                   dimIntervals);
     *     data.dumpData();
     * </PRE>
     *
     */
    public void dumpData(PrintWriter outWriter, 
                         String   recStartIndicator,
                         String   recEndIndicator,
                         String   elementDelimiter,
                         boolean  elementFlag,
                         int      indent,
                         String   indentation,
                         boolean  iso8601) {

        /////////////////////////////////////////////////////////////
        //                                                         //
        //                    Dump 1D Arrays                       //
        //                                                         //
        /////////////////////////////////////////////////////////////

        indent++;
        outputIndentation (outWriter, indent, indentation);
        if (nDims <= 1) {
            outWriter.print (recStartIndicator);
            if (nDims == 0) {
                if (dataType == CDF_CHAR || dataType == CDF_UCHAR) 
                    outWriter.print ("<![CDATA[");
                if (dataType == CDF_EPOCH) {
                    if (!_dataArray.getClass().isArray()) {
                      Double lEpoch = (Double) _dataArray;
                      outWriter.print (((!iso8601)?
                                        Epoch.encode(lEpoch.doubleValue()):
                                        Epoch.encode4(lEpoch.doubleValue())));
                    } else {
                      double lEpoch = ((double[]) _dataArray)[0];
                      outWriter.print (((!iso8601)?
                                        Epoch.encode(lEpoch):
                                        Epoch.encode4(lEpoch)));
                    }
                } else if (dataType == CDF_EPOCH16) {
                    double[] lEpoch = (double[]) _dataArray;
                    outWriter.print (((!iso8601)?
                                      Epoch16.encode(lEpoch):
                                      Epoch16.encode4(lEpoch)));
                } else if (dataType == CDF_TIME_TT2000) {
                    if (!_dataArray.getClass().isArray()) {
                      Long lEpoch = (Long) _dataArray;
                      outWriter.print (CDFTT2000.toUTCstring(lEpoch.longValue()));
                    } else {
                      long lEpoch = ((long[]) _dataArray)[0];
                      outWriter.print (CDFTT2000.toUTCstring(lEpoch));
                    }
                } 
                else { 
                    if (!_dataArray.getClass().isArray())
                      outWriter.print(_dataArray.toString());
                    else {
                      Object ttt = Array.get(_dataArray,0);
                      outWriter.print(ttt.toString());
                    }
                }
                if (dataType == CDF_CHAR || dataType == CDF_UCHAR) {
                    outWriter.print ("]]>");
                }
            }
            else {
                if (dataType == CDF_CHAR || dataType == CDF_UCHAR) {
                    if (!elementFlag) outWriter.print ("<![CDATA[");
                    outWriter.println ("");
                }
                switch ((int) dataType) {
                case (int)CDF_CHAR:
                case (int)CDF_UCHAR:
/*
                    for (int j=0; j<dimSizes[nDims - 1]; j++) {
                        if (j>0) outWriter.print(elementDelimiter);
                        outWriter.print(((String [])_dataArray)[j]);
                    }
*/
                    indent++;
                    int lastElement = dimSizes[nDims-1];

                    for (int j=0; j<dimSizes[nDims - 1]; j++) {
                        if (indent != -99)
                            outputIndentation (outWriter, indent, indentation);

                        // If a string delimiter is not defined from the 
                        // command line (through the -strDelimiter: option), 
                        // recStartIndicator is set to "<record>".  Otherwise,
                        // recStartIndicator is set to 
                        // <record elementDelimiter="@~">

                        if (elementFlag) {
                            outWriter.print("<element index=\""+j+"\">");
                            outWriter.print("<![CDATA[");
                        }
                        else
                            outWriter.print(elementDelimiter);
                        
                        outWriter.print(((String [])_dataArray)[j]);

                        if (elementFlag) {
                            outWriter.print("]]>");
                            outWriter.print("</element>");
                        }
                        else
                            outWriter.print(elementDelimiter);
                        
                        if (j != lastElement) outWriter.println("");
                    }
                    indent--;
                    if (indent != -99)
                        outputIndentation (outWriter, indent, indentation);
                    break;
                case (int)CDF_BYTE:
                case (int)CDF_INT1:
                    for (int j=0; j<dimSizes[nDims - 1]; j++) {
                        if (j>0) outWriter.print(elementDelimiter);
                        outWriter.print(((byte [])_dataArray)[j]);
                    }
                    break;
                case (int)CDF_INT2:
                    for (int j=0; j<dimSizes[nDims - 1]; j++) {
                        if (j>0) outWriter.print(elementDelimiter);
                        outWriter.print(((short [])_dataArray)[j]);
                    }
                    break;
                case (int)CDF_UINT1:
                    for (int j=0; j<dimSizes[nDims - 1]; j++) {
                        if (j>0) outWriter.print(elementDelimiter);
                        if (((short [])_dataArray)[j] >=0) 
			  outWriter.print(((short [])_dataArray)[j]);
			else {
			  short tmp = (short)(((short [])_dataArray)[j]+256); 
			  outWriter.print(tmp);
			}
                    }
                    break;
                case (int)CDF_INT4:
                    for (int j=0; j<dimSizes[nDims - 1]; j++) {
                        if (j>0) outWriter.print(elementDelimiter);
                        outWriter.print(((int [])_dataArray)[j]);
                    }
                    break;
                case (int)CDF_UINT2:
                    for (int j=0; j<dimSizes[nDims - 1]; j++) {
                        if (j>0) outWriter.print(elementDelimiter);
                        if (((int [])_dataArray)[j] >=0)
                          outWriter.print(((int [])_dataArray)[j]);
                        else {
			  int tmp = (int)(((int [])_dataArray)[j]+65536);
                          outWriter.print(tmp);
			}
                    }
                    break;

                case (int)CDF_UINT4:
                    for (int j=0; j<dimSizes[nDims - 1]; j++) {
                        if (j>0) outWriter.print(elementDelimiter);
                        if (((long [])_dataArray)[j] >=0)
                          outWriter.print(((long [])_dataArray)[j]);
                        else {
			  long tmp = (long)(((long [])_dataArray)[j]+
					    4294967296L);
                          outWriter.print(tmp);
			}
                    }
                    break;
                case (int)CDF_INT8:
                    for (int j=0; j<dimSizes[nDims - 1]; j++) {
                        if (j>0) outWriter.print(elementDelimiter);
                        outWriter.print(((long [])_dataArray)[j]);
                    }
                    break;
                case (int)CDF_REAL4:
                case (int)CDF_FLOAT:
                    for (int j=0; j<dimSizes[nDims - 1]; j++) {
                        if (j>0) outWriter.print(elementDelimiter);
                        outWriter.print(((float [])_dataArray)[j]);
                    }
                    break;
                case (int)CDF_REAL8:
                case (int)CDF_DOUBLE:
                    for (int j=0; j<dimSizes[nDims - 1]; j++) {
                        if (j>0) outWriter.print(elementDelimiter);
                        outWriter.print(((double [])_dataArray)[j]);
                    }
                    break;
                case (int)CDF_EPOCH:
                    for (int j=0; j<dimSizes[nDims - 1]; j++) {
                        if (j>0) outWriter.print(elementDelimiter);
			outWriter.print(
                                  ((!iso8601)?
                                   Epoch.encode(((double [])_dataArray)[j]):
                                   Epoch.encode4(((double [])_dataArray)[j])));
                    }
                    break;
                case (int)CDF_EPOCH16:
		    double[] twoDouble = new double[2];
                    for (int j=0, k=0; j<dimSizes[nDims - 1]; j++) {
                        if (j>0) outWriter.print(elementDelimiter);
			twoDouble[0] = ((double [])_dataArray)[k++];
			twoDouble[1] = ((double [])_dataArray)[k++];
                        outWriter.print(((!iso8601)?
                                         Epoch16.encode(twoDouble):
                                         Epoch16.encode4(twoDouble)));
                    }
                    break;
                case (int)CDF_TIME_TT2000:
                    for (int j=0; j<dimSizes[nDims - 1]; j++) {
                        if (j>0) outWriter.print(elementDelimiter);
                        outWriter.print(
                              CDFTT2000.toUTCstring(((long [])_dataArray)[j]));
                    }
                    break;
                default:
                    break;
                } /* switch (dataType) */

               if (dataType == CDF_CHAR || dataType == CDF_UCHAR) {
                   if (!elementFlag) outWriter.print ("]]>"); 
                }
            }
            outWriter.println (recEndIndicator);

        /////////////////////////////////////////////////////////////
        //                                                         //
        //              Dump Multidimensional Arrays               //
        //                                                         //
        /////////////////////////////////////////////////////////////

        } else {
            /*
             * Setup the subArrays array
             * cIndex is the holds the index for the current subarray
             * boundary is the # of values in the subarray
             */
            Object aRow = null;
            Object [] subArrays = new Object[(int)nDims-1];
            int [] cIndex   = new int[(int)nDims-1];
            long [] boundary = new long[(int)nDims-1];

            int jj=0;

            subArrays[0] = _dataArray;
            cIndex[0] = 0;
	    boundary[0] = product(dimSizes,0,(int)nDims);

            for (int i=1; i<(int)nDims-1;i++) {
                subArrays[i] = Array.get(subArrays[i-1], 0);
                boundary[i] = product(dimSizes,i,(int)nDims);
                cIndex[i] = 0;
            }

            int n = 0; // The current element in the _data
            Object cObject = _dataArray;  // A temp object to hold a subarray
            boolean boundaryCrossed;

            outWriter.print (recStartIndicator);
            if (dataType == CDF_CHAR || dataType == CDF_UCHAR) {
                if (elementFlag)   
                    outWriter.println ("");
                else
                    outWriter.println ("<![CDATA[");
            }

            while (n < boundary[0]) {
                // Get the correct 2D subarray to print
                if (n!=0) {
                    jj++;      // Increment the counter for 3-D
                    if (dataType != CDF_CHAR && dataType != CDF_UCHAR) 
                        outWriter.print (elementDelimiter);

                    for (int i = 0; i < (int)nDims-1; i++) {
                        boundaryCrossed = ((n % boundary[i]) == 0);
                        if (boundaryCrossed) {
                            // Get the next sub array
                            cIndex[i]+=1;
                            cObject = Array.get(cObject, cIndex[i]);
                            subArrays[i] = cObject;

                            // Get the first element of each
                            // subsequent subarray
                            for (int j=i+1;j<(int)nDims-1;j++) {
                                cIndex[j] = 0;
                                subArrays[j] =
                                    Array.get(subArrays[j-1],cIndex[j]);
                            }
                            break;

                        } else {
                            cObject = subArrays[i];
                        }
                    }
                }

                // Fill the correct elements of _data

                for (int i=0;i<dimSizes[(int)nDims-2];i++) {
            
                    // if (i > 0) outWriter.print (" ");
                    if (i > 0) { 
                        if (dataType == CDF_CHAR || dataType == CDF_UCHAR) 
                           indent--;
                        else 
                           outWriter.print (elementDelimiter);
                    }

                    switch ((int) dataType) {
                    case (int)CDF_CHAR:
                    case (int)CDF_UCHAR:
                        aRow =
                            (String [])Array.get(subArrays[(int)nDims - 2], i);
/*
                        for (int j=0; j<dimSizes[nDims - 1]; j++) {
                            if (j>0) outWriter.print(elementDelimiter);
                            outWriter.print(((String [])aRow)[j]);
                        }
*/
                        indent++;
                        for (int j=0; j<dimSizes[nDims - 1]; j++) {
                            if (indent != -99)
                                outputIndentation (outWriter, indent, 
                                                   indentation);

                            if (elementFlag) {
                                if (nDims == 2)
                                    outWriter.print("<element index=\""+i+
                                                    ","+j+"\">");
                                else
                                    outWriter.print("<element index=\""+jj+
                                                    ","+i+","+j+"\">");
                                outWriter.print("<![CDATA[");
                            }
                            else
                                outWriter.print(elementDelimiter);

                            outWriter.print(((String [])aRow)[j]); 

                            if (elementFlag) {
                                outWriter.print("]]>");
                                outWriter.println("</element>");
                            }
                            else
                                outWriter.println(elementDelimiter);
                        }
                        break;
                    case (int)CDF_BYTE:
                    case (int)CDF_INT1:
                        aRow = (byte [])Array.get(subArrays[(int)nDims - 2], i);
                        for (int j=0; j<dimSizes[nDims - 1]; j++) {
                            if (j>0) outWriter.print(elementDelimiter);
                            outWriter.print(((byte [])aRow)[j]);
                        }
                        break;
                    case (int)CDF_INT2:
                        aRow = (short [])Array.get(subArrays[(int)nDims - 2], i)
;
                        for (int j=0; j<dimSizes[nDims - 1]; j++) {
                            if (j>0) outWriter.print(elementDelimiter);
                            outWriter.print(((short [])aRow)[j]);
                        }
                        break;
                    case (int)CDF_UINT1:
                        aRow = (short [])Array.get(subArrays[(int)nDims - 2], i);
                        for (int j=0; j<dimSizes[nDims - 1]; j++) {
                            if (j>0) outWriter.print(elementDelimiter);
                            if (((short [])aRow)[j] >=0) 
				outWriter.print(((short [])aRow)[j]);
			    else {
				short tmp = (short) (((short [])aRow)[j] + 256);
				outWriter.print(tmp);
			    }
                        }
                        break;
                    case (int)CDF_INT4:
                        aRow = (int [])Array.get(subArrays[(int)nDims - 2], i);
                        for (int j=0; j<dimSizes[nDims - 1]; j++) {
                            if (j>0) outWriter.print(elementDelimiter);
                            outWriter.print(((int [])aRow)[j]);
                        }
                        break;
                    case (int)CDF_UINT2:
                        aRow = (int [])Array.get(subArrays[(int)nDims - 2], i);
                        for (int j=0; j<dimSizes[nDims - 1]; j++) {
                            if (j>0) outWriter.print(elementDelimiter);
                            if (((int [])aRow)[j] >=0)
                                outWriter.print(((int [])aRow)[j]);
                            else {
                                int tmp = (int) (((int [])aRow)[j] + 65536);
                                outWriter.print(tmp);
                            }
                        }
                        break;
                    case (int)CDF_UINT4:
                        aRow = (long [])Array.get(subArrays[(int)nDims - 2], i);
                        for (int j=0; j<dimSizes[nDims - 1]; j++) {
                            if (j>0) outWriter.print(elementDelimiter);
                            if (((long [])aRow)[j] >=0)
                                outWriter.print(((long [])aRow)[j]);
                            else {
                                long tmp = (long) (((long [])aRow)[j]+
						   4294967296L);
                                outWriter.print(tmp);
                            }
                        }
                        break;
                    case (int)CDF_INT8:
                        aRow = (long [])Array.get(subArrays[(int)nDims - 2], i);
                        for (int j=0; j<dimSizes[nDims - 1]; j++) {
                            if (j>0) outWriter.print(elementDelimiter);
                            outWriter.print(((long [])aRow)[j]);
                        }
                        break;
                    case (int)CDF_REAL4:
                    case (int)CDF_FLOAT:
                        aRow = (float [])Array.get(subArrays[(int)nDims - 2], i);
                        for (int j=0; j<dimSizes[nDims - 1]; j++) {
                            if (j>0) outWriter.print(elementDelimiter);
                            outWriter.print(((float [])aRow)[j]);
                        }
                        break;
                    case (int)CDF_REAL8:
                    case (int)CDF_DOUBLE:
                        aRow = (double [])Array.get(subArrays[(int)nDims - 2], i);
                        for (int j=0; j<dimSizes[nDims - 1]; j++) {
                            if (j>0) outWriter.print(elementDelimiter);
                            outWriter.print(((double [])aRow)[j]);
                        }
                        break;
                    case (int)CDF_EPOCH:
                        aRow = (double [])Array.get(subArrays[(int)nDims - 2], i);
                        for (int j=0; j<dimSizes[nDims - 1]; j++) {
                            if (j>0) outWriter.print(elementDelimiter);
			    outWriter.print(
                                      ((!iso8601)?
                                       Epoch.encode(((double [])aRow)[j]):
                                       Epoch.encode4(((double [])aRow)[j])));
                        }
                        break;
                    case (int)CDF_EPOCH16:
                        aRow = (double [])Array.get(subArrays[(int)nDims - 2], i);
			double[] twoDouble = new double[2];
                        for (int j=0,k=0; j<dimSizes[nDims - 1]; j++) {
                            if (j>0) outWriter.print(elementDelimiter);
			    twoDouble[0] = ((double [])aRow)[k++];
			    twoDouble[1] = ((double [])aRow)[k++];
                            outWriter.print(((!iso8601)?
                                             Epoch16.encode(twoDouble):
                                             Epoch16.encode4(twoDouble)));
                        }
                        break;
                    case (int)CDF_TIME_TT2000:
                        aRow = (long [])Array.get(subArrays[(int)nDims - 2], i);
                        for (int j=0; j<dimSizes[nDims - 1]; j++) {
                            if (j>0) outWriter.print(elementDelimiter);
                              outWriter.print(
                                  CDFTT2000.toUTCstring(((long [])aRow)[j]));
                        }
                        break;
                    default:
                        break;
                    }      /* switch (dataType) */
                    n += dimSizes[nDims - 1];

                }          /* for (int i=0;i<dimSizes[(int)nDims-2];i++) */ 

                if (dataType == CDF_CHAR || dataType == CDF_UCHAR) indent--;
            }              /* while (n < boundary[0]) */

            if (dataType == CDF_CHAR || dataType == CDF_UCHAR) {
                outputIndentation (outWriter, indent, indentation);
                if (!elementFlag) outWriter.print ("]]>");
            }
            outWriter.println (recEndIndicator);
        }
    }

}


