#!/usr/bin/perl -w
##
$perlproc = "edr_map.pl";
# 
#
#       - RJB - Mar 25, 2021 - initial release
#       - RJB - Mar 28, 2021 - allow oct format
#       - RJB - Apr 07, 2021 - breakout of standard header
#       - RJB - May 13, 2021 - continue breakout of subhdrs and data field MFs
#       - RJB - May 18, 2021 - continue breakout of fields, cleanup
#       - RJB - Oct 27, 2021 - add dcom map type f0 or f1 for byte 4
#                           - fixed subheader and data values for dcom_tbl and eng rec
#                             according to Fig B-6 for Engr Rec and Fig B-2 for Decom_tbl
#       - RJB - Oct 29, 2021 - Found updated documenetation 618-308_RevD_Ch5_ dated 5-1-1989
#                             for CR5A and UV5A data modes
#       - RJB - Oct 30, 2021 - updated internal doc for each subroutine call
#                            - added initial processing for engineering and dcom table sub_hdrs
#                              added initial processing for engineering and dcom table data records
#       - RJB - Nov 03, 2021 - fix lecp, crs mag listings 
#       - RJB - Nov 04, 2021 - fix pls listing, fix pra header and sub_header
#       - RJB - Nov 06, 2021 - finish pra data  
#       - RJB - Nov 08, 2021 - finish dcom data,
#       - RJB - Nov 09, 2021 - finish eng data
#       - RJB - Nov 18, 2021 - fixed "Possible precedence problem on bitwise & operator line 4430 
#                               by substituitng  &&
#                               Add MF numbers and sizes
#       - RJB - Nov 20, 2010 - added check for wrong instrument edr
#                               FINAL (WEEKLY) edrs start with a DCOM record
#       - RJB - Dec 03, 2021 - Added byte 5 (spaccraft ID) to output
#       - RJB - Dec 09, 2021 -  Add test to see if byte reversed data
#                                Add test to see if byte format requested and abort if so
#                                
#  usage: edr_map.pl src_dir dest_dir edr_file edr_map_file  format
#
#   Program to create a fully commented record in a file created from the output
#   of program "od" on a Voyager CR-5A Daily EDR file.
#  
#   Environment variables:
#    <none>
#
#   Input:
#       The output of a unix "od" command of a daily (not FINAL) edr file
#       The filename must begin with the 3 letter instrument abbreviation.
#       (crs,eng,lec,mag,pls,pws,uvs,dcom)
#       the input must be generated by one of following:
#       od -xv crs20350.b > crs20350.b.hex
#       od -bv crs20350.b > crs20350.b.oct
#
#   Output:
#       the first record of an edr with comments
#
use warnings;
use Env;
use POSIX qw(uname);
use Term::ANSIColor;
#use Number::Convert;
#
#  PARAMETER PROCESSOR
#
$argCount = scalar @ARGV;
if (($argCount eq 0)) {
    print color ("blue"), "\n";
    print "\n";
    print "    usage: $perlproc src_dir dest_dir    edr_file         edr_map_file       format\n";
    print "            Ex:       ./       ./      crs20350.b.hex    crs20350.b.hexmap    hex \n";
    print "\n";
    print "Separate the records in a file created from the output of od on a Voyager CR-5A Daily EDR file\n";
    print "\n";
    print "\n", color("reset");
    exit(0);
}
if (($argCount == 1)) {
    if ($ARGV[0] =~ /-h/i || $ARGV[0] =~ /help/i || $ARGV[0] =~ /\?/i) {
        print color ("blue"), "\n";
        print "    Voyager Project program to Separate the records in a file created\n";
        print "    from the output of od on a Voyager CR-5A EDR file\n";
        print "\n";
        print "    usage: $perlproc src_dir dest_dir edr_file           edr_map_file   format\n";
        print "            Ex:       ./       ./    crs20350.b.hex    crs20350.b.hexmap    hex \n";
        print "\n";
        print "where:\n";
        print "     src_dir is location of input od dump file \n";
        print "     dest_dir is where data is processed and products created\n";
        print "     edr_file is the name of input od dump file \n";
        print "        (the first 3 letters of name must correspond to a VGR instrument)\n";
        print "     edr_map_file is the name of the output file with the record mapped\n";
        print "     format is the internal format produced by command od\n";
        print "           byte or oct = octal byte format from command 'od -b'";
        print "           hex  = hex byte format from command 'od -x'";
        print "\n";
        print "\n", color("reset");
        exit;
    }
}
if ($argCount < 5) {
    print color ("red"), "\n";
    print "\n";
    print "Did you forget something?\n\n", color("reset");
    exit(1);
}
if ($ARGV[4] =~ lc("byte")) {
    print color ("red"), "\n";
    print "\n";
    print " Byte format is not supported yet\n\n", color("reset");
    exit(1);


}
#**
##** DECLARATIONS:
##**
#
$byte = "byte";
$crs_data = 3120;
$crs_subhdr = 40;
#$data_hdr = 240;  #bytes
$dcom_data = 448;
$dcom_subhdr = 28;
$data_rec = "   Data Rec ";
$eng_data = 3600;       # corr 10-29-2021 from 3820
$eng_subhdr = 20;
$eng_rec = "   Engineering Rec ";
$header = "   Header";
#$header_size = 240;
$hex = "hex";
$lec_data = 320;
$lec_subhdr = 4;
$mag_data = 1040;
$mag_subhdr = 432;
$magic_hex = "d4d1 e2";         #"MJS"
$rev_magic_hex = "d1d4";       #"JM"
$magic_byte = "324 321 342";    #"MJS"
#$rev_magic_byte = "321 324";   #"JM"
$oct = "oct";
#$rec ="   Rec ";
$pls_data = 400;
$pls_subhdr = 32;
$pra_data = 280;            # cosrr 10-29-2021 from 248
$pra_subhdr = 8;
$pws_data = 220;
$pws_subhdr = 80;
$return = "\n";
$space = " ";
$uvs_data = 120;
$uvs_subhdr = 16;
#
#**
##**  ENVIRONMENT
##**
#
##print "ARGV[0] = $ARGV[0]\n";          #src dir
##print "ARGV[1] = $ARGV[1]\n";          #dest dir
##print "ARGV[2] = $ARGV[2]\n";          #edr_file 
##print "ARGV[3] = $ARGV[3]\n";          #edr_mod_file file extension
##print "ARGV[4] = $ARGV[4]\n";          #byte or hex
#
$destpath = $ARGV[1];

if (-d $destpath) {
    #
} else {
    print color ("red"), "      ??E destpath = $destpath does NOT exist\n", color("reset");
    exit(1);
}
my @name = split ( //,$ARGV[2] );
my $instr = join ( "",$name[0],$name[1],$name[2] );
$instr = uc($instr);
# print "instr = $instr\n";
my $format = $ARGV[4];

if ($format =~ m/byte/ || $format =~ m/hex/ || $format =~ m/oct/ ) {
#
    $destpath = $ARGV[1];
} else {
    print color ("red"), "      ??E format not byte or hex";
     print "\n", color("reset");
}

# open that file and find magic 
my $magic_num = 0;          #number of total records
my $eng_num = 0;            #number of engineering records
my $tot_rec = 0;            # total record count
$starfound = 0;
$cur = "000000";
$infile = $ARGV[2];
$outfile = $ARGV[3];

# OPEN and check to see if correct file type
open(INFILE,"$ARGV[2]") || die "Cannot find INFILE: $ARGV[2]\n";
       while (<INFILE>) {
    chop;
    $line = $_;
    @find = split (/\s+/, $line);           #split on spaces
    print "find[1] = $find[1] $find[2]\n";
    if ($find[1] =~ $rev_magic_hex) {
        print color ("red"), "\n";
        print "DATA SET is byte reversed\n";
        print "PLEASE run byteswap program\n\n", color("reset");
        exit;
    }
    @next = split( /\s+/,$find[2]);     #split on spaces
    @ch = split ( //,$next[0]);         #split on characters
    if ($instr =~ "CRS") {
        $iid = "1";
#        print "ch[1] = $ch[1]\n";
#        print "ch[2] = $ch[2]\n";
        if ($ch[2] !~ $iid) {
            print color ("red"), "\n";
            print "ERROR: This is not a Daily CRS EDR\n\n", color("reset");
            if ($ch[2] =~ "f") {
                print "This is a DCOM record. likely a CRS weekly file\n\n", color("reset");
            }
            exit;
        }
        goto CONT;
    }
    if ($instr =~ "LEC") {
        $iid = "3";
        if ($ch[2] !~ $iid) {
            print color ("red"), "\n";
            print "ERROR: This is not a Daily LEC EDR\n\n", color("reset");
            if ($ch[2] =~ "f") {
                print "This is a DCOM record. likely a LEC weekly file\n\n", color("reset");
            }
            exit;
        }
        goto CONT;
    }
    if ($instr =~ "MAG") {
        $iid = "4";
        if ($ch[2] !~ $iid) {
            print color ("red"), "\n";
            print "ERROR: This is not a Daily MAG EDR\n\n", color("reset");
            if ($ch[2] =~ "f") {
                print "This is a DCOM record. likely a MAG weekly file\n\n", color("reset");
            }
            exit;
        }
        goto CONT;
    }
    if ($instr =~ "PLS") {
        $iid = "5";
        if ($ch[2] !~ $iid) {
            print color ("red"), "\n";
            print "ERROR: This is not a Daily PLS EDR\n\n", color("reset");
            if ($ch[2] =~ "f") {
                print "This is a DCOM record. likely a PLS weekly file\n\n", color("reset");
            }
            exit;
        }
        goto CONT;
    }
    if ($instr =~ "PRA") {
        $iid = "7";
        if ($ch[2] !~ $iid) {
            print color ("red"), "\n";
            print "ERROR: This is not a Daily PRA EDR\n\n", color("reset");
            if ($ch[2] =~ "f") {
                print "This is a DCOM record. likely a PRA weekly file\n\n", color("reset");
            }
            exit;
        }
        goto CONT;
    }
    if ($instr =~ "PWS") {
        $iid = "8";
        if ($ch[2] !~ $iid) {
            print color ("red"), "\n";
            print "ERROR: This is not a Daily PWS EDR\n\n", color("reset");
            if ($ch[2] =~ "f") {
                print "This is a DCOM record. likely a PWS weekly file\n\n", color("reset");
            }
            exit;
        }
        goto CONT;
    }
    if ($instr =~ "UVS") {
        $iid = "9";
        if ($ch[2] !~ $iid) {
            print color ("red"), "\n";
            print "ERROR: This is not a Daily UVS EDR\n\n", color("reset");
            if ($ch[2] =~ "f") {
                print "This is a DCOM record. likely a UVS weekly file\n\n", color("reset");
            }
            exit;
        }
        goto CONT;
    }
    if ($instr =~ "ENG") {
        $iid = "b";
        if ($ch[2] !~ $iid) {
            print color ("red"), "\n";
            print "ERROR: This is not a Daily ENG EDR\n\n", color("reset");
            if ($ch[2] =~ "f") {
                print "This is a DCOM record. likely a ENG weekly file\n\n", color("reset");
            }
            exit;
        }
        goto CONT;
    }
    if ($instr =~ "DCO") {
        $iid = "f";
        print "iid = $iid  ch[2] = $ch[2]\n";
        if ($ch[2] !~ $iid) {
            print color ("red"), "\n";
            print "ERROR: This is not a DCOM EDR\n\n", color("reset");
            exit;
        }
        goto CONT;
    }
}
CONT:
#print "here\n";
close (INFILE);
open(INFILE,"$ARGV[2]") || die "Cannot find INFILE: $ARGV[2]\n";
open(OUTFILE,">$outfile") || die "Cannot open OUTFILE: $outfile\n";

print OUTFILE "Documented in 618-306 Rev D Change 5 1-May 1989\n";
print OUTFILE "Voyager Experiment Data Record Format Specification\n";
print OUTFILE "Table C-1 page C-136 Standard Record Header for extracted Data Data Mode CR-5A\n"; 
print OUTFILE "\n";
print OUTFILE "Bytes 1-3 - Project ID:  'MJS'\n";
if ($format =~ m/byte/ ) {
    print OUTFILE "Byte  4   - Instrument ID: 20=crs,  40=pls,  60=lec, 100=mag,\n";
    print OUTFILE "                          160=pra, 200=pws, 220=uvs, 260=eng \n";
    print OUTFILE "                          360=decom_tbl VGR2, 361=decom_tbl for VGR1 \n";
} else {
    print OUTFILE "Byte  4   - Instrument ID: 10=crs, 20=pls, 30=lec, 40=mag,\n";
    print OUTFILE "                           70=pra, 80=pws, 90=uvs, b0=eng,\n";
    print OUTFILE "                           f0=decom_tbl VGR2, f1=decom_tbl for VGR1 \n";
}
if ($format =~ m/byte/ ) {
    print OUTFILE "Byte  5   - Spacecraft ID: 0=VGR2,  1=VGR1\n";
} else {
    print OUTFILE "Byte  5   - Spacecraft ID: 0=VGR2, 1=VGR1 \n";
}
print OUTFILE "Byte  6   - Record Number\n";
if ($format =~ m/byte/ ) {
    print OUTFILE "Byte  7   - Spacecraft Data Mode: 30=CR_5A, 35=UV_5A\n";
} else {
    print OUTFILE "Byte  7   - Spacecraft Data Mode: 18=CR_5A, 1D=UV_5A\n";
}
if ($format =~ m/byte/ ) {
    print OUTFILE "Byte 356  - Data Present Indicator: 41=crs, 42=lec, 43=mag, 44=pls,\n";
    print OUTFILE "                                    46=pra, 47=pws, 50=uvs, 52=dcom, 55=eng\n";  
} else {
    print OUTFILE "Byte 356  - Data Present Indicator: 21=crs, 22=lec, 23=mag, 24=pls,\n";
    print OUTFILE "                                    26=pra, 27=pws, 28=uvs, 2a=dcom, 2d=eng\n";
}
print OUTFILE "Bytes per record:  Header: 718, Engineering: 3860, CRS: 3400, LEC: 564,\n";
print OUTFILE "                    MAG: 1712, PLS: 672, PRA: 528, PWS: 540, UVS: 376 DCOM: 716\n";
print OUTFILE "         (Header and Engineering records are in FINAL products - not QUICKLOOK)\n";
print OUTFILE "\n";
print OUTFILE "Data Record bytes: Header: 240, eng_subhdr: 20, crs_subhdr: 40, lec_subhdr: 4, dcom_tbl_subhdr: 28\n";
print OUTFILE "         mag_subhdr: 432, pls_subhdr: 32, pra_subhdr: 8, pws_subhdr: 80, uvs_subhdr: 16\n";
print OUTFILE "Data Record bytes: Header:   , crs_data: 3120, lec_data: 320, mag_data: 1040, eng_data: 3600\n";
print OUTFILE "         pls_data: 400, pra_data: 280, pws_data: 220, uvs_data: 120, DCOM_data: 448\n";
print OUTFILE "\n";
print OUTFILE "Minor Frames (Size - bytes): ENG=60 (60 bytes), CRS=60 (52 bytes), LECP=20 (16 bytes), MAG=20 (151 bytes)\n";
print OUTFILE "                       PRA=20 (7 bytes), PWS=20 (11 bytes), UVS=25 (i6 bytes),  DCOM=24 (18 bytes)\n";
print OUTFILE "\n"; 

$end = 0;
$rec_rdr = 0;
LINEA: while (<INFILE>) {
    chop;
    $line = $_;
    $skip = 0;
    $last = $cur;
    @find = split (/\s+/, $line);           #split on spaces
    $cur = $find[0];
##    print OUTFILE "$cur\n";
#    print OUTFILE "crs_cnt = $crs_cnt cur = $cur\n";
   if ($format =~ m/$hex/ ) {
        $rec_rdr++;
# if magic_hex then process
        if ($line =~ m/$magic_hex/ ) {
            $crs_cnt = 0;
            $pls_cnt = 0;
            $lec_cnt = 0;
            $mag_cnt = 0;
            $pws_cnt = 0;
            $pra_cnt = 0;
            $eng_cnt = 0;
            $dcom_cnt = 0;
            $skip = 0;
            $magic_num++;               #count of magic finds
            @find = split( /$magic_hex/, $line );
            @next = split( /\s+/,$find[1]);     #split on spaces
            $rec_type = $next[0];
            $ctr = $magic_num;
            $recid = $data_rec;
# check for Header record 
            if ($rec_type =~ "f0") {$recid = $header; $ctr = ""; $magic_num--;}
# check for Engineering record
            if ($rec_type =~ "b0") {$recid = $eng_rec; $eng_num++; $ctr = $eng_num; $magic_num--;}

            if ($find[0] =~ "0000000") {
                $posit = index($line,$magic_hex);
                $filler = " "x$posit;
                $filler = " ";
                $outline = "$find[0]$magic_hex$find[1]$recid$ctr\n";   #"2>$outline" (PRINTS)
                print OUTFILE $outline;
                next LINEA;
            }
         }  #mag_hex

            if ($instr =~ "crs" ) {
                ($skip,$outline) = &proc_crs_hex($outfile);
            }  #end instr = crs
            if ($instr =~ "lec" ) {
#                $skip = 0;
                ($skip,$outline) = &proc_lec_hex($outfile);
#                print "skip = $skip   outline = $outline\n";
            }  #end instr = lec
            if ($instr =~ "mag" ) {
                ($skip,$outline) = &proc_mag_hex($outfile);
            }  #end instr = mag
            if ($instr =~ "pls" ) {
                ($skip,$outline) = &proc_pls_hex($outfile);
            }  #end instr = pls
            if ($instr =~ "pra" ) {
                print OUTFILE "proc_pra_hex\n";
               ($skip,$outline) = &proc_pra_hex($outfile);
            }  #end instr = pra
            if ($instr =~ "pws" ) {
                ($skip,$outline) = &proc_pws_hex($outfile);
            }  #end instr = pws
            if ($instr =~ "uvs" ) {
                ($skip,$outline) = &proc_uvs_hex($outfile);
            }  #end instr = uvs
            if ($instr =~ "eng" ) {
#                print OUTFILE "proc_eng_hex\n";
                ($skip,$outline) = &proc_eng_hex($outfile);
            }  #end instr = eng
            if ($instr =~ "dco" ) {
                ($skip,$outline) = &proc_dcom_hex($outfile);
            }  #end instr = dcom


            if ($find[0] =~ "0006500" && $instr =~ "crs") {
                        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
                        $spaces = "                            ";
                        $data_size = "Data $crs_data Bytes";
                        $outline = "$find[0]$spaces$data_size\n";             #"1>$outline"  (PRINTS)
                        print OUTFILE $outline;                             #Last line of CRS dump
                        $end = 1;
                        next LINEA;
            }
            if ($find[0] =~ "0001060" && $instr =~ "lec") {
                        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
                        $spaces = "      ";
                        $data_size = "Data = $lec_data Bytes";
                        @find = split (/\s+/, $line);
                        $linex = "$find[0]$space$find[1]$space$find[2]";
                        $outline = "$linex$spaces$return$data_size$return";             #"1>$outline"  (PRINTS)
                        print OUTFILE $outline;                             #last line of LECP dump
                        $end = 1;
                        next LINEA;
            }
           if ($find[0] =~ "0001220" && $instr =~ "pls") {
                        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
                        $spaces = "      ";
                        $data_size = "Data = $pls_data Bytes";
               #         $linex = 
                        $outline = "$line$spaces$data_size$return";             #"1>$outline"  (PRINTS)
                        print OUTFILE $outline;                             # last line of PLS dump
                        $end = 1;
                        next LINEA;
            }
           if ($find[0] =~ "0003240" && $instr =~ "mag") {
#                    print "find[0] = $find[0]\n";
                        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
                        $spaces = "      ";
                        $data_size = "Data = $mag_data Bytes";
                        $outline = "$line$spaces$data_size\n";             #"1>$outline"  (PRINTS)
                        print OUTFILE $outline;                             #last line of MAG dump
                        $end = 1;
                        next LINEA;
            }
           if ($find[0] =~ "0001000" && $instr =~ "pra") {
                        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
                        $spaces = "      ";
                        $data_size = "Data = $pra_data Bytes";
                        $outline = "$line$spaces$data_size\n";             #"1>$outline"  (PRINTS)
                        print OUTFILE $outline;
                        $end = 1;
                        next LINEA;
            }
           if ($find[0] =~ "0001020" && $instr =~ "pws") {
                        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
                        $spaces = "               ";
                        $data_size = "Data $pws_data Bytes";
                        $outline = "$find[0]$spaces$data_size\n";             #"1>$outline"  (PRINTS)
                        print OUTFILE $outline;
                        $end = 1;
                        next LINEA;
            }
           if ($find[0] =~ "0000560" && $instr =~ "uvs") {
                        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
                        $spaces = "                         ";
                        $data_size = "Data $uvs_data Bytes";
                        @llast = split (" ",$line );
                        $last_line = join (" ",$llast[0],$llast[1],$llast[2],$llast[3],$llast[4] );
                        $outline = "$last_line$spaces$data_size\n";             #"1>$outline"  (PRINTS)
                        print OUTFILE $outline;
                        $end = 1;
                        next LINEA;
            }
           if ($find[0] =~ "0007420" && $instr =~ "eng") {
#                print OUTFILE "0007420 && instr=eng\n";
                        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
                        $spaces = "                                    ";
                        $data_size = "Data $eng_data Bytes";
                        @llast = split (" ",$line );
                        $last_line = join (" ",$llast[0],$llast[1],$llast[2] );
                        $outline = "$last_line$spaces$data_size\n";             #"1>$outline"  (PRINTS)
                        print OUTFILE $outline;
                        $end = 1;
                        next LINEA;
            }
           if ($find[0] =~ "0001300" && $instr =~ "dco") {
                        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
                        $spaces = "                ";
                        $data_size = "Data $dcom_data Bytes";
                        @llast = split (" ",$line );
                        $last_line = join (" ",$llast[0],$llast[1],$llast[2],$llast[3],$llast[4],$llast[5],$llast[6] );
                        $outline = "$last_line$spaces$data_size\n";             #"1>$outline"  (PRINTS)
                        print OUTFILE $outline;
                        $end = 1;
                        next LINEA;
            }

         if ($end == 0) { 
#            print OUTFILE "rec_rdr = $rec_rdr\n";
            if ($skip == 0) {
                print OUTFILE "$line\n";
            }
         }
     }   # end if hex

     if ($format =~ m/$byte/  || $format =~ m/$oct/ ) {
        $rec_rdr++; 
         if ($line =~ m/$magic_byte/ ) {
                $crs_cnt = 0; 
                $pls_cnt = 0;
                $lec_cnt = 0;
                $mag_cnt = 0;
                $magic_num++; 
                @find = split (/$magic_byte/, $line);
                $rec_type = $find[1];
                $ctr = $magic_num;
                $recid = $data_rec;
                if ($rec_type =~ "360") {$recid = $header; $ctr = ""; $magic_num--;}
                if ($rec_type =~ "260") {$recid = $eng_rec; $eng_num++; $ctr = $eng_num; $magic_num--;}
                if ($find[0] =~ "0000000") {
                    $posit = index($line,$magic_hex);
                    $filler = " "x$posit;
                    $outline = "$find[0]$filler$magic_byte$find[1]$recid$ctr\n";  #"5>$outline"   (PRINTS)
                    print OUTFILE $outline;
                    next LINEA;
                }
            }   # end if magic_byte
#            print "instr = $instr\n";
           if ($instr =~ "crs" ) {
                $skip = 0;
                ($skip,$outline) = &proc_crs_oct($outfile);
#                print "skip = $skip   outline = $outline\n";
            } 
            if ($instr =~ "lec" ) {
                $skip = 0;
                ($skip,$outline) = &proc_lec_oct($outfile);
#                print "skip = $skip   outline = $outline\n";
            }  #end instr = lec
           if ($instr =~ "mag" ) {
                $skip = 0;
                ($skip,$outline) = &proc_mag_oct($outfile);
#                print "skip = $skip   outline = $outline\n";
            } 
           if ($instr =~ "pls" ) {
                $skip = 0;
                ($skip,$outline) = &proc_pls_oct($outfile);
#                print "skip = $skip   outline = $outline\n";
            } 
           if ($instr =~ "pra" ) {
                $skip = 0;
                ($skip,$outline) = &proc_pra_oct($outfile);
#                print "skip = $skip   outline = $outline\n";
            } 
           if ($instr =~ "pws" ) {
                $skip = 0;
                ($skip,$outline) = &proc_pws_oct($outfile);
#                print "skip = $skip   outline = $outline\n";
            } 
           if ($instr =~ "uvs" ) {
                $skip = 0;
                ($skip,$outline) = &proc_uvs_oct($outfile);
#                print "skip = $skip   outline = $outline\n";
            } 
           if ($instr =~ "eng" ) {
                $skip = 0;
                ($skip,$outline) = &proc_eng_oct($outfile);
#                print "skip = $skip   outline = $outline\n";
            } 
            if ($find[0] =~ "0006500" && $instr =~ "crs") {
               $spaces = "                                        ";
               $data_size = "3400 Bytes";
               $outline = "$find[0]$spaces$data_size\n";           #"4>$outlline"   (PRINTS)
               print OUTFILE $outline;
               $end = 1;
            }
            if ($find[0] =~ "0001240" && $instr =~ "pls") {
               open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
               $spaces = "         ";
               $data_size = "672 Bytes";
               $outline = "$line$spaces$data_size\n";             #"1>$outline"  (PRINTS)
               print OUTFILE $outline;
               $end = 1;
           }
           if ($find[0] =~ "0001060" && $instr =~ "lec") {
               open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
               $spaces = "      ";
               $data_size = "564 Bytes";
               $outline = "$line$spaces$data_size\n";             #"1>$outline"  (PRINTS)
               print OUTFILE $outline;
               $end = 1;
           }
           if ($find[0] =~ "0003240" && $instr =~ "mag") {
               open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
#               print OUTFILE "3240\n";
               $spaces = "      ";
               $data_size = "1712 Bytes";
               $outline = "$line$spaces$data_size\n";             #"1>$outline"  (PRINTS)
               print OUTFILE $outline;
               $end = 1; 
          }
          if ($find[0] =~ "0001000" && $instr =~ "pra") {
               open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
#               print OUTFILE "3240\n";
               $spaces = "      ";
               $data_size = "528 Bytes";
               $outline = "$line$spaces$data_size\n";             #"1>$outline"  (PRINTS)
               print OUTFILE $outline;
               $end = 1;
          }
           if ($find[0] =~ "0001020" && $instr =~ "pws") {
                        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
                        $spaces = "                    ";
                        $data_size = "540 Bytes";
                        $outline = "$find[0]$spaces$data_size\n";             #"1>$outline"  (PRINTS)
                        print OUTFILE $outline;
                        $end = 1;
                        next LINEA;
            }
#            if ($find[0] =~ "0000560" ){print "$find[0]\n"};
           if ($find[0] =~ "0000560" && $instr =~ "uvs") {
                        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
                        $spaces = "                                       ";
                        $data_size = "376 Bytes";
                        @llast = split (" ",$line );
                        $last_line = join (" ",$llast[0],$llast[1],$llast[2],$llast[3],$llast[4],$llast[5],$llast[6],$llast[7],$llast[8] );
                        $outline = "$last_line$spaces$data_size\n";             #"1>$outline"  (PRINTS)

                        print OUTFILE $outline;
                        $end = 1;
                        next LINEA;
            }
           if ($find[0] =~ "0007420" && $instr =~ "eng") {
                        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
                        $spaces = "                                       ";
                        $data_size = "3600 Bytes";
                        @llast = split (" ",$line );
                        $last_line = join (" ",$llast[0],$llast[1],$llast[2] );
                        $outline = "$last_line$spaces$data_size\n";             #"1>$outline"  (PRINTS)

                        print OUTFILE $outline;
                        $end = 1;
#                        next LINEA;
            }

         if ($end == 0) {
#            print OUTFILE "rec_rdr = $rec_rdr\n";
            if ($skip == 0) {
                print OUTFILE "$line\n";
            }
         }

     }   # end if byte

     goto NEXT if ($magic_num == 2);
#    
#  
}  #while
# close both files
NEXT:
close (INFILE);
close (OUTFILE);  

##############################################################################################################
# Process EDR STANDARD HEADER
##############################################################################################################
&proc_edrstd_hdr($infile,$outfile);
open(OUTFILE,">>$outfile") || die "Cannot find INFILE: $outfile\n";

print OUTFILE "A = Engineering Extraction Flag - word 2, bits 7,6\n";
print OUTFILE "B = S/C Playback Flag - word 2, bit 5\n";
print OUTFILE "C = Data Source - word 4, bits 7,6\n";
print OUTFILE "D = Golay Encoded Flag - word 3, bits 5,4\n";
print OUTFILE "    Segment Number - word 4, bits 3,0\n";
print OUTFILE "===========================================================\n";
close (OUTFILE);

################################################
# Process the EDR SUBHEADER for each instrument
################################################
    if ($instr =~ "crs" ) {
            &proc_crs_subhdr($infile,$outfile);
    }
    if ($instr =~ "lec" ) {
            &proc_lec_subhdr($infile,$outfile);
    }
    if ($instr =~ "mag" ) {
            &proc_mag_subhdr($infile,$outfile);
    }
    if ($instr =~ "pls" ) {
            &proc_pls_subhdr($infile,$outfile);
    }
    if ($instr =~ "pra" ) {
            &proc_pra_subhdr($infile,$outfile);
    }
    if ($instr =~ "pws" ) {
            &proc_pws_subhdr($infile,$outfile);
    }
    if ($instr =~ "uvs" ) {
            &proc_uvs_subhdr($infile,$outfile);
    }
    if ($instr =~ "eng" ) {
            &proc_eng_subhdr($infile,$outfile);
    }
    if ($instr =~ "dco" ) {
            &proc_dcom_subhdr($infile,$outfile);
    }

#    print "CRS data ========================\n";
############################################################
## Process abbreviated  EDR DATA records for each instrument
############################################################
    if ($instr =~ "crs" ) {
            &proc_crs_data($infile,$outfile);
    }
    if ($instr =~ "lec" ) {
            &proc_lec_data($infile,$outfile);
    }
    if ($instr =~ "mag" ) {
            &proc_mag_data($infile,$outfile);
    }
    if ($instr =~ "pls" ) {
             &proc_pls_data($infile,$outfile);
    }
    if ($instr =~ "pra" ) {
             &proc_pra_data($infile,$outfile);
    }
    if ($instr =~ "pws" ) {
             &proc_pws_data($infile,$outfile);
    }
    if ($instr =~ "uvs" ) {
             &proc_uvs_data($infile,$outfile);
    }
    if ($instr =~ "eng" ) {
            &proc_eng_data($infile,$outfile);
    }
    if ($instr =~ "dco" ) {
            &proc_dcom_data($infile,$outfile);
    }

#####################################################################################################
#     PROCESS EDR STD HEADERS FOR EACH INSTRUMENT
#####################################################################################################
#  proc_edrstd_hdr
#
#  EDR instrument had a standard edr header of 240 bytes
#
sub proc_edrstd_hdr
{
    my ($infile,$outfile) = @_;
#    print "$infile\n";
    $return = "\n"; 
    $space = " ";  
    open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
    open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

    print OUTFILE "=====================================================================\n";
    print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
    print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
    print OUTFILE "Standard EDR header record Table C-1 (pg C-135)\n";
    print OUTFILE "\n\n";
LINEB: while (<INFILE>) {
    chop;
    $line = $_;
    $last = $cur;
    @find = split (/\s+/, $line);           #split on spaces
    @ch = split ( //,$line);         #split on characters
    $cur = $find[0];
#fillers
    if ($find[0] =~ "0000000") {
    $filler1  = "                                    ";
    $filler2a = "               ";
    $filler2b = "                                  ";
    $filler3a = "                  ";
    $filler3b = "                             ";
    $filler4a = "                      ";
    $filler4b = "                           ";
    $filler5a = "                         ";
    $filler5b = "                        ";
    $filler6a = "                            ";
    $filler6b = "                   ";
    $filler7a = "                                 ";
    $filler7b = "              ";
    $filler8a = "                                      ";                                     
    $filler8b = "         ";
    $filler9a = "                                           ";
    $filler9b = "      ";
    $filler10a = "                                             ";
    $filler10b = "    ";
    $def1  = "Project ID         byte  0,1,2";
    $part1 = "$find[0]$space$find[1]$space$ch[13]$ch[14]$filler1$def1$return";
    $def2  = "RecID, SCID        byte  3";
    $part2 = "$filler2a$ch[15]$ch[16]$filler2b$def2$return";
    $def3  = "Physical Record ID byte  4,5";
    $part3 = "$filler3a$find[3]$filler3b$def3$return";
    $def4  = "S/C Data Mode      byte  6";
    $part4 = "$filler4a$ch[23]$ch[24]$filler4b$def4$return";
    $def5  = "A,B,spare          byte  7";
    $part5 = "$filler5a$ch[25]$ch[26]$filler5b$def5$return";
    $def6  = "Start ERT Hr       byte  8,9"; 
    $part6 = "$filler6a$find[5]$filler6b$def6$return";
    $def7  = "Start ERT Sec      byte  10,11";
    $part7 = "$filler7a$find[6]$filler7b$def7$return";
    $def8  = "Start ERT MSec     byte  12,13";
    $part8 = "$filler8a$find[7]$filler8b$def8$return";
    $def9  = "Start ERT Yr       byte  14";
    $part9 = "$filler9a$ch[43]$ch[44]$filler9b$def9$return";
    $def10 = "C,D Seg            byte  15";
    $part10= "$filler10a$ch[45]$ch[46]$filler10b$def10$return";

           $outline = "$part1$part2$part3$part4$part5$part6$part7$part8$part9$part10";   #"2>$outline" (PRINTS)
           print OUTFILE $outline;
                next LINEB;
    }
    if ($find[0] =~ "0000020") {
    $filler11  = "                                       ";
    $filler12a = "             ";
    $filler12b = "                                  ";
    $filler13a = "                  ";
    $filler13b = "                             ";
    $filler14a = "                       ";
    $filler14b = "                          ";
    $filler15a = "                         ";
    $filler15b = "                        ";
    $filler16a = "                            ";
    $filler16b = "                   ";
    $filler17a = "                                 ";
    $filler17b = "              ";
    $filler18a = "                                      ";
    $filler18b = "         ";
    $filler19a = "                                           ";
    $filler19b = "      ";
    $filler20a = "                                             ";
    $filler20b = "    ";
    $def11  = "End ERT Hr         byte 16,17";
    $part11 = "$find[0]$space$find[1]$filler11$def11$return";
    $def12  = "End ERT Sec        byte 18,19";
    $part12 = "$filler12a$find[2]$filler12b$def12$return";
    $def13  = "End ERT MSec       byte 20,21";
    $part13 = "$filler13a$find[3]$filler13b$def13$return";
    $def14  = "End ERT Year       byte 22";
    $part14 = "$filler14a$ch[23]$ch[24]$filler14b$def14$return";
    $def15  = "SW Version         byte 23";
    $part15 = "$filler15a$ch[25]$ch[26]$filler15b$def15$return";
    $def16  = "SCET Hr            byte 24,25";
    $part16 = "$filler16a$find[5]$filler16b$def16$return";
    $def17  = "SCET Sec           byte 26,27";
    $part17 = "$filler17a$find[6]$filler17b$def17$return";
    $def18  = "SCET MMsec         byte 28,29";
    $part18 = "$filler18a$find[7]$filler18b$def18$return";
    $def19  = "SCET Year          byte 30";
    $part19 = "$filler19a$ch[43]$ch[44]$filler19b$def19$return";
    $def20  = "SCE flg,FDS Corr Flg   byte 31";
    $part20 = "$filler20a$ch[45]$ch[46]$filler20b$def20$return";

          $outline = "$part11$part12$part13$part14$part15$part16$part17$part18$part19$part20";   #"2>$outline" (PRINTS)
           print OUTFILE $outline;
                next LINEB;

    }
    if ($find[0] =~ "0000040") {
    $filler21  = "                                       ";
    $filler22a = "             ";
    $filler22b = "                                    ";
    $filler23a = "               ";
    $filler23b = "                               ";
    $filler24a = "                    ";
    $filler24b = "                             ";
    $filler25a = "                       ";
    $filler25b = "                          ";
    $filler26a = "                         ";
    $filler26b = "                        ";
    $filler27a = "                            ";
    $filler27b = "                     ";
    $filler28a = "                              ";
    $filler28b = "                   ";
    $filler29a = "                                 ";
    $filler29b = "              ";
    $filler30a = "                                      ";
    $filler30b = "           ";
    $filler31a = "                                        ";
    $filler31b = "         ";
    $filler32a = "                                           ";
    $filler32b = "    ";
    $def21 = "MOD 2^16 Count      byte 32";
    $part21 = "$find[0]$space$find[1]$filler21$def21$return";
    $def22 = "MOD 2^60 Count      byte 34";
    $part22 = "$filler22a$ch[13]$ch[14]$filler22b$def22$return";
    $def23 = "Line Count          byte 35"; 
    $part23 = "$filler23a$ch[15]$ch[16]$space$ch[18]$ch[19]$filler23b$def23$return";
    $def24 = "Downlink Tlm Rate   byte37";
    $part24 = "$filler24a$ch[20]$ch[21]$filler24b$def24$return";
    $def25 = "Effective Bit Rate  byte 38";
    $part25 = "$filler25a$ch[23]$ch[24]$filler25b$def25$return";
    $def26 = "FID                 byte 39";
    $part26 = "$filler26a$ch[25]$ch[26]$filler26b$def26$return";
    $def27 = "BET                 byte 40";
    $part27 = "$filler27a$ch[28]$ch[29]$filler27b$def27$return";
    $def28 = "DSN Config          byte 41";
    $part28 = "$filler28a$ch[30]$ch[31]$filler28b$def28$return";
    $def29 = "Ground Recvr AGC    byte 42";
    $part29 = "$filler29a$find[6]$filler29b$def29$return";
    $def30 = "DSN Stn             byte 44";
    $part30 = "$filler30a$ch[38]$ch[39]$filler30b$def30$return";
    $def31 = "Spare               byte 45";
    $part31 = "$filler31a$ch[40]$ch[41]$filler31b$def31$return";
    $def32 = "PN EBEC             byte 46";
    $part32 = "$filler32a$find[8]$filler32b$def32$return";

           $outline = "$part21$part22$part23$part24$part25$part26$part27$part28$part29$part30$part31$part32";   #"2>$outline" (PRINTS)
           print OUTFILE $outline;
                next LINEB;


    }
    if ($find[0] =~ "0000060") {
    $filler33  = "                                       ";
    $filler34a = "             ";
    $filler34b = "                                  ";
    $filler35a = "                  ";
    $filler35b = "                             ";
    $filler36a = "                       ";
    $filler36b = "                        ";
    $filler37a = "                           ";
    $filler37b = "                    ";
    $filler38a = "                                 ";
    $filler38b = "              ";
    $filler39a = "                                      ";
    $filler39b = "         ";
    $filler40a = "                                           ";
    $filler40b = "    ";
    $def33 = "Symbol SNR          byte 48";
    $part33 = "$find[0]$space$find[1]$filler33$def33$return";
    $def34 = "Decoder SNR         byte 50";
    $part34 = "$filler34a$find[2]$filler34b$def34$return";
    $def35 = "Physical Record Number/Unused";
    $part35 = "$filler35a$find[3]$filler35b$def35$return";
    $def36 = "DSQW MF1-MF4        byte 54"; 
    $part36 = "$filler36a$find[4]$filler36b$def36$return";
    $def37 = "DSQW MF5-MF8        byte 56";
    $part37 = "$filler37a$find[5]$filler37b$def37$return";
    $def38 = "DSQW MF9-MF12       byte 58";
    $part38 = "$filler38a$find[6]$filler38b$def38$return";
    $def39 = "DSQW MF13-MF16      byte 60";
    $part39 = "$filler39a$find[7]$filler39b$def39$return";
    $def40 = "DSQW MF13-MF16      byte 62";
    $part40 = "$filler40a$find[8]$filler40b$def40$return";
    
           $outline = "$part33$part34$part35$part36$part37$part38$part39$part40";   #"2>$outline" (PRINTS)
           print OUTFILE $outline;
                next LINEB;


    }
    if ($find[0] =~ "0000100") {
    $filler41  = "                                       ";
    $filler42a = "             ";
    $filler42b = "                                  ";
    $filler43a = "                  ";
    $filler43b = "                             ";
    $filler44a = "                       ";
    $filler44b = "                        ";
    $filler45a = "                            ";
    $filler45b = "                   ";
    $filler46a = "                                 ";
    $filler46b = "                ";
    $filler47a = "                                   ";
    $filler47b = "              ";
    $filler48a = "                                      ";
    $filler48b = "           ";
    $filler49a = "                                        ";
    $filler49b = "         ";
    $filler50a = "                                           ";
    $filler50b = "      ";
    $filler51a = "                                             ";
    $filler51b = "    ";
    $def41 = "DSQW MF21-MF24       byte 64";
    $part41 = "$find[0]$space$find[1]$filler41$def41$return";
    $def42 = "DSQW MF25-MF28       byte 66";
    $part42 = "$filler42a$find[2]$filler42b$def42$return";
    $def43 = "DSQW MF29-MF32       byte 68";
    $part43 = "$filler43a$find[3]$filler43b$def43$return";
    $def44 = "DSQW MF33-MF36       byte 70";
    $part44 = "$filler44a$find[4]$filler44b$def44$return"; 
    $def45 = "DSQW MF37-MF40       byte 72";
    $part45 = "$filler45a$find[5]$filler45b$def45$return";
    $def46 = "DPI MF1              byte 73";
    $part46 = "$filler46a$ch[33]$ch[34]$filler46b$def46$return";
    $def47 = "DPI MF2              byte 74";
    $part47 = "$filler47a$ch[35]$ch[36]$filler47b$def47$return";
    $def48 = "DPI MF3              byte 75";
    $part48 = "$filler48a$ch[38]$ch[39]$filler48b$def48$return";
    $def49 = "DPI MF4              byte 76";
    $part49 = "$filler49a$ch[40]$ch[41]$filler49b$def49$return";
    $def50 = "DPI MF5              byte 77";
    $part50 = "$filler50a$ch[43]$ch[44]$filler50b$def50$return";
    $def51 = "DPI MF6              byte 78";
    $part51 = "$filler51a$ch[45]$ch[46]$filler51b$def51$return";

           $outline = "$part41$part42$part43$part44$part45$part46$part47$part48$part49$part50$part51";   #"2>$outline" (PRINTS)
           print OUTFILE $outline;
                next LINEB;


    }
    if ($find[0] =~ "0000120") {
    $filler52  = "                                       ";
    $filler53a = "             ";
    $filler53b = "                                  ";
    $filler54a = "                  ";
    $filler54b = "                             ";
    $filler55a = "                       ";
    $filler55b = "                        ";
    $filler56a = "                           ";
    $filler56b = "                    ";
    $filler57a = "                                 ";
    $filler57b = "              ";
    $filler58a = "                                      ";
    $filler58b = "         ";
    $filler59a = "                                           ";
    $filler59b = "    ";
    $def52 = "DPI MF7             byte 48";
    $part52 = "$find[0]$space$find[1]$filler52$def52$return";
    $def53 = "DPI MF8             byte 50";
    $part53 = "$filler53a$find[2]$filler53b$def53$return";
    $def54 = "DPI MF9                      ";
    $part54 = "$filler54a$find[3]$filler54b$def54$return";
    $def55 = "DPI MF10             byte 54";
    $part55 = "$filler55a$find[4]$filler55b$def55$return";
    $def56 = "DPI MF11             byte 56";
    $part56 = "$filler56a$find[5]$filler56b$def56$return";
    $def57 = "DPI MF12             byte 58";
    $part57 = "$filler57a$find[6]$filler57b$def57$return";
    $def58 = "DPI MF13             byte 60";
    $part58 = "$filler58a$find[7]$filler58b$def58$return";
    $def59 = "DPI MF14             byte 60";
    $part59 = "$filler59a$find[8]$filler59b$def59$return";

           $outline = "$part52$part53$part54$part55$part56$part57$part58$part59";   #"2>$outline" (PRINTS)
           print OUTFILE $outline;
                next LINEB;


    }
    if ($find[0] =~ "0000140") {
    $filler60  = "                                       ";
    $filler61a = "             ";
    $filler61b = "                                  ";
    $filler62a = "                  ";
    $filler62b = "                             ";
    $filler63a = "                       ";
    $filler63b = "                        ";
    $filler64a = "                           ";
    $filler64b = "                    ";
    $filler65a = "                                 ";
    $filler65b = "              ";
    $filler66a = "                                      ";
    $filler66b = "         ";
    $filler67a = "                                           ";
    $filler67b = "    ";
    $def60 = "DPI MF15            byte 48";
    $part60 = "$find[0]$space$find[1]$filler60$def60$return";
    $def61 = "DPI MF16            byte 50";
    $part61 = "$filler61a$find[2]$filler61b$def61$return";
    $def62 = "DPI MF17                     ";
    $part62 = "$filler62a$find[3]$filler62b$def62$return";
    $def63 = "DPI MF18             byte 54";
    $part63 = "$filler63a$find[4]$filler63b$def63$return";
    $def64 = "DPI MF19             byte 56";
    $part64 = "$filler64a$find[5]$filler64b$def64$return";
    $def65 = "DPI MF20             byte 58";
    $part65 = "$filler65a$find[6]$filler65b$def65$return";
    $def66 = "DPI MF21             byte 60";
    $part66 = "$filler66a$find[7]$filler66b$def66$return";
    $def67 = "DPI MF22             byte 60";
    $part67 = "$filler67a$find[8]$filler67b$def67$return";

           $outline = "$part60$part61$part62$part63$part64$part65$part66$part67";   #"2>$outline" (PRINTS)
           print OUTFILE $outline;
                next LINEB;


    }
    if ($find[0] =~ "0000160") {
    $filler70  = "                                       ";
    $filler71a = "             ";
    $filler71b = "                                  ";
    $filler72a = "                  ";
    $filler72b = "                             ";
    $filler73a = "                       ";
    $filler73b = "                        ";
    $filler74a = "                           ";
    $filler74b = "                    ";
    $filler75a = "                                 ";
    $filler75b = "              ";
    $filler76a = "                                      ";
    $filler76b = "         ";
    $filler77a = "                                           ";
    $filler77b = "    ";
    $def70 = "DPI MF23            byte 48";
    $part70 = "$find[0]$space$find[1]$filler70$def70$return";
    $def71 = "DPI MF24            byte 50";
    $part71 = "$filler71a$find[2]$filler71b$def71$return";
    $def72 = "DPI MF25                     ";
    $part72 = "$filler72a$find[3]$filler72b$def72$return";
    $def73 = "DPI MF26             byte 54";
    $part73 = "$filler73a$find[4]$filler73b$def73$return";
    $def74 = "DPI MF27             byte 56";
    $part74 = "$filler74a$find[5]$filler74b$def74$return";
    $def75 = "DPI MF28             byte 58";
    $part75 = "$filler75a$find[6]$filler75b$def75$return";
    $def76 = "DPI MF29             byte 60";
    $part76 = "$filler76a$find[7]$filler76b$def76$return";
    $def77 = "DPI MF30             byte 60";
    $part77 = "$filler77a$find[8]$filler77b$def77$return";

           $outline = "$part70$part71$part72$part73$part74$part75$part76$part77";   #"2>$outline" (PRINTS)
           print OUTFILE $outline;
                next LINEB;


    }
    if ($find[0] =~ "0000200") {
    $filler80  = "                                       ";
    $filler81a = "             ";
    $filler81b = "                                  ";
    $filler82a = "                  ";
    $filler82b = "                             ";
    $filler83a = "                       ";
    $filler83b = "                        ";
    $filler84a = "                           ";
    $filler84b = "                    ";
    $filler85a = "                                 ";
    $filler85b = "              ";
    $filler86a = "                                      ";
    $filler86b = "         ";
    $filler87a = "                                           ";
    $filler87b = "    ";
    $def80 = "DPI MF31            byte 48";
    $part80 = "$find[0]$space$find[1]$filler80$def80$return";
    $def81 = "DPI MF32            byte 50";
    $part81 = "$filler81a$find[2]$filler81b$def81$return";
    $def82 = "DPI MF33                     ";
    $part82 = "$filler82a$find[3]$filler82b$def82$return";
    $def83 = "DPI MF34             byte 54";
    $part83 = "$filler83a$find[4]$filler83b$def83$return";
    $def84 = "DPI MF35             byte 56";
    $part84 = "$filler84a$find[5]$filler84b$def84$return";
    $def85 = "DPI MF36             byte 58";
    $part85 = "$filler85a$find[6]$filler85b$def85$return";
    $def86 = "DPI MF37             byte 60";
    $part86 = "$filler86a$find[7]$filler86b$def86$return";
    $def87 = "DPI MF38             byte 60";
    $part87 = "$filler87a$find[8]$filler87b$def87$return";

           $outline = "$part80$part81$part82$part83$part84$part85$part86$part87";   #"2>$outline" (PRINTS)
           print OUTFILE $outline;
                next LINEB;


    }
    if ($find[0] =~ "0000220") {
    $filler90  = "                                       ";
    $filler91a = "             ";
    $filler91b = "                                  ";
    $filler92a = "                  ";
    $filler92b = "                             ";
    $filler83a = "                       ";
    $filler83b = "                        ";
    $filler84a = "                           ";
    $filler84b = "                    ";
    $filler85a = "                                 ";
    $filler85b = "              ";
    $filler86a = "                                      ";
    $filler86b = "         ";
    $filler87a = "                                           ";
    $filler92b = "    ";
    $def90 = "DPI MF39            byte 48";
    $part90 = "$find[0]$space$find[1]$filler90$def90$return";
    $def91 = "DPI MF40            byte 50";
    $part91 = "$filler91a$find[2]$filler91b$def91$return";
    $def92 = "SPARE                      ";
    $part92 = "$filler92a$find[3]$space$find[4]$space$find[5]$space$find[6]$space$find[7]$space$find[8]$filler92b$def92$return";

           $outline = "$part90$part91$part92";   #"2>$outline" (PRINTS)
           print OUTFILE $outline;
                next LINEB;


    }
    if ($find[0] =~ "0000340") {
    $filler93  = "              ";    
    $filler94a = "                                      ";
    $filler94b = "           ";
    $filler95a = "                                        ";
    $filler95b = "    ";
    $def93 = "SPARE                      ";
    $part93 = "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]$space$find[5]$space$find[6]$filler93$def93$return";
    $def94 = "DRS Data Type         ";
    $part94 = "$filler94a$ch[38]$ch[39]$filler94b$def94$return";
    $def95 = "SPARE                      ";
    $part95 = "$filler95a$ch[40]$ch[41]$space$find[8]$filler95b$def95$return";
           $outline = "$part93$part94$part95";   #"2>$outline" (PRINTS)
           print OUTFILE $outline;
                next LINEB;

    }
    return if ($find[0] =~ "0000360");
  }
}
####################################################################################
#  proc_crs_hex
#   Dump CRS Block Header
#
sub proc_crs_hex 

{
    my ($outfile) = @_;
    $skip = 0;
     $crs_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
# check for leading star - means all zeroes        
     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $crs_cnt =  $diff + $crs_cnt;
           }
     }
     $last = $cur;
#     if ($crs_cnt == 16) {
#          print OUTFILE "sub_hdr = $crs_subhdr bytes\n";           #40
#     }
#     if ($find[0] =~ "0000360") {
#        print OUTFILE "$line$return";
#
#     }
     if ($find[0] =~ "0000360") {
           $data_start = 5;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
            $hdr_marker = "Header = 240 bytes";
            $outline = "Sub_Header = $crs_subhdr bytes";
           $outlinex =  "$hdr_marker$return$sub_hdr_end$space$data_lead$return";  

           print OUTFILE $outlinex;
           $skip= 1;
     }
     if ($find[0] =~ "0000420") {
             $filler2 = "                            ";


            $lead1 = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
            $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
            $sub1 = "Sub_Header = $crs_subhdr bytes";
            $outlinex =  "$lead1$return$sub1$return$filler2$data_lead$return";
            print OUTFILE $outlinex;

    }
#    close (OUTFILE);
    return $skip,$outline;
}
####################################################################################
##  proc_crs_oct
#
sub proc_crs_oct 
{
    my ($outfile) = @_;
    $skip = 0;
     $crs_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $crs_cnt =  $diff + $crs_cnt;
           }
     }
     $last = $cur;
     if ($crs_cnt == 16) {
           print OUTFILE "sub_hdr = $crs_subhdr bytes\n";       #40
     }
     if ($find[0] =~ "0000420") {
           $data_start = 9;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4],$find[5],$find[6],$find[7],$find[8] ); 
           $data_lead = join (" ",$find[9],$find[10],$find[11],$find[12],$find[13],$find[14],$find[15],$find[16] );
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                                        ";
           $data_marker = " data = $crs_data bytes";            #3120
           $outline =  "$sub_hdr_end$return$data_marker$return$filler2$data_lead\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip = 1;
     }
#    close (OUTFILE);
    return $skip,$outline
}
####################################################################################
#  proc_pls_hex
#  Dump PLS Block Header
#
sub proc_pls_hex
{
    my ($outfile) = @_;
    $skip = 0;
     $pls_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $pls_cnt =  $diff + $pls_cnt;
           }
     }
     $last = $cur;
#     if ($pls_cnt == 16) {
#           print OUTFILE "sub_hdr = $pls_subhdr bytes\n";       #32
#     }
     if ($find[0] =~ "0000360") {
           $hdr_marker = "Header = 240 bytes";
           $outline =  "$hdr_marker$return$line$return";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip = 1;
     }
     if ($find[0] =~ "0000400") {
             $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
           $sub_hdr_end = join (" ",$sub_hdr_end,$data_lead);
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
#            $hdr_marker = "Header = 240 bytes";
            $sub1 = "Sub_Header = $pls_subhdr bytes";

           $outline =  "$sub_hdr_end$return$sub1\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip = 1;



     }
#    close (OUTFILE);
    return $skip,$outline;
}

####################################################################################
#  proc_pls_oct
#
sub proc_pls_oct
{
    my ($outfile) = @_;
    $skip = 0;
     $pls_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $pls_cnt =  $diff + $pls_cnt;
           }
     }
     $last = $cur;
     if ($pls_cnt == 16) {
           print OUTFILE "sub_hdr = $pls_subhdr bytes\n";       #32
     }
     if ($find[0] =~ "0000400") {
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4],$find[5],$find[6],$find[7],$find[8] );
           $data_lead = join (" ",$find[9],$find[10],$find[11],$find[12],$find[13],$find[14],$find[15],$find[16] );
             $sub_hdr_end = join (" ",$sub_hdr_end,$data_lead);
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
           $data_marker = " data = $pls_data bytes";        #400
           $outline =  "$sub_hdr_end$return$data_marker\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip = 1;
     }
#    close (OUTFILE);
    return $skip,$outline;

}
####################################################################################
#  proc_lec_hex
#  Dump LECP Block Header 
#
sub proc_lec_hex
{
    my ($outfile) = @_;
     $skip = 0;
     $lec_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $lec_cnt =  $diff + $lec_cnt;
           }
     }
     $last = $cur;
#     if ($lec_cnt == 16) {
#           print OUTFILE "sub_hdr = $lec_subhdr bytes\n";       #4
#     }
#        print "sub: skip = $skip\n";
     if ($find[0] =~ "0000360") {
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2]);  #,$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[3],$find[4],$find[5],$find[6],$find[7],$find[8] );
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                  ";
#           $data_marker = "data = $lec_data bytes";             #320
            $hdr_marker = "Header = 240 bytes";
            $outline = "Sub_Header = $lec_subhdr bytes";
           $outlinex =  "$hdr_marker$return$sub_hdr_end$return$outline$return$filler2$data_lead\n";   #"7>"$outline  (PRINTS)
           $skip = 1;
           print OUTFILE "$outlinex";
     }
#    print "sub2: skip = $skip\n";
#    close (OUTFILE);

    return $skip,$outline;

}

####################################################################################
#  proc_lec_oct
#  
sub proc_lec_oct
{
    my ($outfile) = @_;
     $skip = 0;
     $lec_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $lec_cnt =  $diff + $lec_cnt;
           }
     }
     $last = $cur;
     if ($lec_cnt == 16) {
           print OUTFILE "sub_hdr = $lec_subhdr bytes\n";       #4
     }
     if ($find[0] =~ "0000360") {
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8],$find[9],$find[10], $find[11],$find[12],$find[13],$find[14],$find[15],$find[16] );
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                        ";
           $data_marker = "data = $lec_data bytes";             #320
           $outline = "$sub_hdr_end$return$data_marker$return$filler2$data_lead\n" ;   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip = 1;
     }
#    close (OUTFILE);
    return $skip,$outline;

}
####################################################################################
#  proc_mag_hex
#   Dump MAG Block Header  
sub proc_mag_hex
{
    my ($outfile) = @_;
    $skip = 0;
     $mag_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $mag_cnt =  $diff + $mag_cnt;
           }
     }
     $last = $cur;
#     if ($mag_cnt == 16) {
#           print OUTFILE "sub_hdr = $mag_subhdr bytes\n";           #420
#     a
#    }
     if ($find[0] =~ "0000360") {
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
            $hdr_marker = "Header = 240 bytes";
#            $outline = "Sub_Header = $mag_subhdr bytes";
           $outlinex =  "$hdr_marker$return$sub_hdr_end$space$data_lead$return";  
           print OUTFILE $outlinex;
           $skip= 1;
     }
     if ($find[0] =~ "0001220") {
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
           $sub_hdr_end = join (" ",$sub_hdr_end,$data_lead);
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
           $hdr_marker = "Sub_Header = $mag_subhdr bytes";            #1040
           $outline =  "$sub_hdr_end$return$hdr_marker\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip = 1;
     }
#    close (OUTFILE);
    return $skip,$outline;
}
####################################################################################
#  proc_mag_oct
#  
sub proc_mag_oct
{
    my ($outfile) = @_;
    $skip = 0;
     $mag_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $mag_cnt =  $diff + $mag_cnt;
           }
     }
     $last = $cur;
     if ($mag_cnt == 16) {
           print OUTFILE "sub_hdr = $mag_subhdr bytes\n";           #420
     }
     if ($find[0] =~ "0001220") {
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8],$find[9],$find[10], $find[11],$find[12],$find[13],$find[14],$find[15],$find[16] );
           $sub_hdr_end = join (" ",$sub_hdr_end,$data_lead); 
          $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
           $data_marker = " data = $mag_data bytes";            #1040
           $outline =  "$sub_hdr_end$return$data_marker\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip = 1;
     }
#    close (OUTFILE);
     return $skip,$outline;
}
####################################################################################
#  proc_pra_hex
#   Dump PRA Block Header
#
sub proc_pra_hex
{
    my ($outfile) = @_;
    $skip = 0;
     $pra_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $pra_cnt =  $diff + $pra_cnt;
           }
     }
     $last = $cur;
#     if ($pra_cnt == 16) {
#           print OUTFILE "sub_hdr = $pra_subhdr bytes\n";           #8
#     }

     if ($find[0] =~ "0000360") {
           $data_start = 5;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
            $hdr_marker = "Header = 240 bytes";
            $outline = "Sub_Header = $pra_subhdr bytes";
           $outlinex =  "$hdr_marker$return$sub_hdr_end$return$outline$return$filler2$data_lead\n";   #"7>"$outline  (PRINTS)
           print OUTFILE "$outlinex";

           $skip= 1;
     }

#     if ($find[0] =~ "0000360") {
#           $data_start = 5;
#           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
#           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
#           $posit2 = index($line,$sub_hdr_end);
#           $filler2 = "                            ";
#           $data_marker = " data = $pra_data bytes";            #248
#           $outline =  "$sub_hdr_end$return$data_marker$return$filler2$data_lead\n";   #"7>"$outline  (PRINTS)
#           print OUTFILE $outline;
#           $skip = 1;
#     }
#    close (OUTFILE);
     return $skip,$outline;
#

}
####################################################################################
#  proc_pra_oct
#  
sub proc_pra_oct
{
    my ($outfile) = @_;
    $skip = 0;
     $pra_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $pra_cnt =  $diff + $pra_cnt;
           }
     }
     $last = $cur;
     if ($pra_cnt == 16) {
           print OUTFILE "sub_hdr = $pra_subhdr bytes\n";           #8
     }
     if ($find[0] =~ "0000360") {
           $data_start = 9;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4],$find[5],$find[6],$find[7],$find[8] );
           $data_lead = join (" ",$find[9],$find[10], $find[11],$find[12],$find[13],$find[14],$find[15],$find[16] );
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                                        ";
           $data_marker = " data = $pra_data bytes";            #220
           $outline =  "$sub_hdr_end$return$data_marker$return$filler2$data_lead\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip = 1;
     }
#    close (OUTFILE);
     return $skip,$outline;
#

}
####################################################################################
#  proc_pws_hex
#  Dump PWS BLock Header
#
sub proc_pws_hex
{
    my ($outfile) = @_;
    $skip = 0;
     $pws_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $pws_cnt =  $diff + $pws_cnt;
           }
     }
     $last = $cur;
#     if ($pws_cnt == 16) {
#           print OUTFILE "sub_hdr = $pws_subhdr bytes\n";           #8
#     }
     if ($find[0] =~ "0000360") {
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
           $sub_hdr_end = join (" ",$sub_hdr_end,$data_lead);
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
           $hdr_marker = "Header = 240 bytes";
           $outlinex =  "$hdr_marker$return$sub_hdr_end\n";   #"7>"$outline  (PRINTS)
           print OUTFILE "$outlinex";
           $skip = 1;
     }

     if ($find[0] =~ "0000460") {
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
           $sub_hdr_end = join (" ",$sub_hdr_end,$data_lead);
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
           $hdr_marker = "Sub_Header = $pws_subhdr bytes";            #1040
           $outline =  "$sub_hdr_end$return$hdr_marker\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip = 1;
 #    }
     }
#    close (OUTFILE);
     return $skip,$outline;
#
}
####################################################################################
#  proc_pws_oct
#  
sub proc_pws_oct
{
    my ($outfile) = @_;
    $skip = 0;
     $pws_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $pws_cnt =  $diff + $pws_cnt;
           }
     }
     $last = $cur;
     if ($pws_cnt == 16) {
           print OUTFILE "sub_hdr = $pws_subhdr bytes\n";           #8
     }
     if ($find[0] =~ "0000500") {
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8],$find[9],$find[10], $find[11],$find[12],$find[13],$find[14],$find[15],$find[16] );
            $sub_hdr_end = join (" ",$sub_hdr_end,$data_lead); 
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
            $hdr_marker = "Header = 240 bytes";
            $outline = "Sub_Header = $pws_subhdr bytes";
           $outlinex =  "$hdr_marker$return$sub_hdr_end$return$outline$return$filler2$data_lead\n";   #"7>"$outline  (PRINTS)
           print OUTFILE "$outlinex";
           $skip = 1;
     }
#    close (OUTFILE);
     return $skip,$outline;
#

}
####################################################################################
#  proc_uvs_hex
#  Dump UVS Block Header
#
sub proc_uvs_hex
{
    my ($outfile) = @_;
    $skip = 0;
     $uvs_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $uvs_cnt =  $diff + $uvs_cnt;
           }
     }
     $last = $cur;
#     if ($uvs_cnt == 16) {
#           print OUTFILE "sub_hdr = $uvs_subhdr bytes\n";           #16
#     }
     if ($find[0] =~ "0000360") {
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
           $sub_hdr_end = join (" ",$sub_hdr_end,$data_lead);
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
           $filler2 = "                            ";
            $hdr_marker = "Header = 240 bytes";
            $outline = "Sub_Header = $uvs_subhdr bytes";
           $outlinex =  "$hdr_marker$return$sub_hdr_end$return$outline\n";   #"7>"$outline  (PRINTS)
           print OUTFILE "$outlinex";
#           $data_marker = " data = $uvs_data bytes";            #120
#           $outline =  "$sub_hdr_end$return$data_marker\n";   #"7>"$outline  (PRINTS)
#           print OUTFILE $outline;
           $skip = 1;
     }
#    close (OUTFILE);
     return $skip,$outline;
}
####################################################################################
#  proc_uvs_oct
#  
sub proc_uvs_oct
{
    my ($outfile) = @_;
    $skip = 0;
     $uvs_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $uvs_cnt =  $diff + $uvs_cnt;
           }
     }
     $last = $cur;
     if ($uvs_cnt == 16) {
           print OUTFILE "sub_hdr = $uvs_subhdr bytes\n";           #16
     }
     if ($find[0] =~ "0000360") {
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8],$find[9],$find[10], $find[11],$find[12],$find[13],$find[14],$find[15],$find[16] );
           $sub_hdr_end = join (" ",$sub_hdr_end,$data_lead);
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
           $data_marker = " data = $uvs_data bytes";            #120
           $outline =  "$sub_hdr_end$return$data_marker\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip = 1;
     }
#    close (OUTFILE);
     return $skip,$outline;
#
}
####################################################################################
#  proc_eng_hex
#  Dump ENG Block Header
#
sub proc_eng_hex
{
    my ($outfile) = @_;
    $skip = 0;
     $eng_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $eng_cnt =  $diff + $eng_cnt;
           }
     }
     $last = $cur;
#     if ($eng_cnt == 16) {
#           print OUTFILE "sub_hdr = $eng_subhdr bytes\n";           #0
#     }
     if ($find[0] =~ "0000360") {
#        print OUTFILE "0000360 and eng\n";
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
           $sub_hdr_end1 = join (" ",$sub_hdr_end,$data_lead);
           $posit2 = index($line,$sub_hdr_end1);
           $filler2 = "                            ";
           $hdr_marker = "Header = 240 bytes";
           $outlinex =  "$hdr_marker\n";   #"7>"$outline  (PRINTS)
           print OUTFILE "$outlinex";
           $skip = 1;
     }
    if ($find[0] =~ "0000400") {
#        print OUTFILE "000400 and eng\n";
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2] );
#           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
#           $sub_hdr_end = join (" ",$sub_hdr_end,$data_lead);
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
           $outlinex =  "$sub_hdr_end\n";   #"7>"$outline  (PRINTS)
#           print OUTFILE "$outlinex";
           $hdr_marker = "Sub_Header = $eng_subhdr bytes";            #1040
           $outline =  "$sub_hdr_end$return$hdr_marker\n";   #"7>"$outline  (PRINTS)
           print OUTFILE "$outline";
           $skip = 1;
     }

     if ($find[0] =~ "0007420") {
           $data_start = 0;
#           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2] );
#           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
#           $data_marker = " data = $eng_data bytes";            #248
            $outlinex=  "$sub_hdr_end\n";   #"7>"$outline  (PRINTS)
#           print OUTFILE ">##> $outlinex";
           $skip = 1;
        return $skip,$outline;;
   }
#    close (OUTFILE);
#
}

####################################################################################
#  proc_eng_oct
#  
sub proc_eng_oct
{
    my ($outfile) = @_;
    $skip = 0;
     $eng_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $eng_cnt =  $diff + $eng_cnt;
           }
     }
     $last = $cur;
     if ($eng_cnt == 16) {
           print OUTFILE "sub_hdr = $eng_subhdr bytes\n";           #0
     }
     if ($find[0] =~ "0001300") {
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8],$find[9],$find[10], $find[11],$find[12],$find[13],$find[14],$find[15],$find[16] );
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
           $data_marker = " data = $eng_data bytes";            #248
           $outline =  "$data_marker\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip = 1;
     }
#    close (OUTFILE);
     return $skip,$outline;
#
}
####################################################################################
####    proc_dcom_hex
##  Dump DCOM BLock Header
#
sub proc_dcom_hex
{
    my ($outfile) = @_;
    $skip = 0;
    $dcom_cnt++;
        open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     if ( $cur =~ m/^\*/ ) {
           $starfound = 1;
           $cur = $last;
     } else {
           $starfound = 0;
           $diff = (($cur - $last)/20) - 3;
           if ($diff > 2) {
                 $dcom_cnt =  $diff + $dcom_cnt;
           }
     }
     $last = $cur;

     if ($find[0] =~ "0000360") {
           $data_start = 0;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
           $sub_hdr_end = join (" ",$sub_hdr_end,$data_lead);
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
           $hdr_marker = "Header = 240 bytes";
           $outlinex =  "$hdr_marker$return$sub_hdr_end\n";   #"7>"$outline  (PRINTS)
           print OUTFILE "$outlinex";
           $skip = 1;
     }

     if ($find[0] =~ "0000400") {
           $data_start = 0;
        $filler1 = "                                      ";
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6] );
           $sub_hdr_end = join (" ",$sub_hdr_end,$data_lead);
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                      ";
           $hdr_marker = "Sub_Header = $dcom_subhdr bytes";            #1040
           $outline =  "$sub_hdr_end$return$hdr_marker\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
            $outline = "$filler1$find[7]$space$find[8]$return";
            print OUTFILE $outline;
           $skip = 1;
     }

#    close (OUTFILE);
     return $skip,$outline;
 #              #
}
###################################################################################################
#     PROCESS EDR STD SUBHEADERS FOR EACH INSTRUMENT
###################################################################################################
###################################################################################
##  proc_crs_subhdr
##
##  subhdr for CRS instrument is 40 bytes
##
sub proc_crs_subhdr
{
#The CRS EDR physical science record corresponds to a 9.6 minute measurement period, or 60 minor frames.
#618-306 Rev D chg 5 Voyager Experiment Data Record Format
#Figure 11.1-3 defines the structure and contents of the CRS science record subheader for CR-5A (40 bytse).
#
    my ($infile,$outfile) = @_;
#    print "$infile\n";
    $return = "\n";
    $space = " ";
    open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
    open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

    print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
    print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
    print OUTFILE "Standard EDR header record subheader for CRS (40 bytes) Figure 11.1-3 (pg 11-3)\n";
    print OUTFILE "CRS_SUBHEADER::\n\n";
LINEC: while (<INFILE>) {
    chop;
    $line = $_;
    $last = $cur;
    @find = split (/\s+/, $line);           #split on spaces
    @ch = split ( //,$line);         #split on characters
    $cur = $find[0];

    if ($find[0] =~ "0000360") {
        $filler1  = "                                      ";
        $filler2a = "             ";
        $filler2b = "                                 ";
        $filler3a = "                  ";
        $filler3b = "                            ";
        $filler4a = "                       ";
        $filler4b = "                       ";
        $filler5a = "                            ";
        $filler5b = "                  ";
        $filler6a = "                                 ";
        $filler6b = "             ";
        $filler7a = "                                      ";
        $filler7b = "        ";
        $filler8a = "                                           ";
        $filler8b = "   ";

        $def1 = "CRS COMMAND         MF19      ";
        $def2 = "CRS COMMAND         MF39      ";
        $def3 = "CRS COMMAND         MF59      ";
        $def4 = "SPARE               ";
        $def5 = "CRS STATUS-1        MF1      ";
        $def6 = "CRS STATUS-2        MF6      ";
        $def7 = "CRS STATUS-3        MF11     ";
        $def8 = "CRS STATUS-4        MF16     ";


        $part1  = "$find[0]$space$find[1]$filler1$def1$return";
        $part2  = "$filler2a$find[2]$filler2b$def2$return";
        $part3  = "$filler3a$find[3]$filler3b$def3$return";
        $part4  = "$filler4a$find[4]$filler4b$def4$return";
        $part5  = "$filler5a$find[5]$filler5b$def5$return";
        $part6  = "$filler6a$find[6]$filler6b$def6$return";
        $part7  = "$filler7a$find[7]$filler7b$def7$return";
        $part8  = "$filler8a$find[8]$filler8b$def8$return";
        $outline = "$part1$part2$part3$part4$part5$part6$part7$part8";
        print OUTFILE $outline;
        goto LINEC;
    }

    if ($find[0] =~ "0000400") {
        $filler1  = "                                      ";
        $filler2a = "             ";
        $filler2b = "                                 ";
        $filler3a = "                  ";
        $filler3b = "                            ";
        $filler4a = "                       ";
        $filler4b = "                       ";
        $filler5a = "                            ";
        $filler5b = "                  ";
        $filler6a = "                                 ";
        $filler6b = "             ";
        $filler7a = "                                      ";
        $filler7b = "        ";
        $filler8a = "                                           ";
        $filler8b = "   ";

        $def5 = "CRS STATUS-9        MF21     ";
        $def6 = "CRS STATUS-10       MF26     ";
        $def7 = "CRS STATUS-11       MF31     ";
        $def8 = "CRS STATUS-12       MF36     ";
        $def9 = "CRS STATUS-9        MF41     ";
        $def10 = "CRS STATUS-10      MF46     ";
        $def11 = "CRS STATUS-11      MF51     ";
        $def12 = "CRS STATUS-12      MF56     ";


        $part5  = "$find[0]$space$find[1]$filler1$def5$return";
        $part6  = "$filler2a$find[2]$filler2b$def6$return";
        $part7  = "$filler3a$find[3]$filler3b$def7$return";
        $part8  = "$filler4a$find[4]$filler4b$def8$return";
        $part9  = "$filler5a$find[5]$filler5b$def9$return";
        $part10  = "$filler6a$find[6]$filler6b$def10$return";
        $part11  = "$filler7a$find[7]$filler7b$def11$return";
        $part12  = "$filler8a$find[8]$filler8b$def12$return";


        $outline = "$part5$part6$part7$part8$part9$part10$part11$part12";
        print OUTFILE $outline;
        goto LINEC;
    }

    if ($find[0] =~ "0000420") {
        $filler10a = "                  ";
        $filler10b = "                       ";
        $filler12a = "                                      ";
        $filler12b = "   ";
        $def9  = "SPARE             ";
        $part9  = "$line$filler12b$def9$return";
        $outline = "$part9";

        print OUTFILE $outline;
        goto LINEC;
    }

    return if ($find[0] =~ "0000440");

}
}
###################################################################################
###  proc_lec_subhdr
###
###  subhdr for CR-5A LEC instrument is   4 bytes
###
sub proc_lec_subhdr
{
##The LEC EDR physical science record corresponds to a 3.2 minute measurement period, or 20 minor frames.
##618-306 Rev D chg 5 Voyager Experiment Data Record Format
##Figure 11.3-3 defines the structure and contents of the LECP science record subheader for CR-5A (4 bytse).
##
    my ($infile,$outfile) = @_;
#    print "$infile\n";
    $return = "\n";
    $space = " ";
    open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
    open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

    print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
    print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
    print OUTFILE "Standard EDR header record subheader for LECP (4 bytes) Figure 11.3-3 (pg 11-11)\n";
    print OUTFILE "LECP_SUBHEADER::\n\n";
LINED: while (<INFILE>) {
    chop;
    $line = $_;
    $last = $cur;
    @find = split (/\s+/, $line);           #split on spaces
    @ch = split ( //,$line);         #split on characters
    $cur = $find[0];

    if ($find[0] =~ "0000360") {
        $filler1  = "                                  ";
        $def1 = "LECP COMMAND WORD   MF2     ";
        $def2 = "SPARE";
        $part1  = "$find[0]$space$find[1]$space$find[1]$filler1$def1$return";
        $outline = "$part1";
        print OUTFILE $outline;
        goto LINED;

    }
}
}
#########################################################################################################
#  proc_mag_subhdr
####
####  subhdr for CR-5A MAG instrument is 432 bytes
####
sub proc_mag_subhdr
{
##The MAG EDR physical science record corresponds to a 3.2 minute measurement period, or 20 minor frames.
###618-306 Rec D chg 4 Voyager Experiment Data Record Format
###Figure 11.4-3 defines the structure and contents of the MAG science record subheader for CR-5A (432 bytse).
###
    my ($infile,$outfile) = @_;
#    print "$infile\n";
    $return = "\n";
    $space = " ";
    open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
    open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

    print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
    print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
    print OUTFILE "Standard EDR header record subheader for MAG (432 bytes) Figure 11.4-3 (pg 11-18)\n";
    print OUTFILE "MAG_SUBHEADER::\n\n";
LINEE: while (<INFILE>) {
    chop;
    $line = $_;
    $last = $cur;
    @find = split (/\s+/, $line);           #split on spaces
    @ch = split ( //,$line);         #split on characters
    $cur = $find[0];

    if ($find[0] =~ "0000360") {
        goto LINEE if ($find[0] =~ "10000360");
        goto LINEE if ($find[0] =~ "20000360");
        goto LINEE if ($find[0] =~ "30000360");
        $filler1  = "                                      ";
        $filler2a = "             ";
        $filler2b = "                                 ";
        $filler3a = "                  ";
        $filler3b = "                            ";
        $filler4a = "                       ";
        $filler4b = "                       ";
        $filler5a = "                            ";
        $filler5b = "                  ";
        $filler6a = "                                 ";
        $filler6b = "             ";
        $filler7a = "                                      ";
        $filler7b = "        ";
        $filler8a = "                                           ";
        $filler8b = "   ";

        $def1  = "MAG COMMAND 1     MF3     ";
        $def2  = "MAG COMMAND 2     MF4     ";
        $def3  = "PLS COMMAND M     MF6      ";
        $def4  = "PLS COMMAND E     MF20     ";
        $def5  = "MAG STATUS 1      MF6     ";
        $def6  = "MAG STATUS 2      MF6     ";
        $def7  = "PLS STATUS 1      MF11     ";
        $def8  = "PLS STATUS 2      MF11    ";

        $part1  = "$find[0]$space$find[1]$filler1$def1$return";
        $part2  = "$filler2a$find[2]$filler2b$def2$return";
        $part3  = "$filler3a$find[3]$filler3b$def3$return";
        $part4  = "$filler4a$find[4]$filler4b$def4$return";
        $part5  = "$filler5a$find[5]$filler5b$def5$return";
        $part6  = "$filler6a$find[6]$filler6b$def6$return";
        $part7  = "$filler7a$find[7]$filler7b$def7$return";
        $part8  = "$filler8a$find[8]$filler8b$def8$return";
        $outline = "$part1$part2$part3$part4$part5$part6$part7$part8";
        print OUTFILE $outline;
        goto LINEE;


        $def9  = "MAG STATUS 1     MF16    ";
        $def10 = "MAG STATUS 2     MF16    ";
        $def11 = "PLS STATUS 1     MF46     ";
        $def12 = "PLS STATUS 2     MF60     ";
        $def13 = "START SCE IN MSEC OF DAY ";
        $def14 = "SPARE             ";
        $def15 = "SPARE              ";
        $def16 = "PLS STATUS BLOCK    ";

        print OUTFILE $outline;
        goto LINEE;
    }
    if ($find[0] =~ "0000400") {
        goto LINEE if ($find[0] =~ "10000400");
        goto LINEE if ($find[0] =~ "20000490");
        goto LINEE if ($find[0] =~ "30000400");
        $filler1  = "                                      ";
        $filler2a = "             ";
        $filler2b = "                                 ";
        $filler3a = "                  ";
        $filler3b = "                            ";
        $filler4a = "                       ";
        $filler4b = "                       ";
        $filler5a = "                            ";
        $filler5b = "                  ";
        $filler6a = "                                 ";
        $filler6b = "             ";
        $filler7a = "                                      ";
        $filler7b = "        ";
        $filler8a = "                                           ";
        $filler8b = "   ";
        $part5  = "$find[0]$space$find[1]$filler1$def5$return";
        $part6  = "$filler2a$find[2]$filler2b$def6$return";
        $part7  = "$filler3a$find[3]$filler3b$def7$return";
        $part8  = "$filler4a$find[4]$filler4b$def8$return";
        $part9  = "$filler5a$find[5]$filler5b$def9$return";
        $part10  = "$filler6a$find[6]$filler6b$def10$return";
        $part11  = "$filler7a$find[7]$filler7b$def11$return";
        $part12  = "$filler8a$find[8]$filler8b$def12$return";


        $outline = "$part5$part6$part7$part8$part9$part10$part11$part12";
        print OUTFILE $outline;
        goto LINEE;
    }

   }



}
#########################################################################################################
##  proc_pra_subhdr
###
###  subhdr for CR-5A PRA instrument is 8 bytes
###
sub proc_pra_subhdr
{
#The PRA EDR physical science record corresponds to a 48.0 minute measurement period, or 20 minor frames.
##618-306 Rev D chg 5 Voyager Experiment Data Record Format
##Figure 11.7-3 defines the structure and contents of the PRA science record subheader for CR-5A (8 bytse).
##
    my ($infile,$outfile) = @_;

#    print "$infile\n";
    $return = "\n";
    $space = " ";
    open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
    open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

    print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
    print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
    print OUTFILE "Standard EDR header record subheader for PRA (8 bytes) Figure 11.7-3 (pg 11-27)\n";
    print OUTFILE "PRA_SUBHEADER::\n\n";
LINEF: while (<INFILE>) {
    chop;
    $line = $_;
    $last = $cur;
    @find = split (/\s+/, $line);           #split on spaces
    @ch = split ( //,$line);         #split on characters
    $cur = $find[0];


    if ($find[0] =~ "0000360") {
        $filler1 = "                                       ";
        $filler2 = "             ";
        $filler3 = "                                  ";
        $filler4 = "                  ";
        $filler5 = "                             ";
        $filler6 = "                       ";
        $filler7 = "                        ";

        $def1  = "PRA MODE COMMAND    MF8 ";
        $def2  = "PRA CONFIG COMMAND  MF9 ";
        $def3  = "PRA MODE COMMAND    MF28";
        $def4  = "PRA CONFIG COMMAND  MF29";
        $part1 = "$find[0]$space$find[1]$filler1$def1$return";
        $part2 = "$filler2$find[2]$filler3$def2$return";
        $part3 = "$filler4$find[3]$filler5$def3$return";
        $part4 = "$filler6$find[4]$filler7$def4$return";
        $outline = "$part1$part2$part3$part4";
        print OUTFILE $outline;
        goto LINEF;

   }

  }
}
#
#########################################################################################################
##  proc_pls_subhdr
###
###  subhdr for CR-5A PLS instrument is 32 bytes
###
sub proc_pls_subhdr
{
#The PLS EDR physical science record corresponds to a 3.2 minute measurement period, or 20 minor frames.
##618-306 Rev D chg 5 Voyager Experiment Data Record Format
##Figure 11.5-3 defines the structure and contents of the PLS science record subheader for CR-5A (32 bytse).
##
    my ($infile,$outfile) = @_;
#    print "$infile\n";
    $return = "\n";
    $space = " ";
    open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
    open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

    print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
    print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
    print OUTFILE "Standard EDR header record subheader for PLS (32 bytes) Figure 11.5-3 (pg 11-22)\n";
    print OUTFILE "PLS_SUBHEADER::\n\n";
LINEG: while (<INFILE>) {
    chop;
    $line = $_;
    $last = $cur;
    @find = split (/\s+/, $line);           #split on spaces
    @ch = split ( //,$line);         #split on characters
    $cur = $find[0];


    if ($find[0] =~ "0000360") {

        $def1  = "MAG COMMAND 1 MF3     ";
        $def2  = "MAG COMMAND 2 MF4     ";
        $def3  = "PLS COMMAND M MF6      ";
        $def4  = "PLS COMMAND E MF20     ";
        $def5  = "MAG STATUS  1 MF1     ";
        $def6  = "MAG STATUS  2 MF1     ";
        $def7  = "MAG STATUS  1 MF6     ";
        $def8  = "MAG STATUS  2 MF6      ";
        $def9  = "MAG STATUS  1 MF11    ";
        $def10 = "MAG STATUS  2 MF11     ";
        $def11 = "MAG STATUS  1 MF16    ";
        $def12 = "MAG STATUS  2 MF16     ";
        $def13 = "START SCE IN MSEC OF DAY";
        $def14 = "SPARE             ";
        $def15 = "SPARE             ";

        $filler1  = "                                      ";
        $filler2a = "             ";
        $filler2b = "                                 ";
        $filler3a = "                  ";
        $filler3b = "                            ";
        $filler4a = "                       ";
        $filler4b = "                       ";
        $filler5a = "                            ";
        $filler5b = "                  ";
        $filler6a = "                                 ";
        $filler6b = "             ";
        $filler7a = "                                      ";
        $filler7b = "        ";
        $filler8a = "                                           ";
        $filler8b = "   ";

        $part5  = "$find[0]$space$find[1]$filler1$def1$return";
        $part6  = "$filler2a$find[2]$filler2b$def2$return";
        $part7  = "$filler3a$find[3]$filler3b$def3$return";
        $part8  = "$filler4a$find[4]$filler4b$def4$return";
        $part9  = "$filler5a$find[5]$filler5b$def5$return";
        $part10  = "$filler6a$find[6]$filler6b$def6$return";
        $part11  = "$filler7a$find[7]$filler7b$def7$return";
        $part12  = "$filler8a$find[8]$filler8b$def8$return";

        $outline = "$part5$part6$part7$part8$part9$part10$part11$part12";
        print OUTFILE $outline;
        goto LINEG;

    }
    if ($find[0] =~ "0000400") {

        $def1  = "MAG COMMAND 1 MF3     ";
        $def2  = "MAG COMMAND 2 MF4     ";
        $def3  = "PLS COMMAND M MF6      ";
        $def4  = "PLS COMMAND E MF20     ";
        $def5  = "MAG STATUS  1 MF1     ";
        $def6  = "MAG STATUS  2 MF1     ";
        $def7  = "MAG STATUS  1 MF6     ";
        $def8  = "MAG STATUS  2 MF6      ";
        $def9  = "MAG STATUS  1 MF11    ";
        $def10 = "MAG STATUS  2 MF11     ";
        $def11 = "MAG STATUS  1 MF16    ";
        $def12 = "MAG STATUS  2 MF16     ";
        $def13 = "START SCET IN MSEC OF DAY";
        $def14 = "SPARE             ";
        $def15 = "SPARE             ";

        $filler1  = "                                      ";
        $filler2a = "             ";
        $filler2b = "                                 ";
        $filler3a = "                  ";
        $filler3b = "                            ";
        $filler4a = "                       ";
        $filler4b = "                       ";
        $filler5a = "                            ";
        $filler5b = "                  ";
        $filler6a = "                                 ";
        $filler6b = "             ";
        $filler7a = "                                      ";
        $filler7b = "        ";
        $filler8a = "                                           ";
        $filler8b = "   ";

        $part5  = "$find[0]$space$find[1]$filler1$def9$return";
        $part6  = "$filler2a$find[2]$filler2b$def10$return";
        $part7  = "$filler3a$find[3]$filler3b$def11$return";
        $part8  = "$filler4a$find[4]$filler4b$def12$return";
        $part9  = "$filler5a$find[5]$space$find[6]$filler6b$def13$space$return";
        $part10  = "$filler7a$find[7]$filler7b$def14$return";
        $part11  = "$filler8a$find[8]$filler8b$def15$return";
#        $part12  = "$filler8a$find[8]$filler8b$def8$return";

        $outline = "$part5$part6$part7$part8$part9$part10$part11";
        print OUTFILE $outline;
        goto LINEG;

    }


   }
}
#########################################################################################################
##  proc_pws_subhdr
###
###  subhdr for CR-5A PWS instrument is 80 bytes
###
sub proc_pws_subhdr
{
#The PWS EDR physical science record corresponds to a 3.2 minute measurement period, or 20 minor frames.
##618-306 Rev D chg r54 Voyager Experiment Data Record Format
##Figure 11.8-3 pg 11-32 defines the structure and contents of the PWS science record subheader for CR-5A (80 bytse).
##
    my ($infile,$outfile) = @_;
#    print "$infile\n";
        $def1  = "PPS COMMAND        MF1    ";
        $def2  = "CRS STATUS         MF1   ";
        $def3  = "MAG 1 STATUS       MF1   ";
        $def4  = "MAG 2 STATUS       MF1   ";
        $def5  = "UVS READOUT FLAGS  MF1    ";
        $def6  = "PWS STATUS         MF1   ";
        $def7  = "LECP CONFIG        MF2   ";
        $def8  = "MAG COMMAND 1      MF3   ";



        $def9  = "MAG COMMAND 2      MF4   ";
        $def10 = "UVS MODE           MF5   ";
        $def11 = "PLS MODE COMMAND M MF6 ";
        $def12 = "CRS STATUS         MF6    ";
        $def13 = "MAG 1 STATUS       MF6   ";
        $def14 = "MAG 2 STATUS       MF6  ";
        $def15 = "UVS ROL            MF6   ";
        $def16 = "PWS STATUS         MF6   ";

        $def17 = "PWS LEVEL COMMAND  MF7 ";
        $def18 = "PRA MODE           MF8   ";
        $def19 = "PRA CONFIG         MF9   ";
        $def20 = "FDS COMMAND COUNT  MF10 ";
        $def21 = "FDS COMMAND 1      MF11 ";
        $def22 = "CRS STATUS         MF11  ";
        $def23 = "MAG 1 STATUS       MF11 ";
        $def24 = "MAG 2 STATUS       MF11 ";

        $def25 = "UVS READOUT FLAGS  MF11   ";
        $def26 = "PWS STATUS         MF11  ";
        $def27 = "FDS COMMAND 2      MF12  ";
        $def28 = "FDS COMMAND 3      MF13  ";
        $def29 = "FDS COMMAND 4      MF14 ";
        $def30 = "FDS COMMAND 5      MF15 ";
        $def31 = "FDS COMMAND 6      MF16   ";
        $def32 = "CRS STATUS         MF16   ";


        $def33 = "MAG 1 STATUS       MF16 ";
        $def34 = "MAG 2 STATUS       MF16 ";
        $def35 = "UVS READOURT FLAGS MF16  ";
        $def36 = "PWS STATUS         MF16";
        $def37 = "FDS COMMAND 7      MF17";
        $def38 = "FDS COMMAND 8      MF18";
        $def39 = "CRS COMMAND        MF19";
        $def40 = "PLS MODE COMMAND E MF20";

        $filler1  = "                                      ";
        $filler2a = "             ";
        $filler2b = "                                 ";
        $filler3a = "                  ";
        $filler3b = "                            ";
        $filler4a = "                       ";
        $filler4b = "                       ";
        $filler5a = "                            ";
        $filler5b = "                  ";
        $filler6a = "                                 ";
        $filler6b = "             ";
        $filler7a = "                                      ";
        $filler7b = "        ";
        $filler8a = "                                           ";
        $filler8b = "   ";


    $return = "\n";
    $space = " ";
    open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
    open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

    print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
    print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
    print OUTFILE "Standard EDR header record subheader for PWS (80 bytes) Figure 11.8-3 (pg 11-32)\n";
    print OUTFILE "PWS_SUBHEADER::\n\n";
LINEH: while (<INFILE>) {
    chop;
    $line = $_;
    $last = $cur;
    @find = split (/\s+/, $line);           #split on spaces
    @ch = split ( //,$line);         #split on characters
    $cur = $find[0];

    if ($find[0] =~ "0000360") {
        $part1  = "$find[0]$space$find[1]$filler1$def1$return";
        $part2  = "$filler2a$find[2]$filler2b$def2$return";
        $part3  = "$filler3a$find[3]$filler3b$def3$return";
        $part4  = "$filler4a$find[4]$filler4b$def4$return";
        $part5  = "$filler5a$find[5]$filler5b$def5$return";
        $part6  = "$filler6a$find[6]$filler6b$def6$return";
        $part7  = "$filler7a$find[7]$filler7b$def7$return";
        $part8  = "$filler8a$find[8]$filler8b$def8$return";
        $outline = "$part1$part2$part3$part4$part5$part6$part7$part8";
        print OUTFILE $outline;
        goto LINEH;
          
#      print OUTFILE $outline;
#       goto LINEH;

    }
    if( $find[0] =~ "0000400") {
        $part9  = "$find[0]$space$find[1]$filler1$def9$return";
        $part10 = "$filler2a$find[2]$filler2b$def10$return";
        $part11 = "$filler3a$find[3]$filler3b$def11$return";
        $part12 = "$filler4a$find[4]$filler4b$def12$return";
        $part13 = "$filler5a$find[5]$space$find[6]$filler6b$def13$space$return";
        $part14  = "$filler7a$find[7]$filler7b$def14$return";
        $part15  = "$filler8a$find[8]$filler8b$def15$return";

        $outline = "$part9$part10$part11$part12$part13$part14$part15";
        print OUTFILE $outline;
        goto LINEH;
    }
    if( $find[0] =~ "0000420") {
        $part16 = "$find[0]$space$find[1]$filler1$def16$return";
        $part17 = "$filler2a$find[2]$filler2b$def17$return";
        $part18 = "$filler3a$find[3]$filler3b$def18$return";
        $part19 = "$filler4a$find[4]$filler4b$def19$return";
        $part20 = "$filler5a$find[5]$space$find[6]$filler6b$def20$space$return";
        $part21  = "$filler7a$find[7]$filler7b$def21$return";
        $part22  = "$filler8a$find[8]$filler8b$def22$return";

        $outline = "$part16$part17$part18$part19$part20$part21$part22";
        print OUTFILE $outline;
        goto LINEH;
    }
   if( $find[0] =~ "0000440") {
        $part23 = "$find[0]$space$find[1]$filler1$def23$return";
        $part24 = "$filler2a$find[2]$filler2b$def24$return";
        $part25 = "$filler3a$find[3]$filler3b$def25$return";
        $part26 = "$filler4a$find[4]$filler4b$def26$return";
        $part27 = "$filler5a$find[5]$space$find[6]$filler6b$def27$space$return";
        $part28  = "$filler7a$find[7]$filler7b$def28$return";
        $part29  = "$filler8a$find[8]$filler8b$def29$return";

        $outline = "$part23$part24$part25$part26$part27$part28$part29";
        print OUTFILE $outline;
        goto LINEH;
    }
   if( $find[0] =~ "0000460") {
        $part30 = "$find[0]$space$find[1]$filler1$def30$return";
        $part31 = "$filler2a$find[2]$filler2b$def31$return";
        $part32 = "$filler3a$find[3]$filler3b$def32$return";
        $part33 = "$filler4a$find[4]$filler4b$def33$return";
        $part34 = "$filler5a$find[5]$space$find[6]$filler6b$def34$space$return";
        $part35  = "$filler7a$find[7]$filler7b$def35$return";
        $part36  = "$filler8a$find[8]$filler8b$def36$return";

        $outline = "$part30$part31$part32$part33$part34$part35$part36";
        print OUTFILE $outline;
        goto LINEH;
    }



    }
}
#########################################################################################################
##  proc_uvs_subhdr
###
###  subhdr for CR-5A  UVS instrument is 16 bytes
###
sub proc_uvs_subhdr
{
#The UVS EDR physical science record corresponds to a 4.0 minute measurement period, or 25 minor frames.
##618-306 Rev D chg 5 Voyager Experiment Data Record Format
##Figure 11.9-3 defines the structure and contents of the UVS science record subheader for CR-5A (16 bytse).
##
    my ($infile,$outfile) = @_;
#    print "$infile\n";
        $def51 = " ";
        $def1  = "UVS MODE           MF-A";
        $def2  = "UVS MODE           MF-B";
        $def3  = "UVS READOUT FLAGS  MF-1";
        $def4  = "UVS READOUT FLAGS  MF-6";
        $def5  = "UVS READOUT FLAGS  MF-11";
        $def6  = "UVS READOUT FLAGS  MF-16";
        $def7  = "UVS READOUT FLAGS  MF-21";
        $def8  = "SPARES             ";

    $return = "\n";
    $space = " ";
    open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
    open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

    print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
    print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
    print OUTFILE "Standard EDR header record subheader for UVS (16 bytes) Figure 11.9-3 (pg 11-42)\n";
    print OUTFILE "UVS_SUBHEADER::\n\n";
LINEJ: while (<INFILE>) {
    chop;
    $line = $_;
    $last = $cur;
    @find = split (/\s+/, $line);           #split on spaces
    @ch = split ( //,$line);         #split on characters
    $cur = $find[0];

    if ($find[0] =~ "0000360") {
            $filler52  = "                                       ";
            $filler53a = "             ";
            $filler53b = "                                  ";
            $filler54a = "                  ";
            $filler54b = "                             ";
            $filler55a = "                       ";
            $filler55b = "                        ";
            $filler56a = "                           ";
            $filler56b = "                    ";
            $filler57a = "                                 ";
            $filler57b = "              ";
            $filler58a = "                                      ";
            $filler58b = "         ";
            $filler59a = "                                           ";
            $filler59b = "    ";
            $part0 = "$find[0]$space$find[1]$filler52$def1$return";
            $part1 = "$filler53a$find[2]$filler53b$def2$return";
            $part2 = "$filler54a$find[3]$filler54b$def3$return";
            $part3 = "$filler55a$find[4]$filler55b$def4$return";
            $part4 = "$filler56a$find[5]$filler56b$def5$return";
            $part5 = "$filler57a$find[6]$filler57b$def6$return";
            $part6 = "$filler58a$find[7]$filler58b$def7$return";
            $part7 = "$filler59a$find[8]$filler59b$def8$return";
            $outline = "$part0$part1$part2$part3$part4$part5$part6$part7";
           print OUTFILE $outline;
           $skip= 1;
    }
}
}
#########################################################################################################
###  proc_eng_subhdr
####
####  subhdr for CR-5A  Engineering Block is 20 bytes
####
#
sub proc_eng_subhdr
{
###618-306 Rev D chg 5 Voyager Experiment Data Record Format
###Table B-2 (B-6) defines the structure and contents of the Engineering subheader for CR-5A (20 bytse).
###
    my ($infile,$outfile) = @_;
   #    print "$infile\n";
    $return = "\n";
    $space = " ";
    open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
    open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

    print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
    print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
    print OUTFILE "Standard EDR header record subheader for Engineering (20 bytes) Table B-2 (pg B-4)\n";
    print OUTFILE "ENGR_SUBHEADER::\n\n";
LINEK: while (<INFILE>) {
    chop;
    $line = $_;
    $last = $cur;
    @find = split (/\s+/, $line);           #split on spaces
    @ch = split ( //,$line);         #split on characters
    $cur = $find[0];

    if ($find[0] =~ "0000360") {
        goto LINEK if ($find[0] =~ "10000360");
        $def1  = "SPARE             ";
        $def2  = "SPARE             ";
        $def3  = "SPARE              ";
        $def4  = "SPARE             ";
        $def5  = "SPARE             ";
        $def6  = "SPARE             ";
        $def7  = "SPARE              ";
        $def8  = "SPARE              ";



        $def9  = "SPARE             ";
        $def10 = "SPARE             ";
        $def11 = "SPARE              ";
        $def12 = "SPARE              ";
        $def13 = "SPARE             ";
        $def14 = "SPARE             ";
        $def15 = "SPARE              ";
        $def16 = "SPARE              ";

        $def17 = "SPARE ";
        $def18 = "SPARE ";
        $def19 = "SPARE";
        $def20 = "SPARE";
        $filler1="    ";
        $part1  = "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]";
        $part2  =  "$find[5]$space$find[6]$space$find[7]$space$find[8]$filler1$def1$return";
        $outline = "$part1$space$part2";

        print OUTFILE "$outline";
        goto LINEK;


    }
    if ($find[0] =~ "0000400") {
        goto LINEK if ($find[0] =~ "10000400");
        $filler10a = "              ";
        $filler10b = "                       ";
        $filler12a = "                                      ";
        $filler12b = "   ";
        $def9  = "SPARE             ";
        $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
        $part9  = "$sub_hdr_end$space$filler10b$def9$return";
        $outline = "$part9";

        print OUTFILE $outline;
        goto LINEK;
    }
    }

}
#########################################################################################################
###  proc_dcom_subhdr
####
####  subhdr for CR-5A  DCOM Table is 28 bytes
####
#
sub proc_dcom_subhdr
{
###618-306 Rev D chg 5 Voyager Experiment Data Record Format
####Table B-2 (B-6) defines the structure and contents of the Engineering subheader for CR-5A (20 bytse).
####
    my ($infile,$outfile) = @_;
    #    print "$infile\n";
    $return = "\n";
    $space = " ";
    open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
    open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";
 
    print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
    print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
    print OUTFILE "Standard EDR header record subheader for DCOM Table (28 bytes) Table A-1 (pg A-3)\n";
    print OUTFILE "DCOM_SUBHEADER::\n\n";
LINEL: while (<INFILE>) {
    chop;
    $line = $_;
    $last = $cur;
    @find = split (/\s+/, $line);           #split on spaces
    @ch = split ( //,$line);         #split on characters
    $cur = $find[0];
    if ($find[0] =~ "0000360") {
           $data_start = 5;
           $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
           $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
           $posit2 = index($line,$sub_hdr_end);
           $filler2 = "                            ";
            $filler12b = "    ";
            $def9  = "SPARE             ";
           $outline =  "$sub_hdr_end$space$data_lead$filler12b$def9\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip = 1;
        goto LINEL;
    }
    if ($find[0] =~ "0000400") {
        $filler10a = "              ";
        $filler10b = "                       ";
        $filler12a = "                                      ";
        $filler12b = "   ";
        $def9  = "SPARE             ";
        $sub_hdr_end = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4] );
        $data_chk = join (" ",$find[5],$find[6] );
        $part9  = "$sub_hdr_end$space$data_chk$filler10a$def9$return";
        $outline = "$part9";

        print OUTFILE $outline;
    }

    }

}
##########################################################################################################
#     PROCESS STD EDR DATA BLOCK FOR EACH INSTRUMENT
##########################################################################################################
#
#
##########################################################################################
##  DATA for CR-5A CRS instrument is 3120 bytes
##
sub proc_crs_data
{
#The CRS EDR physical science record corresponds to a 9.6 minute measurement period, or 60 minor frames.
#618-306 Rev D chg 5 Voyager Experiment Data Record Format
#Figure 11.1-4 defines the structure and contents of the CRS science data record subheader for CR-5A (3120 bytse).
#
     my ($infile,$outfile) = @_;
#     print "$infile\n";
     $return = "\n";
     $space = " ";
     open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
     open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     print OUTFILE "=================================================================\n";
     print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
     print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
     print OUTFILE "Standard EDR data for CRS (780 bytes) Figure 11.1-4 (pg 11-5)\n";
     print OUTFILE "each minor frame is separated for 4 minor frames of 52 bytes each\n";
     print OUTFILE "CRS_DATA::\n\n";
     LINEK: while (<INFILE>) {
          chop;
          $line = $_;
        $last = $cur;
        @find = split (/\s+/, $line);           #split on spaces
        @ch = split ( //,$line);         #split on characters
        $cur = $find[0];

        if ($find[0] =~ "0000420") {
            $filler2 = "                            ";
            $data_lead = join (" ",$find[5],$find[6],$find[7],$find[8] );
           $outline =  "$filler2$data_lead\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0000440" || $find[0] =~ "0000460") {
           $outline =  "$line\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0000500") {
            $data_end = "MF 1";
            $filler3 = "             ";
            $filler4 = "                                      ";
            $data_lead = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4],$find[5],$find[6]);
           $outline =  "$data_lead$filler3$data_end\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
            $data_lead = join (" ",$find[7],$find[8]);
            $outline = "$filler4$data_lead";
          print OUTFILE $outline; 
           $skip= 1;

        }
        if ($find[0] =~ "0000520" ) {
           $outline =  "$return$line\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0000540" ) {
            $outline =  "$line\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0000560") {
           $data_end = "   MF 2";
           $outline =  "$line$data_end$return\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;
    
        }
        if ($find[0] =~ "0000600"  || $find[0] =~ "0000620" || $find[0] =~ "0000640") {
            $outline =  "$line\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;
    
        }
        if ($find[0] =~ "0000660") {
           $data_end = "   MF 3";
           $filler5 = "                              ";
           $outline =  "$find[0]$space$find[1]$space$find[2]$filler5$data_end$return\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;

           $filler6 = "                  ";
           $outline = "$filler6$find[3]$space$find[4]$space$find[5]$space$find[6]$space$find[7]$space$find[8]$return";
           print OUTFILE $outline;
           $skip= 1;
    
        }
        if ($find[0] =~ "0000700"  || $find[0] =~ "0000720")  {
            $outline =  "$line\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;
    
        }
        if ( $find[0] =~ "0000740") {
            $filler7 = "                    ";
            $data_end = "   MF4";
            $outline =  "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]$filler7$data_end$return";
             print OUTFILE $outline;
           $skip= 1;

        }
    }
}
##########################################################################################
##  DATA for  CR-5A LECP instrument is 320 bytes
##
sub proc_lec_data
{
#The LECP EDR physical science record corresponds to a 3.2 minute measurement period, or 20 minor frames in CR-5.
#618-306 Rec D chg 5 Voyager Experiment Data Record Format
#Figure 11.3-2 defines the structure and contents of the LECP science data record subheader for CR-5A (320 bytse).
##
     my ($infile,$outfile) = @_;
#     print "$infile\n";
     $return = "\n";
     $space = " ";
     open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
     open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     print OUTFILE "=================================================================\n";
     print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
     print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
     print OUTFILE "Standard EDR data for LEC (320 bytes) Figure 11.3-2 (pg 11-13)\n";
     print OUTFILE "each minor frame is separated for 1 minor frame of 20 bytes each\n";
     print OUTFILE "LECP_DATA::\n\n";
     LINEL: while (<INFILE>) {
          chop;
          $line = $_;
        $last = $cur;
        @find = split (/\s+/, $line);           #split on spaces
        @ch = split ( //,$line);         #split on characters
        $cur = $find[0];

        if ($find[0] =~ "0000360") {
            $filler2 = "           ";
            $part0 = "$find[0]";
            $data_lead = join (" ",$find[3],$find[4],$find[5],$find[6],$find[7],$find[8]);
           $outline =  "$part0$filler2$data_lead\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0000400") {
            $filler3 = "                        ";
           $filler4 = "                            ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]";
            $part1 = join (" ",$find[5],$find[6],$find[7],$find[8]);
            $data_end = "MF 1";
           $outline =  "$part0$filler3$data_end$return$filler4$part1$filler4\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0000420") {
            $filler5 = "    ";
           $filler4 = "              ";
           $filler5 = "                                      ";
            $filler6 = "                       ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]";
            $part1 = join (" ",$find[5],$find[6]);
            $part2 = "$filler4";
            $data_end = "MF 2";
            $part3 = "$find[7]$space$find[8]";
           $outline =  "$part0$space$part1$part2$data_end$return$filler5$part3\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0000440") {
           $data_end = "    MF 3";
           $outline =  "$line$data_end$return\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0000460") {
           $outline =  "$line\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;
        }
        if ($find[0] =~ "0000500") {
            $filler7 = "                                ";
            $data_end = "  MF 4";
           $outline =  "$find[0]$space$find[1]$space$find[1]$filler7$data_end$return\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;
        }

#
    }
#
}

##########################################################################################
##  DATA for CR-5A MAG instrument is 1040 bytes
##
sub proc_mag_data
{
##The MAG EDR physical science record corresponds to a 3.2 minute measurement period, or 20 minor frames in CR-5A.
##618-306 Rec D chg 5 Voyager Experiment Data Record Format
##Figure 11.4-4 defines the structure and contents of the MAG science data record subheader for CR-5A (1040 bytse).
###
     my ($infile,$outfile) = @_;
#     print "$infile\n";
     $return = "\n";
     $space = " ";
     open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
     open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     print OUTFILE "=================================================================\n";
     print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
     print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
     print OUTFILE "Standard EDR data record for MAG (1040 bytes) Table 11.4-4 (pg 11-19)\n";
     print OUTFILE "each minor frame is separated for 1 minor frame of 52 bytes each\n";
     print OUTFILE "MAG_DATA::\n\n";
LINEM: while (<INFILE>) {
          chop;
          $line = $_;
        $last = $cur;
        @find = split (/\s+/, $line);           #split on spaces
        @ch = split ( //,$line);         #split on characters
        $cur = $find[0];

        if ($find[0] =~ "0001240"  || $find[0] =~ "0001260" || $find[0] =~ "0001300") {
            goto LINEM if ($find[0] =~ "10001240");
            goto LINEM if ($find[0] =~ "20001240");
            goto LINEM if ($find[0] =~ "30001240");
            goto LINEM if ($find[0] =~ "10001260");
            goto LINEM if ($find[0] =~ "20001260");
            goto LINEM if ($find[0] =~ "30001260");
            goto LINEM if ($find[0] =~ "10001300");
            goto LINEM if ($find[0] =~ "20001300");
            goto LINEM if ($find[0] =~ "30001300");

           $outline =  "$line$return";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0001320") {
            goto LINEM if ($find[0] =~ "10001320");
            goto LINEM if ($find[0] =~ "20001320");
            goto LINEM if ($find[0] =~ "30001320");

            $filler3 = "               ";
           $filler4 = "                                   ";
            $filler5 = "                                 ";
           $data_end = "MF 1";
            $pad = "PAD";
            $part0 = "$find[0]$space$find[1]$space$ch[13]$ch[14]$filler4$data_end$return";
            $part1 = "$ch[15]$ch[16]$filler5$pad";
           $outline =  "$part0$filler3$part1$filler4\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
            $filler6 = "                  ";
            $part2 = join (" ",$find[3],$find[4],$find[5],$find[6],$find[7],$find[8]);
            $outline = "$filler6$part2$return";
            print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0001340"  || $find[0] =~ "0001360" ) {
            goto LINEM if ($find[0] =~ "10001340");
            goto LINEM if ($find[0] =~ "20001340");
            goto LINEM if ($find[0] =~ "30001340");
            goto LINEM if ($find[0] =~ "10001360");
            goto LINEM if ($find[0] =~ "20001360");
            goto LINEM if ($find[0] =~ "30001360");

           $outline =  "$line$return";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
       if ($find[0] =~ "0001400" ) {
            goto LINEM if ($find[0] =~ "10001400");
            goto LINEM if ($find[0] =~ "20001400");
            goto LINEM if ($find[0] =~ "30004040");

            $filler1 = "                         ";
            $filler2 = "                         ";
            $filler3 = "                       "; 
            $filler4 = "                            ";
            $data_end = "MF 2";
            $part1 = join(" ",$find[0],$find[1],$find[2],$find[3]);
            $part2 = "$space$ch[23]$ch[24]$filler1$data_end";
           $outline =  "$part1$part2$return";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
            $pad = "PAD";
            $part3 = "$filler2$ch[25]$ch[26]$filler3$pad$return";
            $outline = "$part3";
            print OUTFILE $outline;
            $part4 = "$filler4$find[5]$space$find[6]$space$find[7]$space$find[8]";
            $outline = "$part4";
            print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0001420"  || $find[0] =~ "0001440" ) {
            goto LINEM if ($find[0] =~ "10001420");
            goto LINEM if ($find[0] =~ "20001420");
            goto LINEM if ($find[0] =~ "30001420");
            goto LINEM if ($find[0] =~ "10001440");
            goto LINEM if ($find[0] =~ "20001440");
            goto LINEM if ($find[0] =~ "30004240");

           $outline =  "$return$line";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
               if ($find[0] =~ "0001460" ) {
            goto LINEM if ($find[0] =~ "10001460");
            goto LINEM if ($find[0] =~ "20001460");
            goto LINEM if ($find[0] =~ "30001460");

            $filler1 = "               ";
            $filler2 = "                                   ";
            $filler3 = "             "; 
            $filler4 = "                                      ";
            $data_end = "MF 3";
            $part1 = join(" ",$find[0],$find[1],$find[2],$find[3],$find[4],$find[5]);
            $part2 = "$space$ch[33]$ch[34]$filler1$data_end";
           $outline =  "$return$part1$part2$return";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
            $pad = "PAD";
            $part3 = "$filler2$ch[35]$ch[36]$filler3$pad$return";
            $outline = "$part3";
            print OUTFILE $outline;
            $part4 = "$filler4$find[7]$space$find[8]";
            $outline = "$part4";
            print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0001500"  || $find[0] =~ "0001520" ) {
            goto LINEM if ($find[0] =~ "10001500");
            goto LINEM if ($find[0] =~ "20001500");
            goto LINEM if ($find[0] =~ "30001500");
            goto LINEM if ($find[0] =~ "10001520");
            goto LINEM if ($find[0] =~ "20001520");
            goto LINEM if ($find[0] =~ "30001520");

           $outline =  "$return$line";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0001540" ) {
            goto LINEM if ($find[0] =~ "10001540");
            goto LINEM if ($find[0] =~ "20001540");
            goto LINEM if ($find[0] =~ "30001540");
            $filler1 = "     ";
            $filler2 = "                                             ";
            $filler3 = "   ";          
            $filler4 = "                                      ";
            $data_end = "MF 4";
            $part1 = join(" ",$find[0],$find[1],$find[2],$find[3],$find[4],$find[5],$find[6],$find[7]);
            $part2 = "$space$ch[43]$ch[44]$filler1$data_end";
           $outline =  "$return$part1$part2$return";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
            $pad = "PAD";
            $part3 = "$filler2$ch[45]$ch[46]$filler3$pad$return";
           $outline =  "$part3";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;

        }
    }
}
##########################################################################################
#  DATA for CR-5A PLS instrument is 400 bytes
###
sub proc_pls_data
{
#The PLS EDR physical science record corresponds to a 3.2 minute measurement period, or 20 minor frames in CR-5A.
#618-306 Rev D chg 5 Voyager Experiment Data Record Format
#Figure 11.5-4 defines the structure and contents of the PLS science data record subheader for CR-5A (400 bytes).
#
     my ($infile,$outfile) = @_;
#     print "$infile\n";
     $return = "\n";
     $space = " ";
     open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
     open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     print OUTFILE "=================================================================\n";
     print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
     print OUTFILE "Rev D, Chg 5 - 1 May 1989";
     print OUTFILE "Standard EDR data for PLS (400 bytes) Figure 11.5-4 (pg 11-23)\n";
     print OUTFILE "each minor frame is separated for 1 minor frame of 20 bytes each\n";
     print OUTFILE "PLS_DATA::\n\n";
     LINEM: while (<INFILE>) {
          chop;
          $line = $_;
        $last = $cur;
        @find = split (/\s+/, $line);           #split on spaces
        @ch = split ( //,$line);         #split on characters
        $cur = $find[0];

        if ($find[0] =~ "0000420" ) {
            $sline="S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16    MF-1";
           $filler3 = "    ";
           $outline =  "$return$line$filler3$sline$return";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }


        if ($find[0] =~ "0000440") {
            $filler3 = "    ";
           $filler4 = "              ";
           $filler5 = "                                  ";
            $filler6 = "                  ";
            $sline1 = "S-17,S-18 S-19,S-20    MF1";
            $sline2 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12    MF2";
            $part0 = "$find[0]$space$find[1]$space$find[2]";
            $part1 = join (" ",$find[3],$find[4],$find[5],$find[6],$find[7],$find[8]);
            $part2 = "$filler5";
            $data_end = $sline1;
           $outline =  "$part0$part2$data_end$filler3\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
            $outline = "$filler6$part1$filler3$sline2$return";
            print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0000460") {
            $filler5 = "    ";
           $filler4 = "              ";
           $filler5 = "                        ";
            $filler6 = "                            ";
            $sline = "S-13,S-14 S-15,S-16 S-17,S-18 S-19,S-20   MF2";
            $part0 = "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]";
            $part1 = join (" ",$find[5],$find[6],$find[7],$find[8]);
            $part2 = "$filler5";
            $data_end = $sline; 
            $outline = "$part0$filler5$data_end";
   #        $outline =  "$part2";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
            $outline = "$return$filler6$part1";
            print OUTFILE $outline;
           $skip= 1;

        }

#        if ($find[0] =~ "0000460") {
#            $filler5 = "    ";
#           $filler4 = "              ";
#           $filler5 = "              ";
#            $filler6 = "                                      ";
#            $part0 = "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]$space$find[5]$space$find[6]";
#            $part1 = join (" ",$find[7],$find[8]);
#            $part2 = "$filler5";
#            $data_end = "MF 3";
#            $outline = "$part0$filler5$data_end";
#           print OUTFILE $outline;
#            $outline = "$return$filler6$part1$return";
#            print OUTFILE $outline;
#           $skip= 1;

#        }
        if ($find[0] =~ "0000500") {
            $filler5 = "    ";
           $filler4 = "              ";
            $filler6 = "                                      ";
            $sline1 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8   MF3";
            $sline2 = "S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16 S-17,S-18 S-19,S-20   MF3";
            $sline3 = "S-1,S-2 S-3,S-4   MF-4";
            $part2 = "$filler5";
            $part1 = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4],$find[5],$find[6]);
            $outline = "$filler5$sline1$return$part1$filler4$sline2$return";
           print OUTFILE $outline;
            $part2 = join (" ",$find[7],$find[8]);
            $outline = "$filler6$part2$filler5$sline3$return";
            print OUTFILE $outline;
           $skip= 1;
            next LINEM; 
        }
        if ($find[0] =~ "0000520") {
            $sline1 = "S-1,S-2 S-3,S-4   MF-4";
            $sline2 = "S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16 S-17,S-18 S-19,S20   MF4";
            $filler5 = "    ";
            $filler6 = "                                      ";
#            $outline = "$filler6$part1$filler5$sline1$return";
#            print OUTFILE $outline;
            $outline = "$line$filler5$sline2$return";
            print OUTFILE $outline;
        }
        if ($find[0] =~ "0000540") {
            $filler3 = "    ";
           $filler4 = "              ";
           $filler5 = "                                  ";
            $filler6 = "                  ";
            $sline1 = "S-17,S-18 S-19,S-20    MF5";
            $sline2 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12 S13,S-14 S-15,S-16   MF5";
            $part0 = "$find[0]$space$find[1]$space$find[2]";
            $part1 = join (" ",$find[3],$find[4],$find[5],$find[6],$find[7],$find[8]);
            $part2 = "$filler5";
            $data_end = $sline1;
           $outline =  "$part0$space$part1$filler3$sline2\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0000560") {
            $filler3 = "    ";
           $filler4 = "                  ";
           $filler5 = "                        ";
            $filler6 = "                             ";
            $sline = " S-17,S-18 S-19,S-20    MF5";
            $part0 = "$find[0]$space$find[1]$space$find[2]";
            $part1 = join (" ",$find[3],$find[4],$find[5],$find[6],$find[7],$find[8]);
            $part2 = "$filler5";
            $data_end = $sline;
            $outline = "$part0$filler6$filler3$sline";
            print OUTFILE $outline;
            $outline = "$return$filler4$part1";
            print OUTFILE $outline;
            $skip= 1; 

        }
        if ($find[0] =~ "0000600") {
            $filler5 = "    ";
           $filler4 = "                            ";
            $filler6 = "                                      ";
            $filler7 = "                        ";
            $sline1 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12  MF6";
            $sline2 = "S-13,S-14 S-15,S-16 S-17,S-18 S-19,S20   MF6";
            $sline3 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8     MF-7";
            $part2 = "$filler5";
            $part1 = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4]);
            $outline = "$filler5$sline1$return$part1$filler7$sline2$return";
           print OUTFILE $outline;
            $part2 = join (" ",$find[5],$find[6],$find[7],$find[8]);
            $outline = "$filler4$part2$filler5$sline3$return";
            print OUTFILE $outline;
           $skip= 1;
            next LINEM;
        }
        if ($find[0] =~ "0000620") {
            $sline2 = "S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16 S-17,S-18 S-19,S20   MF7";
            $sline1 = "S-1,S-2 S-3,S-4    MF8";
            $filler4 = "              ";
            $filler5 = "    ";
            $filler6 = "                                      ";
            $part1 = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4],$find[5],$find[6]);
            $outline = "$part1$filler4$sline2$return";
            print OUTFILE $outline;
            $part2 = join (" ",$find[7],$find[8]);
            $outline = "$filler6$part2$filler5$sline1$return";
            print OUTFILE $outline;
        }
        if ($find[0] =~ "0000640") {
            $filler3 = "    ";
           $filler4 = "              ";
           $filler5 = "                                  ";
            $filler6 = "                  ";
            $sline1 = "S-17,S-18 S-19,S-20    MF5";
            $sline2 = "S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16 S-17,S-18, S-19,S-20   MF8";
            $part0 = "$find[0]$space$find[1]$space$find[2]";
            $part1 = join (" ",$find[3],$find[4],$find[5],$find[6],$find[7],$find[8]);
            $part2 = "$filler5";
            $data_end = $sline1;
           $outline =  "$line$filler3$sline2$filler3\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0000660") {
            $filler3 = "    ";
           $filler4 = "              ";
           $filler5 = "                        ";
            $filler6 = "                            ";
            $sline = "S-13,S-14 S-15,S-16    MF7";
            $sline2 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16    MF 9";
            $part0 = "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]";
            $part1 = join (" ",$find[5],$find[6],$find[7],$find[8]);
            $part2 = "$filler5";
            $data_end = $sline;
            $outline = "$line$filler3$sline2$return";
            print OUTFILE $outline;
            $skip= 1;

        }
        if ($find[0] =~ "0000700") {
            $filler5 = "    ";
           $filler4 = "                  ";
            $filler6 = "                                  ";
            $sline1 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8   MF6";
            $sline2 = "S-17,S-18 S-19,S-20   MF9";
            $sline3 = "S-1,S-2 S-3,S-4  S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12    MF-10";
            $part2 = "$filler5";
            $part1 = join (" ",$find[0],$find[1],$find[2]);
            $outline = "$part1$filler6$sline2$return";
           print OUTFILE $outline;
            $part2 = join (" ",$find[3],$find[4],$find[5],$find[6],$find[7],$find[8]);
            $outline = "$filler4$part2$filler5$sline3$return";
            print OUTFILE $outline;
           $skip= 1;
            next LINEM;
        }
        if ($find[0] =~ "0000720") {
            $sline2 = "S-13,S-14 S-15,S-16 S-17,S-18 S-19,S20   MF10";
            $filler5 = "    ";
           $filler4 = "                            ";

            $filler5 = "    ";
            $filler6 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]"; 
           $outline = "$part0$filler6$sline2$return";
            print OUTFILE $outline;
            $part2 = join (" ",$find[5],$find[6],$find[7],$find[8]);
                $sline3 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8     MF-11";
            $outline = "$filler4$part2$filler5$sline3$return";
            print OUTFILE $outline;
           $skip= 1;
            next LINEM;

        }
        if ($find[0] =~ "0000740") {
            $sline2 = "S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16 S-17,S-18 S-19,S20   MF11";
            $sline1 = "S-1,S-2 S-3,S-4    MF12";
            $filler4 = "              ";
            $filler5 = "    ";
            $filler6 = "                                      ";
            $part1 = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4],$find[5],$find[6]);
            $outline = "$part1$filler4$sline2$return";
            print OUTFILE $outline;
            $part2 = join (" ",$find[7],$find[8]);
            $outline = "$filler6$part2$filler5$sline1$return";
            print OUTFILE $outline;
        }
        if ($find[0] =~ "0000760") {
            $filler3 = "    ";
           $filler4 = "              ";
           $filler5 = "                                  ";
            $filler6 = "                  ";
            $sline1 = "S-17,S-18 S-19,S-20    MF5";
            $sline2 = "S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16 S-17,S-18, S-19,S-20   MF12";
            $part0 = "$find[0]$space$find[1]$space$find[2]";
            $part1 = join (" ",$find[3],$find[4],$find[5],$find[6],$find[7],$find[8]);
            $part2 = "$filler5";
            $data_end = $sline1;
           $outline =  "$line$filler3$sline2$filler3\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0001000") {
            $filler3 = "    ";
           $filler4 = "              ";
           $filler5 = "                        ";
            $filler6 = "                            ";
            $sline = "S-13,S-14 S-15,S-16    MF7";
            $sline2 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16    MF 13";
            $part0 = "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]";
            $part1 = join (" ",$find[5],$find[6],$find[7],$find[8]);
            $part2 = "$filler5";
            $data_end = $sline;
            $outline = "$line$filler3$sline2$return";
            print OUTFILE $outline;
            $skip= 1;

        }
        if ($find[0] =~ "0001020") {
            $filler5 = "    ";
           $filler4 = "                  ";
            $filler6 = "                                  ";
            $sline1 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8   MF6";
            $sline2 = "S-17,S-18 S-19,S-20   MF13";
            $sline3 = "S-1,S-2 S-3,S-4  S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12    MF-14";
            $part2 = "$filler5";
            $part1 = join (" ",$find[0],$find[1],$find[2]);
            $outline = "$part1$filler6$sline2$return";
           print OUTFILE $outline;
            $part2 = join (" ",$find[3],$find[4],$find[5],$find[6],$find[7],$find[8]);
            $outline = "$filler4$part2$filler5$sline3$return";
            print OUTFILE $outline;
           $skip= 1;
            next LINEM;
        }
        if ($find[0] =~ "0001040") {
            $sline2 = "S-13,S-14 S-15,S-16 S-17,S-18 S-19,S20   MF14";
            $filler5 = "    ";
           $filler4 = "                            ";

            $filler5 = "    ";
            $filler6 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]";
           $outline = "$part0$filler6$sline2$return";
            print OUTFILE $outline;
            $part2 = join (" ",$find[5],$find[6],$find[7],$find[8]);
                $sline3 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8     MF-15";
            $outline = "$filler4$part2$filler5$sline3$return";
            print OUTFILE $outline;
           $skip= 1;
            next LINEM;

        }
        if ($find[0] =~ "0001060") {
            $sline2 = "S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16 S-17,S-18 S-19,S20   MF15";
            $sline1 = "S-1,S-2 S-3,S-4    MF16";
            $filler4 = "              ";
            $filler5 = "    ";
            $filler6 = "                                      ";
            $part1 = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4],$find[5],$find[6]);
            $outline = "$part1$filler4$sline2$return";
            print OUTFILE $outline;
            $part2 = join (" ",$find[7],$find[8]);
            $outline = "$filler6$part2$filler5$sline1$return";
            print OUTFILE $outline;
        }
        if ($find[0] =~ "0001100") {
            $filler3 = "    ";
           $filler4 = "              ";
           $filler5 = "                                  ";
            $filler6 = "                  ";
            $sline1 = "S-17,S-18 S-19,S-20    MF5";
            $sline2 = "S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16 S-17,S-18, S-19,S-20   M16";
            $part0 = "$find[0]$space$find[1]$space$find[2]";
            $part1 = join (" ",$find[3],$find[4],$find[5],$find[6],$find[7],$find[8]);
            $part2 = "$filler5";
            $data_end = $sline1;
           $outline =  "$line$filler3$sline2$filler3\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0001120") {
            $filler3 = "    ";
           $filler4 = "              ";
           $filler5 = "                        ";
            $filler6 = "                            ";
            $sline = "S-13,S-14 S-15,S-16    MF7";
            $sline2 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16    MF 17";
            $part0 = "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]";
            $part1 = join (" ",$find[5],$find[6],$find[7],$find[8]);
            $part2 = "$filler5";
            $data_end = $sline;
            $outline = "$line$filler3$sline2$return";
            print OUTFILE $outline;
            $skip= 1;

        }
        if ($find[0] =~ "0001140") {
            $filler5 = "    ";
           $filler4 = "                  ";
            $filler6 = "                                  ";
            $sline1 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8   MF18";
            $sline2 = "S-17,S-18 S-19,S-20   MF 17";
            $sline3 = "S-1,S-2 S-3,S-4  S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12    MF-10";
            $part2 = "$filler5";
            $part1 = join (" ",$find[0],$find[1],$find[2]);
            $outline = "$part1$filler6$sline2$return";
           print OUTFILE $outline;
            $part2 = join (" ",$find[3],$find[4],$find[5],$find[6],$find[7],$find[8]);
            $outline = "$filler4$part2$filler5$sline3$return";
            print OUTFILE $outline;
           $skip= 1;
            next LINEM;
        }
        if ($find[0] =~ "0001160") {
            $sline2 = "S-13,S-14 S-15,S-16 S-17,S-18 S-19,S20   MF14";
            $filler5 = "    ";
           $filler4 = "                            ";

            $filler5 = "    ";
            $filler6 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]";
           $outline = "$part0$filler6$sline2$return";
            print OUTFILE $outline;
            $part2 = join (" ",$find[5],$find[6],$find[7],$find[8]);
                $sline3 = "S-1,S-2 S-3,S-4 S-5,S-6 S-7,S-8     MF-15";
            $outline = "$filler4$part2$filler5$sline3$return";
            print OUTFILE $outline;
           $skip= 1;
            next LINEM;

        }
        if ($find[0] =~ "0001200") {
            $sline2 = "S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16 S-17,S-18 S-19,S20   MF19";
            $sline1 = "S-1,S-2 S-3,S-4    MF-20";
            $filler4 = "              ";
            $filler5 = "    ";
            $filler6 = "                                      ";
            $part1 = join (" ",$find[0],$find[1],$find[2],$find[3],$find[4],$find[5],$find[6]);
            $outline = "$part1$filler4$sline2$return";
            print OUTFILE $outline;
            $part2 = join (" ",$find[7],$find[8]);
            $outline = "$filler6$part2$filler5$sline1$return";
            print OUTFILE $outline;
        }
        if ($find[0] =~ "0001220") {
            $filler3 = "    ";
           $filler4 = "              ";
           $filler5 = "                                  ";
            $filler6 = "                  ";
            $sline1 = "S-17,S-18 S-19,S-20    MF5";
            $sline2 = "S-5,S-6 S-7,S-8 S-9,S-10 S-11,S-12 S-13,S-14 S-15,S-16 S-17,S-18, S-19,S-20   MF20";
            $part0 = "$find[0]$space$find[1]$space$find[2]";
            $part1 = join (" ",$find[3],$find[4],$find[5],$find[6],$find[7],$find[8]);
            $part2 = "$filler5";
            $data_end = $sline1;
           $outline =  "$line$filler3$sline2$filler3\n";   #"7>"$outline  (PRINTS)
           print OUTFILE $outline;
           $skip= 1;

        }

    }


}
##########################################################################################
#  DATA for CR-5A PRA instrument is 280 bytes
####
sub proc_pra_data
{
#The PRA EDR physical science record corresponds to a 48.0 minute measurement period, or 20 minor frames in CR-5A.
#618-306 Rec D chg 5 Voyager Experiment Data Record Format
#Figure 11.7-4 defines the structure and contents of the PRA science data record subheader for CR-5A (280 bytes).
#
#
     my ($infile,$outfile) = @_;
    $prasci1  = "SCAN 64L(1),SCAN 64L(2) SCAN 64R(1),SCAN 64R(2)";
    $prasci2  = "PRA STAT(1),PRA STAT(2) POR COUNTER,SCAN  1L(1)";
    $prasci3  = "SCAN  1L(2),SCAN  1R(1) SCAN  1R(2),SCAN  2L(1)";
    $prasci4  = "SCAN  2L(2),SCAN  2R(1) SCAN  2R(2),SCAN  3L(1)";
    $prasci5  = "SCAN  3L(2),SCAN  3R(1) SCAN  3R(2),SCAN  4L(1)";
    $prasci6  = "SCAN  4L(2),SCAN  4R(1) SCAN  4R(2),SCAN  5L(1)";
    $prasci7  = "SCAN  5L(2),SCAN  5R(1) SCAN  5R(2),SCAN  6L(1)";
    $prasci8  = "SCAN  6L(2),SCAN  6R(1) SCAN  6R(2),SCAN  7L(1)";
    $prasci9  = "SCAN  7L(2),SCAN  7R(1) SCAN  7R(2),SCAN  8L(1)";
    $prasci10 = "SCAN  8L(2),SCAN  8R(1) SCAN  8R(2),PRA STAT(1)";
    $prasci11 = "PRA STAT(2),POR COUNTER SCAN  9L(1),SCAN  9L(2)";
    $prasci12 = "SCAN  9R(1),SCAN  9R(2) SCAN 10L(1),SCAN 10L(2)";
    $prasci13 = "SCAN 10R(1),SCAN 10R(2) SCAN 11L(1),SCAN 11L(2)";
    $prasci14 = "SCAN 11R(1),SCAN 11R(2) SCAN 12L(1),SCAN 12L(2)";
    $prasci15 = "SCAN 12R(1),SCAN 12R(2) SCAN 13L(1),SCAN 13L(2)";
    $prasci16 = "SCAN 13R(1),SCAN 13R(2) SCAN 14L(1),SCAN 14L(2)";
    $prasci17 = "SCAN 14R(1),SCAN 14R(2) SCAN 15L(1),SCAN 15L(2)";
    $prasci18 = "SCAN 15R(1),SCAN 15R(2) SCAN 16L(1),SCAN 16L(2)";
    $prasci19 = "SCAN 16R(1),SCAN 16R(2) PRA STAT(1),PRA STAT(2)";
    $prasci20 = "POR COUNTER,SCAN 17L(1) SCAN 17L(2),SCAN 17R(1)";
    $prasci21 = "SCAN 17R(2),SCAN 18L(1) SCAN 18L(2),SCAN 18R(1)";
    $prasci22 = "SCAN 18R(2),SCAN 19L(1) SCAN 19L(2),SCAN 19R(1)";
    $prasci23 = "SCAN 19R(2),SCAN 20L(1) SCAN 20L(2),SCAN 20R(1)";
    $prasci24 = "SCAN 20R(2),SCAN 21L(1) SCAN 21L(2),SCAN 21R(1)";
    $prasci25 = "SCAN 21R(2),SCAN 22L(1) SCAN 22L(2),SCAN 22R(1)";
    $prasci26 = "SCAN 22R(2),SCAN 23L(1) SCAN 23L(2),SCAN 23R(1)";
    $prasci27 = "SCAN 23R(2),SCAN 24L(1) SCAN 24L(2),SCAN 24R(1)";
    $prasci28 = "SCAN 24R(2),PRA STAT(1) PRA STAT(2),POR COUNTER";
    $prasci29 = "SCAN 25L(1),SCAN 25L(2) SCAN 25R(1),SCAN 25R(2)";
    $prasci30 = "SCAN 26L(1),SCAN 26L(2) SCAN 26R(1),SCAN 26R(2)";
    $prasci31 = "SCAN 27L(1),SCAN 27L(2) SCAN 27R(1),SCAN 27R(2)";
    $prasci32 = "SCAN 28L(1),SCAN 28L(2) SCAN 28R(1),SCAN 28R(2)";
    $prasci33 = "SCAN 29L(1),SCAN 29L(2) SCAN 29R(1),SCAN 29R(2)";
    $prasci34 = "SCAN 30L(1),SCAN 30L(2) SCAN 30R(1),SCAN 30R(2)";
    $prasci35 = "SCAN 31L(1),SCAN 31L(2) SCAN 31R(1),SCAN 31R(2)";
    $prasci36 = "SCAN 32L(1),SCAN 32L(2) SCAN 62R(1),SCAN 32R(2)";
    $prasci37 = "PRA STAT(1),PRA STAT(2) POR COUNTER,SCAN 33L(1)";
    $prasci38 = "SCAN 33L(2),SCAN 33R(1) SCAN 33R(2),SCAN 34L(1)";
    $prasci39 = "SCAN 34L(2),SCAN 34R(1) SCAN 34R(2),SCAN 35L(1)";
    $prasci40 = "SCAN 35L(2),SCAN 35R(1) SCAN 35R(2),SCAN 36L(1)";

    $prasci41 = "SCAN 36L(2),SCAN 36R(1) SCAN 36R(2),SCAN 37L(1)";
    $prasci42 = "SCAN 37L(2),SCAN 37R(1) SCAN 37R(2),SCAN 38L(1)";
    $prasci43 = "SCAN 38L(2),SCAN 38R(1) SCAN 38R(2),SCAN 39L(1)";
    $prasci44 = "SCAN 39L(2),SCAN 39R(1) SCAN 39R(2),SCAN 40L(1)";
    $prasci45 = "SCAN 40L(2),SCAN 40R(1) SCAN 40R(2),PRA STAT(1)";
    $prasci46 = "PRA STAT(2),POR COUNTER SCAN 41L(1),SCAN 41L(2)";
    $prasci47 = "SCAN 41R(1),SCAN 41R(2) SCAN 42L(1),SCAN 42L(2)";
    $prasci48 = "SCAN 42R(1),SCAN 42R(2) SCAN 43L(1),SCAN 43L(2)";
    $prasci49 = "SCAN 43R(1),SCAN 43R(2) SCAN 44L(1),SCAN 44L(2)";
    $prasci50 = "SCAN 44R(1),SCAN 44R(2) SCAN 45L(1),SCAN 45L(2)";
    $prasci51 = "SCAN 45R(1),SCAN 45R(2) SCAN 46L(1),SCAN 46L(2)";
    $prasci52 = "SCAN 46R(1),SCAN 46R(2) SCAN 47L(1),SCAN 47L(2)";
    $prasci53 = "SCAN 47R(1),SCAN 47R(2) SCAN 48L(1),SCAN 48L(2)";
    $prasci54 = "SCAN 48R(1),SCAN 48R(2) PRA ST9T(1),PRA STAT(2)";
    $prasci55 = "POR COUNTER,SCAN 49L(1) SCAN 49L(2),SCAN 49R(1)";
    $prasci56 = "SCAN 49R(2),SCAN 50L(1) SCAN 59L(2),SCAN 50R(1)";
    $prasci57 = "SCAN 50R(2),SCAN 51L(1) SCAN 51L(2),SCAN 51R(1)";
    $prasci58 = "SCAN 51R(2),SCAN 52L(1) SCAN 52L(2),SCAN 52R(1)";
    $prasci59 = "SCAN 52R(2),SCAN 53L(1) SCAN 53L(2),SCAN 53R(1)";
    $prasci60 = "SCAN 53R(2),SCAN 54L(1) SCAN 54L(2),SCAN 54R(1)";
    $prasci61 = "SCAN 54R(2),SCAN 55L(1) SCAN 55L(2),SCAN 55R(1)";
    $prasci62 = "SCAN 55R(2),SCAN 56L(1) SCAN 56L(2),SCAN 56R(1)";
    $prasci63 = "SCAN 56R(2),PRA STAT(1) PRA STAT(2),POR COUNTER";
    $prasci64 = "SCAN 57L(1),SCAN 57L(2) SCAN 57R(1),SCAN 57R(2)";
    $prasci65 = "SCAN 58L(1),SCAN 58L(2) SCAN 58R(1),SCAN 58R(2)";
    $prasci66 = "SCAN 59L(1),SCAN 59L(2) SCAN 59R(1),SCAN 59R(2)";
    $prasci67 = "SCAN 60L(1),SCAN 60L(2) SCAN 60R(1),SCAN 60R(2)";
    $prasci68 = "SCAN 61L(1),SCAN 61L(2) SCAN 61R(1),SCAN 61R(2)";
    $prasci69 = "SCAN 62L(1),SCAN 62L(2) SCAN 62R(1),SCAN 62R(2)";
    $prasci70 = "SCAN 63L(1),SCAN 63L(2) SCAN 63R(1),SCAN 63R(2)";


#     print "$infile\n";
     $return = "\n";
     $space = " ";
     open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
     open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     print OUTFILE "=================================================================\n";
     print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
     print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
     print OUTFILE "Standard EDR data for PRA (280 bytes) Table 11.7-4 (pg 11-28)\n";
     print OUTFILE "each minor frame is separated for 1 minor frame of 7 bytes each\n";
     print OUTFILE "PRA_DATA::\n\n";
     LINEM: while (<INFILE>) {
          chop;
          $line = $_;
        $last = $cur;
        @find = split (/\s+/, $line);           #split on spaces
        @ch = split ( //,$line);         #split on characters
        $cur = $find[0];
        if ($find[0] =~ "0000360") {
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                      ";
            $filler8 = "             ";
            $part0 = "$find[0]$filler6$find[5]$space$find[6]$filler8$prasci1$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci2$return";
           print OUTFILE $outline;
           $skip= 1;
        }
        if ($find[0] =~ "0000400") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci3$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci4$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci5$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci6$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0000420") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci7$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci8$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci9$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci10$return";
           print OUTFILE $outline;
           $skip= 1;
       }


        if ($find[0] =~ "0000440") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci11$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci12$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci13$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci14$return";
           print OUTFILE $outline;
           $skip= 1;
       }

        if ($find[0] =~ "0000460") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci15$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci16$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci17$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci18$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0000500") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci19$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci20$return";
           print OUTFILE $outline; 
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci21$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci22$return";
           print OUTFILE $outline;
           $skip= 1;
       }

        if ($find[0] =~ "0000520") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci23$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci24$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci25$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci26$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0000540") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci27$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci28$return";
           print OUTFILE $outline; 
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci29$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci30$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0000560") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci31$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci32$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci33$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci34$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0000600") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci35$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci36$return";
           print OUTFILE $outline; 
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci37$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci38$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0000620") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci39$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci40$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci41$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci42$return";
           print OUTFILE $outline;
           $skip= 1;
        }
        if ($find[0] =~ "0000640") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci43$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci44$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci45$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci46$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0000660") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci47$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci48$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci49$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci50$return";
           print OUTFILE $outline;
           $skip= 1;
       }


        if ($find[0] =~ "0000700") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci51$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci52$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci53$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci54$return";
           print OUTFILE $outline;
           $skip= 1;
       }

        if ($find[0] =~ "0000720") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci55$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci56$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci57$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci58$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0000740") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci59$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci60$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci61$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci62$return";
           print OUTFILE $outline;
           $skip= 1;
       }


        if ($find[0] =~ "0000760") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci63$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci64$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci65$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci66$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0001000") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$prasci67$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$prasci68$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$prasci69$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$prasci70$return";
           print OUTFILE $outline;
           $skip= 1;
       } 

    }
print OUTFILE "=======================================================================\n";
}
##########################################################################################
#  DATA for CR-5A PWS instrument is 220 bytes
#####
sub proc_pws_data
{
#The PWS EDR physical science record corresponds to a 4.0 minute measurement period, or 20 minor frames in CR-5A.
#618-306 Rev D chg 5 Voyager Experiment Data Record Format CR-5A
#Figure 11.8-4 defines the structure and contents of the PWS science data record  for CR-5A (220 bytse).
#
#
     my ($infile,$outfile) = @_;
#     print "$infile\n";
     $return = "\n";
     $space = " ";
     open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
     open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     print OUTFILE "=================================================================\n";
     print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
     print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
     print OUTFILE "Standard EDR data for PWS (220 bytes) Figure 11.8-4 (pg 11-33)\n";
     print OUTFILE "each minor frame is 88 bits (11-bytes) each\n";
     print OUTFILE "PWS_DATA::\n\n";
     LINEN: while (<INFILE>) {
          chop;
          $line = $_;
        $last = $cur;
        @find = split (/\s+/, $line);           #split on spaces
        @ch = split ( //,$line);         #split on characters
        $cur = $find[0];
        if ($find[0] =~ "0000500") {
#            print "$find[0]\n";
            $filler5 = "                ";
            $filler4 = "                                   ";
            $filler6 = "                     ";
            $filler7 = "                        ";
            $data_end = "MF 1";
            $part0 = "$find[0]$space$find[1]$space$find[2]$space$find[3]$space$find[4]$space$find[5]$space$ch[33]$ch[34]";
            $part2 = "$filler5$data_end";
            $part1 = "$return$filler4$ch[35]$ch[36]$space$find[7]$space$find[8]";
            $outline = "$part0$part2$part1$return";
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0000520") {
#            print "$find[0]\n";
            $filler5 = "                             ";
            $filler4 = "                       ";
            $filler6 = "                     ";
            $filler7 = "                        ";
            $data_end = "MF 2";
            $part0 = "$find[0]$space$find[1]$space$find[2]$space$find[3]";
            $part2 = "$filler5$data_end";
            $part1 = "$return$filler4$find[4]$space$find[5]$space$find[6]$space$find[7]$space$find[8]";
            $outline = "$part0$part2$part1$return";
           print OUTFILE $outline;
           $skip= 1;

        }
        if ($find[0] =~ "0000540") {
#            print "$find[0]\n";
            $filler5 = "                                         ";
            $filler4 = "          ";
            $filler6 = "                     ";
            $filler7 = "              ";
            $data_end = "MF 3";
            $data_end1 = "MF 4";
            $part0 = "$find[0]$space$ch[8]$ch[9]";
            $part2 = "$filler5$data_end";
            $part1 = "$return$filler4$ch[10]$ch[11]$space$find[2]$space$find[3]$space$find[4]$space$find[5]$space$find[6]";
            $part3 = "$filler7$data_end1";
            $outline = "$part0$part2$part1$part3$return";
           print OUTFILE $outline;
           $skip= 1;

        }


    }
}
##########################################################################################
#  DATA for CR-5A UVS instrument is 120 bytes
######
sub proc_uvs_data
{
#The UVS EDR physical science record corresponds to a 4.0 minute measurement period, or 25 minor frames in CR-5A.
#618-306 Rev D chg 5 Voyager Experiment Data Record Format
#Figure 11.9-4 defines the structure and contents of the UVS science data record subheader for CR-5A (120 bytse).
#
     my ($infile,$outfile) = @_;
#     print "$infile\n";
    $uvssci1  = "CHANNEL   1,CHANNEL   2 CHANNEL   5,CHANNEL   6";
    $uvssci2  = "CHANNEL   7,CHANNEL   8 CHANNEL   9,CHANNEL  10";
    $uvssci3  = "CHANNEL  11,CHANNEL  12 CHANNEL  13,CHANNEL  14";
    $uvssci4  = "CHANNEL  15,CHANNEL  16 CHANNEL  17,CHANNEL  18";
    $uvssci5  = "CHANNEL  19,CHANNEL  20 CHANNEL  21,CHANNEL  22";
    $uvssci6  = "CHANNEL  23,CHANNEL  24 CHANNEL  25,CHANNEL  26";
    $uvssci7  = "CHANNEL  27,CHANNEL  28 CHANNEL  29,CHANNEL  30";
    $uvssci8  = "CHANNEL  31,CHANNEL  32 CHANNEL  33,CHANNEL  34";
    $uvssci9  = "CHANNEL  35,CHANNEL  36 CHANNEL  37,CHANNEL  38";
    $uvssci10 = "CHANNEL  39,CHANNEL  40 CHANNEL  41,CHANNEL  42";
    $uvssci11 = "CHANNEL  43,CHANNEL  44 CHANNEL  45,CHANNEL  46";
    $uvssci12 = "CHANNEL  47,CHANNEL  48 CHANNEL  49,CHANNEL  50";
    $uvssci13 = "CHANNEL  51,CHANNEL  52 CHANNEL  53,CHANNEL  54";
    $uvssci14 = "CHANNEL  55,CHANNEL  56 CHANNEL  57,CHANNEL  58";
    $uvssci15 = "CHANNEL  59,CHANNEL  60 CHANNEL  61,CHANNEL  62";
    $uvssci16 = "CHANNEL  63,CHANNEL  64 CHANNEL  65,CHANNEL  66";
    $uvssci17 = "CHANNEL  67,CHANNEL  68 CHANNEL  69,CHANNEL  70";
    $uvssci18 = "CHANNEL  71,CHANNEL  72 CHANNEL  73,CHANNEL  74";
    $uvssci19 = "CHANNEL  75,CHANNEL  76 CHANNEL  77,CHANNEL  78";
    $uvssci20 = "CHANNEL  79,CHANNEL  80 CHANNEL  81,CHANNEL  82";
    $uvssci21 = "CHANNEL  83,CHANNEL  84 CHANNEL  85,CHANNEL  86";
    $uvssci22 = "CHANNEL  87,CHANNEL  88 CHANNEL  89,CHANNEL  90";
    $uvssci23 = "CHANNEL  91,CHANNEL  92 CHANNEL  93,CHANNEL  94";
    $uvssci24 = "CHANNEL  95,CHANNEL  96 CHANNEL  97,CHANNEL  98";
    $uvssci25 = "CHANNEL  99,CHANNEL 100 CHANNEL 101,CHANNEL 102";
    $uvssci26 = "CHANNEL 103,CHANNEL 104 CHANNEL 105,CHANNEL 106";
    $uvssci27 = "CHANNEL 107,CHANNEL 108 CHANNEL 109,CHANNEL 110";
    $uvssci28 = "CHANNEL 111,CHANNEL 112 CHANNEL 113,CHANNEL 114";
    $uvssci29 = "CHANNEL 115,CHANNEL 116 CHANNEL 117,CHANNEL 118";
    $uvssci30 = "CHANNEL 119,CHANNEL 120 CHANNEL 121,CHANNEL 122";

     $return = "\n";
     $space = " ";
     open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
     open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     print OUTFILE "=================================================================\n";
     print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
     print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
     print OUTFILE "Standard EDR data for UVS (120 bytes) Figure 11.9-4 (pg 11-43)\n";
     print OUTFILE "each minor frame is separated for 1 minor frame of 7 bytes each\n";
     print OUTFILE "UVS_DATA::\n\n";
LINEM: while (<INFILE>) {
     chop;
          $line = $_;
        $last = $cur;
        @find = split (/\s+/, $line);           #split on spaces
        @ch = split ( //,$line);         #split on characters
        $cur = $find[0];
        if ($find[0] =~ "0000400") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$uvssci1$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$uvssci2$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$uvssci3$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$uvssci4$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0000420") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$uvssci5$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$uvssci6$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$uvssci7$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$uvssci8$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0000440") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$uvssci9$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$uvssci10$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$uvssci11$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$uvssci12$return";
           print OUTFILE $outline;
           $skip= 1;
       }

        if ($find[0] =~ "0000460") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$uvssci13$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$uvssci14$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$uvssci15$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$uvssci16$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0000500") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$uvssci17$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$uvssci18$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$uvssci19$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$uvssci20$return";
           print OUTFILE $outline;
           $skip= 1;
       }

        if ($find[0] =~ "0000520") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$uvssci21$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$uvssci22$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$uvssci23$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$uvssci24$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0000540") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$uvssci25$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$uvssci26$return";
           print OUTFILE $outline;
            $filler2 = "   ";
            $filler4 = "                                    ";
            $filler6 = "                             ";
            $filler8 = "             ";
            $part0 = "$filler6$find[5]$space$find[6]$filler8$uvssci27$return";
            $part1 = "$filler4$find[7]$space$find[8]";
            $outline = "$part0$filler2$part1$filler2$uvssci28$return";
           print OUTFILE $outline;
           $skip= 1;
       }
        if ($find[0] =~ "0000560") {
            $filler1 = "                                  ";
            $filler3 = "         ";
            $filler5 = "          ";
            $filler7 = "                        ";
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$uvssci29$return";
            $part1 = "$filler3$find[3]$space$find[4]";
            $outline = "$part0$filler3$part1$filler7$uvssci30$return";
           print OUTFILE $outline;
            $skip = 1;
        }
    }
print OUTFILE "=============================================================================\n";
}
##########################################################################################
##  DATA for CR-5A Engineering record is 3600 bytes
#######
#
sub proc_eng_data  
{
##618-306 Rev D chg 5 Voyager Experiment Data Record Format
##Figure B-4 defines the structure and contents of the Engineering data record for CR-5A (3600 bytse).
##
     my ($infile,$outfile) = @_;
     #     print "$infile\n";
     $return = "\n";
     $space = " ";
        $eng_cnt = 0;
        $engd = "COM POSITION ";
        $mfrec = 1;
        $fcnt = 100;
        $mfmf = "MF ";
        $mfcnt = 0;
        $filler1  = "                                 ";
        $filler2a = "             ";
        $filler2b = "                                 ";
        $filler3a = "                       ";
        $filler3b = "                            ";
        $filler4a = "                       ";
        $filler4b = "                       ";
        $filler5a = "                            ";
        $filler5b = "                  ";
        $filler6a = "                                 ";
        $filler6b = "             ";
        $filler7a = "                                      ";
        $filler7b = "         ";
        $filler8a = "                                           ";
        $filler8b = "  ";

     open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
     open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     print OUTFILE "=================================================================\n";
     print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
     print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
     print OUTFILE "Standard EDR data for Engineering (3600 bytes) Figure B-4 (pg B-5)\n";
     print OUTFILE "ENGR_DATA::\n\n";
LINEN: while (<INFILE>) {
     chop;
        $line = $_;
        $last = $cur;
        @find = split (/\s+/, $line);           #split on spaces
        @ch = split ( //,$line);         #split on characters
        $cur = $find[0];
        $eng_cnt++;
#        $outline = "$line$return";
#        print OUTFILE $outline;
#        print $outline;
        if ($find[0] =~ "0007420")  {
            $part0 = "$find[0]$space$find[1]$space$find[2]$filler1$engd$fcnt";
            $fcnt++;
            $part0a = "$space$fcnt$filler7b$mfmf$space$mfrec";
            $outline = "$part0$part0a$return";
            print OUTFILE $outline; 
            print OUTFILE "=============================================================\n";
           $skip = 1;
#            $return;
        } elsif ( $eng_cnt > 16) {
            $mfcnt++;
            if ($mfcnt >= 15)  {$mfrec++; $mfcnt=0; $fcnt = 100}
            $fe = $fcnt++;
            $ff = $fcnt++;
            $fg = $fcnt++;
            $fh = $fcnt++;
##            print OUTFILE "mfcnt = $mfcnt, fe = $fe, ff = $ff, fg = $fg, fh = $fh\n";
            $part0 = "$find[0]$space$find[5]$space$find[6]$space$find[7]$space$find[8]$filler3a";
            $part0a = "$engd$fe$space$ff$space$fg$space$fh$space$mfmf$space$mfrec";
            $outline = "$part0$part0a$return";
            print OUTFILE $outline;
            $mfcnt++;
##            print OUTFILE "mfcnt = $mfcnt\n";
            if ($mfcnt >= 15)  {$mfrec++; $mfcnt=0; $fcnt = 100}
            $fj = $fcnt++;
            $fk = $fcnt++;
            $fl = $fcnt++;
            $fm = $fcnt++;
##            print OUTFILE "mfcnt = $mfcnt, fj = $fj, fk = $fk, fl = $fl, fm = $fm\n";
#            if ($mfcnt > 7)  {$mfrec++; $mfcnt=0; $fcnt = 100}
            $part1 = "$return$filler5a$space$find[1]$space$find[2]$space$find[3]$space$find[4]$filler8b";
            $part1a= "$engd$fj$space$fk$space$fl$space$fm$space$mfmf$space$mfrec";
            $outline = "$part1$part1a$return";

#            $outline = "$line$return";
             print OUTFILE "$outline";
 #               if ($mfcnt > 6)  {$mfrec++; $mfcnt=0; $fcnt = 100}
        }
        return if ($find[0] =~ "0007420");
    }
#   return if ($find[0] =~ "0007420");
}
##########################################################################################
##  DATA for CR-5A DCOM TABLE is 448 bytes
#######
sub proc_dcom_data
{
##618-306 Rev D chg 5 Voyager Experiment Data Record Format
###Figure A-1 defines the structure and contents of the Decommutation Map data block  for CR-5A (3600 bytse).
###
#
     my ($infile,$outfile) = @_;
     #print "$infile\n";
     $return = "\n";
     $space = " ";
     $cnt = 0;
     $cnt100 = 99;
     $filler = "    ";
     open(INFILE,"$infile") || die "Cannot find INFILE: $infile\n";
     open(OUTFILE,">>$outfile") || die "Cannot open OUTFILE: $outfile\n";

     print OUTFILE "=================================================================\n";
     print OUTFILE "618-306 Voyager Experiment Data Record Format Specification\n";
     print OUTFILE "Rev D, Chg 5 - 1 May 1989\n";
     print OUTFILE "Standard EDR data for Decom TBL (448 bytes) Figure A-1 (pg A-3)\n";
     print OUTFILE "24 Minor Frames (MF) - 19.2 minutes\n";
     print OUTFILE "DCOM_DATA::\n\n";
LINEP: while (<INFILE>) {
     chop;
        $line = $_;
        $last = $cur;
        @find = split (/\s+/, $line);           #split on spaces
        @ch = split ( //,$line);         #split on characters
        $cur = $find[0];
        next LINEP if ($find[0] =~ "0001320");
        return if ($find[0] =~ "0001320");
        $value = int($find[0]);
        $letter = "";
        if ($value >= 400 && $value <= 660) {
            $cnt++;
#            print OUTFILE "$cnt$return";
            $cnt100++;
            if ($cnt == 17) {
#                print "here";
                $cnt100 = 299;
                $cnt100++;
            }
#            if ($cnt >= 107) {
#                $cnt = 213;
#                $cnt100 = 213;
#                $letter = "";
#            }
#            $cnt++;
            $descr1 = "MAP WORD $cnt DECK $letter$cnt100,";
            $cnt++;
     #       $cnt100++;
#            if ($cnt == 50) {
#                $letter = "A";
#                $cntx = "00";
#                $cnt100 = $cntx;
#            }
            if ($cnt > 51 && $cnt < 213) {$letter = "A";}
            $cnt100++;
            $descr2 = "MAP WORD $cnt DECK $letter$cnt100,";
            $cnt++;
            $cnt100++;
            if ($cnt == 47) {$cnt100 = 117;}
            $descr3 = "MAP WORD $cnt DECK $letter$cnt100,";
            $cnt++;
            $cnt100++;
            $descr4 = "MAP WORD $cnt DECK $letter$cnt100";
            $outline = "$line$filler$descr1$space$descr2$space$descr3$space$descr4$return";
            print OUTFILE $outline;
#           print $outline;
        }
        if ($value >=700 && $value < 1240) { 
            $cnt++;
#            print OUTFILE "$cnt$return";
            if ($cnt == 47) {
                $cnt100 = 117;
            }
            $cnt100++;
            if ($cnt >= 53) {$letter = "A";}

            $descr1 = "MAP WORD $cnt DECK $letter$cnt100,";
            $cnt++;
            if ($cnt == 50) {
                $letter = "A";
                $cntx = "00";
                $cnt100 = $cntx;
            }
#            $cnt100++;
            $descr2 = "MAP WORD $cnt DECK $letter$cnt100,";
            $cnt++;
            $cnt100++;
            $descr3 = "MAP WORD $cnt DECK $letter$cnt100,";
            $cnt++;
            $cnt100++;
            $descr4 = "MAP WORD $cnt DECK $letter$cnt100";
            $outline = "$line$filler$descr1$space$descr2$space$descr3$space$descr4$return";
            print OUTFILE $outline;
            next LINEP;
        }
        if ($value == 1240) {
#            print OUTFILE "++++$return";
            if ($value == 1240) {$cnt = 213;}
#            print OUTFILE "$cnt$return";
            if ($cnt == 213) {$cnt100 = 213;}
           $descr1 = "MAP WORD $cnt DECK $letter$cnt100,";
            $cnt++;
            $cnt100++;
            $descr2 = "MAP WORD $cnt DECK $letter$cnt100,";
            $cnt++;
            $cnt100++;
            $descr3 = "MAP WORD $cnt DECK $letter$cnt100,";
            $cnt++;
            $cnt100++;
            if ($cnt == 220) {$cnt100 = 156;}
            $descr4 = "MAP WORD $cnt DECK $letter$cnt100";
            $outline = "$line$filler$descr1$space$descr2$space$descr3$space$descr4$return";
            print OUTFILE $outline;
            next LINEP;


        }
        if ($value == 1260) {
            $cnt++;
            $cnt100++;
           $descr1 = "MAP WORD $cnt DECK $letter$cnt100,";
            $cnt++;
            $cnt100++;
            $descr2 = "MAP WORD $cnt DECK $letter$cnt100,";
            $cnt++;
            $cnt100++;
            $descr3 = "MAP WORD $cnt DECK $letter$cnt100,";
            $cnt++;
            $cnt100++;
            if ($cnt == 220) {$cnt100 = 156;}
            $descr4 = "MAP WORD $cnt DECK $letter$cnt100";
            $outline = "$line$filler$descr1$space$descr2$space$descr3$space$descr4$return";
            print OUTFILE $outline;
            $cnt++;
            $cnt100++;
            next LINEP;




        }
        if ($value  == 1300) {
#            $cnt++;
#            $cnt100++;
            $descr1 = "MAP WORD $cnt DECK $cnt100,";
            $cnt++;
            $cnt100++;
            $descr2 = "MAP WORD $cnt DECK $cnt100,";
            $cnt++;
            $cnt100++;
            $descr3 = "MAP WORD $cnt DECK $cnt100,";
            $filler= "              ";
            $descr4 = "SPARE";
            $outline = "$line$filler$descr1$space$descr2$space$descr3$space$descr4$return";
            print OUTFILE $outline;
        }

#   return if ($find[0] =~ "0007440");
#
    }
}
