/*
         1         2         3         4         5         6         7
123456789012345678901234567890123456789012345678901234567890123456789012
*/
#ifndef Header
/*****************************************************************
 * TITLE: sfparser.c
 *
 * AUTHOR:  Unknown
 *          Aug 31, 1994
 *
 * MODIFIED:    Ray Bambery
 *          Aug 24, 2020 -  removed label after #ENDIF Header 
 *****************************************************************
** MANUAL
**	SFPARSER 3x "April 22, 1990"
**
** NAME
**	sfparser - parse keyword equal to value or just word.
**
** SYNOPSIS
**
**	   int  sfnetckv_parser(rdinfo,indx_in_buf,
**	                          bufnumber,buf,buflen,
**	                          pars_func,spcl_chars,
**	                          use_len);
**
**	   int  sfnetcall_parser(rdinfo,indx_in_buf,
**	                          bufnumber,buf,buflen,
**	                          pars_func,spcl_chars,
**	                          use_len);
**
**	   int  sfetckv_parser(rdinfo,indx_in_buf,
**	                          bufnumber,buf,buflen,
**	                          pars_func,spcl_chars,
**	                          use_len);
**
**	   int  sfetcall_parser(rdinfo,indx_in_buf,
**	                          bufnumber,buf,buflen,
**	                          pars_func,spcl_chars,
**	                          use_len);
**
**	   char *sfndifbin(rdinfo,indx_in_buf,
**	                          bufnumber,buf,buflen,
**	                          use_len);
**
**	   struct  rdinfo  *rdinfo;
**	   int     indx_in_buf;
**	   int     bufnumber;
**	   char    *buf;
**	   int     buflen;
**	   int     (*pars_func)();
**	   char    *spcl_chars;
**	   int     use_len;
**
** PARAMETERS
**
**	    1. struct  rdinfo - for buffer management.
**	    2. indx_in_buf - where in the buffer in rdbuf.
**	    3. bufnumber - rdbuf->bufs[bufnumber] for buffer.
**	    4. buf - If rdinfo is NULL then the information
**	       to parse is pointed to by buf.
**	    5. buflen - length of buf.
**	    6. pars_func - pointer to function to call
**	       for keyword value pairs and words.
**	    7. spcl_chars - NULL terminated list of characters
**	       that need special proceesing.
**	    8. use_len - length to use for countin
**	       how many bytes to use. Over rides
**	       all other lengths. Ignore if zero.
**
** STRUCTURES
**
**
**	  structure bufinfo, bufsin, rdbuf used for buffer management
**
**	  struct  bufinfo {
**	     char  *buf;     points to buffer containing map
**	     int   buflen;   length of buffer in bytes
**	     int   bufnumber; current buffer number.
**	     BOOL  contflag; TRUE for more buffers.
**	     int   bufrmode; returned mode for buffers.
**	  } ;
**
**	  bufrmode can be
**	  1. NO_NEW_BUFR - ignore this request.
**	  2. NEXT_BUFR - get next buffer.
**	  3. PREV_BUFR - get previous buffer.
**	  4. BOF - bottom of file get first buffer.
**	  5. EOF - get last buffer.
**
**	  struct rdinfo {
**	     FOPDEF(frd) - frd is file descriptor.
**	     char  *filename; file name of frd.
**	     char  **list_pntr; set of string pointers
**	                        used if filname = NULL
**	     int   indx_in_list - used along with list_pntr.
**	     char  *rdbuf; pointer to buffer struct.
**	     char  *start_adis_pntr; points to first adis structure.
**	  }
**
**	  struct bufsin {
**	     int   tot_bufs;  total number of buffers.
**	     int   tot_bytes; total bytes in all buffers.
**	     BOOL  more_reads; if All buffers full but
**	                       more reads or lists needed.
**	     struct  {
**	        char *pntr; points to buffer.
**	        int  lngth; number of bytes in buf.
**	        int  count; true buffer numbernot modulo MAXNUMBUFRS.
**	     } [MAXNUMBUFRS];
**	  }
**
** FUNCTION
**
**	'pars_func' points to a user supplied function.
**
**	The function must have the following attributes.
**
**	  int  (*pars_func)(indx_of_1st_wrd,len_1st_wrd,
**	                    buf_num_1st_wrd,
**	                    indx_of_2nd_wrd,len_2nd_wrd,
**	                    buf_num_2nd_wrd,
**	                    running_count);
**
**	  int  indx_of_1st_wrd;   index in buffer
**	  int  len_1st_wrd;       length, in bytes, of 1st word.
**	  int  buf_num_1st_wrd;   number of buffer in rdbuf.
**	  int  indx_of_2nd_wrd;   index in buffer.
**	  int  len_2nd_wrd;       length, in bytes, of 2nd word.
**	  int  buf_num_2nd_wrd;   number of buffer in rdbuf.
**	  long running_count;     Number of bytes from start
**	                          of actual parsing to start
**	                          of word.
**
** DESCRIPTION
**
**	sfparser parses the buffers input in 'rdinfo' for
**	all the keyword value pairs. If 'rdinfo' is NULL then
**	'sfparser' uses the information in 'buf' and 'buflen'.
**	If 'rdinfo' is NULL then indx_in_buf and bufnumber is
**	ignored.
**
**	'sfnetckv_parser' and 'sfnetcall_parser' do not check
**	for comments or for double or single quotes. The words
**	are bounded by any non ascii character i.e. '\n', '\0'
**	etc.
**
**	'sfetckv_parser' and 'sfetcall_parser' check for comments
**	double and single quotes. The words are bounded by '\n', '\t',
**	',', ';',CR,'\0',' '. They also check for non ASCII characters.
**
**	'sfnetcall_parser' and 'sfetcall_parser' return keyword
**	value pairs as well as individual words, not part of keyword
**	value pairs. This also includes ';'.
**
**	'sfnetckv_parser' and 'sfetckv_parser' anly find keyword
**	value pairs.
**
**	When a keyword value pair have been found, 'sfparser'
**	calls
**
**	  int  (*pars_func)(indx_of_1st_wrd,len_1st_wrd,
**	                    buf_num_1st_wrd,
**	                    indx_of_2nd_wrd,len_2nd_wrd,
**	                    buf_num_2nd_wrd,
**	                    running_count);
**
**	where 1st_wrd points to keyword, and 2nd_wrd points to
**	value. Because the file in 'rdinfo' can span several
**	buffers the start of the 1st word can be in a different buffer
**	than the 2nd_wrd. 'running_count' is the number of bytes
**	to second word.
**
**	For 'sfnetcall_parser' and 'sfetcall_parser' when a 'word'
**	(not a keyword value pair) has been found 'sfparser' calls
**
**	  int  (*pars_func)(indx_of_1st_wrd,len_1st_wrd,
**	                    buf_num_1st_wrd,
**	                    -1,-1,0,running_count);
**
**	where 1st word points to the lone word.
**
**	The users function 'pars_func' returns '0' to continue.
**	'1' for finished and '-1' an error has occured.
**
**	'spcl_chars' is a list terminated by '\0'.
**	When one of these characters are found 'pars_func'
**	is called as a single word. This is for
**	certain types of keyword value languages.
**	NOTE: if a special character of '\0' is needed then make
**	this the first character in the list.
**	If no special characters are wanted then spcl_chars
**	must be equal to NULL (not '\0').
**	When a special character has been found 'sfparser'
**	calls (*pars_func) with the following parameters
**
**	  int  (*pars_func)(indx_of_char,1,
**	                    buf_of_char,
**	                    -1,indx_of_spcl_char,0,
**	                    running_count);
**
**	where 'indx_of_char' is the index into buffer,
**	the length is 1 byte, buf_of_char is the buffer number
**	for 'rdbuf', the -1 tells that this has no value
**	the indx_of_spcl_char is the index into 'spcl_chars'.
**
**	'sfparser' returns current buffer number for buffer management.
**	'sfparser' returns '-1' for error.
**
**	If the user wishes to use a single buffer and not a
**	file read using 'sfbufman' then the user can inbut
**	this in 'buf' and 'buf' len rather than 'rdonfo'
**	NOTE: 'rdinfo' MUST be set to NULL for 'sfparser'
**	to use 'buf' and 'buflen'.
**
**	The 'running_count' keeps actual count of offset from
**	startint posiotion to the start of the last
**	word in pars_func.
**
**	If the user does not know the length of the buffer
**	then the length should be set to -1 BUT beware
**	if the 'sfparser' does not find what the user wants
**	it will parse until the cow comes home or until it
**	core dumps!!!!!!
**
**      If the parser comes to the end of file or
**	'use_len' is used up then
**	calls (*pars_func) with the following parameters
**
**	  int  (*pars_func)(indx_of_last_char,-1,
**	                    buf_of_last_char,
**	                    -1,-1,0,
**	                    running_count);
**
**	If 'sfetcall_parser' or 'sfetckv_parser' find a non ASCII character
**	then they call (*pars_func) with the following parameters
**
**	  int  (*pars_func)(indx_of_nonascii_char,1,
**	                    buf_of_nonascii_char,
**	                    -2,-1,0,
**	                    running_count);
**
**
**	'sfndifbin' finds if the buffer contains binary characters i.e.
**	non ascii. If found 'sfndifbin' returns pointer to first non
**	ascii (excluding tabs,carriage control, newlines etc)
**	if none found a NULL pointer is returned.
**
** RETURN
**
**	'sfparser' returns '0' for success and a '-1' for error.
**
** SEE ALSO
**
**	sfbufman
**
** EXAMPLE

  This is a set of examples.

  EXAMPLE 1. use special characters with 'sfetcall_parser'

   char   *spcl_list '\0','\032','\0' };

   This will call (*pars_func) with an index for '\0' and for '\032'
   which is a control Z.
   '\0' needs to be the first character if it is one of the special
   characters.

** END MAN PAGES
**
*/
#endif  /*  Header   */
#if (IBM_MAIN_FRAME)
/*place pragma to allow for entry points for JTPM*/
#pragma csect(CODE,"SCPARSE0") csect(STATIC,"SCPARSE1")
#endif

#include	<stdio.h>
#include	<ctype.h>
#include	<sftypes.h>
#include	<sfbufmgr.h>

static	BOOL	astric_found = FALSE;
static	BOOL	slash_found = FALSE;
static	BOOL	is_double_quote = FALSE;
static	BOOL	is_single_quote = FALSE;
static	BOOL	is_parens = FALSE;
static	BOOL	is_comment = FALSE;
static	BOOL	is_keyword;
static	BOOL	start_word;
static	BOOL	next_char_is_end; /*used for singl double quotes*/
static	int	start_prev_indx;
static	int	prev_len;
static	int	prev_bufnumber;
static	int	start_bufnumber;
static	int	start_indx;
static	struct	bufinfo bufinfo;
static	struct	bufsin	*rdbuf;
static	int	word_count;
static	BOOL	use_spcl;
static	BOOL	is_spcl;
static	int	spcl_i;
static	BOOL	is_ascii;
static	int	ascii_i;
static	long	running_count;
static	long	prev_running_count;
static	long	start_running_count;
static	long	offset_running_count;

sfparser()
{
	/*empty call for IBM main frame*/
}

sfetcall_parser(rdinfo,indx_in_buf,
                  bufnumber,buf,buflen,
		  pars_func,spcl_chars,
		  use_len)

struct	rdinfo	*rdinfo;
int	indx_in_buf;
int	bufnumber;
char	*buf;
int	buflen;
int	(*pars_func)();
char	*spcl_chars;
int	use_len;

{

int	rt_code;
register	char	*pntr;
register	int	i;
register	int	len;
int	blen;

/*begin*/

if (rdinfo != (struct rdinfo *)NULL)
{
	rdbuf = (struct bufsin *)rdinfo->rdbuf;
	bufinfo.bufnumber = bufnumber;
	bufinfo.buf = rdbuf->bufs[bufnumber%MAXNUMBUFRS].pntr;
	bufinfo.bufrmode = NEXT_BUFR;
	bufinfo.buflen = rdbuf->bufs[bufnumber%MAXNUMBUFRS].lngth;
	i = indx_in_buf;
	if (rdbuf->more_reads ||
	    bufnumber < rdbuf->tot_bufs - 1)
		bufinfo.contflag = TRUE;
	else
		bufinfo.contflag = FALSE;
	pntr = &bufinfo.buf[i];
}
else
{
	bufinfo.buf = buf;
	bufinfo.buflen = buflen;
	bufinfo.contflag = FALSE;
	bufinfo.bufnumber = 0;
	i = 0;
	pntr = buf;
}
prev_bufnumber = bufinfo.bufnumber;
start_bufnumber = bufinfo.bufnumber;
if (spcl_chars == NULL)
	use_spcl = FALSE;
else
	use_spcl = TRUE;
is_keyword = FALSE;
start_word = FALSE;
start_prev_indx = -1;
start_indx = -1;
is_comment = FALSE;
next_char_is_end = FALSE;
word_count = 0;
prev_len = 0;
len = 0;
running_count = 0;
start_running_count = 0;
prev_running_count = 0;
offset_running_count = i;
astric_found = FALSE;
slash_found = FALSE;
is_double_quote = FALSE;
is_single_quote = FALSE;
is_parens = FALSE;
is_comment = FALSE;
for (;;)
{
	if (use_len   &&
	    use_len == running_count + i - offset_running_count)
	{
		rt_code = 0;
		if (is_comment)
		{
			/*this is an error no end to comments*/
			return(-1);
		}
		if (is_double_quote)
		{
			/*no ending quotes found*/
			return(-1);
		}
		if (is_single_quote)
		{
			/*no ending single quote*/
			return(-1);
		}
		if (is_parens)
		{
			/*no ending parens*/
			return(-1);
		}
		if (start_word)
		{
			start_word = FALSE;
			if (is_keyword)
			{
				is_keyword = FALSE;
				rt_code = (*pars_func)(
					start_prev_indx,
					prev_len,
					prev_bufnumber,
					start_indx,
					len,
					start_bufnumber,
					prev_running_count);
				word_count = 0;
			}
		}
		if (word_count)
		{
			rt_code = 0;
			if (word_count == 2)
				rt_code = (*pars_func)(
					 start_prev_indx,
					 prev_len,
					 prev_bufnumber,
					 -1,-1,0,
					 prev_running_count);
			if (!rt_code)
				rt_code = (*pars_func)(
					  start_indx,
					  len,
					  start_bufnumber,
					  -1,-1,0,
					  start_running_count);
		}
		else
		{
			i--;
			start_running_count = running_count + i - 
				offset_running_count;
			rt_code = (*pars_func)(
				  i,
				  -1,
				  bufinfo.bufnumber,
				  -1,-1,0,
				  start_running_count);
		}
		if (rt_code < 0)
			return(rt_code);
		return(0);
	}
	else if (i == bufinfo.buflen)
	{
		/*this is the end of this buffer*/
		if (!bufinfo.contflag)
		{
			if (is_comment)
			{
				/*this is an error no end to comments*/
				return(-1);
			}
			if (is_double_quote)
			{
				/*no ending quotes found*/
				return(-1);
			}
			if (is_single_quote)
			{
				/*no ending single quote*/
				return(-1);
			}
			if (is_parens)
			{
				/*no ending parens*/
				return(-1);
			}
			if (start_word)
			{
				start_word = FALSE;
				if (is_keyword)
				{
					is_keyword = FALSE;
					rt_code = (*pars_func)(
						start_prev_indx,
						prev_len,
						prev_bufnumber,
						start_indx,
						len,
						start_bufnumber,
						prev_running_count);
					word_count = 0;
				}
			}
			if (word_count)
			{
				rt_code = 0;
				if (word_count == 2)
					rt_code = (*pars_func)(
						 start_prev_indx,
						 prev_len,
						 prev_bufnumber,
						 -1,-1,0,
						 prev_running_count);
				if (!rt_code)
					rt_code = (*pars_func)(
						  start_indx,
						  len,
						  start_bufnumber,
						  -1,-1,0,
						  start_running_count);
			}
			else
			{
				i--;
				start_running_count = running_count + i - 
					offset_running_count;
				rt_code = (*pars_func)(
					  i,
					  -1,
					  bufinfo.bufnumber,
					  -1,-1,0,
					  start_running_count);
			}
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		running_count += i - offset_running_count;
		blen = sfbufman(&bufinfo,rdinfo);
		if (blen <= 0)
			return(-1);
		i = 0;
		offset_running_count = 0; /*start at new place*/
		pntr = bufinfo.buf;
	}
	if (astric_found  && pntr[0] == '/')
	{
		/*end of comment*/
		astric_found = FALSE;
		is_comment = FALSE;
		pntr++; /*beyond */
		i++;
		continue;
	}
	if (is_comment)
	{
		if (pntr[0] == '*')
			astric_found = TRUE;
		else
			astric_found = FALSE;
		i++;
		pntr++;
		continue;
	}
	if (next_char_is_end)
	{
		next_char_is_end = FALSE;
		start_word = FALSE;
		if (!is_keyword)
		{
			if (word_count == 2)
			{
				rt_code = (*pars_func)(
					 start_prev_indx,
					 prev_len,
					 prev_bufnumber,
					 -1,-1,0,prev_running_count);
				if (rt_code)
				{
					if (rt_code < 0)
						return(rt_code);
					return(0);
				}
				word_count--;
			}
			i++;
			pntr++;
			continue;
		}
		word_count = 0;
		is_keyword = FALSE;
		rt_code = (*pars_func)(start_prev_indx,
				prev_len,
				prev_bufnumber,
				start_indx,
				len,
				start_bufnumber,
				prev_running_count);
		if (rt_code)
		{
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		word_count = 0;
		i++;
		pntr++;
		continue;
	}
	if (is_single_quote)
	{
		if (pntr[0] != '\'')
		{
			pntr++;
			i++;
			len++;
			continue;
		}
		is_single_quote = FALSE;
		next_char_is_end = TRUE;
		len++;
		pntr++;
		i++; /*go one past*/
		continue;
	}
	if (is_double_quote)
	{
		if (pntr[0] != '\"')
		{
			pntr++;
			i++;
			len++;
			continue;
		}
		is_double_quote = FALSE;
		next_char_is_end = TRUE;
		len++;
		pntr++;
		i++; /*go one past*/
		continue;
	}
	if (is_parens)
	{
		if (pntr[0] != ')')
		{
			pntr++;
			i++;
			len++;
			continue;
		}
		is_parens = FALSE;
		next_char_is_end = TRUE;
		len++;
		pntr++;
		i++; /*go one past*/
		continue;
	}
	/*check for start of comment*/
	if (slash_found &&  pntr[0] == '*')
	{
		slash_found = FALSE;
		/*start of comment*/
		is_comment = TRUE;
		if (start_word)
		{
			start_word = FALSE;
			if (!is_keyword)
			{
				if (word_count == 2)
				{
					rt_code = (*pars_func)(
						 start_prev_indx,
						 prev_len,
						 prev_bufnumber,
						 -1,-1,0,prev_running_count);
					if (rt_code)
					{
						if (rt_code < 0)
							return(rt_code);
						return(0);
					}
					word_count--;
				}
				i++;
				pntr++;
				continue;
			}
			word_count = 0;
			is_keyword = FALSE;
			rt_code = (*pars_func)(start_prev_indx,
					prev_len,
					prev_bufnumber,
					start_indx,
					len,
					start_bufnumber,
					prev_running_count);
			if (rt_code)
			{
				if (rt_code < 0)
					return(rt_code);
				return(0);
			}
			word_count = 0;
		}
		i++;
		pntr++;
		continue;
	}
	slash_found = FALSE;
	if (pntr[0] == '=')
	{
		/*this is equals have word etc*/
		if (start_word)
		{
			start_word = FALSE;
			if (word_count == 2)
			{
				rt_code = (*pars_func)(
					 start_prev_indx,
					 prev_len,
					 prev_bufnumber,
					 -1,-1,0,prev_running_count);
				if (rt_code)
				{
					if (rt_code < 0)
						return(rt_code);
					return(0);
				}
				word_count--;
			}
		}
		is_keyword = TRUE;
		i++;
		pntr++;
		/*cases go here*/
		continue;
	}
	is_ascii = FALSE;
	/*check for non ascii character*/
	if (iscntrl(pntr[0])  &&
	    pntr[0] != '\n'   &&
	    pntr[0] != CR     &&
	    pntr[0] != '^'    &&
	    pntr[0] != '\t')
	{
		is_ascii = TRUE;
	}
	is_spcl = FALSE;
	if (use_spcl)
	{
		spcl_i = 0;
		for (;;)
		{
			if (spcl_i  &&
			    spcl_chars[spcl_i] == '\0')
				break;
			if (spcl_chars[spcl_i] == pntr[0])
			{
				is_spcl = TRUE;
				break;
			}
			spcl_i++;
		}
	}
	if (*pntr != ' '   &&
	    *pntr != ','   &&
	    *pntr != '\''  &&
	    *pntr != '\"'  &&
	    *pntr != '/'   &&
	    *pntr != '('   &&
	    *pntr != ';'   &&
	    !is_spcl       &&
	    !iscntrl(pntr[0]))
	{
		if (start_word)
		{
			pntr++;
			i++;
			len++;
			continue;
		}
		word_count++;
		start_word = TRUE;
		prev_len = len;
		prev_running_count = start_running_count;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_indx = i;
		start_bufnumber = bufinfo.bufnumber;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		pntr++;
		i++;
		continue;
	}
	/*this is 'white'spaces*/
	if (start_word)
	{
		/*finish up start word here*/
		start_word = FALSE;
		if (!is_keyword)
		{
			if (word_count == 2)
			{
				rt_code = (*pars_func)(
					 start_prev_indx,
					 prev_len,
					 prev_bufnumber,
					 -1,-1,0,prev_running_count);
				if (rt_code)
				{
					if (rt_code < 0)
						return(rt_code);
					return(0);
				}
				word_count--;
			}
			if (!is_spcl)
			{
				pntr++;
				i++;
			}
			continue;
		}
		word_count = 0;
		is_keyword = FALSE;
		rt_code = (*pars_func)(start_prev_indx,
			prev_len,
			prev_bufnumber,
			start_indx,
			len,
			start_bufnumber,
			prev_running_count);
		if (rt_code)
		{
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		word_count = 0;
	}
	if (!is_spcl          &&
	    iscntrl(pntr[0])  &&
	    *pntr != '\t'     &&
	    *pntr != '\n'     &&
	    *pntr != ';'   &&
	    *pntr != CR)
	{
		is_spcl = TRUE;
		spcl_i = -1;
	} 
	if (is_ascii)
	{
		prev_len = len;
		prev_running_count = start_running_count;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		if (word_count)
		{
			rt_code = (*pars_func)(
					 start_prev_indx,
					 prev_len,
					 prev_bufnumber,
					 -1,-1,0,prev_running_count);
			if (rt_code)
			{
				if (rt_code < 0)
					return(rt_code);
				return(0);
			}
		}
		rt_code = (*pars_func)(
					 start_indx,
					 1,
					 start_bufnumber,
					 -2,
					 -1,
					 0,
					 start_running_count);
		if (rt_code)
		{
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		word_count = 0;
		pntr++;
		i++;
		continue;
	}
	if (is_spcl)
	{
		prev_len = len;
		prev_running_count = start_running_count;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		if (word_count)
		{
			rt_code = (*pars_func)(
					 start_prev_indx,
					 prev_len,
					 prev_bufnumber,
					 -1,-1,0,prev_running_count);
			if (rt_code)
			{
				if (rt_code < 0)
					return(rt_code);
				return(0);
			}
		}
		rt_code = (*pars_func)(
					 start_indx,
					 1,
					 start_bufnumber,
					 -1,
					 spcl_i,
					 0,
					 start_running_count);
		if (rt_code)
		{
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		word_count = 0;
		pntr++;
		i++;
		continue;
	}
	if (pntr[0] == '\'')
	{
		is_single_quote = TRUE;
		word_count++;
		start_word = TRUE;
		prev_running_count = start_running_count;
		prev_len = len;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		pntr++;
		i++;
		continue;
	}
	if (pntr[0] == '\"')
	{
		is_double_quote = TRUE;
		word_count++;
		start_word = TRUE;
		prev_running_count = start_running_count;
		prev_len = len;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		pntr++;
		i++;
		continue;
	}
	if (pntr[0] == '(')
	{
		is_parens = TRUE;
		word_count++;
		start_word = TRUE;
		prev_running_count = start_running_count;
		prev_len = len;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		pntr++;
		i++;
		continue;
	}
	if (pntr[0] == '/')
	{
		slash_found = TRUE;
		pntr++;
		i++;
		continue;
	}
	i++;
	pntr++;
} /*end of inner for loop*/
}

sfetckv_parser(rdinfo,indx_in_buf,
                  bufnumber,buf,buflen,
		  pars_func,spcl_chars,
		  use_len)

struct	rdinfo	*rdinfo;
int	indx_in_buf;
int	bufnumber;
char	*buf;
int	buflen;
int	(*pars_func)();
char	*spcl_chars;
int	use_len;

{

int	rt_code;
register	char	*pntr;
register	int	i;
register	int	len;
int	blen;

/*begin*/
if (rdinfo != (struct rdinfo *)NULL)
{
	rdbuf = (struct bufsin *)rdinfo->rdbuf;
	bufinfo.bufnumber = bufnumber;
	bufinfo.buf = rdbuf->bufs[bufnumber%MAXNUMBUFRS].pntr;
	bufinfo.bufrmode = NEXT_BUFR;
	bufinfo.buflen = rdbuf->bufs[bufnumber%MAXNUMBUFRS].lngth;
	i = indx_in_buf;
	if (rdbuf->more_reads ||
	    bufnumber < rdbuf->tot_bufs - 1)
		bufinfo.contflag = TRUE;
	else
		bufinfo.contflag = FALSE;
	pntr = &bufinfo.buf[i];
}
else
{
	bufinfo.buf = buf;
	bufinfo.buflen = buflen;
	bufinfo.contflag = FALSE;
	bufinfo.bufnumber = 0;
	i = 0;
	pntr = buf;
}
start_bufnumber = bufinfo.bufnumber;
prev_bufnumber = bufinfo.bufnumber;
if (spcl_chars == NULL)
	use_spcl = FALSE;
else
	use_spcl = TRUE;
word_count = 0;
is_keyword = FALSE;
start_word = FALSE;
start_prev_indx = -1;
start_indx = -1;
is_comment = FALSE;
astric_found = FALSE;
slash_found = FALSE;
is_double_quote = FALSE;
is_single_quote = FALSE;
is_parens = FALSE;
next_char_is_end = FALSE;
prev_len = 0;
len = 0;
running_count = 0;
start_running_count = 0;
prev_running_count = 0;
offset_running_count = i;
for (;;)
{
	if (use_len   &&
	    use_len == running_count + i - offset_running_count)
	{
		if (is_comment)
		{
			/*this is an error no end to comments*/
			return(-1);
		}
		if (is_double_quote)
		{
			/*no ending quotes found*/
			return(-1);
		}
		if (is_single_quote)
		{
			/*no ending single quote*/
			return(-1);
		}
		if (is_parens)
		{
			/*no ending parens*/
			return(-1);
		}
		if (start_word)
		{
			start_word = FALSE;
			if (is_keyword)
			{
				is_keyword = FALSE;
				rt_code = (*pars_func)(
					start_prev_indx,
					prev_len,
					prev_bufnumber,
					start_indx,
					len,
					start_bufnumber,
					prev_running_count);
			}
		}
		else
		{
			i--;
			start_running_count = running_count + i - 
				offset_running_count;
			rt_code = (*pars_func)(
				  i,
				  -1,
				  bufinfo.bufnumber,
				  -1,-1,0,
				  start_running_count);
		}
		if (rt_code < 0)
			return(rt_code);
		return(0);
	}
	else if (i == bufinfo.buflen)
	{
		/*this is the end of this buffer*/
		if (!bufinfo.contflag)
		{
			if (is_comment)
			{
				/*this is an error no end to comments*/
				return(-1);
			}
			if (is_double_quote)
			{
				/*no ending quotes found*/
				return(-1);
			}
			if (is_single_quote)
			{
				/*no ending single quote*/
				return(-1);
			}
			if (is_parens)
			{
				/*no ending parens*/
				return(-1);
			}
			if (start_word)
			{
				start_word = FALSE;
				if (is_keyword)
				{
					is_keyword = FALSE;
					rt_code = (*pars_func)(
						start_prev_indx,
						prev_len,
						prev_bufnumber,
						start_indx,
						len,
						start_bufnumber,
						prev_running_count);
				}
			}
			else
			{
				i--;
				start_running_count = running_count + i - 
					offset_running_count;
				rt_code = (*pars_func)(
					  i,
					  -1,
					  bufinfo.bufnumber,
					  -1,-1,0,
					  start_running_count);
			}
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		running_count += i - offset_running_count;
		blen = sfbufman(&bufinfo,rdinfo);
		if (blen <= 0)
			return(-1);
		i = 0;
		offset_running_count = 0;
		pntr = bufinfo.buf;
	}
	if (astric_found  && pntr[0] == '/')
	{
		/*end of comment*/
		astric_found = FALSE;
		is_comment = FALSE;
		pntr++; /*beyond */
		i++;
		continue;
	}
	if (is_comment)
	{
		if (pntr[0] == '*')
			astric_found = TRUE;
		else
			astric_found = FALSE;
		i++;
		pntr++;
		continue;
	}
	if (next_char_is_end)
	{
		next_char_is_end = FALSE;
		start_word = FALSE;
		if (!is_keyword)
		{
			i++;
			pntr++;
			continue;
		}
		is_keyword = FALSE;
		rt_code = (*pars_func)(start_prev_indx,
				prev_len,
				prev_bufnumber,
				start_indx,
				len,
				start_bufnumber,
				prev_running_count);
		if (rt_code)
		{
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		i++;
		pntr++;
		continue;
	}
	if (is_single_quote)
	{
		if (pntr[0] != '\'')
		{
			pntr++;
			i++;
			len++;
			continue;
		}
		is_single_quote = FALSE;
		next_char_is_end = TRUE;
		len++;
		pntr++;
		i++; /*go one past*/
		continue;
	}
	if (is_double_quote)
	{
		if (pntr[0] != '\"')
		{
			pntr++;
			i++;
			len++;
			continue;
		}
		is_double_quote = FALSE;
		next_char_is_end = TRUE;
		len++;
		pntr++;
		i++; /*go one past*/
		continue;
	}
	if (is_parens)
	{
		if (pntr[0] != ')')
		{
			pntr++;
			i++;
			len++;
			continue;
		}
		is_parens = FALSE;
		next_char_is_end = TRUE;
		len++;
		pntr++;
		i++; /*go one past*/
		continue;
	}
	/*check for start of comment*/
	if (slash_found &&  pntr[0] == '*')
	{
		slash_found = FALSE;
		/*start of comment*/
		is_comment = TRUE;
		if (start_word)
		{
			start_word = FALSE;
			if (!is_keyword)
			{
				i++;
				pntr++;
				continue;
			}
			is_keyword = FALSE;
			rt_code = (*pars_func)(
					start_prev_indx,
					prev_len,
					prev_bufnumber,
					start_indx,
					len,
					start_bufnumber,
					prev_running_count);
			if (rt_code)
			{
				if (rt_code < 0)
					return(rt_code);
				return(0);
			}
		}
		i++;
		pntr++;
		continue;
	}
	slash_found = FALSE;
	if (pntr[0] == '=')
	{
		/*this is equals have word etc*/
		if (start_word)
			start_word = FALSE;
		is_keyword = TRUE;
		i++;
		pntr++;
		/*cases go here*/
		continue;
	}
	is_ascii = FALSE;
	/*check for non ascii character*/
	if (iscntrl(pntr[0])  &&
	    pntr[0] != '\n'   &&
	    pntr[0] != CR     &&
	    pntr[0] != '^'    &&
	    pntr[0] != '\t')
	{
		is_ascii = TRUE;
	}
	is_spcl = FALSE;
	if (use_spcl)
	{
		spcl_i = 0;
		for (;;)
		{
			if (spcl_i  &&
			    spcl_chars[spcl_i] == '\0')
				break;
			if (spcl_chars[spcl_i] == pntr[0])
			{
				is_spcl = TRUE;
				break;
			}
			spcl_i++;
		}
	}
	if (*pntr != ' '   &&
	    *pntr != ','   &&
	    *pntr != '\''  &&
	    *pntr != '\"'  &&
	    *pntr != '('   &&
	    *pntr != '/'   &&
	    *pntr != ';'   &&
	    !is_spcl       &&
	    !iscntrl(pntr[0]))
	{
		if (start_word)
		{
			pntr++;
			i++;
			len++;
			continue;
		}
		start_word = TRUE;
		prev_running_count = start_running_count;
		prev_len = len;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		pntr++;
		i++;
		continue;
	}
	/*this is 'white'spaces*/
	if (start_word)
	{
		/*finish up start word here*/
		start_word = FALSE;
		if (!is_keyword)
		{
			if (!is_spcl)
			{
				pntr++;
				i++;
			}
			continue;
		}
		is_keyword = FALSE;
		rt_code = (*pars_func)(
			start_prev_indx,
			prev_len,
			prev_bufnumber,
			start_indx,
			len,
			start_bufnumber,
			prev_running_count);
		if (rt_code)
		{
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
	}
	if (is_ascii)
	{
		prev_len = len;
		prev_running_count = start_running_count;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		if (word_count)
		{
			rt_code = (*pars_func)(
					 start_prev_indx,
					 prev_len,
					 prev_bufnumber,
					 -1,-1,0,prev_running_count);
			if (rt_code)
			{
				if (rt_code < 0)
					return(rt_code);
				return(0);
			}
		}
		rt_code = (*pars_func)(
					 start_indx,
					 1,
					 start_bufnumber,
					 -2,
					 -1,
					 0,
					 start_running_count);
		if (rt_code)
		{
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		word_count = 0;
		pntr++;
		i++;
		continue;
	}
	if (is_spcl)
	{
		prev_len = len;
		prev_running_count = start_running_count;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		rt_code = (*pars_func)(
					 start_indx,
					 1,
					 start_bufnumber,
					 -1,
					 spcl_i,
					 0,
					 start_running_count);
		if (rt_code)
		{
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		pntr++;
		i++;
		continue;
	}
	if (pntr[0] == '\'')
	{
		is_single_quote = TRUE;
		start_word = TRUE;
		prev_running_count = start_running_count;
		prev_len = len;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		pntr++;
		i++;
		continue;
	}
	if (pntr[0] == '\"')
	{
		is_double_quote = TRUE;
		start_word = TRUE;
		prev_running_count = start_running_count;
		prev_len = len;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		pntr++;
		i++;
		continue;
	}
	if (pntr[0] == '(')
	{
		is_parens = TRUE;
		start_word = TRUE;
		prev_running_count = start_running_count;
		prev_len = len;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		pntr++;
		i++;
		continue;
	}
	if (pntr[0] == '/')
	{
		slash_found = TRUE;
		pntr++;
		i++;
		continue;
	}
	i++;
	pntr++;
} /*end of inner for loop*/
}

sfnetcall_parser(rdinfo,indx_in_buf,
                  bufnumber,buf,buflen,
		  pars_func,spcl_chars,
		  use_len)

struct	rdinfo	*rdinfo;
int	indx_in_buf;
int	bufnumber;
char	*buf;
int	buflen;
int	(*pars_func)();
char	*spcl_chars;
int	use_len;

{

int	rt_code;
register	char	*pntr;
register	int	i;
register	int	len;
int	blen;

/*begin*/
if (rdinfo != (struct rdinfo *)NULL)
{
	rdbuf = (struct bufsin *)rdinfo->rdbuf;
	bufinfo.bufnumber = bufnumber;
	bufinfo.buf = rdbuf->bufs[bufnumber%MAXNUMBUFRS].pntr;
	bufinfo.bufrmode = NEXT_BUFR;
	bufinfo.buflen = rdbuf->bufs[bufnumber%MAXNUMBUFRS].lngth;
	i = indx_in_buf;
	if (rdbuf->more_reads ||
	    bufnumber < rdbuf->tot_bufs - 1)
		bufinfo.contflag = TRUE;
	else
		bufinfo.contflag = FALSE;
	pntr = &bufinfo.buf[i];
}
else
{
	bufinfo.buf = buf;
	bufinfo.buflen = buflen;
	bufinfo.contflag = FALSE;
	bufinfo.bufnumber = 0;
	i = 0;
	pntr = buf;
}
prev_bufnumber = bufinfo.bufnumber;
start_bufnumber = bufinfo.bufnumber;
if (spcl_chars == NULL)
	use_spcl = FALSE;
else
	use_spcl = TRUE;
is_keyword = FALSE;
start_word = FALSE;
start_prev_indx = -1;
start_indx = -1;
word_count = 0;
prev_len = 0;
len = 0;
running_count = 0;
start_running_count = 0;
prev_running_count = 0;
offset_running_count = i;
for (;;)
{
	if (use_len   &&
	    use_len == running_count + i - offset_running_count)
	{
		if (start_word)
		{
			start_word = FALSE;
			if (is_keyword)
			{
				is_keyword = FALSE;
				rt_code = (*pars_func)(
					start_prev_indx,
					prev_len,
					prev_bufnumber,
					start_indx,
					len,
					start_bufnumber,
					prev_running_count);
				word_count = 0;
			}
		}
		if (word_count)
		{
			rt_code = 0;
			if (word_count == 2)
				rt_code = (*pars_func)(
					 start_prev_indx,
					 prev_len,
					 prev_bufnumber,
					 -1,-1,0,
					 prev_running_count);
			if (!rt_code)
				rt_code = (*pars_func)(
					  start_indx,
					  len,
					  start_bufnumber,
					  -1,-1,0,
					  start_running_count);
		}
		else
		{
			i--;
			start_running_count = running_count + i - 
				offset_running_count;
			rt_code = (*pars_func)(
				  i,
				  -1,
				  bufinfo.bufnumber,
				  -1,-1,0,
				  start_running_count);
		}
		if (rt_code < 0)
			return(rt_code);
		return(0);
	}
	else if (i == bufinfo.buflen)
	{
		/*this is the end of this buffer*/
		if (!bufinfo.contflag)
		{
			if (start_word)
			{
				start_word = FALSE;
				if (is_keyword)
				{
					is_keyword = FALSE;
					rt_code = (*pars_func)(
						start_prev_indx,
						prev_len,
						prev_bufnumber,
						start_indx,
						len,
						start_bufnumber,
						prev_running_count);
					if (rt_code)
					{
						if (rt_code < 0)
							return(rt_code);
						return(0);
					}
					word_count = 0;
				}
			}
			if (word_count)
			{
				rt_code = 0;
				if (word_count == 2)
					rt_code = (*pars_func)(
						 start_prev_indx,
						 prev_len,
						 prev_bufnumber,
						 -1,-1,0,
						 prev_running_count);
				if (!rt_code)
				{
					rt_code = (*pars_func)(
						  start_indx,
						  len,
						  start_bufnumber,
						  -1,-1,0,
						  start_running_count);
				}
			}
			else
			{
				i--;
				start_running_count = running_count + i - 
					offset_running_count;
				rt_code = (*pars_func)(
					  i,
					  -1,
					  bufinfo.bufnumber,
					  -1,-1,0,
					  start_running_count);
			}
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		running_count += i - offset_running_count;
		blen = sfbufman(&bufinfo,rdinfo);
		if (blen <= 0)
			return(-1);
		i = 0;
		offset_running_count = 0; /*start at new place*/
		pntr = bufinfo.buf;
	}
	if (pntr[0] == '=')
	{
		/*this is equals have word etc*/
		if (start_word)
		{
			start_word = FALSE;
			if (word_count == 2)
			{
				rt_code = (*pars_func)(
					 start_prev_indx,
					 prev_len,
					 prev_bufnumber,
					 -1,-1,0,prev_running_count);
				if (rt_code)
				{
					if (rt_code < 0)
						return(rt_code);
					return(0);
				}
				word_count--;
			}
		}
		is_keyword = TRUE;
		i++;
		pntr++;
		/*cases go here*/
		continue;
	}
	is_spcl = FALSE;
	if (use_spcl)
	{
		spcl_i = 0;
		for (;;)
		{
			if (spcl_i  &&
			    spcl_chars[spcl_i] == '\0')
				break;
			if (spcl_chars[spcl_i] == pntr[0])
			{
				is_spcl = TRUE;
				break;
			}
			spcl_i++;
		}
	}
	if (*pntr != ' '   &&
	    *pntr != ','   &&
	    *pntr != ';'   &&
	    !is_spcl       &&
	    !iscntrl(pntr[0]))
	{
		if (start_word)
		{
			pntr++;
			i++;
			len++;
			continue;
		}
		word_count++;
		start_word = TRUE;
		prev_running_count = start_running_count;
		prev_len = len;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		pntr++;
		i++;
		continue;
	}
	/*this is 'white'spaces*/
	if (start_word)
	{
		/*finish up start word here*/
		start_word = FALSE;
		if (!is_keyword)
		{
			if (word_count == 2)
			{
				rt_code = (*pars_func)(
					 start_prev_indx,
					 prev_len,
					 prev_bufnumber,
					 -1,-1,0,prev_running_count);
				if (rt_code)
				{
					if (rt_code < 0)
						return(rt_code);
					return(0);
				}
				word_count--;
			}
			pntr++;
			i++;
			continue;
		}
		word_count = 0;
		is_keyword = FALSE;
		rt_code = (*pars_func)(
			start_prev_indx,
			prev_len,
			prev_bufnumber,
			start_indx,
			len,
			start_bufnumber,
			prev_running_count);
		if (rt_code)
		{
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		word_count = 0;
	}
	if (is_spcl)
	{
		prev_len = len;
		prev_running_count = start_running_count;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		if (word_count)
		{
			rt_code = (*pars_func)(
					 start_prev_indx,
					 prev_len,
					 prev_bufnumber,
					 -1,-1,0,prev_running_count);
			if (rt_code)
			{
				if (rt_code < 0)
					return(rt_code);
				return(0);
			}
		}
		rt_code = (*pars_func)(
					 start_indx,
					 1,
					 start_bufnumber,
					 -1,
					 spcl_i,
					 0,
					 start_running_count);
		if (rt_code)
		{
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		word_count = 0;
		pntr++;
		i++;
		continue;
	}
	i++;
	pntr++;
} /*end of inner for loop*/
}

sfnetckv_parser(rdinfo,indx_in_buf,
                  bufnumber,buf,buflen,
		  pars_func,spcl_chars,
		  use_len)

struct	rdinfo	*rdinfo;
int	indx_in_buf;
int	bufnumber;
char	*buf;
int	buflen;
int	(*pars_func)();
char	*spcl_chars;
int	use_len;

{

int	rt_code;
register	char	*pntr;
register	int	i;
register	int	len;
int	blen;

/*begin*/

if (rdinfo != (struct rdinfo *)NULL)
{
	rdbuf = (struct bufsin *)rdinfo->rdbuf;
	bufinfo.bufnumber = bufnumber;
	bufinfo.buf = rdbuf->bufs[bufnumber%MAXNUMBUFRS].pntr;
	bufinfo.bufrmode = NEXT_BUFR;
	bufinfo.buflen = rdbuf->bufs[bufnumber%MAXNUMBUFRS].lngth;
	i = indx_in_buf;
	if (rdbuf->more_reads ||
	    bufnumber < rdbuf->tot_bufs - 1)
		bufinfo.contflag = TRUE;
	else
		bufinfo.contflag = FALSE;
	pntr = &bufinfo.buf[i];
}
else
{
	bufinfo.buf = buf;
	bufinfo.buflen = buflen;
	bufinfo.contflag = FALSE;
	bufinfo.bufnumber = 0;
	i = 0;
	pntr = buf;
}
prev_bufnumber = bufinfo.bufnumber;
start_bufnumber = bufinfo.bufnumber;
if (spcl_chars == NULL)
	use_spcl = FALSE;
else
	use_spcl = TRUE;
is_keyword = FALSE;
start_word = FALSE;
start_prev_indx = -1;
start_indx = -1;
prev_len = 0;
len = 0;
running_count = 0;
start_running_count = 0;
prev_running_count = 0;
offset_running_count = i;
for (;;)
{
	if (use_len   &&
	    use_len == running_count + i - offset_running_count)
	{
		if (start_word)
		{
			start_word = FALSE;
			if (is_keyword)
			{
				is_keyword = FALSE;
				rt_code = (*pars_func)(
					start_prev_indx,
					prev_len,
					prev_bufnumber,
					start_indx,
					len,
					start_bufnumber,
					prev_running_count);
			}
		}
		else
		{
			i--;
			start_running_count = running_count + i - 
				offset_running_count;
			rt_code = (*pars_func)(
				  i,
				  -1,
				  bufinfo.bufnumber,
				  -1,-1,0,
				  start_running_count);
		}
		if (rt_code < 0)
			return(rt_code);
		return(0);
	}
	else if (i == bufinfo.buflen)
	{
		/*this is the end of this buffer*/
		if (!bufinfo.contflag)
		{
			if (start_word)
			{
				start_word = FALSE;
				if (is_keyword)
				{
					is_keyword = FALSE;
					rt_code = (*pars_func)(
						start_prev_indx,
						prev_len,
						prev_bufnumber,
						start_indx,
						len,
						start_bufnumber,
						prev_running_count);
				}
			}
			else
			{
				i--;
				start_running_count = running_count + i - 
					offset_running_count;
				rt_code = (*pars_func)(
					  i,
					  -1,
					  bufinfo.bufnumber,
					  -1,-1,0,
					  start_running_count);
			}
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		running_count += i - offset_running_count;
		blen = sfbufman(&bufinfo,rdinfo);
		if (blen <= 0)
			return(-1);
		i = 0;
		offset_running_count = 0;
		pntr = bufinfo.buf;
	}
	if (pntr[0] == '=')
	{
		/*this is equals have word etc*/
		if (start_word)
			start_word = FALSE;
		is_keyword = TRUE;
		i++;
		pntr++;
		/*cases go here*/
		continue;
	}
	is_spcl = FALSE;
	if (use_spcl)
	{
		spcl_i = 0;
		for (;;)
		{
			if (spcl_i  &&
			    spcl_chars[spcl_i] == '\0')
				break;
			if (spcl_chars[spcl_i] == pntr[0])
			{
				is_spcl = TRUE;
				break;
			}
			spcl_i++;
		}
	}
	if (*pntr != ' '   &&
	    *pntr != ','   &&
	    *pntr != ';'   &&
	    !is_spcl       &&
	    !iscntrl(pntr[0]))
	{
		if (start_word)
		{
			pntr++;
			i++;
			len++;
			continue;
		}
		start_word = TRUE;
		prev_running_count = start_running_count;
		prev_len = len;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		pntr++;
		i++;
		continue;
	}
	/*this is 'white'spaces*/
	if (start_word)
	{
		/*finish up start word here*/
		start_word = FALSE;
		if (!is_keyword)
		{
			pntr++;
			i++;
			continue;
		}
		is_keyword = FALSE;
		rt_code = (*pars_func)(start_prev_indx,
			prev_len,
			prev_bufnumber,
			start_indx,
			len,
			start_bufnumber,
			prev_running_count);
		if (rt_code)
		{
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
	}
	if (is_spcl)
	{
		prev_len = len;
		prev_running_count = start_running_count;
		start_prev_indx = start_indx;
		prev_bufnumber = start_bufnumber;
		start_bufnumber = bufinfo.bufnumber;
		start_indx = i;
		start_running_count = running_count + i - offset_running_count;
		len = 1;
		rt_code = (*pars_func)(
					 start_indx,
					 1,
					 start_bufnumber,
					 -1,
					 spcl_i,
					 0,
					 start_running_count);
		if (rt_code)
		{
			if (rt_code < 0)
				return(rt_code);
			return(0);
		}
		pntr++;
		i++;
		continue;
	}
	i++;
	pntr++;
} /*end of inner for loop*/
}

char	*
sfndifbin(rdinfo,indx_in_buf,
                  bufnumber,buf,buflen,
		  use_len)

struct	rdinfo	*rdinfo;
int	indx_in_buf;
int	bufnumber;
char	*buf;
int	buflen;
int	use_len;

{

int	rt_code;
register	char	*pntr;
register	int	i;
register	int	len;
int	blen;

/*begin*/

if (rdinfo != (struct rdinfo *)NULL)
{
	rdbuf = (struct bufsin *)rdinfo->rdbuf;
	bufinfo.bufnumber = bufnumber;
	bufinfo.buf = rdbuf->bufs[bufnumber%MAXNUMBUFRS].pntr;
	bufinfo.bufrmode = NEXT_BUFR;
	bufinfo.buflen = rdbuf->bufs[bufnumber%MAXNUMBUFRS].lngth;
	i = indx_in_buf;
	if (rdbuf->more_reads ||
	    bufnumber < rdbuf->tot_bufs - 1)
		bufinfo.contflag = TRUE;
	else
		bufinfo.contflag = FALSE;
	pntr = &bufinfo.buf[i];
}
else
{
	bufinfo.buf = buf;
	bufinfo.buflen = buflen;
	bufinfo.contflag = FALSE;
	bufinfo.bufnumber = 0;
	i = 0;
	pntr = buf;
}
len = 0;
running_count = 0;
start_running_count = 0;
prev_running_count = 0;
offset_running_count = i;
prev_bufnumber = bufinfo.bufnumber;
start_bufnumber = bufinfo.bufnumber;
for (;;)
{
	if (use_len   &&
	    use_len == running_count + i - offset_running_count)
	{
		rt_code = 0;
		return(NULL);
	}
	else if (i == bufinfo.buflen)
	{
		/*this is the end of this buffer*/
		if (!bufinfo.contflag)
		{
			return(NULL);
		}
		running_count += i - offset_running_count;
		blen = sfbufman(&bufinfo,rdinfo);
		if (blen <= 0)
			return(NULL);
		i = 0;
		offset_running_count = 0; /*start at new place*/
		pntr = bufinfo.buf;
	}
	/*check for non ascii character*/
	if (iscntrl(pntr[0]))
	{
		if (pntr[0] != '\n'   &&
		    pntr[0] != CR     &&
		    pntr[0] != '^'    &&
	   	    pntr[0] != '\t')
		{
			return(pntr); /*there it is */
		}
	}
	i++;
	pntr++;
} /*end of inner for loop*/
}
