/*
         1         2         3         4         5         6         7
123456789012345678901234567890123456789012345678901234567890123456789012
*/
#ifndef Header
/*****************************************************************
 * TITLE: diowlt2dp.c
 *
 * AUTHOR:  Unknown
 *          Aug 31, 1994
 *
 * MODIFIED:    Ray Bambery
 *          Aug 24, 2020 -  removed label after #ENDIF Header 
 ******************************************************************
** MANUAL
**	DIOWLT2DP 3x, "December 7,1993"
**
** NAME
**	diowlt2dp - create OWLT double precision file.
**
** SYNOPSIS
**	
**	  void diowlt2dp(owlt_txt,owlt,
**	                epoch,msn,rtcode);
**
**	  char  *owlt_txt;
*	  char  *owlt;
*	  char	*msn;
**	  char  *epoch
**	  char	*rtcode;
**
** PARAMETERS
**
**	  1. owlt_txt - name of OWLT text file to convert.
**	  2. owlt - name of ouput converted OWLT file to double
**	     precision.
**	  5. msn - 3 character mission id 
**	     1. ULS - Ulysses.
**	     2. GLL - Galileo.
**	  6. epoch - double precision seconds past epoch
**	     epoch can be '1950' '2000' or users choice.
**	  7. rtcode - 4 character string for error control.
**	     rtcode = 'NORM' no error.
**
** DESCRIPTION
**
**	'diowlt2dp' converts an OWLT  file from text to a more usefull
**	 double precision file.
**
**	Conversion of OWLT text file to a double precision OWLT file.
**
**	Example 3 shows an OWLT input file.
**
**	An OWLT ASCII file is input and is to be converted to an eight (8)
**	byte double precision numbers.
**
**	The output file 'owlt' is an SFDU file. Example
**	  CCSD3ZF0000100000001NJPL3KS0L015$$owlt$$
**	  MISSION_NAME=ULYSSES;
**	  SPACECRAFT_NAME=ULS;
**	  DATA_SET_ID=DOUBLE_PRECISION_OWLT;
**	  FILE_NAME=uls.dp1owlt.data;
**	  PRODUCT_CREATION_TIME=1990-285T14:49:10;
**	  PRODUCT_VERSION_ID=1;
**	  PRODUCER_ID=DMT;
**	  APPLICABLE_START_TIME=1990-285T14:49:10;
**	  PPLICABLE_STOP_TIME=1999-12-30T12:00:00.000;
**	  MISSION_ID=3;
**	  SPACECRAFT_ID=55;
**	  TARGET_PARAMETER_NAME=SCET_AND_ERT;
**	  TARGET_PARAMETER_EPOCH=1950-001T00:00:00.000;
**	  CCSD$$MARKER$$owlt$$NJPL3IFxxxx00000001(folowed by the
**	  double precision OWLT data)
**
**	for GLL the mission name is 'GALILEO' and msn id is 1.
**	
**	The input data to be converted is 
**	  1. SCET - Character time 'YY-DOY/HH:MM:SS.FFF'
**	  2. Down leg time - 10 bytes xxxxxx.fff.
**	  3. Up leg time - 10 bytes xxxxxx.fff
**	  4. Sation# 2 bytes.
**	  5. S/C id 1 character.
**	The rest of the data is ignored.
**
**	The output is 
**	  1. SCET - double precision seconds past epoch.
**	  2. Down Leg time - double precisions seconds.
**	  3. Up leg time - double precision seconds.
**	  4. ERT - double precision ERT.
**	  5. A1 - coefficient for finding ERT given SCET
**	     or for finding SCET given ERT.
**	  6. Station number - int.
**	  7. S/C id 1 character.
**	
**	ERT is equal to SCET + Down leg time.
**	A1 is (NEW_ERT - LAST_ERT)/(NEW_SCET - LAST_SCET)
**	or the slope M.
**
**	Given a SCET find ERT
**	   ERT = ERT_OWLT + A1*(SCET - SCET_OWLT)
**	or the convers
**	  SCET = SCET_OWLT + (ERT - ERT_OWLT)/A1 
**	where SCET_OWLT and ERT_OWLT are found from the file 'owlt'.
**
** EXAMPLE


EXAMPLE 1.

EXAMPLE

$$GLL       LIGHT TIME FILE                                                    1 
*LITIME     OWLTG(  )                                                          2 
*PREP                                                                          3 
*TITLE      GI1013 TRAJECTORY, LAUNCH TO LAUNCH + 1 YEAR                       4 
*SCID       G                                                                  5 
*RUNID      ENTER LITIME  2-MAR-1989 08:23:31 LINKED 16-SEP-1988 15:1          6 
*CREATION   JPL 89-061/08:23:31                                                7 
*BEGIN      SCE 89-286/00:00:56.300  ERT 89-286/00:01:38.505                   8 
*CUTOFF     SCE 90-288/00:00:00.000                                            9 
*PFILE                                                                        10 
'     SCE                     DOWN-LEG        UP-LEG         RUN TIME  S      11 
$$EOS                                                                         12 
89-286/00:00:56                    .103           .103   3 890302082331G      13 
*/
#endif /*  Header	 */

#if (IBM_MAIN_FRAME)
/*place pragma to allow for entry points for JTPM*/
#pragma csect(CODE,"DCOWLT20") csect(STATIC,"DCOWLT21")
#endif

#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<glbltme.h>
#include	<glowlsce.h>
#include	<dictmerr.h>
#include	<diowlclk.h>
#include	<clkowlti.h>
	
void 
diowlt2dp(owlt_txt,owlt,
	 epoch,msn_in,rtcode)
	 
char	*owlt_txt;
char	*owlt;
char	*epoch;
char	*msn_in;
char	*rtcode;

{

int	rtrn;
char	*pntr;

strcpy(rtcode,"NORM");
if (strcmp(times_data_init,"YES"))
{
	ditminit();
}
if (strcmp(vax_ordr_called,"YES"))
{
	/*check on order of integters (VAX VMS< etc vs SUN, MACINTOSH, 
	  IBM main frame*/
	strcpy(vax_ordr_called,"YES");
	rtrn = mchintst(NULL,NULL);
	if (rtrn)
	{
		/*this is a VAX order reorder array*/
		vax4_or_othr[0] = 3;
		vax4_or_othr[1] = 2;
		vax4_or_othr[2] = 1;
		vax4_or_othr[3] = 0;
		vax2_or_othr[0] = 1;
		vax2_or_othr[1] = 0;
	}
}


/*find epoch and choice of epoch*/
if (epoch == NULL)
{
	/*epoch is '1950' by default*/
	CLOWEPS.epoch_indx = DP50_INDX;
	strcpy(CLOWEPS.epoch,"1950-001T00:00:00");
}
else
{
	while (epoch[0] == ' ')
	{
		epoch++;
	}
	if (!strlen(epoch))
	{
		/*epoch is '1950' by default*/
		CLOWEPS.epoch_indx = DP50_INDX;
		strcpy(CLOWEPS.epoch,"1950-001T00:00:00");
	}
	else if (!strncmp(epoch,"1950",4))
	{
		CLOWEPS.epoch_indx = DP50_INDX;
		strcpy(CLOWEPS.epoch,"1950-001T00:00:00");
	}
	else if (!strncmp(epoch,"2000",4))
	{
		CLOWEPS.epoch_indx = DP2000_INDX;
		strcpy(CLOWEPS.epoch,"2000-001T12:00:00");
	}
	else
	{
		g6epoch(epoch,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
		CLOWEPS.epoch_indx = DPEP_OF_CHOICE;
	}
}
/*epoch is here*/
/*msn here*/
switch(msn_in[0])
{
	case 'G': /*galileo*/
	case 'g':
		/*this is galileo here*/
		msn_case_nmbr = GLL_INDX;
	break;
	case 'U': /*Ulysses*/
	case 'u':
		msn_case_nmbr = ULS_INDX;
	break;
	case 'V': /*voyager*/
	case 'v':
		msn_case_nmbr = VGR_INDX;
	break;
	default:
		msn_case_nmbr = UNK_INDX; /*unknown*/
	break;
}
/*ok now do it*/
if (strlen(owlt_txt))
{
	/*ok we have it here*/
	crowltfile(owlt_txt,owlt,rtcode);
}
return;
}

static
g6epoch(epoch,rtcode)

char	*epoch;
char	*rtcode;

{

char	*pntr;

/*set 'type_in' as 'EPOCH CHAR"*/
strcpy(type_in,"EPOCH CHAR");
strcpy(type_out,"CCSDSDOY");
pntr = epoch;
while (pntr[0] == ' ')
{
	pntr++;
}
/*get new epoch*/
dictmcnv(tarray,NULL,NULL,dptimes,pntr,type_in,type_out,CLOWEPS.epoch,
	rtcode);
/*************************
          1         2
0123456789012345678901234
YYYY-DOYTHH:MM:SS.FFF change to
YYYY-DOYTHH:MM:SS
****************************/
return;
}
 
static
crowltfile(owlt_txt,owlt,rtcode)

char	*owlt_txt;
char	*owlt;
char	*rtcode;

{

strcpy(owlt_file,owlt_txt);
strcpy(dpowlt_file,owlt);
/*open files for owlt in and out*/
SFFOPEN(fdowlt,owlt_txt,SFRDONLY);
if (SFFERROPEN(fdowlt))
{
	strcpy(rtcode,err_rtrn_code[OWI_INDX]);
	return;
}
SFFOPEN(fdowlt_out,owlt,SFWRTONLY);
if (SFFERROPEN(fdowlt_out))
{
	strcpy(rtcode,err_rtrn_code[OWO_INDX]);
	return;
}
/*read in and create header*/
dihdrowl(rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
/*set to start of records here*/
SFFSEEK(fdowlt,owloffset,SFSEEK_SET);
/*write out records*/
wrtowltrecs(rtcode);
return;
}

static
wrtowltrecs(rtcode)

char	*rtcode;

{

BOOL	compute_a1;
int	read;
int	totwrt;
BOOL	start_owlt;

/*************************************************************
**	The input data to be converted is 
**	  1. SCET - Character time 'YY-DOY/HH:MM:SS.FFF'
**	  2. Down leg time - 10 bytes xxxxxx.fff.
**	  3. Up leg time - 10 bytes xxxxxx.fff
**	The rest of the data is ignored.
**
**	The output is 
**	  1. SCET - double precision seconds past epoch.
**	  2. Down Leg time - double precisions seconds.
**	  3. Up leg time - double precision seconds.
**	  4. ERT - double precision ERT.
**	  5. A1 - coefficient for finding ERT given SCET
**	     or for finding SCET given ERT.
**	  6. station# integer.
**	  7. S/C id 1 character.
**	
**	ERT is equal to SCET + Down leg time.
**	A1 is (NEW_ERT - LAST_ERT)/(NEW_SCET - LAST_SCET)
**	or the slope M.
**********************************************************/
SOWLTREC = (OWLTREC *)strng;
start_owlt = TRUE;
compute_a1 = FALSE;
owlt_rec_lngth = sizeof(OWLTREC);
/*do until end*/
while(TRUE)
{
	read_owlt(owlrecsize,&read,rtcode);
	if (rtcode[3] == 'X')
	{
		return;
	}
	if (!strncmp(hdr_rec,"$$EOF",5))
	{
		/*is eof finished*/
		return;
	}
	/*compute stuff*/
	/*get scet*/
	memcpy(tstrng,hdr_rec,20);
	tstrng[20] = '\0';
	if (start_owlt)
	{
		start_owlt = FALSE;
		strcpy(stowlt,tstrng);
	}
	strcpy(enowlt,tstrng);
	strcpy(type_in,"CHAR");
	dictmcnv(tarray,mon,dow,dptimes,tstrng,type_in,NULL,NULL,
        	 rtcode);
        if (rtcode[3] == 'X')
        {
        	/*error*/
        	return;
        }
	scet = dptimes[CLOWEPS.epoch_indx];
	/*compute down leg*/
	strncpy(tstrng,&hdr_rec[29],10);
	tstrng[10] = '\0';
	cnv2dbl(tstrng,&down_leg);
	/*compute up leg*/
	strncpy(tstrng,&hdr_rec[44],10);
	tstrng[10] = '\0';
	cnv2dbl(tstrng,&up_leg);
	/*compute ert*/
	ert = scet + down_leg;
	memcpy(tstrng,&hdr_rec[56],2);
	tstrng[2] = '\0';
	sscanf(tstrng,"%d",&station);
	if (compute_a1)
	{
		SOWLTREC->EA1 = ((DBL)ert - SOWLTREC->eu.ert)/
		               ((DBL)scet - SOWLTREC->su.scet);
		/*write out record*/
		write_owlt(owlt_rec_lngth,&totwrt,strng,rtcode);
		if (rtcode[3] == 'X')
		{
			return;
		}
	}
	else
	{
		compute_a1 = TRUE;
	}
	SOWLTREC->su.scet = scet;
	SOWLTREC->down_leg = down_leg;
	SOWLTREC->up_leg = up_leg;
	SOWLTREC->eu.ert = ert;
	SOWLTREC->station = station;
}
}	
	
static
read_owlt(toread,read,rtcode)

int	toread;
int	*read;
char	*rtcode;

{

read[0] = SFFREAD(fdowlt,hdr_rec,toread);
if (read[0] !=  toread)
{
	strcpy(rtcode,err_rtrn_code[OWR_INDX]);
}
return;
}

static	
write_owlt(towrt,totwrt,wrt_buf,rtcode)

int	towrt;
int	*totwrt;
char	*wrt_buf;
char	*rtcode;

{
totwrt[0] = SFFWRITE(fdowlt_out,wrt_buf,towrt);
if (totwrt[0] != towrt)
{
	/*error in writting owlt file*/
	strcpy(rtcode,err_rtrn_code[OWW_INDX]);
}
return;
}

static
cnv2dbl(input,wiz)

char	*input;
DBL	*wiz;

{

char	*pntr;
char	*pntr1;
char	*pntr2;
BOOL	is_decimal;
int	i;
long	int	dev;
int	tot_dec;
long	int	wiz1;

pntr = input;
while(pntr[0] == ' ')
{
	pntr++;
}
is_decimal = FALSE;
pntr1 = strchr(pntr,'.');
if (pntr1 != NULL)
{
	is_decimal = TRUE;
	
	pntr2 = pntr1;
	pntr1++;
	pntr2[0] = '\0';
	pntr2 = strchr(pntr1,' ');
	if (pntr2 != NULL)
	{
		pntr2[0] = '\0';
	}
	tot_dec = strlen(pntr1);
	dev = 1;
	for ( i = 0 ; i < tot_dec ; i++)
	{
		dev *= 10;
	}
}
if (strlen(pntr))
{
	sscanf(pntr,"%ld",&wiz1);
	wiz[0] = (DBL)wiz1;
}
else
{
	wiz[0] = 0;
}
if (is_decimal)
{
	sscanf(pntr1,"%ld",&wiz1);
	wiz[0] += (DBL)wiz1/(DBL)dev;
}
return;
}
