/*
         1         2         3         4         5         6         7
123456789012345678901234567890123456789012345678901234567890123456789012
*/
#ifndef Header
/*
** MANUAL
**	DIOWLCLK 3x, "April 15,1993"
**
** NAME
**	diowlclk - create OWLT and/or SCLK-SCET double precision files.
**
** SYNOPSIS
**	
**	  void diowlclk(owlt_txt,owlt,
**	                sceclk_txt,sceclk,
**	                epoch,msn,rtcode);
**
**	  char  *owlt_txt;
**	  int   *owlt_rec_size;
**	  char  *owlt;
**	  char  *sceclk_txt;
**	  int   *sceclk_rec_size;
**	  char	*sceclk;
**	  char	*msn;
**	  char  *epoch
**	  char	*rtcode;
**
** PARAMETERS
**
**	  1. owlt_txt - name of OWLT text file to convert.
**	  2. owlt - name of ouput converted OWLT file to double
**	     precision.
**	  3. sceclk_txt - name of SCET/SCLK text file to convert.
**	  4. sceclk - name of ouput SCET/SCLK file, converted to
**	     double precision.
**	  5. msn - 3 character mission id 
**	     1. ULS - Ulysses.
**	     2. GLL - Galileo.
**	  6. epoch - double precision seconds past epoch
**	     epoch can be '1950' '2000' or users choice.
**	  7. rtcode - 4 character string for error control.
**	     rtcode = 'NORM' no error.
**
** DESCRIPTION
**
**	'diowlclk' converts an OWLT and/or an SCET/SCLK file from
**	text to a more usefull double precision file.
**
**	Conversion of OWLT text file to a double precision OWLT file.
**
**	Example 3 shows an OWLT input file.
**
**	If 'owlt_txt' is not empty or NULL then an OWLT text
**	file is input and is to be converted to 8 byte double
**	precision numbers.
**
**	The output file 'owlt' is an SFDU file. Example
**	  CCSD3ZF0000100000001NJPL3KS0L015$$owlt$$
**	  MISSION_NAME=ULYSSES;
**	  SPACECRAFT_NAME=ULS;
**	  DATA_SET_ID=DOUBLE_PRECISION_OWLT;
**	  FILE_NAME=uls.dp1owlt.data;
**	  PRODUCT_CREATION_TIME=1990-285T14:49:10;
**	  PRODUCT_VERSION_ID=1;
**	  PRODUCER_ID=DMT;
**	  APPLICABLE_START_TIME=1990-285T14:49:10;
**	  PPLICABLE_STOP_TIME=1999-12-30T12:00:00.000;
**	  MISSION_ID=3;
**	  SPACECRAFT_ID=55;
**	  TARGET_PARAMETER_NAME=SCET_AND_ERT;
**	  TARGET_PARAMETER_EPOCH=1950-001T00:00:00.000;
**	  CCSD$$MARKER$$owlt$$NJPL3IFxxxx00000001(folowed by the
**	  double precision OWLT data)
**
**	for GLL the mission name is 'GALILEO' and msn id is 1.
**	
**	The input data to be converted is 
**	  1. SCET - Character time 'YY-DOY/HH:MM:SS.FFF'
**	  2. Down leg time - 10 bytes xxxxxx.fff.
**	  3. Up leg time - 10 bytes xxxxxx.fff
**	  4. Sation# 2 bytes.
**	  5. S/C id 1 character.
**	The rest of the data is ignored.
**
**	The output is 
**	  1. SCET - double precision seconds past epoch.
**	  2. Down Leg time - double precisions seconds.
**	  3. Up leg time - double precision seconds.
**	  4. ERT - double precision ERT.
**	  5. A1 - coefficient for finding ERT given SCET
**	     or for finding SCET given ERT.
**	  6. Station number - int.
**	  7. S/C id 1 character.
**	
**	ERT is equal to SCET + Down leg time.
**	A1 is (NEW_ERT - LAST_ERT)/(NEW_SCET - LAST_SCET)
**	or the slope M.
**
**	Given a SCET find ERT
**	   ERT = ERT_OWLT + A1*(SCET - SCET_OWLT)
**	or the convers
**	  SCET = SCET_OWLT + (ERT - ERT_OWLT)/A1 
**	where SCET_OWLT and ERT_OWLT are found from the file 'owlt'.
**
**	SCLK_SCET files.
**
**	The spacecraft clock file has 2 formats. The first is the
**	older format from the UNYSIS system. See example 1.
**	The second is the AMMOS format. See example 2.
**
**	The output 'scetclk' file is an SFDU file. Example
**	  CCSD3ZF0000100000001NJPL3KS0L015$$scet$$
**	  MISSION_NAME=ULYSSES;
**	  SPACECRAFT_NAME=ULS;
**	  DATA_SET_ID=DOUBLE_PRECISION_SCLK_SCET;
**	  FILE_NAME=uls.dp1coeff.data;
**	  PRODUCT_CREATION_TIME=1990-285T14:49:10;
**	  PRODUCT_VERSION_ID=1;
**	  PRODUCER_ID=DMT;
**	  APPLICABLE_START_TIME=1990-285T14:49:10;
**	  PPLICABLE_STOP_TIME=1999-12-30T12:00:00.000;
**	  MISSION_ID=3;
**	  SPACECRAFT_ID=55;
**	  TARGET_PARAMETER_NAME=SCET_AND_ERT;
**	  TARGET_PARAMETER_EPOCH=1950-001T00:00:00.000;
**	  CCSD$$MARKER$$scet$$NJPL3IFxxxx00000001(folowed by the
**	  double precision scet coefficient data)
**
**	for GLL the mission name is 'GALILEO' and msn id is 1.
**
**	The Double precision output is
**	 SCLK,SCET,A1,DUT
**
** EXAMPLE


EXAMPLE 1.

	Galileo SCLK/GLL coefficient file (UNYSIS)

$$GLL       SCLK/SCET COEFFICIENT FILE                                         1 
*SCID       ORBITER                                                            2 
*FILE       DMT*SCLKSCETCOEF(0).                                               3 
*CREATION   88-152/10:16:17                                                    4 
*UPDATE     88-152/10:16:27       20                                           5 
*RATE       60.666608164                                                       6 
*PART 1     85-226/16:57:19.184 @      100                                     7 
*                                                                             50 
*           SCET(UTC) = A0 + A1*(SCLK -SCLK0)                                 51 
*                                                                             52 
*    SCLK0             A0            DUT       A0           ENTRY       ENTRY 53 
* EFFECT. SCLK  EFFECT. SCET(UTC)  ET-UTC EFFECT. RATE      EPOCH        SEQ  54 
*RIM:MOD91.RTI YY-DDD/HH:MM:SS.FFF  SECS   SECONDS/RIM    LOCAL TIME     NUM  55 
$$EOH                                                                         99 
        0:00:0 85-226/16:57:19.184 58.200 60.666608164 88-152/10:16:17         1 


	Ulyses SCLK/GLL coefficient file (UNYSIS)
	
$$ULS       SCLK/SCET COEFFICIENT FILE                                         1
*SCID       ULS                                                                2
*FILE       UDMT*SCLKSCETCOEF.                                                 3
*CREATION   90-285/14:49:10                                                    4
*UPDATE     93-106/10:44:39                                                    5
*RATE        2.000030997                                                       6
*PART 1     90-279/19:10:49.727 @      100                                     7
*                                                                             50
*           SCET(UTC) = A0 + A1*(SCLK - SCLK0)                                51
*                                                                             52
*    SCLK0             A0            DUT       A1           ENTRY       ENTRY 53
* EFFECT. SCLK  EFFECT. SCET(UTC)  ET-UTC EFFECT. RATE      EPOCH        SEQ  54
*SCLKCOUNT.000 YY-DDD/HH:MM:SS.FFF  SECS   SECONDS/RIM    LOCAL TIME     NUM  55
$$EOH                                                                         99
         0.000 90-279/19:10:49.727 57.184  1.994697308 93-026/11:31:22       100

EXAMPLE 2.
CCSD3ZS00001$$sclk$$NJPL3KS0L015$$scet$$
MISSION_NAME=ULYSSES;
SPACECRAFT_NAME=ULS;
DATA_SET_ID=SCLK_SCET;
FILE_NAME=SCLK_SCET.00001;
PRODUCT_CREATION_TIME=1990-285T14:49:10;
PRODUCT_VERSION_ID=1;
PRODUCER_ID=DMT;
APPLICABLE_START_TIME=1990-285T14:49:10;
APPLICABLE_STOP_TIME=1999-12-30T12:00:00.000;
MISSION_ID=3;
SPACECRAFT_ID=55;
CCSD3RE00000$$scet$$NJPL3IS00613$$data$$
*____SCLK0_______  ____SCET0____________ _DUT__ _SCLKRATE___ 
          0.000.0  1990-279T19:10:49.727 57.184 1.994697308
       1000.000.0  1990-279T19:44:04.424 57.184 2.000052742
 
EXAMPLE 3

$$GLL       LIGHT TIME FILE                                                    1 
*LITIME     OWLTG(  )                                                          2 
*PREP                                                                          3 
*TITLE      GI1013 TRAJECTORY, LAUNCH TO LAUNCH + 1 YEAR                       4 
*SCID       G                                                                  5 
*RUNID      ENTER LITIME  2-MAR-1989 08:23:31 LINKED 16-SEP-1988 15:1          6 
*CREATION   JPL 89-061/08:23:31                                                7 
*BEGIN      SCE 89-286/00:00:56.300  ERT 89-286/00:01:38.505                   8 
*CUTOFF     SCE 90-288/00:00:00.000                                            9 
*PFILE                                                                        10 
'     SCE                     DOWN-LEG        UP-LEG         RUN TIME  S      11 
$$EOS                                                                         12 
89-286/00:00:56                    .103           .103   3 890302082331G      13 
*/
#endif Header	
#if (IBM_MAIN_FRAME)
/*place pragma to allow for entry points for JTPM*/
#pragma csect(CODE,"DCOWLCL0") csect(STATIC,"DCOWLCL1")
#endif

#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<glbltme.h>
#include	<glowlsce.h>
#include	<dictmerr.h>
#include	<diowlclk.h>
#include	<clkowlti.h>
	
void 
diowlclk(owlt_txt,owlt,
	 sceclk_txt,sceclk,
	 epoch,msn_in,rtcode)
	 
char	*owlt_txt;
char	*owlt;
char	*sceclk_txt;
char	*sceclk;
char	*epoch;
char	*msn_in;
char	*rtcode;

{

int	rtrn;
char	*pntr;

strcpy(rtcode,"NORM");
if (strcmp(times_data_init,"YES"))
{
	ditminit();
}
if (strcmp(vax_ordr_called,"YES"))
{
	/*check on order of integters (VAX VMS< etc vs SUN, MACINTOSH, 
	  IBM main frame*/
	strcpy(vax_ordr_called,"YES");
	rtrn = mchintst(NULL,NULL);
	if (rtrn)
	{
		/*this is a VAX order reorder array*/
		vax4_or_othr[0] = 3;
		vax4_or_othr[1] = 2;
		vax4_or_othr[2] = 1;
		vax4_or_othr[3] = 0;
		vax2_or_othr[0] = 1;
		vax2_or_othr[1] = 0;
	}
}


/*find epoch and choice of epoch*/
if (epoch == NULL)
{
	/*epoch is '1950' by default*/
	CLOWEPS.epoch_indx = DP50_INDX;
	strcpy(CLOWEPS.epoch,"1950-001T00:00:00");
}
else
{
	while (epoch[0] == ' ')
	{
		epoch++;
	}
	if (!strlen(epoch))
	{
		/*epoch is '1950' by default*/
		CLOWEPS.epoch_indx = DP50_INDX;
		strcpy(CLOWEPS.epoch,"1950-001T00:00:00");
	}
	else if (!strncmp(epoch,"1950",4))
	{
		CLOWEPS.epoch_indx = DP50_INDX;
		strcpy(CLOWEPS.epoch,"1950-001T00:00:00");
	}
	else if (!strncmp(epoch,"2000",4))
	{
		CLOWEPS.epoch_indx = DP2000_INDX;
		strcpy(CLOWEPS.epoch,"2000-001T12:00:00");
	}
	else
	{
		g6epoch(epoch,rtcode);
		if (rtcode[3] == 'X')
		{
			/*error*/
			return;
		}
		CLOWEPS.epoch_indx = DPEP_OF_CHOICE;
	}
}
/*epoch is here*/
/*msn here*/
switch(msn_in[0])
{
	case 'G': /*galileo*/
	case 'g':
		/*this is galileo here*/
		msn_case_nmbr = GLL_INDX;
	break;
	case 'U': /*Ulysses*/
	case 'u':
		msn_case_nmbr = ULS_INDX;
	break;
	case 'V': /*voyager*/
	case 'v':
		msn_case_nmbr = VGR_INDX;
	break;
	default:
		msn_case_nmbr = UNK_INDX; /*unknown*/
	break;
}
/*ok now do it*/
if (owlt_txt != NULL)
{
	if (strlen(owlt_txt))
	{
		/*ok we have it here*/
		crowltfile(owlt_txt,owlt,rtcode);
	}
}
if (sceclk_txt != NULL)
{
	if (strlen(sceclk_txt))
	{
		crsceclkfile(sceclk_txt,sceclk,msn_in,rtcode);
	}
}
return;
}

static
g6epoch(epoch,rtcode)

char	*epoch;
char	*rtcode;

{

char	*pntr;

/*set 'type_in' as 'EPOCH CHAR"*/
strcpy(type_in,"EPOCH CHAR");
strcpy(type_out,"CCSDSDOY");
pntr = epoch;
while (pntr[0] == ' ')
{
	pntr++;
}
/*get new epoch*/
dictmcnv(tarray,NULL,NULL,dptimes,pntr,type_in,type_out,CLOWEPS.epoch,
	rtcode);
/*************************
          1         2
0123456789012345678901234
YYYY-DOYTHH:MM:SS.FFF change to
YYYY-DOYTHH:MM:SS
****************************/
return;
}
 
static
crowltfile(owlt_txt,owlt,rtcode)

char	*owlt_txt;
char	*owlt;
char	*rtcode;

{

strcpy(owlt_file,owlt_txt);
strcpy(dpowlt_file,owlt);
/*open files for owlt in and out*/
SFFOPEN(fdowlt,owlt_txt,SFRDONLY);
if (SFFERROPEN(fdowlt))
{
	strcpy(rtcode,err_rtrn_code[OWI_INDX]);
	return;
}
SFFOPEN(fdowlt_out,owlt,SFWRTONLY);
if (SFFERROPEN(fdowlt_out))
{
	strcpy(rtcode,err_rtrn_code[OWO_INDX]);
	return;
}
/*read in and create header*/
dihdrowl(rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
/*set to start of records here*/
SFFSEEK(fdowlt,owloffset,SFSEEK_SET);
/*write out records*/
wrtowltrecs(rtcode);
return;
}

static
wrtowltrecs(rtcode)

char	*rtcode;

{

BOOL	compute_a1;
int	read;
int	totwrt;
BOOL	start_owlt;

/*************************************************************
**	The input data to be converted is 
**	  1. SCET - Character time 'YY-DOY/HH:MM:SS.FFF'
**	  2. Down leg time - 10 bytes xxxxxx.fff.
**	  3. Up leg time - 10 bytes xxxxxx.fff
**	The rest of the data is ignored.
**
**	The output is 
**	  1. SCET - double precision seconds past epoch.
**	  2. Down Leg time - double precisions seconds.
**	  3. Up leg time - double precision seconds.
**	  4. ERT - double precision ERT.
**	  5. A1 - coefficient for finding ERT given SCET
**	     or for finding SCET given ERT.
**	  6. station# integer.
**	  7. S/C id 1 character.
**	
**	ERT is equal to SCET + Down leg time.
**	A1 is (NEW_ERT - LAST_ERT)/(NEW_SCET - LAST_SCET)
**	or the slope M.
**********************************************************/
SOWLTREC = (OWLTREC *)strng;
start_owlt = TRUE;
compute_a1 = FALSE;
owlt_rec_lngth = sizeof(OWLTREC);
/*do until end*/
while(TRUE)
{
	read_owlt(owlrecsize,&read,rtcode);
	if (rtcode[3] == 'X')
	{
		return;
	}
	if (!strncmp(hdr_rec,"$$EOF",5))
	{
		/*is eof finished*/
		return;
	}
	/*compute stuff*/
	/*get scet*/
	memcpy(tstrng,hdr_rec,20);
	tstrng[20] = '\0';
	if (start_owlt)
	{
		start_owlt = FALSE;
		strcpy(stowlt,tstrng);
	}
	strcpy(enowlt,tstrng);
	strcpy(type_in,"CHAR");
	dictmcnv(tarray,mon,dow,dptimes,tstrng,type_in,NULL,NULL,
        	 rtcode);
        if (rtcode[3] == 'X')
        {
        	/*error*/
        	return;
        }
	scet = dptimes[CLOWEPS.epoch_indx];
	/*compute down leg*/
	strncpy(tstrng,&hdr_rec[29],10);
	tstrng[10] = '\0';
	cnv2dbl(tstrng,&down_leg);
	/*compute up leg*/
	strncpy(tstrng,&hdr_rec[44],10);
	tstrng[10] = '\0';
	cnv2dbl(tstrng,&up_leg);
	/*compute ert*/
	ert = scet + down_leg;
	memcpy(tstrng,&hdr_rec[56],2);
	tstrng[2] = '\0';
	sscanf(tstrng,"%d",&station);
	if (compute_a1)
	{
		SOWLTREC->EA1 = ((DBL)ert - SOWLTREC->eu.ert)/
		               ((DBL)scet - SOWLTREC->su.scet);
		/*write out record*/
		write_owlt(owlt_rec_lngth,&totwrt,strng,rtcode);
		if (rtcode[3] == 'X')
		{
			return;
		}
	}
	else
	{
		compute_a1 = TRUE;
	}
	SOWLTREC->su.scet = scet;
	SOWLTREC->down_leg = down_leg;
	SOWLTREC->up_leg = up_leg;
	SOWLTREC->eu.ert = ert;
	SOWLTREC->station = station;
}
}	
	
static
read_owlt(toread,read,rtcode)

int	toread;
int	*read;
char	*rtcode;

{

read[0] = SFFREAD(fdowlt,hdr_rec,toread);
if (read[0] !=  toread)
{
	strcpy(rtcode,err_rtrn_code[OWR_INDX]);
}
return;
}

static	
write_owlt(towrt,totwrt,wrt_buf,rtcode)

int	towrt;
int	*totwrt;
char	*wrt_buf;
char	*rtcode;

{
totwrt[0] = SFFWRITE(fdowlt_out,wrt_buf,towrt);
if (totwrt[0] != towrt)
{
	/*error in writting owlt file*/
	strcpy(rtcode,err_rtrn_code[OWW_INDX]);
}
return;
}
 
static
crsceclkfile(sceclk_txt,sceclk,msn_in,rtcode)

char	*sceclk_txt;
char	*sceclk;
char	*msn_in;
char	*rtcode;

{

/*open files for sceclk in and out*/
SFFOPEN(fdsceclk,sceclk_txt,SFRDONLY);
if (SFFERROPEN(fdsceclk))
{
	strcpy(rtcode,err_rtrn_code[CLI_INDX]);
	return;
}
SFFOPEN(fdsccl_out,sceclk,SFWRTONLY);
if (SFFERROPEN(fdsccl_out))
{
	strcpy(rtcode,err_rtrn_code[CLO_INDX]);
	return;
}
strcpy(sceclk_file,sceclk_txt);
strcpy(dpsceclk_file,sceclk);
/*find mission id*/
diclkcnv(NULL,msn_in,NULL,NULL,NULL,NULL,NULL,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
/*read in and create header*/
dihdrclk(rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
/*set to start of records here*/
SFFSEEK(fdsceclk,clkoffset,SFSEEK_SET);
/*write out records*/
wrtsceclkrecs(rtcode);
return;
}
static
wrtsceclkrecs(rtcode)

char	*rtcode;

{

int	read;
int	totwrt;
BOOL	start_clock;

/*************************************************************
          1         2         3         4         5         6
01234567890123456789012345678901234567890123456789012345678901234567890
        0:00:0 85-226/16:57:19.184 58.200 60.666608164 88-152/10:16:17         1 
         0.000 90-279/19:10:49.727 57.184  1.994697308 93-026/11:31:22       100
**	The input data to be converted is 
**	  1. Effective clock rate (mission dependent) 
**	  2. SCET YY-DOY/HH:MM:SS
**	  3.
**********************************************************/
sceclk_rec_lngth = sizeof(SCLKR);
RSCLK = (SCLKR *)strng;
start_clock = TRUE;
/*do until end*/
while(TRUE)
{
	read_sceclk(clkrecsize,&read,rtcode);
	if (rtcode[3] == 'X')
	{
		/*find ending stuff*/
		if (!strncmp(hdr_rec,"CCSD",4))
		{
			/*is eof*/
			strcpy(rtcode,"NORM");
		}
		return;
	}
	if (use_unisys)
	{
		if (!strncmp(hdr_rec,"$$EOF",5))
		{
			/*is eof finished*/
			return;
		}
	}
	else
	{
		/*this is from AMMOS*/
		/*find ending stuff*/
		if (!strcmp(hdr_rec,"CCSD"))
		{
			/*is eof*/
			return;
		}
	}
	/*compute stuff*/
	/*get scet*/
	memcpy(tstrng,&hdr_rec[sce_indx_any],
		sce_len_any);
	tstrng[sce_len_any] = '\0';
	if (start_clock)
	{
		strcpy(stclksce,tstrng);
	}
	strcpy(enclksce,tstrng);
	strcpy(type_in,"CHAR");
	dictmcnv(tarray,mon,dow,dptimes,tstrng,type_in,NULL,NULL,
        	 rtcode);
        if (rtcode[3] == 'X')
        {
        	/*error*/
        	return;
        }
	scet = dptimes[CLOWEPS.epoch_indx];
	/*compute down leg*/
	strncpy(tstrng,&hdr_rec[dut_indx_any],dut_len_any);
	tstrng[dut_len_any] = '\0';
	cnv2dbl(tstrng,&dut);
	/*compute up leg*/
	strncpy(tstrng,&hdr_rec[a1_indx_any],a1_len_any);
	tstrng[a1_len_any] = '\0';
	cnv2dbl(tstrng,&A1);
	strncpy(tstrng,&hdr_rec[clk_indx_any],clk_len_any);
	tstrng[clk_len_any] = '\0';
	if (start_clock)
	{
		/*set it to no more we have don scet and clk*/
		start_clock = FALSE;
		strcpy(stclk,tstrng);
	}
	strcpy(enclk,tstrng);
	computeclk(rtcode);
	if (rtcode[3] == 'X')
	{
		return;
	}
	/*set into structure*/
	RSCLK->sclk = sclk;
	RSCLK->su.scet = scet;
	RSCLK->A1 = A1;
	RSCLK->dut = dut;
	write_sceclk(sceclk_rec_lngth,&totwrt,strng,rtcode);
}
}	
	
static
read_sceclk(toread,read,rtcode)

int	toread;
int	*read;
char	*rtcode;

{

read[0] = SFFREAD(fdsceclk,hdr_rec,toread);
if (read[0] !=  toread)
{
	strcpy(rtcode,err_rtrn_code[CLR_INDX]);
}
return;
}

static	
write_sceclk(towrt,totwrt,wrt_buf,rtcode)

int	towrt;
int	*totwrt;
char	*wrt_buf;
char	*rtcode;

{
totwrt[0] = SFFWRITE(fdsccl_out,wrt_buf,towrt);
if (totwrt[0] != towrt)
{
	/*error in writting sceclk file*/
	strcpy(rtcode,err_rtrn_code[CLW_INDX]);
}
return;
}

static
computeclk(rtcode)

char	*rtcode;

{

switch (msn_case_nmbr)
{
	case GLL_INDX:
		/*input for unisys is 'RIM:MOD91:MOD10' needs mod 8*/
		sprintf(&tstrng[strlen(tstrng)],":0");
	break;
	default:
	break;
}

/*compute clk from various clk formats from tstrng*/
strcpy(type_in,"CHAR");
diclkcnv(carray,msn,&sclk,tstrng,type_in,NULL,NULL,rtcode);
return;
}

static
cnv2dbl(input,wiz)

char	*input;
DBL	*wiz;

{

char	*pntr;
char	*pntr1;
char	*pntr2;
BOOL	is_decimal;
int	i;
long	int	dev;
int	tot_dec;
long	int	wiz1;

pntr = input;
while(pntr[0] == ' ')
{
	pntr++;
}
is_decimal = FALSE;
pntr1 = strchr(pntr,'.');
if (pntr1 != NULL)
{
	is_decimal = TRUE;
	
	pntr2 = pntr1;
	pntr1++;
	pntr2[0] = '\0';
	pntr2 = strchr(pntr1,' ');
	if (pntr2 != NULL)
	{
		pntr2[0] = '\0';
	}
	tot_dec = strlen(pntr1);
	dev = 1;
	for ( i = 0 ; i < tot_dec ; i++)
	{
		dev *= 10;
	}
}
if (strlen(pntr))
{
	sscanf(pntr,"%ld",&wiz1);
	wiz[0] = (DBL)wiz1;
}
else
{
	wiz[0] = 0;
}
if (is_decimal)
{
	sscanf(pntr1,"%ld",&wiz1);
	wiz[0] += (DBL)wiz1/(DBL)dev;
}
return;
}
