/*
         1         2         3         4         5         6         7
123456789012345678901234567890123456789012345678901234567890123456789012
*/
#ifndef Header
/*****************************************************************
 * TITLE: diclkscecnv.c
 *
 * AUTHOR:  Unknown
 *          Aug 31, 1994
 *
 * MODIFIED:    Ray Bambery
 *          Aug 24, 2020 -  removed label after #ENDIF Header 
 *                          added prototypes to fix error:
 *                          error: static declaration of 'd9opscet' follows non-static declaration
 ***************************************************************
 *          
** MANUAL
**	DICLKSCECNV 3x "November 12, 1993"
**
** NAME
**	diclkscecnv - convert between SCET and SCLK for various 
**	missions.
**
** SYNOPSIS
**
**	  #include   <glbltme.h>
**
**	  void diclkscecnv(clksceary,msn,dptimes,sstrng,type_in,rtcode);
**
**	  long int *clksceary;
**	  char     *msn;
**	  DBL      *dptimes;
**	  char     *sstrng;
**	  char     *type_in;
**	  char     *rtcode;
**
** PARAMETERS AT A GLANCE
**
**	  1. clksceary - long int clksceary[9] - an array of times 
**	     and  spacecraft clocks.
**	  2. msn - char[4] - 3 character  string abbreviations for 
**	     missions, GLL, VGR, ULS etc.
**	  3. dptimes -  DBL[4] - double precisions times from epoch.
**	  4. sstrng - char[variable length] -  string input for  
**	     times, and/or clocks.
**	  5. type_in - char[variable length] - string of type of 
**	     time  input to 'disctlk'.
**	  6. rtcode - char[5] - return code, 'NORM' for ok.
**	
** PARAMETERS IN DETAIL
**
**	  1. clksceary- an array of times and spacecraft clocks.
**	     1. clksceary[0] - SCET - seconds past epoch
**	        (usually 1950)
**	     2. clksceary[1] - SCET - fractional seconds past 
**	        epoch.
**	     3. clksceary[2] - SCET - milliseconds past epoch.
**	     4. clksceary[3] thru clksceary[6] - SCLK components, 
**	        msn dependent.
**	        1. ULS
**	           1. clksceary[3] - corrected counter, 
**	              incremented every 2 seconds.
**	           2. clksceary[4] - fractional part of counter.
**	        2. GLL 
**	           1. clksceary[3] - rim counter, incremented 
**	              every 60 2/3 seconds.
**	           2. clksceary[4] - Mod 91 counter, incremented  
**	              every 2/3 seconds.
**	           3. clksceary[5] - mod 10 counter, incremented 
**	              every 66 2/3 milliseconds.
**	           4. clksceary[6] - mod 8 counter, incremented 
**	              every 8 2/3 milliseconds.
**	        3. VGR
**	           1. clksceary[3] - Mod 16 counter.
**	           2. clksceary[4] - mod 60 counter.
**	           3. clksceary[5] - line counter.
**	     5. clksceary[7], clksceary[8], actual SCLK, mission
**	        dependent.
**	        1. ULS - counter, 32 bits, binary fraction,16 bits.
**	        2. GLL - rim counter, 24 bits, mod 91 counter, 
**	           8 bits,mod 10 counter, 8 bits, mod 8 counter, 
**	           8 bits.
**	        3. VGR - Mod 16 counter, 16 bits, mod 60 counter,
**	           6 bits, line count, 10 bits.
**	  2. msn - 3 character abreviations for missions, GLL, VGR, 
**	     ULS etc.
**	  3. dptimes - If type_in is 'DPERT','DPSCET', or 'DPSCLK',
**	     dptimes contains.
**	     1. dptimes[0] - SCET - 'type_in' = 'DPSCET'.
**	     2. dptimes[1] - ERT - 'type_in' = 'DPERT'.
**	     3. dptimes[2] - SCLK - 'type_in' = 'DPSCLK'.
**	  4. type_in - Not NULL or empty. If 'type_in' is
**	     1. SCET - input in clksceary[0], clksceary[1].
**	     2. SCETM - input in clksceary[0], clksceary[2].
**	     3. DPSCET - doble precsion input in dptimes[0].
**	     4. SCLK - Inputs in clksceary[6] etc, mission 
**	        dependent.
**	     5. SCLKR - Raw SCLK placed in clksceary[7], etc. msn 
**	        dependent.
**	     6. DPSCLK - Doble precision SCLK input in dptimes[1].
**	     7. OPEN - Open and initialize scet sclk file.
**	        file is input in 'sstrng'.
**	     8. CLOSE - Close files and reset parameters.
**	     9. CHAR - the input ascii and is in 'sstrng'.
**	  5. sstrng - ascii string
**	     1. For 'type_in' = 'CHAR'.
**	        The allowed inputs are
**	        'ERT timestring'
**	        'SCET timestring'
**	        'SCLK sclkstring'
**	        'SCLKH sclkhexstrng'
**	        where 'timestring' is the legal inputs for
**	        'dictmcnv', 'sclkstring' and 'sclkhexstring'
**	        are mission dependent and are:
**	        1. msn = 'ULS' 
**	           1. sclkstring - 'ccccc.FFF' where ccccc is
**	              the count and FFF is the fractional portion 
**	              of the count.
**	           2. sclkhexstring - xxxxxxxx xxxx the hex numbers
**	              for the counter and binary fraction of 
**	              counter.
**	     2. For 'type_in' = 'OPEN'.
**	        'sstrng' contains the name of the coefficient file.
**	  6. rtcode - return code, 'NORM' for ok.
**	     
**	      
** GLOSSARY
**
**	 SCET - Spacecraft event time.
**	 SCLK - Spacecract clock.
**	 GLL - Galileo.
**	 ULS - Ulysses.
**	 VGR - Voyager.
**	 MGN - Magellon.
**
** EXTERNS
**	externs other than inputs used by 'diclkscecnv.c' (glowsce.h)
**
**	  1. fdsceclk - SFFOPDEF(fdsceclk) - file descriptor for 
**	     SCET/SCLK file.
**	  2. clkoffset - long int - Offset to start of data.
**	  3. msn - char[4] - 3 character mission id i.e. ULS, GLL, 
**	     VGR etc.
**	  4. msn_case_nmbr - int - for computing different 
**	     SCLK/SCETs. (ULS_INDX,GLL_INDX,VGR_INDX)
**	  5. CLKEPS -struct CLOWEP_STRUCT - epoch information
**	     used in creating double precision SCET seconds past
**	     that epoch.
**
** STRUCTURES
**
**	  struct epoch_struct {
**	        char    char_epoch[24];
**	        long    int   epoch_tarray[SIZE_OF_TARRAY];
**	        DBL     epoch_dptimes[SIZE_OF_DPS];
**	        char    epoch_mon[4];
**	        char    epoch_dow[4];
**	        char    new_epoch[4];
**	        BOOL    find_epoch_times;
**	  };
**	  typedef struct epoch_struct EPOCH_STRUCT;
**	
**	  struct clowep_struct {	
**	        long    int    epoch_indx;
**	        char    epoch_type[12];
**	        char    epoch[24];
**	        EPOCH_STRUCT   clow_ep_save;
**	  };
**	
**	  typedef struct clowep_struct CLOWEP_STRUCT;
**	
** DEFINES
**
**	These defines are set at compilation time.
**
**	  1. IBM_MAIN_FRAME - set if on IBM 3090, or any IBM 
**	     main frame.
**	  2. FBINARY - set for machines requiring binary opens.
**	     Use for IBM main frame machines and IBM PCs.
**	  3. MAC - set for Macintoshes using THINK C.
**	  4. NOPROTO - set for machines where there is no
**	     prototyping.
**	  5. NOSEEK - set for use buffered file management rather
**	     than seeking to a position in  a file. Use for
**	     IBM main frame machines and VAX VMS systems.
**	  6. SMLLMACH - set for machines using 'int' of 2 bytes.
**	     Use for Macintoshes, IBM PCs. Used for buffer 
**	     management.
**	  7. UNIX - set for UNIX operating systems on SUNSs etc.
**	  8. VMSVAX - set for DECS VMS VAXs.
**
** DESCRIPTION.
**
**	The double precision SCET/SCLK coefficient file is an SFDU
**	file and conists of the CCSDZ0001 label (which envelopes
**	other SFDUS), NJPLKL015	whose value is the meta data or header,
**	NJPLI0049 whose value are the double precision date needed
**	to convert from SCLK to SCET and visa versa.
**
**	'diclkscecnv' reads the SCET/SCLK coefficient file and 
**	calculates all other outputs from the given input in 'type_in'.
**
**	'diclkscecnv' opens and reads in the header for each the file.
**	A sample header looks like:
**
**	 CCSD3ZF0000100000001NJPL3KS0L015$$scet$$
**	 MISSION_NAME=ULYSSES;
**	 SPACECRAFT_NAME=ULS;
**	 DATA_SET_ID=DOUBLE_PRECISION_SCLK_SCET;
**	 FILE_NAME=uls.dp1coeff.data;
**	 PRODUCT_CREATION_TIME=1990-285T14:49:10;
**	 PRODUCT_VERSION_ID=1;
**	 PRODUCER_ID=DMT;
**	 APPLICABLE_START_TIME=1990-285T14:49:10;
**	 APPLICABLE_STOP_TIME=1999-12-30T12:00:00.000;
**	 MISSION_ID=3;
**	 SPACECRAFT_ID=55;
**	 TARGET_PARAMETER_NAME=SCET;
**	 TARGET_PARAMETER_EPOCH=1950-001T00:00:00.000;
**	 CCSD$$MARKER$$scet$$NJPL3IF004900000001
**
**	From this header 'diclkscecnv', finds the epoch used for the SCET double 
**	precision seconds. 
**
**	NOTE: All SCET seconds MUST be seconds past the epoch found in the header.
** 
**	The different clocks are:
**	  1. GLL - Raw format (bit stream) starts in clksceary[13].
**	     1. RIM Counter - 24 bits (clksceary[9]).
**	     2. MOD 91 Counter - 8 bits (clksceary[10]).
**	     3. MOD 10 Counter - 8 bits (clksceary[11]).
**	     4. MOD 8 Counter - 8 bits (clksceary[12]).
**	  2. ULS - Raw format (bit stream) start in clksceary[13].
**	     1. Current Corrected Counter - 4 bytes (clksceary[9]).
**	     2. Fractional count - 16 bits (clksceary[10]).
**	  3. VGR - Raw format (bit stream) starts in clksceary[13].
**	     1. MOD 16 - 16 bits (clksceary[9]).
**	     2. MOD 60 - 6 bits  (clksceary[10]).
**	     3. Line Count - 10 bits (clksceary[11]).
**
**
*/
#endif   /* Header */
#if (IBM_MAIN_FRAME)
/*place pragma to allow for entry points for JTPM*/
#pragma csect(CODE,"DCCLKSC0") csect(STATIC,"DCCLKSC1")
#endif

#include	<stdio.h>
#include	<stdlib.h>
#include	<ctype.h>
#include	<string.h>
#include	<sfbufmgr.h>
#include	<glbltme.h>
#include	<glowlsce.h>
#include	<dictmerr.h>
#include	<diclksce.h>

/* prototypes */

void diclkscecnv(long int *clksceary,char *msn_in,DBL *dptimes,char *sstrng,char *type_in,char *rtcode);
static d9opscet(char *msn_in,char *sstrng,char *rtcode);
static  int d9chkeps(void);
static d9clkfilein(char *msn_in,char *rtcode);
static d9setclkbfm(char *rtcode);
static d9crclkblks(char *rtcode);
//int d9chkrlo(int *tot_bytes_left,char *rtcode);
static int d9closeclk(char *rtcode);
static d9crclkblks(char *rtcode);

//int d9chkrlo(char *rtcode);
static int d9closeclk(char *rtcode);
static d9chrinp(char *sstrng,long int *clksceary,DBL *dptimes,char *rtcode);
static d9dpinp(char *type_in,long int *clksceary,DBL *dptimes,char *rtcode);
static d9scet(char *strng_pntr,long int *clksceary,DBL *dptimes,char *rtcode);
static d9sceinp(char *type_in,long int *clksceary,DBL *dptimes,char *rtcode);
static d9dpscei(long int *clksceary,DBL *dptimes,char *rtcode);
static d9cmpsce(long int *clksceary,DBL *dptimes,char *rtcode);
static d9clkinp(char *type_in,long int *clksceary,DBL *dptimes,char *rtcode);
static d9dpclki(long int *clksceary,DBL *dptimes,char *rtcode);
static d9cmpclk(long int *clksceary,DBL *dptimes,char *rtcode);
static d9sefclk(char *rtcode);
static d9clkfse(char *rtcode);
static fndclkblk(char *rtcode);
static d9blkrlo(char *rtcode);
static fndclkrec(char *rtcode);
static d9cnvsce(long int *clksceary,DBL *dptimes,char *rtcode);
static d9cnvclk(long int *clksceary,DBL *dptimes,char *rtcode);
static d9cnvfhx(char *strng,long int *intout,int len);
static d9ridquotes(char *value,char **value_pntr,int *value_len);
static d9cary2sary(long int *clksceary,char *rtcode);
static d9sary2cary(long int *clksceary,char *rtcode);






/***************************************************************/
void
diclkscecnv(clksceary,msn_in,dptimes,sstrng,type_in,rtcode)

  long int *clksceary;
  char     *msn_in;
  DBL	   *dptimes;
  char     *sstrng;
  char     *type_in;
  char     *rtcode;
  
{

int	rtrn;
#if (IBM_MAIN_FRAME)
ibm_here = TRUE;
#else
ibm_here = FALSE;
#endif
/*begin 'disctclk'*/
strcpy(rtcode,"NORM");
if (strcmp(did_init,"YES"))
{
	/*initia;ize here*/
	if (strcmp(times_data_init,"YES"))
	{
		ditminit();
		if (ibm_here)
		{
			sfparser();
		}
	}
	strcpy(did_init,"YES");
}
if (strcmp(vax_ordr_called,"YES"))
{
	/*check on order of integters (VAX VMS etc vs SUN, MACINTOSH,
	 IBM main frame*/
	strcpy(vax_ordr_called,"YES");
	rtrn = mchintst(NULL,NULL);
	if (rtrn)
	{
		/*this is a VAX order reorder array*/
		vax4_or_othr[0] = 3;
		vax4_or_othr[1] = 2;
		vax4_or_othr[2] = 1;
		vax4_or_othr[3] = 0;
		vax2_or_othr[0] = 1;
		vax2_or_othr[1] = 0;
	}
}
/*********************************************************************
**	     1. SCET - input in clksceary[0], clksceary[1].
**	     2. SCETM - input in clksceary[0], clksceary[2].
**	     3. DPSCET - doble precsion input in dptimes[0].
**	     7. SCLK - Inputs in clksceary[6] etc, mission dependent.
**	     8. SCLKR - Raw SCLK placed in clksceary[10], etc. msn 
**	        dependent.
**	     9. DPSCLK - Doble precision SCLK input in dptimes[3].
**	    10. OPEN - Open and initialize scet sclk file.
**	    11. CHAR - character string in sstrng for times etc.
*********************************************************************/
scet_fnd = FALSE;
sclk_fnd = FALSE;
ert_fnd = FALSE;
switch (type_in[0])
{
	case 'O': 
	case 'o':
		/*open scet file*/
		d9opscet(msn_in,sstrng,rtcode);
	break;
	case 'C':
	case 'c':
		/*close or char inputs*/
		if (type_in[1] == 'l'  ||  type_in[1] == 'L')
		{
			d9closeclk(rtcode);
		}
		else
		{
			/*character string input*/
			d9chkeps();
			d9chrinp(sstrng,clksceary,dptimes,rtcode);
		}
	break;
	case 'D':
	case 'd':
		/*double precision input here*/
		d9chkeps();
		d9dpinp(type_in,clksceary,dptimes,rtcode);
	break;
	case 'S':
	case 's':
		d9chkeps();
		if (!strncmp(type_in,"SCLK",4))
		{
			/*sclk here*/
			d9clkinp(type_in,clksceary,dptimes,rtcode);
		}
		else
		{
			/*scet here*/
			d9sceinp(type_in,clksceary,dptimes,rtcode);
		}
	break;
	default:
		strcpy(rtcode,err_rtrn_code[BDI_INDX]);
		return;
}

return;
}
/**********************************************************/
static
d9opscet(msn_in,sstrng,rtcode)

char	*msn_in;
char	*sstrng;
char	*rtcode;

{

size_t	sizme;
SFFOPDEF(fdinit);
int	rtc;
int	recsize;
char	*malc;

/*begin diopscet*/
strcpy(sceclk_file,sstrng);
d9clkfilein(msn_in,rtcode);
if (rtcode[3] == 'X')
{
	/*error */
	return;
}

return;
}	

/****************************************************************/
static	int
d9chkeps()

{

tepoch_indx = 10 + 2*CLKEPS.epoch_indx;
/*check epochs of coeff file*/
if (CLKEPS.epoch_indx != DPEP_OF_CHOICE)
{
	/*the epoch is either 1950 or 2000*/
	return;
}
/*ok check to see if the epoch for times is proper*/
if (!strcmp(EPS.char_epoch,CLKEPS.clow_ep_save.char_epoch))
{
	/*they are the same no need for a change*/
	return;
}
/*change epochs*/
EPS = CLKEPS.clow_ep_save;
return;
}
/***************************************************************/	
static
d9clkfilein(msn_in,rtcode)

char	*msn_in;
char	*rtcode;

{

size_t	sizme;
char	data_set_id[80];

SFFOPEN(fdsceclk,sceclk_file,SFRDONLY);
if SFFERROPEN(fdsceclk)
{
	strcpy(rtcode,err_rtrn_code[CLO_INDX]);
	return;
}
/*get size of file*/
/*read in header records*/
clk_file_size = sfilesiz(fdsceclk);
/*get header*/
dihdrsceowl(fdsceclk,data_set_id,SFFNULL,rtcode);
if (rtcode[3] == 'X')
{
	/*error encountered*/
	return;
}
if (strcmp(data_set_id,COEFF_DATA_SET_ID))
{
	/*error here*/
	strcpy(rtcode,err_rtrn_code[DSC_INDX]);
	/*set error here*/
	return;
}
/*check on msn or space craft id*/
if (msn_in != NULL)
{	
	switch (msn_case_nmbr)
	{
		case ULS_INDX:
			if (msn_in[0] != 'u'  &&
			    msn_in[0] != 'U')
			{
				/*error here mission inconsistant*/
				strcpy(rtcode,err_rtrn_code[MTO_INDX]);
				return;
			}
		break;
		case GLL_INDX:
			if (msn_in[0] != 'g'   &&
			    msn_in[0] != 'G')
			{
				/*error here mission inconsistant*/
				strcpy(rtcode,err_rtrn_code[MTO_INDX]);
				return;
			}
		break;
		default:
			/*error here mission inconsistant*/
			strcpy(rtcode,err_rtrn_code[MTO_INDX]);
			return;
	}
}		
rec_size_clk  = sizeof(SCLKR);
/*two files sclk/scet and files*/
numclk_recs_per_blk = CLK_RECS_PER_BLK;
std_clk_offset = rec_size_clk * numclk_recs_per_blk;
#if (NOSEEK)
/*set up structure rdinfo*/
d9setclkbfm(rtcode);
d9crclkblks(rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
#else
/*compute number of records*/
number_clk_records = (clk_file_size - clkoffset)/rec_size_clk;
/*malloc bloksize*/
sizme = numclk_recs_per_blk*rec_size_clk +1;
sclkarray = (char *)malloc(sizme);
if (sclkarray == NULL)
{
	strcpy(rtcode,err_rtrn_code[MAL_INDX]);
	return;
}
/*place for reading records*/
SFFSEEK(fdsceclk,clkoffset,SFSEEK_SET);
d9crclkblks(rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
#endif
return;
}

#if (NOSEEK)

/*****************************************************************/
/*set up clk buffer management */

static
d9setclkbfm(rtcode)

char	*rtcode;

{

int	i;
int	len;
int	len1;
int	rtc;

/*compute number records per block*/
/*MAXBUFL is the maximum buffer size.
  number of bytes must be some multiple of records
 */
len = MAXBUFL/rec_size_clk;
len *= rec_size_clk;
len1 = len/numclk_recs_per_blk;
if (!len1)
{
	len1 = 1;
}
numclk_recs_per_blk = len/len1;
std_clk_offset = numclk_recs_per_blk * rec_size_clk;
len /= std_clk_offset;
len *= std_clk_offset;
rdclkbuf.tot_bytes = 0;
rdclkbuf.more_reads = FALSE;
rdclkbuf.start_at_byte = clkoffset;
rdclkbuf.bytes_per_buf = len;
for ( i = 0 ; i < MAXNUMBUFRS ; i ++)
{
	/*initialize rdclkbuf structure*/
	rdclkbuf.bufs[i].pntr = NULL;
}
rdinfclk.rdbuf = (char *)&rdclkbuf;
rdinfclk.frd = fdsceclk;
rtc = sfread_file_in(&rdinfclk);
if (rtc < 0)
{	
	/*error here*/
	strcpy(rtcode,err_rtrn_code[BFM_INDX]);
	return;
}
return;
}

/****************************************************************/
/*this uses buffer management to create clock blocks*/
static
d9crclkblks(rtcode)

char	*rtcode;

{

size_t	sizme;
int	rtc;
int	i;
char	*malc;
DBL	*curr_clk_pntr;
int	tot_bytes_left;
int	blksize;
	
sizme = (size_t)sizeof(BLKSCLK);
blk_size_clk = std_clk_offset;
number_clk_blks = 0;
clkbufinfo.bufrmode = BOF;
/*point to first buffer*/
rtc = sfbufman(&clkbufinfo,&rdinfclk);
clkbufinfo.bufrmode = NEXT_BUFR;
if (rtc < 0)
{
	/*error*/
	strcpy(rtcode,err_rtrn_code[BFM_INDX]);
	return;
} 
while (TRUE)
{
	/*do for all buffers here and now*/
	tot_bytes_left = clkbufinfo.buflen;
	sclkarray = clkbufinfo.buf;
	while(TRUE)
	{
		/*clkbufinfo.buf points to currebt buffer
		  clkbufinfo.buflen is the length i bytes
		  clkbufinfo.bufnumber is the buff number to 
		  store in CURRCLKBLK->bufcount*/
		curr_clk_pntr = (DBL *)sclkarray;
		SCECLKREC = (SCLKR *)sclkarray;
		/*malloc it here*/
		malc = (char *)malloc(sizme);
		if (malc == NULL)
		{
			strcpy(rtcode,err_rtrn_code[MAL_INDX]);
			return;
		}
		CURRCLKBLK = (BLKSCLK *)malc;
		rtc = 0;
		if (!number_clk_blks)
		{
		/*start here*/
			start_clk_blk = malc;
			CURRCLKBLK->prev = NULL;
		}
		else
		{
			CURRCLKBLK->prev = (char *)PREVCLKBLK;
			PREVCLKBLK->next = (char *)CURRCLKBLK;		
			/*check for rollover here*/
			rtc = d9chkrlo(&tot_bytes_left,rtcode);
			if (rtc < 0)
			{
				/*error*/
				return;
			}
			if (rtc)
			{
				/*get next*/
				clkbufinfo.bufrmode = NEXT_BUFR;
				continue;
			}
		}
		CURRCLKBLK->sclk = SCECLKREC->sclk;
		CURRCLKBLK->scet = SCECLKREC->su.scet;
		CURRCLKBLK->next = NULL;
		CURRCLKBLK->bufcount = clkbufinfo.bufnumber;
		CURRCLKBLK->offset = clkbufinfo.buflen - tot_bytes_left;
		if (tot_bytes_left < blk_size_clk)
		{
			blksize = tot_bytes_left;
		}
		else
		{
			blksize = blk_size_clk;
		}
		CURRCLKBLK->nmbr_recs = blksize/rec_size_clk;
		/*set prev to curr*/
		PREVCLKBLK = CURRCLKBLK;
		number_clk_blks++;
		/*do an offset from here*/
		tot_bytes_left -= (int)blksize;
		if (tot_bytes_left <= 0)
		{
			break;
		}
		sclkarray = &clkbufinfo.buf[clkbufinfo.buflen -
		                            tot_bytes_left];
	}
	/*get next*/
	if (!clkbufinfo.contflag)
	{
		break;
	}
	/*get next buffer*/
	rtc = sfbufman(&clkbufinfo,&rdinfclk);
	if (rtc < 0)
	{
		strcpy(rtcode,err_rtrn_code[BFM_INDX]);
		return;
	}
	
}
CURRCLKBLK = (BLKSCLK *)NULL;
SCECLKREC = (SCLKR *)NULL;
return;
}
/**********************************************************/
int
d9chkrlo(tot_bytes_left,rtcode)

int	*tot_bytes_left;
char	*rtcode;

{

size_t	sizme;
char	*malc;
int	nmbr_reads;
int	bytes_read;
int	nmbr_recs;
int	i;
char	*pntr;
SCLKR	*last_sclk;
int	rtc;
long	int	pos;
BOOL	previous;
int	blksize;

/*PREV_BLK has the last read point CMBCLKR or SCLKREC
  points to current record read to compare sclks*/
if (SCECLKREC->sclk > PREVCLKBLK->sclk)
{
	/*it still ascending*/
	return(0);
}
/*there is a problem*/
/*go to start of buffer read forwards*/
sizme = sizeof(ROLL_OVER);
if (PREVCLKBLK->bufcount != CURRCLKBLK->bufcount)
{
	/*this is in previous buffer*/
	previous = TRUE;
	clkbufinfo.bufrmode = PREV_BUFR;
	rtc = sfbufman(&clkbufinfo,&rdinfclk);
	if (rtc < 0)
	{
		strcpy(rtcode,err_rtrn_code[BFM_INDX]);
		return;
	}
}
else
{
	previous = FALSE;
}
sclkarray = &clkbufinfo.buf[(int)PREVCLKBLK->offset];
nmbr_recs = PREVCLKBLK->nmbr_recs;
pntr = sclkarray;
last_sclk = (SCLKR *)pntr;
pntr += rec_size_clk;
for (i = 1 ; i < nmbr_recs ; i++)
{
	SCECLKREC = (SCLKR *)pntr;
	if (SCECLKREC->sclk < last_sclk->sclk)
	{
		/*here is discontinuity*/
		nmbr_recs = i;
		break;
	}
	last_sclk = SCECLKREC;
	pntr += rec_size_clk;
}
blksize = nmbr_recs*(int)rec_size_clk;
CURRCLKBLK->sclk = SCECLKREC->sclk;
CURRCLKBLK->scet = SCECLKREC->su.scet;
CURRCLKBLK->next = NULL;
CURRCLKBLK->bufcount = clkbufinfo.bufnumber;
CURRCLKBLK->offset = PREVCLKBLK->offset+blksize;
if (tot_bytes_left[0] < blk_size_clk)
{
	blksize = tot_bytes_left[0];
}
else
{
	blksize = blk_size_clk;
}
CURRCLKBLK->nmbr_recs = blksize/rec_size_clk;
/*do an offset from here*/
tot_bytes_left[0] -= blksize;
sclkarray = &clkbufinfo.buf[(int)CURRCLKBLK->offset+blksize];
PREVCLKBLK->nmbr_recs = nmbr_recs;
number_clk_blks++; /*add a block to it*/
/*get ROLL_OVER area for structure etc*/
malc = (char *)malloc(sizme);
if (malc == NULL)
{
	strcpy(rtcode,err_rtrn_code[MAL_INDX]);
	return;
}
CURR_ROLL = (ROLL_OVER *)malc;
if (sclk_roll_over == NULL)
{
	/*start here*/
	sclk_roll_over = malc;
	CURR_ROLL->prev = NULL;
}
else
{
	CURR_ROLL->prev = (char *)PREV_ROLL;
	PREV_ROLL->next = (char *)CURR_ROLL;		
}
CURR_ROLL->next = NULL;
/*place record in place*/
CURR_ROLL->prev_sclkrec.sclk = last_sclk->sclk;
CURR_ROLL->prev_sclkrec.su.scet = last_sclk->su.scet;
CURR_ROLL->prev_sclkrec.A1 = last_sclk->A1;
/*reset PREV to CURR*/
PREVCLKBLK = CURRCLKBLK;
return(1);
}
/***************************************************************/
static	int
d9closeclk(rtcode)

char	*rtcode;

{

int	i;

/*close files and buffers*/
for ( i = 0 ; i < MAXNUMBUFRS ; i ++)
{
	if (rdclkbuf.bufs[i].pntr == NULL)
	{
		break;
	}
	free(rdclkbuf.bufs[i].pntr);
}
SFFCLOSE(fdsceclk);
return;
}
/*****************************************************************/
#else
static
d9crclkblks(rtcode)

char	*rtcode;

{

size_t	sizme;
int	rtc;
int	i;
char	*malc;
DBL	*curr_clk_pntr;
	
sizme = (size_t)sizeof(BLKSCLK);
blk_size_clk = std_clk_offset;
number_clk_blks = 0;
while (TRUE)
{
	while(TRUE)
	{
		rtc = SFFREAD(fdsceclk,sclkarray,rec_size_clk);
		if (rtc < 0)
		{
			strcpy(rtcode,err_rtrn_code[CLR_INDX]);
			return;
		}
		if (!rtc)
		{
			/*ok no more set last etc*/
			CURRCLKBLK->nmbr_recs = 
				(clk_file_size - CURRCLKBLK->offset)/
				rec_size_clk;
			/*last block*/
			last_clk_blk = malc;
			break;
		}
		curr_clk_pntr = (DBL *)sclkarray;
		if (!curr_clk_pntr[CLKA1_FIELD])
		{
			continue;
		}
		break;
	}
	if (!rtc)
	{
		break;
	}
	SCECLKREC = (SCLKR *)sclkarray;
	/*malloc it here*/
	malc = (char *)malloc(sizme);
	if (malc == NULL)
	{
		strcpy(rtcode,err_rtrn_code[MAL_INDX]);
		return;
	}
	CURRCLKBLK = (BLKSCLK *)malc;
	rtc = 0;
	if (!number_clk_blks)
	{
		/*start here*/
		start_clk_blk = malc;
		CURRCLKBLK->prev = NULL;
	}
	else
	{
		CURRCLKBLK->prev = (char *)PREVCLKBLK;
		PREVCLKBLK->next = (char *)CURRCLKBLK;		
		/*check for rollover here*/
		rtc = d9chkrlo(rtcode);
		if (rtc < 0)
		{
			/*error*/
			return;
		}
		if (rtc)
		{
			/*get next*/
			continue;
		}
	}
	CURRCLKBLK->sclk = SCECLKREC->sclk;
	CURRCLKBLK->scet = SCECLKREC->su.scet;
	CURRCLKBLK->next = NULL;
	CURRCLKBLK->nmbr_recs = std_clk_offset/rec_size_clk;
	CURRCLKBLK->offset = SFFTELL(fdsceclk) - rec_size_clk;
	/*set prev to curr*/
	PREVCLKBLK = CURRCLKBLK;
	number_clk_blks++;
	/*do an offset from here*/
	rtc = SFFSEEK(fdsceclk,std_clk_offset-rec_size_clk,
		      SFSEEK_CUR);
	if (rtc)
	{
		i = 0;
	}
}
CURRCLKBLK = (BLKSCLK *)NULL;
SCECLKREC = (SCLKR *)NULL;
/*position to start of data record*/
rtc = SFFSEEK(fdsceclk,clkoffset,SFSEEK_SET);
curr_clk_seek_pos = clkoffset;
return;
}
/************************************************************/
int
d9chkrlo(rtcode)

char	*rtcode;

{

size_t	sizme;
char	*malc;
int	nmbr_reads;
int	bytes_read;
int	nmbr_recs;
int	i;
char	*pntr;
SCLKR	*last_sclk;
int	rtc;
long	int	pos;

/*PREV_BLK has the last read point CMBCLKR or SCLKREC
  points to current record read to compare sclks*/
if (SCECLKREC->sclk > PREVCLKBLK->sclk)
{
	/*it still ascending*/
	return(0);
}
/*there is a problem*/
/*go to start of buffer read forwards*/
sizme = sizeof(ROLL_OVER);
rtc = SFFSEEK(fdsceclk,PREVCLKBLK->offset,SFSEEK_SET);
/*seek from start of file to here*/
/*read into array*/
bytes_read = SFFREAD(fdsceclk,sclkarray,(int)std_clk_offset);
if (bytes_read <= 0)
{
	/*error */
	strcpy(rtcode,err_rtrn_code[CLR_INDX]);
	return(-1);
}
nmbr_recs = bytes_read/rec_size_clk;
pntr = sclkarray;
last_sclk = (SCLKR *)pntr;
pntr += rec_size_clk;
for (i = 1 ; i < nmbr_recs ; i++)
{
	SCECLKREC = (SCLKR *)pntr;
	if (SCECLKREC->sclk < last_sclk->sclk)
	{
		/*here is discontinuity*/
		nmbr_recs = i;
		break;
	}
	last_sclk = SCECLKREC;
	pntr += rec_size_clk;
}
CURRCLKBLK->next = NULL;
CURRCLKBLK->sclk = SCECLKREC->sclk;
CURRCLKBLK->scet = SCECLKREC->su.scet;
CURRCLKBLK->nmbr_recs = std_clk_offset/rec_size_clk;
CURRCLKBLK->offset = nmbr_recs*rec_size_clk + PREVCLKBLK->offset;
/*do an offset from here*/
/*find posisiton to go to the seek is at end of buffer read
  but the current blk is back inside nblks from start or
   PREVCLKBLK->nmbr_recs - nmbr_recs 
   so pos = -(PREVCLKBLK - nmbr_blks)*recsize + std_offset*/
pos = std_clk_offset - (PREVCLKBLK->nmbr_recs - nmbr_recs)*rec_size_clk;
rtc = SFFSEEK(fdsceclk,pos,SFSEEK_CUR);
PREVCLKBLK->nmbr_recs = nmbr_recs;
number_clk_blks++; /*add a block to it*/
/*get ROLL_OVER area for structure etc*/
malc = (char *)malloc(sizme);
if (malc == NULL)
{
	strcpy(rtcode,err_rtrn_code[MAL_INDX]);
	return;
}
CURR_ROLL = (ROLL_OVER *)malc;
if (sclk_roll_over == NULL)
{
	/*start here*/
	sclk_roll_over = malc;
	CURR_ROLL->prev = NULL;
}
else
{
	CURR_ROLL->prev = (char *)PREV_ROLL;
	PREV_ROLL->next = (char *)CURR_ROLL;		
}
CURR_ROLL->next = NULL;
/*place record in place*/
CURR_ROLL->prev_sclkrec.sclk = last_sclk->sclk;
CURR_ROLL->prev_sclkrec.su.scet = last_sclk->su.scet;
CURR_ROLL->prev_sclkrec.A1 = last_sclk->A1;
/*reset PREV to CURR*/
PREVCLKBLK = CURRCLKBLK;
return(1);
}
/*******************************************************************/
static	int
d9closeclk(rtcode)

char	*rtcode;

{

char	*free_pntr;

CURRCLKBLK = (BLKSCLK *)start_clk_blk;
for (;;)
{
	if (CURRCLKBLK == (BLKSCLK *)NULL)
	{
		break;
	}
	free_pntr = (char *)CURRCLKBLK;
	free(free_pntr);
	CURRCLKBLK = (BLKSCLK *)CURRCLKBLK->next;
}
SFFCLOSE(fdsceclk);
return;
}
/*********************************************************************/
#endif
static	
d9chrinp(sstrng,clksceary,dptimes,rtcode)

char	*sstrng;
long	int	*clksceary;
DBL	*dptimes;
char	*rtcode;

{

int	i;
int	j;
int	tot;
int	indx;
char	cmp_strng[8];
int	len;
int	rt_code;

/*********************************************************************
	parse string here find what kind

	1. ??? YYYY DOY HH:MM:SS.FFF (and other calender forms)
	2. ??? SP50 (or SP2000 or SPEPOCH) xxxxxxxxx.xxx
	3. ??? SPxH (for hex) XXXXXXXX XXXX (where x is 50 2000 or 
	   EPOCH). 
	Where ??? is either SCET or ERT
	4. SCLK the clock for mission
	   1. ULS - 'SCLK xxxxxxxx.xxx'
	   2. ULS - 'SCLKH (for hex) XXXXXXXX XXXX'
	   
	NOTE: See 'dictmcnv' for types of time for ERT and SCET inputs.
	
*********************************************************************/

strng_pntr = sstrng;
tot = strlen(sstrng);
scet_fnd = FALSE;
sclk_fnd = FALSE;
/*actually find first word find if SCET SCLK*/
for (i = 0 ; i < tot ; i++)
{
	if (sstrng[i] == ' ')
	{
		continue;
	}
	/*we found something*/
	strng_pntr = strchr(&strng_pntr[i],' '); /*find blank*/
	if (strng_pntr == NULL)
	{
		/*error */
		strcpy(rtcode,err_rtrn_code[CHR_INDX]);
		return;
	}
	len = (long int)strng_pntr - (long int)&sstrng[i];
	if (len > 7) 
	{
		strcpy(rtcode,err_rtrn_code[STL_INDX]);
		return;
	}
	indx = i;
	for (j = 0 ; j < len ; j++)
	{
		if (islower(sstrng[i+j]))
		{
			cmp_strng[j] = (char)toupper(sstrng[i+j]);
		}
		else
		{
			cmp_strng[j] = sstrng[i+j];
		}
	}
	cmp_strng[len] = '\0';
	if (!strcmp(cmp_strng,"SCET"))
	{
		scet_fnd = TRUE;
		break;
	}
	else if (!strncmp(cmp_strng,"SCLK",4))
	{
		sclk_fnd = TRUE;
		break;
	}
	strcpy(rtcode,err_rtrn_code[NOT_INDX]);
	return;
}
if (scet_fnd)
{
	d9scet(strng_pntr,clksceary,dptimes,rtcode);
}
else
{
	/*ok now get it going*/
	strcpy(typein,"CHAR");
	diclkcnv(carray,NULL,&sclk,strng_pntr,typein,NULL,
		 NULL,rtcode);
	if (rtcode[3] == 'X')
	{
		/*error*/
		return;
	}
	DPSCLK = sclk;
	/*place output in proper place*/
	d9cary2sary(clksceary,rtcode);
	d9cmpclk(clksceary,dptimes,rtcode);
}
return;
}
		
/*********************************************************************/	
static
d9dpinp(type_in,clksceary,dptimes,rtcode)

char	*type_in;
long	int	*clksceary;
DBL	*dptimes;
char	*rtcode;

{

/*input is dp someting or other*/
/*********************************************************************
**	     1. DPERT - double precision, input in dptimes[1].
**	     2. DPSCET - doble precsion input in dptimes[0].
**	     3. DPSCLK - Doble precision SCLK input in dptimes[3].
*********************************************************************/

switch (type_in[4])
{
	case 'E':
	case 'e':
		/*dpsc[E]t*/
		d9dpscei(clksceary,dptimes,rtcode);
	break;
	case 'L':
	case 'l':
		/*dpsc[L]k*/
		d9dpclki(clksceary,dptimes,rtcode);
	break;
	default:
		strcpy(rtcode,err_rtrn_code[BDI_INDX]);
}
return;
}
/*********************************************************************/
static
d9scet(strng_pntr,clksceary,dptimes,rtcode)

char	*strng_pntr;
long	int	*clksceary;
DBL	*dptimes;
char	*rtcode;

{

int	i;
int	j;
int	k;
int	tot;
BOOL	hex;
int	len;
int	toti;
int	totl;
char	*pntr;
/*find next word see if SPetc*/

tot = strlen(strng_pntr);
for ( i = 0 ; i < tot ; i++)
{
	if (strng_pntr[i] == ' ')
	{
		continue;
	}
	pntr = strchr(&strng_pntr[i],' ');
	if (pntr == NULL)
	{
		strcpy(rtcode,err_rtrn_code[NOT_INDX]);
		return;
	}
	/*looking for SPEPOCH(H)*/
	totl = (long int)pntr - (long int)&strng_pntr[i];
	if (totl != 7 && len != 8)
	{
		break;
	}
	if (strng_pntr[i] != 'S'  && strng_pntr[i] != 's')
	{
		break;
	}
	if (strng_pntr[i+1] != 'P'   && strng_pntr[i+1] != 'p')
	{
		break;
	}
	if (strng_pntr[i+2] != 'E'   && strng_pntr[i+2] != 'e')
	{
		break;
	}
	/*should be SPEPOCH*/
	if (strng_pntr[i+7] == 'H'  || strng_pntr[i+7] == 'h')
	{
		hex = TRUE;
	}
	else
	{
		hex = FALSE;
	}
	len = strlen(time_type); /*50 or 2000*/
	strncpy(&strng_pntr[i],time_type,len);
	toti = i + 2 + len;
	for (j = toti ; j < totl ; j++)
	{
		strng_pntr[j] = ' '; /*set as blanks or white spaces*/
	}
	break;
}
/*convert*/
strcpy(typein,"CHAR");
dictmcnv(tarray,mon,NULL,dpsecs,strng_pntr,typein,NULL,NULL,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
/*scet here*/
scet = dpsecs[CLKEPS.epoch_indx];
DPSCET = scet;
SCETSECS = tarray[tepoch_indx];
SCETFRACS = tarray[tepoch_indx+1]; /*fractional seconds*/
SCETMILLS = MILLSEC; /*milliseconds*/
d9cmpsce(clksceary,dptimes,rtcode);

return;
}
/**********************************************************************/	
static
d9sceinp(type_in,clksceary,dptimes,rtcode)

char	*type_in;
long	int	*clksceary;
DBL	*dptimes;
char	*rtcode;

{

int	len;

/*********************************************************************
**	     1. SCET - input in clksceary[0], clksceary[1].
**	     2. SCETM - input in clksceary[0], clksceary[2].
**********************************************************************/


/******************************************************************
**	   tarray - long int tarray[14] - time array of different times 
**	     1. tarray[0]  - 4 digit year
**	     2. tarray[1]  - doy, day of year
**	     3. tarray[2]  - hoy, hour of year
**	     4. tarray[3]  - second of hour (hoy format)
**	     5. tarray[4]  - month of year
**	     6. tarray[5]  - day of month
**	     7. tarray[6]  - hour of day
**	     8. tarray[7]  - minutes of hour
**	     9. tarray[8]  - seconds of minute
**	    10. tarray[9]  - milliseconds
**	    11. tarray[10] - seconds past 1950
**	    12. tarray[11] - fractional part of seconds past 1950
**	    13. tarray[12] - seconds from 2000
**	    14. tarray[13] - fractional part of seconds from 2000
***********************************************************************/
if (type_in[4] == 'M'  ||  type_in[4] == 'm')
{
	switch (CLKEPS.epoch_indx)
	{
		case DPEP_OF_CHOICE:
			/*epoch of choice*/
			strcpy(typein,"SPEPOCHM");
			SECSFEPOCH = SCETSECS; 
		break;
		case DP2000_INDX:
			/* seconds past 2000*/
			strcpy(typein,"SP2000M");
			SECSF2000 = SCETSECS; 
		break;
		default:
			/* seconds past 50*/
			strcpy(typein,"SP50M");
			SECSF1950 =  SCETSECS; 
		break;
	}
	/*milliseconds not fractional seconds*/
	/*set up to do time conversions here*/
	MILLSEC = SCETMILLS;
}
else
{
	/*use fractional seconds*/
	switch (CLKEPS.epoch_indx)
	{
		case DPEP_OF_CHOICE:
			/*epoch of choice*/
			strcpy(typein,"SPEPOCH");
			SECSFEPOCH = SCETSECS; 
		break;
		case DP2000_INDX:
			/* seconds past 2000*/
			strcpy(typein,"SP2000");
			SECSF2000 = SCETSECS; 
		break;
		default:
			/* seconds past 50*/
			strcpy(typein,"SP50");
			SECSF1950 = SCETSECS; 
		break;
	}
	tarray[tepoch_indx+1] = SCETFRACS;
}
dictmcnv(tarray,mon,NULL,dpsecs,NULL,typein,NULL,NULL,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
/*ok now set up and go*/
scet = dpsecs[CLKEPS.epoch_indx];
DPSCET = scet;
if (type_in[3] == 'M'  ||  type_in[3] == 'm')
{
	SCETFRACS = tarray[tepoch_indx+1]; /*fractional seconds*/
}
else
{
	SCETMILLS = MILLSEC; /*milliseconds*/
}
d9cmpsce(clksceary,dptimes,rtcode);
return;
}
/************************************************************************/
static
d9dpscei(clksceary,dptimes,rtcode)

long	int	*clksceary;
DBL	*dptimes;
char	*rtcode;

{

/*double precision input of scet*/
scet = DPSCET;
d9cnvsce(clksceary,dptimes,rtcode);
if (rtcode[3] == 'X')
{
	/*error here*/
	return;
}
/*call the rest now*/
d9cmpsce(clksceary,dptimes,rtcode);
return;
}
/**************************************************************************/
static
d9cmpsce(clksceary,dptimes,rtcode)

long	int	*clksceary;
DBL	*dptimes;
char	*rtcode;

{

scet_fnd = TRUE; 
clk_value = scet;
clk_field_indx = CLKSCET_FIELD;
/*********************************************************
 starting with zero it is the 1st element or field in the record
		  rec = sclk,scet,A1,
***********************************************************/
val_clk_blk_indx = 1;	/*starting with zero it is the 1st element of 
			  BLKSCLK
			  struct is = sclk,scet,ert and others*/
fndclkblk(rtcode); /*find block of data*/
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
fndclkrec(rtcode); /*find record to do*/
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
/**********************************************************************
struct	blksclk {	  blocks of 100 sclk scet records start at 
			  begin of block
	DBL	sclk;
	DBL	scet;
	int	nmbr_recs;	  number of records in this block
	long	int	offset;   offset for seeking to place in file
	char	*prev;		  points to previous structure
	char	*next;		  points to next structure.
};

*********************************************************************/
/*find sclk from scet*/
d9clkfse(rtcode);
if (rtcode[3] == 'X')
{
	return;
}
DPSCLK = sclk;
d9cnvclk(clksceary,dptimes,rtcode);
return;
}

/**********************************************************************/
static
d9clkinp(type_in,clksceary,dptimes,rtcode)

char	*type_in;
long	int	*clksceary;
DBL	*dptimes;
char	*rtcode;

{

DBL	dbl;
char	*sclk_pntr;

/*sclk inputs here*/

/*********************************************************************
**	     7. SCLK - Inputs in clksceary[6] etc, mission dependent.
**	     8. SCLKR - Raw SCLK placed in clksceary[10], etc. msn 
**	        dependent.
*********************************************************************/
if (type_in[4] == 'R'  ||  type_in[4] == 'r')
{
	/*raw data input*/
	strcpy(typein,"RAW");
	memcpy((char *)&SCLKRAW,(char *)&SSCLKRAW,6);
	diclkcnv(carray,NULL,&DPSCLK,strng_pntr,typein,NULL,
		 NULL,rtcode);
	if (rtcode[3] == 'X')
	{
		/*error*/
		return;
	}
	d9cary2sary(clksceary,rtcode);
}
else
{
	strcpy(typein,"SCLK");
	d9sary2cary(clksceary,rtcode);
	diclkcnv(carray,NULL,&DPSCLK,strng_pntr,typein,NULL,
		 NULL,rtcode);
	if (rtcode[3] == 'X')
	{
		/*error*/
		return;
	}
	memcpy((char *)&SSCLKRAW,(char *)&SCLKRAW,6);
	d9cary2sary(clksceary,rtcode);
}
d9cmpclk(clksceary,dptimes,rtcode);
return;
}
/********************************************************************/
static
d9dpclki(clksceary,dptimes,rtcode)

long	int	*clksceary;
DBL	*dptimes;
char	*rtcode;

{

sclk = DPSCLK;
d9cnvclk(clksceary,dptimes,rtcode);
d9cmpclk(clksceary,dptimes,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
return;
}
/********************************************************************/
static
d9cmpclk(clksceary,dptimes,rtcode)

long	int	*clksceary;
DBL	*dptimes;
char	*rtcode;

{
 
sclk_fnd = TRUE;
clk_value = sclk;
clk_field_indx = CLKSCLK_FIELD; /*starting with zero it is the zeroth 
		  element or field in the record
		  rec = sclk,scet,A1 */
val_clk_blk_indx = 0;	/*starting with zero it is the zeroth element 
			  of SCLKBLK
			  struct is = sclk,scet,ert and others*/
fndclkblk(rtcode); /*find block of data*/
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
fndclkrec(rtcode); /*find record to do*/
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
d9sefclk(rtcode);
if (rtcode[3] == 'X')
{
	return;
}
DPSCET = scet;
d9cnvsce(clksceary,dptimes,rtcode);
if (rtcode[3] == 'X')
{
	return;
}
return;
}
/******************************************************************************/		
static
d9sefclk(rtcode)
 
 char	*rtcode;
 
 {
 
 /* get scet from sclk value*/
 /* scet = A0 + A1*(sclk - SCLK) */
 
 scet = SCECLKREC->su.A0 + (sclk - SCECLKREC->sclk)*SCECLKREC->A1;
 
 return;
 }

/*****************************************************************************/ 
 static
 d9clkfse(rtcode)
 
 char	*rtcode;
 
 {
 
 /*find sclk fro scet*/
 /* sclk = SCLK + (scet - SCET)/A1*/
 
 sclk = SCECLKREC->sclk + (scet - SCECLKREC->su.scet)/SCECLKREC->A1;
 
 return;
 }
/*****************************************************************************/ 
static
fndclkblk(rtcode)

char	*rtcode;

{

DBL	*blk_clk_pntr;
DBL	*nxt_clk_pntr;
long	int	pos;
int	rtc;
long	int	fromftell;
/*find block for next whatever*/
/*val_in_blk_indx is the index into structure BLKSCLK for sclk, scet or
  ert field_indx is index into structure SCECLKREC for sclk, scet or 
  ert*/
  
if (CURRCLKBLK == (BLKSCLK *)NULL)
{
	clk_frwrd = TRUE;
	CURRCLKBLK = (BLKSCLK *)start_clk_blk;
}
else
{
	blk_clk_pntr = (DBL *)CURRCLKBLK;
	if (clk_value < blk_clk_pntr[val_clk_blk_indx])
	{
		clk_frwrd = FALSE;
		if (sclk_roll_over != NULL && sclk_fnd)
		{
			/*ok find from roll over*/
			d9blkrlo(rtcode);
		}
	}
	else if (CURRCLKBLK->next == NULL)
	{
		/*we have found it*/
		if (sclk_roll_over != NULL && sclk_fnd)
		{
			/*ok find from roll over*/
			d9blkrlo(rtcode);
		}
		else
		{
			return;
		}
	}
	else
	{
		nxt_clk_pntr = (DBL *)CURRCLKBLK->next;
		if (clk_value < nxt_clk_pntr[val_clk_blk_indx])
		{
			/*we have the right block*/
			return;
		}
		if (sclk_roll_over != NULL && sclk_fnd)
		{
			/*ok find from roll over*/
			d9blkrlo(rtcode);
		}
		else
		{
			clk_frwrd = TRUE;
			CURRCLKBLK = NEXTCLKBLK;
		}
	}
}
if (!clk_frwrd)
{
	while (CURRCLKBLK != (BLKSCLK *)NULL)
	{
		/*go backwards to wahtever*/
		/*CURRCLKBLK has current info*/
		PREVCLKBLK = (BLKSCLK *)CURRCLKBLK->prev;
		if (PREVCLKBLK == (BLKSCLK *)NULL)
		{
			break;
		}
		CURRCLKBLK = PREVCLKBLK;
		blk_clk_pntr = (DBL *)CURRCLKBLK;
		if (clk_value >= blk_clk_pntr[val_clk_blk_indx])
		{
			/*ok it is here*/
			break;
		}
	}
}
else
{
	/*search forward here*/
	
	while (CURRCLKBLK != (BLKSCLK *)NULL)
	{
		/*go backwards to wahtever*/
		/*CURRCLKBLK has current info*/
		NEXTCLKBLK = (BLKSCLK *)CURRCLKBLK->next;
		if (NEXTCLKBLK == (BLKSCLK *)NULL)
		{
			break;
		}
		nxt_clk_pntr = (DBL *)CURRCLKBLK->next;	
		if (clk_value < nxt_clk_pntr[val_clk_blk_indx])
		{
			/*we have the right block*/
			break;
		}
		CURRCLKBLK = NEXTCLKBLK;
	}
}
#if (NOSEEK)
/*use buffered management*/
/*get current buffer*/
clkbufinfo.bufnumber = CURRCLKBLK->bufcount;
clkbufinfo.bufrmode = USE_BUFINDX;
rtc = sfbufman(&clkbufinfo,&rdinfclk);
if (rtc < 0)
{
	strcpy(rtcode,err_rtrn_code[BFM_INDX]);
	return;
}
sclkarray = &clkbufinfo.buf[(int)CURRCLKBLK->offset];
number_clk_in_curr_blk = CURRCLKBLK->nmbr_recs;
curr_blk_size_clk = CURRCLKBLK->nmbr_recs * rec_size_clk;
#else
/*read in proper block*/
/*firstposition for read*/
fromftell = SFFTELL(fdsceclk);
pos = CURRCLKBLK->offset - fromftell;	
rtc = SFFSEEK(fdsceclk,pos,SFSEEK_CUR);
if (rtc)
{
	strcpy(rtcode,err_rtrn_code[SEK_INDX]);
	return;
}
/*ok it is there*/
/*now read it in*/
rtc = SFFREAD(fdsceclk,sclkarray,CURRCLKBLK->nmbr_recs*rec_size_clk);
if (rtc <= 0)
{
	strcpy(rtcode,err_rtrn_code[CLR_INDX]);
	return;
}
curr_clk_seek_pos = CURRCLKBLK->offset + rtc;
#endif /*end of SEEK NO SEEK */
/*compute number of records in this block*/
number_clk_in_curr_blk = rtc/rec_size_clk;
curr_blk_size_clk = rtc;
curr_clk_rec = NULL;
prev_clk_rec = NULL;
next_clk_rec = NULL;
return;
}
/**************************************************************************/
static
d9blkrlo(rtcode)

char	*rtcode;

{


/*ok we have some roll over so now what
  find the block that this is between*/

CURRCLKBLK = (BLKSCLK *)start_clk_blk;
CURR_ROLL = (ROLL_OVER *)sclk_roll_over;

while (TRUE)
{

	if (CURRCLKBLK->sclk < CURR_ROLL->prev_sclkrec.sclk)
	{
		/*ok this is in the ball park*/
		clk_frwrd = TRUE;
		return;
	}
	CURRCLKBLK = (BLKSCLK *)CURR_ROLL->roll_over_blk;
	if (CURR_ROLL->next == NULL)
	{
		clk_frwrd = TRUE;
		return;
	}
	CURR_ROLL = (ROLL_OVER *)CURR_ROLL->next;
}
}
/**********************************************************************/
static
fndclkrec(rtcode)

char	*rtcode;

{

DBL	*curr_clk_pntr;
DBL	*next_clk_pntr;
long	int	next_rec_add;
long	int	last_rec_add;
	
if (curr_clk_rec == NULL)
{
	clk_frwrd = TRUE;
	curr_clk_rec = sclkarray;
	SCECLKREC = (SCLKR *)curr_clk_rec;
}
else
{
	curr_clk_pntr = (DBL *)curr_clk_rec;
	if (curr_clk_pntr[clk_field_indx] > clk_value)
	{
		clk_frwrd = FALSE;
	}
	else if (next_clk_rec == NULL)
	{
		/*found*/
		return;
	}
	else 
	{
		next_clk_pntr = (DBL *)next_clk_rec;
		if (clk_value < next_clk_pntr[clk_field_indx])
		{
			/*found */
			return;
		}
		clk_frwrd = TRUE;
		curr_clk_rec = next_clk_rec;
		SCECLKREC = (SCLKR *)curr_clk_rec;
	}
}
if (!clk_frwrd)
{
	/*back wards search*/
	while (curr_clk_rec != NULL)
	{
		prev_clk_rec = curr_clk_rec - rec_size_clk;
		if ((long int)prev_clk_rec < (long int)sclkarray)
		{
			prev_clk_rec = NULL;
			next_clk_rec = curr_clk_rec + rec_size_clk;
			break;
		}
		next_clk_rec = curr_clk_rec;
		curr_clk_rec = prev_clk_rec;
		curr_clk_pntr = (DBL *)curr_clk_rec;
		if (!curr_clk_pntr[CLKA1_FIELD])
		{
			continue;
		}
		if (curr_clk_pntr[clk_field_indx] <= clk_value)
		{
			/*found*/
			prev_clk_rec = curr_clk_rec - rec_size_clk;
			break;
		}
	}
}
else
{
	last_rec_add = (long int)&sclkarray[curr_blk_size_clk];
	/*forward*/
	while (curr_clk_rec != NULL)
	{
		next_clk_rec = curr_clk_rec + rec_size_clk;
		next_rec_add = (long int)next_clk_rec;
		if (next_rec_add >= last_rec_add)
		{
			/*over out of space*/
			next_clk_rec = NULL;
			break;
		}
		next_clk_pntr = (DBL *)next_clk_rec;
		if (!next_clk_pntr[CLKA1_FIELD])
		{
			prev_clk_rec = curr_clk_rec;
			curr_clk_rec = next_clk_rec;
			continue;
		}
		if (next_clk_pntr[clk_field_indx] > clk_value)
		{
			/*found*/
			prev_clk_rec = curr_clk_rec - rec_size_clk;
			break;
		}
		prev_clk_rec = curr_clk_rec;
		curr_clk_rec = next_clk_rec;
	}
}
SCECLKREC = (SCLKR *)curr_clk_rec;
return;
}
/*****************************************************************************/
static
d9cnvsce(clksceary,dptimes,rtcode)

long	int	*clksceary;
DBL	*dptimes;
char	*rtcode;

{


/*convert dp seconds of scet to long ints*/

switch (CLKEPS.epoch_indx)
{
	case DPEP_OF_CHOICE:
		/*epoch of choice*/
		strcpy(typein,"DPEPOCH");
	break;
	case DP2000_INDX:
		/* seconds past 2000*/
		strcpy(typein,"DP2000");
	break;
	default:
		/* seconds past 50*/
		strcpy(typein,"DP50");
	break;
}
dpsecs[CLKEPS.epoch_indx] = scet;
dictmcnv(tarray,mon,NULL,dpsecs,NULL,typein,NULL,NULL,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
SCETSECS = tarray[tepoch_indx]; /*seconds past epoxh*/
SCETFRACS = tarray[tepoch_indx+1]; /*fractional seconds*/
SCETMILLS = MILLSEC; /*milliseconds*/
return;
}
/****************************************************************/
static
d9cnvclk(clksceary,dptimes,rtcode)

long	int	*clksceary;
DBL	*dptimes;
char	*rtcode;

{

/*call whatever is neccessary*/
strcpy(typein,"DPSCLK");
diclkcnv(carray,NULL,&sclk,NULL,typein,NULL,NULL,rtcode);
if (rtcode[3] == 'X')
{
	/*error*/
	return;
}
/*place output in proper place*/
d9cary2sary(clksceary,rtcode);
return;
}
/*****************************************************************/
static
d9cnvfhx(strng,intout,len)

char	*strng;
long	int	*intout; /*will cast to specifications*/
int	len;

{

/*convert from hex chracters*/
int	j;
int	k;
char	hexchars[3];
int	hexint;
union
{
	long	int	li;
	short	int	si[2];
	char	ch[4];
} iu;

/*len is 4 (bytes) or 2 (bytes)*/
iu.li = 0;
if (len == 4)
{
	k = 0;
}
else
{
	k = 2;
}
for ( j = 0 ; j < len ; j++)
{
	strncpy(hexchars,(char *)&strng[j*2],2);
	/*find hex  from character string*/
	sscanf(hexchars,"%X",
	       &hexint);
	iu.ch[vax4_or_othr[k]] = hexint;
	k++;
}
intout[0] = iu.li;
return;
}
/*************************************************************************/
static
d9ridquotes(value,value_pntr,value_len)

char	*value;
char	**value_pntr;
int	*value_len;

{

if (value[0] == '\''   ||   value[0] == '"')
{
	/*this is a string */
	/*get rid of leading and trailing quotes*/
	value_pntr[0]++;
	value_len[0] -= 2;
	value_pntr[0][value_len[0]] = '\0';
} 	
return;
}
		
/*******************************************************************/
static
d9cary2sary(clksceary,rtcode)

long	int	*clksceary;
char	*rtcode;

{

	                  
switch (msn_case_nmbr)
{
	case ULS_INDX:
		SULSCOUNT = ULSCOUNT;
		SULSFRAC = ULSFRAC;
		SULSMILLS = ULSMILLS;
	break;
	case GLL_INDX:
		SGLLRIMCNT = GLLRIMCNT;
		SGLLMOD91 = GLLMOD91;
		SGLLMOD10 = GLLMOD10;
		SGLLMOD8 = GLLMOD8;
	break;
	case VGR_INDX:
		SVGRMOD16 = VGRMOD16;
		SVGRMOD60 = VGRMOD60;
		SVGRLNCNT = VGRLNCNT;
	break;
	default:
	break;
}
memcpy((char *)&SSCLKRAW,(char *)&SCLKRAW,6);
return;
}
/*************************************************************************/
static
d9sary2cary(clksceary,rtcode)

long	int	*clksceary;
char	*rtcode;

{

	                  
switch (msn_case_nmbr)
{
	case ULS_INDX:
		ULSCOUNT = SULSCOUNT;
		ULSFRAC  = SULSFRAC;
		ULSMILLS = SULSMILLS;
	break;
	case GLL_INDX:
		GLLRIMCNT = SGLLRIMCNT;
		GLLMOD91  = SGLLMOD91;
		GLLMOD10  = SGLLMOD10;
		GLLMOD8   = SGLLMOD8;
	break;
	case VGR_INDX:
		VGRMOD16 = SVGRMOD16;
		VGRMOD60 = SVGRMOD60;
		VGRLNCNT = SVGRLNCNT;
	break;
	default:
	break;
}
memcpy((char *)&SCLKRAW,(char *)&SSCLKRAW,6);
return;
}

