/* Compile flag used to determine mission */
#define _VGR_TED_
#undef _GLL_TED_
#undef _ULS_TED_


#ifdef _GLL_TED_
#define  MISSION_HDR "gll_tds_if.h"
#endif
#ifdef _VGR_TED_
#define  MISSION_HDR "vgr_tds_if.h"
#endif
#ifdef _ULS_TED_
#define  MISSION_HDR "uls_tds_if.h"
#endif

/* $Log: ted_tds_if.c,v $
 * Revision 1.14  12-08-2021 - Ray Bambery
 * for Centos 7 on pistol.jpl.nasa.hov
 * add #include <time.h> for messages"
 *  warning: implicit declaration of function ‘gmtime
 *  warning: implicit declaration of function ‘strftime
 *  add prototpes for XmTextReplace, XmTextGetInsertionPosition
 *
 * Revision 1.13  11-18-2020 - Ray Bambery
 * removed debug printf statements
 *
 * Revision 1.12  1996/07/11  18:55:30  ted
 * replaced peng with pkt_eng
 *
 * Revision 1.11  1996/01/30  00:47:37  ted
 * add cm crap
 *
 * Revision 1.11  1996/01/30  00:47:37  ted
 * add cm crap
 *
 * Revision 1.10  1996/01/18  19:07:38  ted
 * fixing new cm crap
 *
 * Revision 1.9  1996/01/18  05:37:04  ted
 * added new cm crap
 *
 * Revision 1.8  1995/02/10  23:13:01  ted
 * correct Time String processing
 *
 * Revision 1.8  1995/02/10  23:13:01  ted
 * correct Time String processing
 *
 * Revision 1.7  1995/02/01  23:02:40  ted
 * added processing for TESTBED_SC_QUERY_ID environment variable
 *
 * Revision 1.7  1995/02/01  23:02:40  ted
 * added processing for TESTBED_SC_QUERY_ID environment variable
 *
 * Revision 1.6  1995/01/31  23:47:42  ted
 * corrected sc_id parameter conversion
 *
 * Revision 1.5  1995/01/21  01:33:16  ted
 * always display record total when closeInputFileDescriptor is called
 *
 * Revision 1.2  1995/01/21  01:30:43  ted
 * *** empty log message ***
 *
 * Revision 1.4  1995/01/19  03:06:21  ted
 * Force galileo spacecraft ID to 87 when Test Bed is selected
 *
 * Revision 1.3  1995/01/12  22:42:00  ted
 * Initialize record_count to 0
 *
 * Revision 1.2  1995/01/09  22:43:44  ted
 * Original Delivery
 *
 * Revision 1.1  1995/01/12  18:52:28  ted
 * Initial revision
 *
 * Revision 1.1  1995/01/04  17:18:25  ted
 * Initial revision
 * */
/* RCS message propagation: */
static  char    const rcsid[] = "@(#)$Id: ted_tds_if.c,v 1.12 1996/07/11 18:55:30 ted Exp $";
/************************************************************************/
/*	Date		Author		Notes				*/
/*    3AUG94		T Specht	Original Delivery		*/
/*    18OCT94		T Specht	Added Galileo Mission		*/
/*    29NOV94		T Specht	Added Ulysses Mission		*/
/*************************************************************************
* To use the Telemetry Extraction and Distribution libraries requires
* calls to the following routines:
**************************************************************************
*
* Non motif routines will write to standard out, routines must be called
* in the order listed below.
*
**************************************************************************
* This routine will initialize variables required to connect to TDS, all
* messages will be written to standard out:
*
*	void initializeTedNonMotifCommon(sc_id_test_enabled);
*	int sc_id_test_enabled When set to true enables the SC ID
*	of SFDUs from TDS to be verified.  the incoming SC ID field of the
*	SFDU will be compared to the sc_id parameter passed in the subroutine
*	buildTdsFrameQuery or buildTdsChannelQuery.
*
**************************************************************************
* This routine will initialize variables required to connect to TDS, all
* messages will be written to a Motif scrolling log widget:
*
*       void initializeTedMotifCommon(scrolling_log,sc_id_test_enabled);
*	char*	scrolling_log	the Motif scrolling text widget ID.
*	int sc_id_test_enabled When set to true enables the SC ID
*	of SFDUs from TDS to be verified.  the incoming SC ID field of the
*	SFDU will be compared to the sc_id parameter passed in the subroutine
*	buildTdsFrameQuery or buildTdsChannelQuery.
*
**************************************************************************
* This routine will open the specified data source:
*
*       int openDataSource(source_name,source_id);
*       char *source_name;( either a file name or TDS host name)
*       int  source_id; ( 0 to specify a disk file or 6666 for the TDS socket)
*       Returns 0 if open succeeds else -1 if error occurs.
*
* ************************************************************************
* This routine will read a TDS query file from disk if a file name is specified
* else if a null pointer is passed then the query was built by the ted_query
* code.  The query will be passed to TDS:
*
*       int processTdsQuery(query_file_name);
*       char *query_file_name;
*       query_file_name is NULL if query is already in tds_query_buffer,
*       tds_if.h contains tds_query_buffer and tds_query_length declarations.
*       Returns 0 if open succeeds else -1 if error occurs.
*
* ************************************************************************ 
* This routine will create a TDS Frame query from the parameter list
*
*	void buildTdsFrameQuery(name,mission,sc_id,time_type,
*	start_time, stop_time, time_order, tds_source, data_type);
*	char*	name:		A string specifying the PVL Name Field.
*				(no blanks in name field allowed)
*	char*	mission:	A string specifying the Query Mission ID.
*				("GALILEO","VOYAGER_1","VOYAGER_2","ULYSSES")
*	char*	sc_id:		A string specifying the Query Spacecraft ID.
*				(Valid SC ID=GLL,SGLL,VOYAGER_1,VOYAGER_2,etc.)
*	char*	time_type:	A string specifying the Query time Types.
*				("ERT","SCLK","RCT","SCET")
*	char*	start_time:	A string specifying the Query Start Time.
*				(ERT=YEAR-DOYTHH:MM:SS,SCLK varies by mission)
*	char*	stop_time:	A string specifying the Query Stop Time.
*				(same format as start time)
*	char*	time_order:	A string specifying the Query time order.
*				("ERT" or "SCLK")
*	char*	tds_source:	A string specifying the Query TDS source.
*				("NERT" or "RT" or "NERT,RT" for both)
*	char*	data_type:	A string specifying the Query Data Types.
*				(i.e. "eng,img,mon,qqc" see gll_tds_if.h or
*				 vgr_tds_if.h)
*
* *********************************************************************** 
* This routine will create a TDS channel query from the parameter list
*
*	void buildTdsChannelQuery(name,mission,sc_id,time_type,
*	start_time, stop_time,time_order,tds_source,channel_range,sample_mode);
*	char*	name:		A string specifying the PVL Name Field.
*				(no blanks in name field allowed)
*	char*	mission:	A string specifying the Query Mission ID.
*				(GALILEO VOYAGER_1 VOYAGER_2)
*	char*	sc_id:		A string specifying the Query Spacecraft ID.
*				(Valid SC ID 77,87,31,32 etc.)
*	char*	time_type:	A string specifying the Query Time Types.
*				(ERT,SCLK,RCT,SCET)
*	char*	start_time:	A string specifying the Query Start Time.
*				(ERT=YEAR-DOYTHH:MM:SS,SCLK varies by mission)
*	char*	stop_time:	A string specifying the Query Stop Time.
*				(same format as start time)
*	char*	time_order:	A string specifying the Query Time Order.
*				("ERT" or "SCLK")
*	char*	tds_source:	A string specifying the Query TDS data source.
*				("NERT" or "RT" or "NERT,RT" for both)
*	char*	channel_range:	A string specifying the Query Channel range.
*				("E-0..E-1000,E-2000..E-3000")
*	char*	sample_mode:	A string specifying the Channel sample mode.
*				("ON_SAMPLE","ON_CHANGE","CHANGES_ONLY",
*				 "DELTA_TIME", or "FREQUENCY")
*
* ***********************************************************************
* This routine will write the query to TDS.
*
*       int writeQueryToTds(void);
*       Returns 0 if open succeeds else -1 if error occurs.
*
* **********************************************************************
* This routine will read any input from TDS and return the following codes:
*
*       char *processTdsData();
*       Returns NULL if no data received from TDS, -1 if End of Data received
*       from TDS or an address of a SFDU.
*
**************************************************************************/
#define _TED_INIT_
#include "ted.h"
#include MISSION_HDR
#include  <Xm/Xm.h>
#include <time.h>
void    buildTdsFrameQuery(char*,char*,char*,char*,char*,
						char*,char*,char*,char*);
void    buildTdsChannelQuery(char*,char*,char*,char*,char*,char*,
				char*,char*,char*,char*);
void    buildTdsQueryFields(char*,int);
void 	initializeTedNonMotifCommon(int);
int	readMessageForSpecifiedLength(int);
int	openDataSource(char*,int);
void	initializeTedMotifCommon(Widget,int);
void    closeInputFileDescriptor(void);
char*	processTdsData(void);
void    copyStringToLog(char*);
int	readData(char*,int);
int	processMessageFromTds(void);
int	processSfduFromTds(void);
int	processQueryFromTds(void);
int	readMessageForSpecifiedLength(int);
int	processMessageField(char*);
int	processMessageTextField(void);
int	alignPointerWithLine(void);
int	readSfdu(void);
void	writeSfduToLogFile(char*,int);
void	writeMessageToLogFile(char*,int);
void	writeQueryToLogFile(char*,int);
void	processEndOfDataMessage(int);
int	setupClientSocket(char*,int);
int	incrementRecordTypeReceived(void);
void	displayRecordTypesReceived(void);
int	verifySpacecraftID(void);

void XmTextReplace(Widget widget, XmTextPosition from_pos, XmTextPosition to_pos, char* value);
XmTextPosition XmTextGetInsertionPosition(Widget widget);

/***************************************************/
/***** The Following section processes input *******/
/***************************************************/
/* scratch variables */
int temp_length;
int read_length;
/* offset to the read position in input_data */
int read_position;
/* offset to the write position in input_data */
int write_position;
/* pointer to  matching string in input_data */
char *string_found;

Widget log_widget;

/****************************************************/
/* Read record from specified input source, then    */
/* determine the type of record received, TDS Query */
/* Echo, TDS Status/Error message, and/or a SFDU.   */
/* Any other type is unknown                        */
/****************************************************/
int readSfdu()
{
int sfdu_stat;
//printf ("ted_tds_if:  readSfdu Entry\n");
read_position = write_position = 0;
sfdu_pointer = NULL;
while (TRUE)
   {
   memset((char *)&input_data,'\0',MAX_TDS_QUERY_LENGTH);
   temp_length = LABEL_LENGTH;
   while (temp_length GT 0)
   { /* determine record type from record label */
      read_length=readData((char *)(&input_data.byte_buffer[write_position]),
                                                      temp_length);
      if (read_length LE 0)
         return(END_OF_DATA);
      else
         {
         temp_length -= read_length;
         write_position += read_length;
         }
      }
   /* NJPL prefix indicates a TDS Status/Error Message or a SFDU */
   if (strncmp((char *)(&input_data.byte_buffer[read_position]),NJPL,4) EQ 0)
      {/* NJPL Header Received */
      if (strncmp((char *)(&input_data.byte_buffer[read_position]),
             TDS_MESSAGE_LABEL,LABEL_LENGTH) EQ 0)
         {/* Status/Error message from TDS */
         if (processMessageFromTds() EQ ERROR_STATUS)
            return(END_OF_DATA);
         else
            return 0;       /* Not NULL */
         }
      else
         { /* SFDU from TDS */
         sfdu_stat = processSfduFromTds();

        if (sfdu_stat EQ ERROR_STATUS)
            return(END_OF_DATA);
         else
            return(SFDU_RECEIVED);
         }
      }
   else
      {/* TDS query label */
      if (strncmp((char *)(&input_data.byte_buffer[read_position]),
                             QUERY_ECHO_LABEL,write_position) EQ 0)
         {/* Query is echoed by TDS */
         if (processQueryFromTds() EQ ERROR_STATUS)
             return(END_OF_DATA);
         }/* there is always some msgs from TDS appended to */
          /* the query echo so continue loop until they are */
	  /* processed. */
      else
         {/* TDS End of Data label */
         if (strncmp((char *)(&input_data.byte_buffer[read_position]),
                             SFDU_EOD_LABEL,write_position) EQ 0)
            {/* normal end of data received */
            processEndOfDataMessage(FALSE);
            return(END_OF_DATA);
            }
         else
            {/* OH NO! Dont Know what this could be */
            /* write this garbage to message file   */
            if (message_capture)
               writeMessageToLogFile(
		      (char *)(&input_data.byte_buffer[read_position]),
							write_position);
            sprintf(log_string,"Unknown Record Received\n");
            copyStringToLog(log_string);
            sprintf(log_string,"REFUSING ALL DATA UNTIL RESTARTED\n");
            copyStringToLog(log_string);
            closeInputFileDescriptor();
            return(END_OF_DATA);
            }
         }
      }
   }
}
/****************************************************************************/
#define FOUND		999
int processMessageFromTds()
{
extern int read_position,temp_length,write_position;
extern char	*string_found;
int		trailer_found = FALSE;
int		end_object_field = FALSE;
int		message_field_index=0;
/* write the current label to the message log and continue */
if (message_capture)
   writeMessageToLogFile((char *)(&input_data.byte_buffer[read_position]),
				LABEL_LENGTH);
/* finished with label so point past it */
read_position += LABEL_LENGTH;
while (end_object_field NE FOUND)
   {
   switch (message_field_index)
      {
      case 0:
      if (readMessageForSpecifiedLength(OBJECT_LENGTH) EQ ERROR_STATUS)
         return(ERROR_STATUS);
      else
         end_object_field = processMessageField(OBJECT_LABEL);
      message_field_index=1;
      break;

      case 1:
      if (readMessageForSpecifiedLength(MESSAGE_TYPE_LENGTH) EQ ERROR_STATUS)
         return(ERROR_STATUS);
      else
         end_object_field = processMessageField(MESSAGE_TYPE_LABEL);
      message_field_index=2;
      break;

      case 2:
      if (readMessageForSpecifiedLength(MESSAGE_NUM_LENGTH) EQ ERROR_STATUS)
         return(ERROR_STATUS);
      else
         end_object_field = processMessageField(MESSAGE_NUM_LABEL);
      message_field_index=3;
      break;

      case 3:
      if (readMessageForSpecifiedLength(MESSAGE_LENGTH) EQ ERROR_STATUS)
         return(ERROR_STATUS);
      else
         end_object_field = FOUND;
      break;
      }
   if (end_object_field EQ ERROR_STATUS)
      return(ERROR_STATUS);
   }
if (processMessageTextField() EQ ERROR_STATUS)
   return(ERROR_STATUS);
while (NOT trailer_found)
   {
   string_found = strstr((char *)(&input_data.byte_buffer[read_position]),
				TDS_MESSAGE_TRAILER);
   if (string_found)
      trailer_found = TRUE;
   else
      {/* read some more data and try again */
      if (readMessageForSpecifiedLength(1) EQ ERROR_STATUS)
         return(ERROR_STATUS);
      }
   }      
read_position = ((unsigned int)(string_found) -
                  (unsigned int)(&input_data.byte_buffer[0])) + LABEL_LENGTH;
return(SUCCESS_STATUS);
}
/**************************************************************************************/
int processMessageField(field_name)
char *field_name;
{
extern int 	read_position,temp_length,write_position;
extern char     *string_found;
string_found = NULL;
while(NOT string_found)
   {
   string_found = strstr(
                (char *)(&input_data.byte_buffer[read_position]),field_name);
   if (string_found)
      {/* adjust pointer to start of message field */
      temp_length = (unsigned int)string_found -
		(unsigned int)(char *)&input_data.byte_buffer[read_position];
      read_position += temp_length;
      }
   else
      {/* possible incorrect message format received */
      /* check for END_OBJECT field */
      string_found = strstr(
	    (char *)(&input_data.byte_buffer[read_position]),END_OBJECT_LABEL);
      if (string_found)/* return to process message text */
         return(FOUND);
      else
         {/* read more from the input queue */
         if (readMessageForSpecifiedLength(1) EQ ERROR_STATUS)
            return(ERROR_STATUS);
         }
      }
   }
if (alignPointerWithLine() EQ ERROR_STATUS)
   return(ERROR_STATUS);
return(SUCCESS_STATUS);
}
/*************************************************************************************/
int alignPointerWithLine()
{
extern int	read_position,temp_length;
extern char	*string_found;
char		*new_line_found;
int		i;
string_found = NULL;
while (NOT string_found)
    { /* compute pointer to next line */
    string_found = strchr(
       (char *)(&input_data.byte_buffer[read_position]),';');
    if (string_found)
       {/* bump pointer past ';' */
       temp_length = ((unsigned int)(string_found) -
          (unsigned int)(&input_data.byte_buffer[read_position])) + 1;
       read_position += temp_length;
       for (i=0;i<2;i++)
           {
           new_line_found = strchr(
		(char *)(&input_data.byte_buffer[read_position]),NEW_LINE);
           if (new_line_found)
              {
              temp_length = ((unsigned int)(new_line_found) -
                 (unsigned int)(&input_data.byte_buffer[read_position])) + 1;
              read_position += temp_length;
              break;
              }
           else
              {
              if (readMessageForSpecifiedLength(1) EQ ERROR_STATUS)
                  return(ERROR_STATUS);
               }
          }
       }
    else
       {
       if (readMessageForSpecifiedLength(1) EQ ERROR_STATUS)
          return(ERROR_STATUS);
       }
    }
return(SUCCESS_STATUS);
}
/**********************************************************************************/
/************************************************************/
/* The TDS message text will precede the END_OBJECT string  */
/* at this point.  The problem is there is an indeterminate */
/* number of message lines.  So in addition to NEW_LINE I   */
/* have to search for the END_OBJECT string.		    */
/************************************************************/
int processMessageTextField()
{
extern int	read_position,temp_length;
extern char	*string_found;
int i;
char		*line_found;
char		*message_found;
char		*equal_found;
int		end_object = FALSE;
while (NOT end_object)
    {/* search for message field */
    message_found = strstr(
			(char *)(&input_data.byte_buffer[read_position]),MESSAGE_LABEL);
    if (message_found)
       {/* search for = */
       temp_length = (unsigned int)message_found -
			(unsigned int)(&input_data.byte_buffer[read_position]);
       read_position += temp_length;
       equal_found = strchr(
			(char *)(&input_data.byte_buffer[read_position]),'=');
       if (equal_found)
          {
          temp_length = ((unsigned int)equal_found - 
			(unsigned int)(&input_data.byte_buffer[read_position])) + 1;
          read_position += temp_length;
          }
       }
    /* else search for NEW_LINE */
    line_found = strchr(
          (char *)(&input_data.byte_buffer[read_position]),NEW_LINE);
    if (line_found)
       {/* process the message text */
       temp_length = (unsigned int)(line_found) -
              (unsigned int)(&input_data.byte_buffer[read_position]);
       /* copy text for display */
       strncpy(scratch,(char *)(&input_data.byte_buffer[read_position]),
                                temp_length);
       /* terminate string and bump pointer past NEW_LINE character */
       scratch[temp_length++]=NULL;
       /* display text */
       sprintf(log_string,"TDS:%s\n",scratch);
       copyStringToLog(log_string);
       /* compute beginning of next line */
       read_position += temp_length;
       }
    /* now search for END_OBJECT string */
    string_found = strstr(
		(char *)(&input_data.byte_buffer[read_position]),"END_");
    if (string_found AND NOT line_found)
       {/* search for END_OBJECT */
       temp_length = (unsigned int)(string_found) -
			(unsigned int)(&input_data.byte_buffer[read_position]);
       read_position += temp_length;
       if (processMessageField(END_OBJECT_LABEL) EQ ERROR_STATUS)
          return(ERROR_STATUS);
       else
          end_object = TRUE;
       }
    else
       {/* read more data */
       if (readMessageForSpecifiedLength(1) EQ ERROR_STATUS)
          return(ERROR_STATUS);
       }
    }
return(SUCCESS_STATUS);
}
/**********************************************************************************/
int readMessageForSpecifiedLength(length)
int length;
{
extern int	write_position;
int read_length;
while (length GT 0)
   {
   read_length = readData(
		 (char *)(&input_data.byte_buffer[write_position]),length);
   if (read_length EQ ERROR_STATUS)
      return(ERROR_STATUS);
   else
      {
      if (message_capture)
	     writeMessageToLogFile(
               (char *)(&input_data.byte_buffer[write_position]),read_length);
      length -= read_length;
      write_position += read_length;
      }
   }
return(SUCCESS_STATUS);
}
/*********************************************************************************/
int processSfduFromTds()
{
extern int read_position, write_position;
extern int temp_length, read_length, byte_count;
extern char *string_found;
int	data_type_index;
/* write SFDU Label to log file */
if (data_capture) {
   writeSfduToLogFile((char *)(&input_data.byte_buffer[read_position]),
				LABEL_LENGTH);
   }
   
/* get length of SFDU from label after entire label has been read */
if (input_data.byte_buffer[read_position+SFDU_LENGTH_TYPE_OFFSET] EQ
						 ASCII_LENGTH_TYPE)
   {/* length is ascii */
   temp_length = 
                     (atoi((char *)(&input_data.byte_buffer
				[read_position+ASCII_LENGTH_OFFSET])));
   }
else
   {
   if (input_data.byte_buffer[read_position+SFDU_LENGTH_TYPE_OFFSET] EQ 
						BINARY_LENGTH_TYPE)
      {/* length is unsigned 16 bit number */
      temp_length =
	input_data.short_buffer[(read_position/2)+BINARY_LENGTH_OFFSET];
      }
   else
      {/* variable length type */
      if (input_data.byte_buffer[read_position+SFDU_LENGTH_TYPE_OFFSET] EQ
							VARIABLE_LENGTH_TYPE)
         {
         temp_length = processMessageFromTds();
         return(temp_length);
         }
      else
         {/* unknown length type */
         sprintf(log_string,"Invalid length id in Label of record #%d\n",
							record_count);
         copyStringToLog(log_string);
         sprintf(log_string,"Waiting for a connection reset\n");
         copyStringToLog(log_string);
         closeInputFileDescriptor();
         }
      }
   }
/* read remaining data */
while (temp_length GT 0)
   {
   read_length = readData((char *)(&input_data.byte_buffer[write_position]),
					temp_length);
   if (read_length LE 0)
      return(ERROR_STATUS);
   else
      {
      if (data_capture)
         writeSfduToLogFile((char *)(&input_data.byte_buffer[write_position]),
                                                        read_length);
      temp_length -= read_length;
      write_position += read_length;
      }
   }
data_type_index = incrementRecordTypeReceived();
if ((input_source_type EQ TDS) AND (sc_id_test_enabled) AND
   (data_type_index NE MON_DATA AND data_type_index NE CHMON_DATA AND
    data_type_index NE QQC_DATA AND data_type_index NE CHQQC_DATA))
    {/* verify SC ID of telemetry data from TDS only */
    if (verifySpacecraftID())
       return(END_OF_DATA);
    }
record_count++;
sfdu_pointer = (char *)(&input_data.byte_buffer[read_position]);
/* return pointer to start of SFDU */
return(SFDU_RECEIVED);
}
/*********************************************************************************************/
int processQueryFromTds()
{/* Query echoed by TDS */
extern int temp_length,read_position,write_position,record_boundary_found;
extern int read_length;
extern char *string_found;
int tds_message_read = FALSE;
struct timeval time_struct;
struct timezone time_zone_struct;
struct tm *gmt_time_struct_ptr;
int time_stat;
if (message_capture)
   writeMessageToLogFile((char *)(&input_data.byte_buffer[read_position]),
					LABEL_LENGTH);
/* we are finished with the query primary */
/* label so point past it		  */
read_position += LABEL_LENGTH;
/* read query */
if (readMessageForSpecifiedLength(tds_query_length-(2*LABEL_LENGTH)) EQ ERROR_STATUS)
   return(ERROR_STATUS);
read_position = write_position - LABEL_LENGTH;
while (tds_message_read EQ FALSE)
   {   /* search for end of query echo */
   string_found = strstr((char *)(&input_data.byte_buffer[read_position]),
                                        		SFDU_TRAILER);
   if (NOT string_found)
      {/* trailer not found check for NULL character */
      temp_length = strlen((char *)(&input_data.byte_buffer[0]));
      if (temp_length LT write_position)
         {/* bump read position past NULL character */
         read_position = temp_length+1;
         string_found = strstr(
		(char *)(&input_data.byte_buffer[read_position]),
                                                        SFDU_TRAILER);
         }
      if (NOT string_found)
         {/* read more data to find trailer */
         if (readMessageForSpecifiedLength(1) EQ ERROR_STATUS)
            return(ERROR_STATUS);
         }
      }
   if (string_found)/* end of query found! */
      {/* align offset with next label */
      read_position = ((unsigned int)(string_found) -
                  (unsigned int)(&input_data.byte_buffer[0])) + LABEL_LENGTH;
      /* set flag to exit loop */
      tds_message_read = TRUE;
      gettimeofday(&time_struct,&time_zone_struct);
      gmt_time_struct_ptr = gmtime(&time_struct.tv_sec);
      if (gmt_time_struct_ptr EQ NULL)
         {
         copyStringToLog("TED:gmtime function call failed\n");
         exit(1);
         }
      time_stat = strftime(time_string, TIME_STRING_LENGTH, "%Y-%jT%H:%M:%S",
                                                gmt_time_struct_ptr);
      if (time_stat LT 0)
         {/* garbage returned */
         copyStringToLog("TED:Time conversion failed\n");
         }
      else
         {
         sprintf(log_string,"Data stream starting at %s\n",time_string);
         copyStringToLog(log_string);
         return(1);
         }
      }
   }
return(SUCCESS_STATUS);
}
/******************************************************************************************/
/***********************************************************/
/* This routine processes the End Of Data Message from TDS */
/* by notifying the operator and closing the data stream   */
/***********************************************************/
/* error_status is false on normal end of data else true   */
/***********************************************************/
void processEndOfDataMessage(error_status)
int	error_status;
{/* END OF DATA Received from TDS */
extern int read_position,write_position;
struct timeval time_struct;
struct timezone time_zone_struct;
struct tm *gmt_time_struct_ptr;
/* return values */
int time_stat;

if (message_capture AND NOT error_status)
   writeMessageToLogFile((char *)(&input_data.byte_buffer[read_position]),
							write_position);
gettimeofday(&time_struct,&time_zone_struct);
gmt_time_struct_ptr = gmtime(&time_struct.tv_sec);
if (gmt_time_struct_ptr EQ NULL)
   {
   sprintf(log_string,"TED:gmtime function call failed\n");
   copyStringToLog(log_string);
   exit(1);
   }
time_stat = strftime(time_string, TIME_STRING_LENGTH, "%Y-%jT%H:%M:%S",
                                                gmt_time_struct_ptr);
if (time_stat LT 0)
   {/* garbage returned */
   sprintf(log_string,"TED:Time conversion failed\n");
   copyStringToLog(log_string);
   }
sprintf(log_string,"TED:End of data at %s after processing %d SFDUs\n",
                     	time_string,record_count);
copyStringToLog(log_string);
closeInputFileDescriptor();
}
/**********************************************************************************/
void writeSfduToLogFile(stream_ptr,write_length)
char *stream_ptr;
int  write_length;
{
int write_status;
write_status = write(sfdu_log_file, stream_ptr, write_length);
if (write_status LT ZERO)
   {
   perror("write to SFDU log");
   copyStringToLog("TED:write to SFDU log file failed\n");
   data_capture=FALSE;
   closeInputFileDescriptor();
   }
}
/**************************************************************************************/
void writeMessageToLogFile(stream_ptr,write_length)
char *stream_ptr;
int  write_length;
{
int write_status;
write_status = write(message_log_file, stream_ptr, write_length);
if (write_status LT ZERO)
   {
   perror("write to message log");
   copyStringToLog("TED:write to message log file failed\n");
   message_capture=FALSE;
   closeInputFileDescriptor();
   }
}
/*************************************************************************************/
void writeQueryToLogFile(stream_ptr,write_length)
char *stream_ptr;
int  write_length;
{
int write_status;
write_status = write(query_log_file, stream_ptr, write_length);
if (write_status LT ZERO)
   {
   perror("write to query log");
   copyStringToLog("TED:write to message log file failed\n");
   query_capture=FALSE;
   closeInputFileDescriptor();
   }
}
/***************************************************************************************/
/*************************************************/
/* Read data from predetermined input source, if */
/* an error occurs then notify operator and take */
/* appropriate action.                           */
/*************************************************/
int readData(buf_ptr,read_length)
int  read_length;
char *buf_ptr;
{
int read_stat,rcv_flags;
if (input_source_type EQ TDS)
   read_stat = t_rcv(input_fd,buf_ptr,read_length,&rcv_flags);
else
   read_stat = read(input_fd,buf_ptr,read_length);
if (read_stat < ZERO)
   {
   if (input_source_type EQ DISK_FILE)
      {
      perror("read error");
      sprintf(log_string,"Error while attempting to read file\n");
      }
   else
      {
      t_error("t_rcv failed");
      sprintf(log_string,"Error while reading socket\n");
      }
   copyStringToLog(log_string);
   processEndOfDataMessage(TRUE);
   }
else
   {
   if (read_stat EQ ZERO)
      {
      if (input_source_type EQ DISK_FILE)
         {
         sprintf(log_string,"TED: End of file reached\n");
         copyStringToLog(log_string);
         processEndOfDataMessage(TRUE);
         }
      else
         {/* source_type EQ TDS(TCP-IP) */
         perror("Socket read");
         sprintf(log_string,"Error while attempting to read input socket\n");
         copyStringToLog(log_string);
         processEndOfDataMessage(TRUE);
         }
      }
   }
return(read_stat);
}
/*************************************************************************************/
void copyStringToLog(msg_string)
char*	msg_string;
{
extern	Widget	log_widget;
XmTextPosition text_position;
if (motif_version)
   {
   /* get current text position */
   text_position = XmTextGetInsertionPosition(log_widget);
   /* insert message */
   XmTextReplace(log_widget, text_position, text_position, msg_string);
   }
else
   printf("%s\n",msg_string);
}
/*************************************************************************************/
void closeInputFileDescriptor()
{
displayRecordTypesReceived();
if (input_source_type EQ TDS
    AND input_fd NE END_OF_DATA
    AND NOT debug_enabled)
   {
   if (t_sndrel(input_fd))
      t_error("t_sndrel failed");
   else
      (void)t_rcvrel(input_fd);
   }
close(input_fd);
input_fd = END_OF_DATA;
}
/*************************************************************************************/
int setupClientSocket(host_name,port_number)
char   *host_name;
int    port_number;
/*********************************************************************/
/*                                                                   */
/* tcp/ip connect as client routine                                  */
/*                                                                   */
/*********************************************************************/
{
int    raw_socket;
raw_socket = t_open("/dev/tcp", O_RDWR, NULL);
if (raw_socket EQ ERROR_STATUS)
   {
   t_error("t_open");
   sprintf(log_string,"TED:Error on call to function t_open()\n");
   copyStringToLog(log_string);
   return(ERROR_STATUS);
   }
if (t_bind(raw_socket, NULL, NULL) LT 0)
   {
   t_error("t_bind");
   sprintf(log_string,"TED:Error on call to function t_bind()\n");
   copyStringToLog(log_string);
   return(ERROR_STATUS);
   }
if ((sndcall = (struct t_call *)t_alloc(raw_socket,T_CALL,T_ADDR)) EQ NULL)
   {
   t_error("t_alloc failed");
   sprintf(log_string,"TED:Error on call to function t_bind()\n");
   copyStringToLog(log_string);
   return(ERROR_STATUS);
   }
sndcall->addr.len = sizeof(struct sockaddr_in);
memset((void *)&servaddr, 0, sizeof(struct sockaddr_in));
servaddr.sin_family = AF_INET;
servaddr.sin_port = htons(port_number);
host_info = gethostbyname(host_name);
if (host_info EQ NULL)
   {
   sprintf(log_string,"TED:Error on call to function gethostbyname()\n");
   copyStringToLog(log_string);
   return(ERROR_STATUS);
   }
(void)memcpy((caddr_t)&servaddr.sin_addr,host_info->h_addr_list[0],
       host_info->h_length);
(void)memcpy(sndcall->addr.buf,&servaddr,sizeof(servaddr));
if (t_connect(raw_socket,sndcall,NULL)<0)
   {
   t_error("t_connect failed");
   sprintf(log_string,"TED (ted_tds_if): host_name = %s\n",host_name);
   copyStringToLog(log_string);
   sprintf(log_string,"TED (ted_tds_if): servaddr.sin_port = %d\n",servaddr.sin_port);
   copyStringToLog(log_string);
   sprintf(log_string,"TED (ted_tds_if):Error on call to function t_connect()\n");
   copyStringToLog(log_string);
   return(ERROR_STATUS);
   }
else
   {
   sprintf(log_string,"TED:Connected to TDS host %s successfully\n",host_name);
   copyStringToLog(log_string);
   getrlimit(RLIMIT_NOFILE, &resource);
   width = resource.rlim_cur;
   FD_SET(raw_socket,&read_fd_template);
   to.tv_sec = 1;  /* 1 second timer for select */
   return(raw_socket);
   }
}
/**************************************************************************************/
int openDataSource(source_name,source_id)
char	*source_name;
int	source_id;
{
//  printf (" ted_tds_if: openDataSource:  Entry\n");
if (source_id EQ TDS_TCPIP_PORT)/* connect to TDS */
   {
   input_fd = setupClientSocket(source_name,source_id);
   if (input_fd LT 0)
      return(-1);
   else
      input_source_type = TDS;
   }
else
   {
   if (NOT source_id)
      {/* open SFDU bytestream file */
      input_source_type = DISK_FILE;
      input_fd = open(source_name,O_RDONLY);
      if (input_fd LT 0)
         {
         sprintf(log_string,"TED:Unable to open SFDU file %s\n",source_name);
         copyStringToLog(log_string);
         perror("TED:open");
         return(-1);
         }
      else
         FD_SET(input_fd,&read_fd_template);
      }
   else
      {/* invalid parameter passed */
      copyStringToLog("TED:Invalid Source Id passed to openDataSource\n");
      return(-1);
      }
   }
return(0);
}
/**************************************************************************************/
char *processTdsData()
{/***********************************/
/* This is the TDS Polling routine. */
/************************************/
int sfdu_stat;
/* if socket not connected then return no data indicator
if (input_fd EQ END_OF_DATA)
   return(NULL);*/
/* when socket connected poll TDS */
//printf ("processTdsData Entry\n");
read_fd = read_fd_template;
//printf ("processTdsData Entry -  read_fd = %d\n",read_fd);
if (select(width,&read_fd,NULL,NULL,&to) EQ ERROR_STATUS)
   {
   perror("TED:select");
   closeInputFileDescriptor();
   return(NULL);
   }
else
   {/* check for TDS data */
   if (FD_ISSET(input_fd, &read_fd))
      {/* process TDS Data */
      sfdu_stat = readSfdu();
      /* if TDS SFDU received then return pointer to buffer */
      if (sfdu_stat EQ SFDU_RECEIVED)
         return(sfdu_pointer);
      else
         {/* check for End of Data from TDS */
//        printf ("processTdsData - END_OF_DATA\n");
         if (sfdu_stat EQ END_OF_DATA)
            return((char*)-1);
         else/* no SFDU received */
            return(NULL);
         }
      }
   }
return(NULL);
}
/***********************************************************************************/
#define NEW_LINE		0x0a
int processTdsQuery(query_file_name)
char *query_file_name;
{
int query_fd;
int data_present;
char query_byte; 
input_source_type = TDS;
//printf ("ted_tds_if: processTdsQuery Entry\n");
if (query_file_name)

   {/* read query from bytestream file */
//   printf ("ted_tds_if: processTdsQuery if loop\n");
   query_fd = open(query_file_name,O_RDONLY);
   data_present = TRUE;
   while (data_present)
      {/* read query into query buffer */
      data_present = read(query_fd,&query_byte,1);
//    printf ("ted_tds_if: data_present = %d\n",data_present);
      if (data_present LT 0)
         {
         sprintf(log_string,"TED:Read error on query file %s\n",
							query_file_name);
         copyStringToLog(log_string);
         perror("TED:processTdsQuery");
         return(-1);
         }
      else
         {
         if (data_present GT 0)
            {/* skip all null and new line characters */
            if (query_byte NE NULL AND query_byte NE NEW_LINE)
               tds_query_buffer[tds_query_length++] = query_byte;
            }
         else/* end of file reached */
            {
            close(query_fd);
            data_present = FALSE;
            }
         }
      }
    
   }
//printf ("ted_tds_if: processTdsQuery - no file\n");

/* tds_query_buffer already contains TDS query */
/*
 * if (writeQueryToTds() LT 0) {
    printf ("ted_tds_if: return (-2)\n");
   return(-2);
}
*/
tds_query_ready=TRUE;
//printf ("ted_tds_if: return(0)\n");
return(0);
}
/*********************************************************************************/
int writeQueryToTds()
{
int temp_length,read_position,write_length;
temp_length = tds_query_length;
read_position = 0;
while(temp_length)
   {/* do not write to TDS if debug_enabled */
   if (NOT debug_enabled)
      {
      write_length = write(input_fd,tds_query_buffer,temp_length);
      if (write_length LT 0)
         {/* error */
         copyStringToLog("TED:Error occured while writing query to TDS\n");
         perror("TED:TDS Query write");
         return(-1);
         }
      if (query_capture)/* write query to buffer */
         writeQueryToLogFile(
                (char *)(&tds_query_buffer[read_position]), write_length);
      read_position += write_length;
      temp_length -= write_length;
      }
   else
      {
      if (query_capture)/* write query to buffer */
         writeQueryToLogFile(
                (char *)(&tds_query_buffer[read_position]), temp_length);
      temp_length=0;
      }
   }
copyStringToLog("TED:Query sent to TDS\n");
tds_query_sent=TRUE;
return(0);
}
/*******************************************************************************/
void initializeTedNonMotifCommon(sc_test)
int	sc_test;
{
sc_id_test_enabled=sc_test;
motif_version=FALSE;
input_fd = END_OF_DATA;
scratch_ptr = scratch;
FD_ZERO(&read_fd);
FD_ZERO(&read_fd_template);
testbed_query = FALSE;
debug_enabled = FALSE;
message_capture=FALSE;
query_capture=FALSE;
data_capture=FALSE;
if (debug_enabled)
   {
   sfdu_log_file = open("tds_data", O_RDWR|O_CREAT|O_TRUNC,0644);
   if (sfdu_log_file EQ ERROR_STATUS)
      {
      copyStringToLog("TED:error on SFDU log file open\n");
      perror("TED:open");
      return;
      }
   message_log_file = open("tds_messages", O_RDWR|O_CREAT|O_TRUNC,0644);
   if (message_log_file EQ ERROR_STATUS)
      {
      copyStringToLog("TED:erro on TDS Message log file open\n");
      perror("TED:open");
      return;
      }
   }
/* set select width */
getrlimit(RLIMIT_NOFILE, &resource);
width = resource.rlim_cur;
to.tv_usec = 100;
}
/**************************************************************************************/
void buildTdsFrameQuery(name,mission,sc_id,time_type,start_time,stop_time,
			time_order, tds_source,data_type)
char *name,*mission,*sc_id,*start_time,*time_type;
char *stop_time,*time_order,*tds_source,*data_type;
{
char	*testbed_sc_query_id;

//printf ("ted_tds_if:   buildTdsFrameQuery Entry\n");
group_index=FRAME_GROUP;
/* pvl label field is constant */
buildTdsQueryFields(PVL_LABEL,FALSE);
/* sfdu label field is constant */
buildTdsQueryFields(SFDU_LABEL,FALSE);
/* object field */
buildTdsQueryFields(O_LABEL,TRUE);
buildTdsQueryFields(DESCRIPTION_LABEL,TRUE);
/* insert name label field */
buildTdsQueryFields(NAME_LABEL,FALSE);
/* get user name from Widget */
buildTdsQueryFields(name,TRUE);
/* the mission field is constant */
buildTdsQueryFields(MISSION_LABEL,FALSE);
buildTdsQueryFields(mission,TRUE);
/* the spacecraft name field is constant */
buildTdsQueryFields(SC_LABEL,FALSE);
/***********************************************************/
/* store the spacecraft ID for possible verification       */
/* of SFDUs from TDS(This is mostly to prevent GLL         */
/* Testbed data from getting confused with spacecraft data */ 
/***********************************************************/
//printf ("ted_tds_if: sc_id = %s\n",sc_id);
if (strcmp(sc_id,"GLL") == 0)
   {
   sc_num = 77;
   testbed_query = FALSE;
   }
else
   {
   if (strcmp(sc_id,"SGLL") == 0)
      {/* test bed query */
      testbed_sc_query_id = getenv("TESTBED_SC_QUERY_ID");
      if (testbed_sc_query_id EQ NULL)
         {
         printf("environment variable TESTBED_SC_QUERY_ID not set\n");
         printf("if testbed Spacecraft ID = 77 then\n");
         printf("    setenv TESTBED_SC_QUERY_ID GLL\n");
         printf("else\n");
         printf("    setenv TESTBED_SC_QUERY_ID SGLL\n");
         exit(0);
         }
      else
         {
         sc_num = 87;
         strcpy(sc_id,testbed_sc_query_id);
         testbed_query = TRUE;
         }
      }
   }
buildTdsQueryFields(sc_id,TRUE);
/* TDS source field is constant */
buildTdsQueryFields(DATA_SOURCE_LABEL,FALSE);
buildTdsQueryFields(tds_source,FALSE);
buildTdsQueryFields("}",TRUE);
buildTdsQueryFields(TIME_TYPE_LABEL,FALSE);
buildTdsQueryFields(time_type,TRUE);
buildTdsQueryFields(START_TIME_LABEL,FALSE);
buildTdsQueryFields(start_time,TRUE);
buildTdsQueryFields(STOP_TIME_LABEL,FALSE);
buildTdsQueryFields(stop_time,TRUE);
buildTdsQueryFields(TIME_ORDER_LABEL,FALSE);
buildTdsQueryFields(time_order,TRUE);
buildTdsQueryFields(DSS_ID_LABEL,FALSE);
buildTdsQueryFields(GROUP_LABEL,FALSE);
buildTdsQueryFields(group_names[FRAME_GROUP],TRUE);
/* the reason this is false is because the string already */
/* contains a terminator when it was initialized */
buildTdsQueryFields(DT_NAME_LABEL,FALSE);
buildTdsQueryFields("\"",FALSE);
buildTdsQueryFields(data_type,FALSE);
buildTdsQueryFields("\"",TRUE);
buildTdsQueryFields(END_GROUP_LABEL,FALSE);
buildTdsQueryFields(group_names[FRAME_GROUP],TRUE);
buildTdsQueryFields(EO_LABEL,TRUE);
buildTdsQueryFields(SFDU_TRAILER,FALSE);
buildTdsQueryFields(PVL_TRAILER,FALSE);
//printf ("ted_tds_if: Exit\n");
}
/*******************************************************************************/
void buildTdsQueryFields(src,end_of_line)
char	*src;
int	end_of_line;
{
int string_length;
if (src NE NULL)
   {
   string_length = strlen(src);
   strncpy(&tds_query_buffer[tds_query_length],src,string_length);
   tds_query_length += string_length;
   }
if (end_of_line)
   tds_query_buffer[tds_query_length++] = ';';
}
/*********************************************************************************/
void initializeTedMotifCommon(w,sc_test)
Widget	w;
int	sc_test;
{
sc_id_test_enabled=sc_test;
log_widget = w;
motif_version=TRUE;
input_fd = END_OF_DATA;
scratch_ptr = scratch;
tds_query_sent=FALSE;
testbed_query = FALSE;
FD_ZERO(&read_fd);
FD_ZERO(&read_fd_template);
debug_enabled = FALSE;
data_capture = FALSE;
message_capture = FALSE;
query_capture = FALSE;
/* set select width */
getrlimit(RLIMIT_NOFILE, &resource);
width = resource.rlim_cur;
to.tv_usec = 100;
}
/**************************************************************************************/
void buildTdsChannelQuery(name,mission,sc_id,time_type,start_time,
		 stop_time, time_order,tds_source,channel_range,sample_mode)
char *name,*mission,*sc_id,*time_type,*start_time;
char *stop_time,*time_order,*tds_source,*channel_range,*sample_mode;
{
group_index=CHANNEL_GROUP;
/* pvl label field is constant */
buildTdsQueryFields(PVL_LABEL,FALSE);
/* sfdu label field is constant */
buildTdsQueryFields(SFDU_LABEL,FALSE);
/* object field */
buildTdsQueryFields(O_LABEL,TRUE);
buildTdsQueryFields(DESCRIPTION_LABEL,TRUE);
/* insert name label field */
buildTdsQueryFields(NAME_LABEL,FALSE);
/* get user name from Widget */
buildTdsQueryFields(name,TRUE);
/* the mission field is constant */
buildTdsQueryFields(MISSION_LABEL,FALSE);
buildTdsQueryFields(mission,TRUE);
/* the spacecraft name field is constant */
buildTdsQueryFields(SC_LABEL,FALSE);
buildTdsQueryFields(sc_id,TRUE);
/***********************************************************/
/* store the spacecraft ID for possible verification       */
/* of SFDUs from TDS(This is mostly to prevent GLL         */
/* Testbed data from getting confused with spacecraft data */
/***********************************************************/
sc_num = strtol(sc_id,(char **)NULL,10);
/* TDS source field is constant */
buildTdsQueryFields(DATA_SOURCE_LABEL,FALSE);
buildTdsQueryFields(tds_source,FALSE);
buildTdsQueryFields("}",TRUE);
buildTdsQueryFields(TIME_TYPE_LABEL,FALSE);
buildTdsQueryFields(time_type,TRUE);
buildTdsQueryFields(START_TIME_LABEL,FALSE);
buildTdsQueryFields(start_time,TRUE);
buildTdsQueryFields(STOP_TIME_LABEL,FALSE);
buildTdsQueryFields(stop_time,TRUE);
buildTdsQueryFields(TIME_ORDER_LABEL,FALSE);
buildTdsQueryFields(time_order,TRUE);
buildTdsQueryFields(DSS_ID_LABEL,FALSE);
buildTdsQueryFields(GROUP_LABEL,FALSE);
buildTdsQueryFields(group_names[CHANNEL_GROUP],TRUE);
/* the reason this is false is because the string already */
/* contains a terminator when it was initialized */
buildTdsQueryFields(CHANNEL_SET_LABEL,FALSE);
buildTdsQueryFields("{",FALSE);
buildTdsQueryFields(channel_range,FALSE);
buildTdsQueryFields("}",TRUE);
buildTdsQueryFields(SAMPLE_MODE_LABEL,FALSE);
buildTdsQueryFields(sample_mode,TRUE);
buildTdsQueryFields(END_GROUP_LABEL,FALSE);
buildTdsQueryFields(group_names[CHANNEL_GROUP],TRUE);
buildTdsQueryFields(EO_LABEL,TRUE);
buildTdsQueryFields(SFDU_TRAILER,FALSE);
buildTdsQueryFields(PVL_TRAILER,FALSE);
}
/**********************************************************************************/
int	incrementRecordTypeReceived()
{
/* compare major minor format of SFDU and increment counter */
/* return major type for use in SC validation */
int i,j,k,mismatch;
for (i=0;i<MAX_TDS_TYPES;i++)
   {/* search for a match */
   for (j=0;j<frame_sfdus[i].num_sfdu_types;j++)
      {/* compare sfdu received to all possible SFDU types in table */
      mismatch = FALSE;
      for (k=0;k<PRIMARY_SIZE;k++)
         {/* compare major,minor,mission,format in sfdu rcvd to table */
         if (input_data.byte_buffer[PRIMARY_HDR_DATA_OFFSET+k] NE 
		frame_sfdus[i].sfdus[j].sfdu_id[k])
            {
            mismatch=TRUE;
            break;
            }
         }
      if (NOT mismatch)
         {/* increment record count */
         frame_sfdus[i].sfdus[j].rcvd++;
         return(i);
         }
      }
   }
/* unknown record rcvd */
unknown_sfdus_rcvd++;
return(i);
}
/***********************************************************************************/
void	displayRecordTypesReceived()
{
int i,j,k,mismatch;
if (group_index EQ FRAME_GROUP)
   {/* frame query sent to TDS */
   for (i=0;i<MAX_TDS_TYPES;i++)
      {/* search for a match */
      for (j=0;j<frame_sfdus[i].num_sfdu_types;j++)
         {/* compare sfdu received to all possible SFDU types in table */
         if (frame_sfdus[i].sfdus[j].rcvd)
            {
            sprintf(scratch,"%d frame SFDUs of type %s were received\n",
			frame_sfdus[i].sfdus[j].rcvd,
		*(frame_sfdus[i].name_list+frame_sfdus[i].sfdus[j].dt_index));
            copyStringToLog(scratch);
            /* reset counter after displaying value */
            frame_sfdus[i].sfdus[j].rcvd=0;
            }
         }
      }
   }
if (unknown_sfdus_rcvd)
   {
   sprintf(scratch,"%d Unknown SFDUS received\n",unknown_sfdus_rcvd);
   copyStringToLog(scratch);
   unknown_sfdus_rcvd=0;
   }
}
/*********************************************************************************/
#define	MSG_SENT	0xdead
int verifySpacecraftID()
{
static int	error_msg_output;
/* if SC ID Mismatch then return Non zero else return zero */
if (input_data.byte_buffer[SC_ID_OFFSET] NE sc_num)
   {/* unless galileo test bed selected quit processing */
   if (NOT testbed_query)
     {
     sprintf(scratch,
        "Spacecraft ID of %d received, %d expected, stopping data stream\n",
        input_data.byte_buffer[SC_ID_OFFSET],sc_num);
     copyStringToLog(scratch);
     processEndOfDataMessage(TRUE);
     return(TRUE);
     }
  else
     {/* if galileo test bed then change sc id to 87 */
     if (error_msg_output NE MSG_SENT)
        {/* output error msg once */
        sprintf(scratch, 
           "Spacecraft ID of %d received, %d expected, correcting ID\n",
           input_data.byte_buffer[SC_ID_OFFSET],sc_num);
        copyStringToLog(scratch);
        error_msg_output = MSG_SENT;
        }
     input_data.byte_buffer[SC_ID_OFFSET]=87;
     }
  }
return(FALSE);
}
