/*  ftime2.c
 *
 *  std_time_to_seconds         read_sclk, scet2sclk      
 *  any_time_to_seconds         read_sclk   
*/
#include <stdlib.h> 
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
/*
#define DEBUG_GDRTIME   1
*/

#define ERROR		(-1)
#define OK		(0)

#define SPD		86400L		/* seconds per day */
#define DSPD            86400.0         /* same in double format */
#define FOURYEARS	1461L		/* number of days in 4 years */
#define FOURCENTS	146097L		/* days in 4 gregorian centuries */

/* These are Julian day numbers for various Epochs you may want to use */
#define J2000		2451544L	/* J2000 Epoch */
#define J1950		2433282L	/* Jan0 1950 */
#define J1958		2436204L	/* Jan0 1958 */
#define J1985		2446066L	/* Jan0 1958 */
#define J0000		1867156L	/* Jan0 0000 */

#define DAYS_1985	578909L
#define JEPOCH		J1985		/* Epoch we are currently using */
/*
#define JEPOCH		J0000
*/

/* OFFSET1 is used in the ascii->seconds routines */
#define OFFSET1		(JEPOCH - 1721028L)

/* OFFSET2 is used in the seconds->ascii routine */
#define OFFSET2		(OFFSET1 - 90)

/* Time format specifications for sprintf */
#define	YMD_FORM	"%04ld-%02ld-%02ldT%02ld:%02ld:%06.3lf"
#define YDOY_FORM	"%04ld-%03ldT%02ld:%02ld:%06.3lf"

/* test year to determine whether it is a leapyear or not */
#define LEAPYEAR(y) ( (y%400 == 0) || ((y%4 == 0) && (y%100 != 0))  )

typedef	double		BTIME;	/* binary time representation */
typedef	char *		string;

/* array containing day of year for beginning of each month */
static	long	monthdays[] = {
	0,	31,	59,	90,
	120,	151,	181,	212,
	243,	273,	304,	334
};
int seconds_to_ascii_time(BTIME t, int  mode, string  buffer);
int any_time_to_seconds(register char* s, double* t);


//------------------------------------------------------------------------
/* std_time_to_seconds
** This routine accepts TIME values in the following standard format:
**	yyyy-mm-dddThh:mm:ss.sss -or- yyyy-oooThh:mm:ss.sss
** where the first format contains year, month, and day-of-month, and
** the second format contains year and day-of-year.  The function
** determines whether there are two or three fields ahead of the 'T'
** to determine which format it is parsing.  If only the date is to
** be represented, the 'T' must also be left out.
*/
int
std_time_to_seconds(s, t)
	register string	s;
	BTIME *		t;
{
	register char *	p;
	long		year, month, day, hour, minute;
	double		second;
	int		state;

	/* Set defaults */
	hour = minute = 0L;
	month = day = 1L;
	second = 0.0;

	/* first do the parsing of fields */
	for (state = 0; *s; state++)
	{
		switch(state)
		{
		case 0:		/* start: parse year */
		case 4:
		case 6:
		case 8:
		case 10:
			switch(state)
			{
			case 0:
				year = (long)atoi(s);
				if (year < 1700 || year > 9999)
					return(ERROR);
				break;
			case 4:
				month = day;
				day = (long)atoi(s);
				break;
			case 6:
				hour = (long)atoi(s);
				break;
			case 8:
				minute = (long)atoi(s);
				break;
			case 10:
				second = (double)atof(s);
				while (isdigit(*s) || *s == '.') s++;
				break;
			}
			while (isdigit(*s))	s++;
			break;
		case 1:		/* expect '-' after year or month */
		case 3:
			if (*s++ != '-')
				return(ERROR);
			break;
		case 7:		/* expect ':' after hour or minutes */
		case 9:
			if (*s++ != ':')
				return(ERROR);
			break;
		case 2:		/* could be doy or month */
			day = (long)atoi(s);
			while (isdigit(*s))	s++;
			if (*s == 'T')
				state = 4;
			break;
		case 5:
			if (*s++ != 'T')
				return(ERROR);
			break;
		/* unexpected characters past end of parsed fields */
		default:
			return(ERROR);
		}
	}
	if (state < 2)
		return(ERROR);		/* not enough fields parsed */

	/* This is the Muller-Wimberly formula for calculating time
	** from the year, month, day, and time (see ref).
	*/
	day = 367L * year - 7L * (year + (month + 9L) /12L) /4L
			- 3L * ((year + (month - 9L) /7L) /100L + 1L) /4L
			+ 275L * month /9L + day - OFFSET1;

	*t = ((double)day - 0.5) * DSPD
			+ (double)(60L * (60L * hour + minute))
			+ second;

	return (OK);
}
//----------------------------------------------------------------------------------
/* any_time_to_seconds
** This routine accepts TIME values in any of the following formats:
**	yyyy{t}mm{t}ddd{T}hh:mm:ss.sss -or- yyyy{t}ddd{T}hh:mm:ss.sss
** where {t} can be any of the following single-character tokens:
**	{'/', '-', ' '}
** and {T} can be any of those or 'T'.
** Because such a variety of formats are accepted and analyzed to
** determine which field is which, errors are reported only if the
** input is ambiguous.  Missing fields are allowed and the following
** defaults are applied:
**	year:	no default -- must be provided
**	month:	January (1)
**	day:	1
**	hour:	0
**	minute:	0
**	second: 0.000
**
** the second format contains year and day-of-year.  The function
** determines whether there are two or three fields ahead of the 'T'
** to determine which format it is parsing.  If only the date is to
** be represented, the 'T' must also be left out.
*/
int any_time_to_seconds(register char* s, double * t)
/*
any_time_to_seconds(s, t)
	register	string s;
	BTIME *		t;
*/
{
	register char *	p;
	char		*tp;
	long		year, month, day, hour, minute;
	double		second;
	int		state;

	/* set defaults */
	month = day = 1L;
	hour = minute = 0L;
	second = 0.0;

	/* In order to distinguish between yyyy mm dd hhh and yyyy ddd hh mm
	** we look ahead to see if there is a colon between hh and mm
	** since this is a required token in this position.  We can then
	** back up the pointer to the beginning of hh and use this as a
	** test after we've parsed the first two fields: if after parsing
	** two fields the pointer is up to the test pointer then we
	** assume the input format was yyyy-doy and go on to parse the 
	** time.  Note: if tp==NULL (no colon) then the assmption is
	** made that the input format is yyyy mm dd unless dd is > 31.
	*/
	tp = strchr(s, ':');
	if (tp)
		do --tp; while (tp > s && isdigit(*tp));

	/* first do the parsing of fields */
	for (state = 0; *s; state++)
	{
		switch(state)
		{
		case 0:		/* start: parse year */
		case 4:
		case 6:
		case 8:
		case 10:
			switch(state)
			{
			case 0:
				year = (long)atoi(s);
				if (year < 0 || year > 9999)
					return(ERROR);
				if (year < 50)
					year += 2000L;
				else if (year < 100)
					year += 1900L;
				break;
			case 4:
				month = day;
				day = (long)atoi(s);
				break;
			case 6:
				hour = (long)atoi(s);
				break;
			case 8:
				minute = (long)atoi(s);
				break;
			case 10:
				second = (double)atof(s);
				while (isdigit(*s) || *s == '.') s++;
				break;
			}
			while (isdigit(*s))	s++;
			break;
		case 1:		/* expect {t} after year or month */
		case 3:
			if (!(*s=='-' || *s=='/' || *s==' '))
				return(ERROR);
			s++;
			if (tp && s >= tp)
				state=5;
			break;
		case 7:		/* expect ':' after hour or minutes */
		case 9:
			if (*s++ != ':')
				return(ERROR);
			break;
		case 2:		/* could be doy or month */
			day = (long)atoi(s);
			while (isdigit(*s))	s++;
			if (*s == 'T' || day > 31L)
				state = 4;
			break;
		case 5:
			if (!(*s=='-' || *s=='/' || *s==' ' || *s=='T'))
				return(ERROR);
			s++;
			break;
		/* unexpected characters past end of parsed fields */
		default:
			return(ERROR);
		}
	}
	if (state < 1)
		return(ERROR);		/* not enough fields parsed */

	/* This is the Muller-Wimberly formula for calculating time
	** from the year, month, day, and time (see ref).
	*/
	day = 367L * year - 7L * (year + (month + 9L) /12L) /4L
			- 3L * ((year + (month - 9L) /7L) /100L + 1L) /4L
			+ 275L * month/9L + day - OFFSET1;

	*t = ((double)day - 0.5) * DSPD
			+ (double)(60L * (60L * hour + minute))
			+ second;

	return (OK);
}
/*
int
seconds_to_ascii_time(t, mode, buffer)
	BTIME   t;
	int	mode;*/	/* 0 = year-doy, otherwise year-month-day */
/*	string	buffer;
*/
int seconds_to_ascii_time(BTIME t, int  mode, string  buffer)
{
	register long	x, day;
	long		year, month, hour, minute;
	double		dp2000, second;

	/* Convert t in seconds past 2000 to days past 2000.  Round up
	** seconds to 3 decimal places since only 3 will be printed.
	** Rounding here eliminates wierdnesses like T12:59:60.000 at
	** the end.  The 0.5 days accounts for the fact that we are
	** measuring Julian days which change at noon rather than
	** midnight.
	*/
	dp2000 = (t+0.00005)/DSPD + 0.5;

	/* make sure that the integer day corresponds to the preceding
	** midnight even for times before the J2000 origin.
	*/
	if ( (dp2000 >= 0.0) || ( dp2000 == floor(dp2000) ) )
		day = (long)dp2000;
	else
		day = (long)(dp2000 - 1.0);

	/* This is the inverse of the Muller-Wimberly formula.  Note that
	** 1461 = 4 years (4*365 + 1), and 14097 = 4 Gregorian centuries
	** (400*365 + 97). The constant OFFSET2 is the number of Julian
	** days from March 1, 1 BC (in the proleptic Gregorian calendar)
	** to the EPOCH.  The algorithm moves the time origin
	** from EPOCH to 1BC to make the problem more tractable.  In
	** particular, it enables the elegant (5*day-3)/153 expression
	** which gives the months of 31,30,31,30,31,31,30,31,30,31,31,30...
	** days.  This series is the lengths of the first 11 months if
	** one starts with March (thus the offset of 3 months).
	*/
	second	= DSPD*(dp2000 - (double)day);
	x	= day + OFFSET2 ;  
	year	= (4L*x-1L) / 146097L;
	day	= ((4L*x-1L) - (146097L*year)) / 4L;
	x	= (4L*day+3L) / 1461;
	day	= (((4L*day+3L) - (1461*x)) + 4L) / 4L;
	month	= (5L*day-3L) / 153L;
	day	= (((5L*day-3L) - (153L*month)) + 5L) / 5L;
	year	= 100L*year + x;

	if (month < 10L)
		month += 3L;
	else
	{
		month -= 9L;
		year++;
	}

	/* Solving for hours, minutes, and seconds is straightforward */
	hour	= (long)(second / 3600.0);
	second	-= 3600.0*(double)hour;
	minute	= (long)(second / 60.0);
	second	-= 60.0*(double)minute;

	/* Choose the output format */
	if (mode)
	{
		sprintf(buffer, YMD_FORM, year, month, day,
			hour, minute, second);

	} else {	/* derive doy */
		if (month > 2L)
			if ( LEAPYEAR(year) )
				day++;
		day = monthdays[month-1] + day;
		sprintf(buffer, YDOY_FORM, year, day,
			hour, minute, second);
	}
	return(OK);
}

#ifdef DEBUG_GDRTIME
static char * data1[] = {
	"1994-011T04:33:24.675234",
	"1994-011T04:34:12.675",
	"1994-030T10:09:24.467",
	"1994-030T18:09:24.530",
	"1994-035T21:21:24.576",
	"1994-036T05:21:24.502",
	"1983-04-13T12:09:14.000",
	"1957-04-01T01:23:45.567",
	"1983-103T01:23:45.000",
	""
};
static char * data2[] = {
	"1983-04-13-12:09:14.000",
	"1983/01/01-12:00:00.000",
	"1983/01-12:00:00.000",
	""
};

/* This is a test routine that can be used to test these routines.
** They simply parse a time string into double and then print it
** back out in both formats.  Two data arrays are provided; the
** first contains data formatted in standard ISO time format and
** the second contains more free-form time.
*/
main()
{
	BTIME t;
	char **	p;
	char buff[100];

	for (p = data1; **p; p++)
	{
		if (std_time_to_seconds(*p, &t) != OK)
			printf("Error on %s\n", *p);
		else
			printf("%s --> %lf\n", *p, t);

		if (seconds_to_ascii_time(t, 1, buff) != OK)
			printf("Error converting to date\n");
		else
			printf("%lf --> %s\n", t, buff);

		if (seconds_to_ascii_time(t, 0, buff) != OK)
			printf("Error converting to doy\n");
		else
			printf("%lf --> %s\n", t, buff);

	}

	for (p = data2; **p; p++)
	{
		if (any_time_to_seconds(*p, &t) != OK)
			printf("Error on %s\n", *p);
		else
			printf("%s --> %lf\n", *p, t);

		if (seconds_to_ascii_time(t, 1, buff) != OK)
			printf("Error converting to date\n");
		else
			printf("%lf --> %s\n", t, buff);

		if (seconds_to_ascii_time(t, 0, buff) != OK)
			printf("Error converting to doy\n");
		else
			printf("%lf --> %s\n", t, buff);

	}


}

#endif

