/**************************************************************************
TITLE:  vgr_sfdu.h

AUTHOR:         Judy Yin
CREATED:        Aug 4, 1994
modified:       3-11-2021 - Ray Bambery
                Added detailed description of an SFDU type 2 file
                1-22-2022 - Ray Bambery
                put in decriptions of SFDU label and ADID values
                2-8-2022 - Ray Bambery
                add items 40 and 41 back in after accidental deletion earlier
OS:             Solaris 10.2
COMPILER:       gcc 4.5.2
****************************************************************************
 * Include file for vgr_sfdu.C to map a Voyager telemetry file structure
 * in SFDU2 format. For details see JPL Doc. 820-013 DSN External
 * Interface Specification (D-16765) "Telemetry Standard Format
 * Data Unit (SFDU) Interface" - Rev A - Aug 29, 2008
 *
 * Section 3 describes the Detailed Interface Description and physical
 * layout of the telemetry SFDU.
 *
 * Also the CHDO descriptions are detailed in JPL Doc 820-013-0172
 * Telecom CHDO Structures - Rev E - Jul 30, 2010
 *
 *  The following is a description of an SFDU Type 2 file as implemented
 *  for a Voyager telemetry file normally called datafile.a for
 *  Voyager 1 and datafile.b for Voyager 2. In the archive they
 *  are renamed in the fashion datafileYRdoy.a and datafileYRdoy.b.
 *  For example datafile20350.b would be the datafile created
 *  for the Year 2020 and Day of year 350 for Voyager 2.
 *  DOY of 350 for that year corresponds to 12-14-2020.
 *
 *  Quotes implies ASCII - numbers are binary unsigned integers
 *  In documentation a byte is called an "octet"
 * ********************************************* 
 *  each sfdu2 file has 5 labels (headers) 
 *  SFDU Label - 20 bytes
 *  Aggregation CHDO - 4 bytes
 *  Primary CHDO - 8 bytes
 *  Secondary CHDO - 84 bytes
 *  Telemetry Data CHDO - (N-116) bytes
 * *******************************************
 *  SFDU label has 20 bytes
 *  0-3   - Control Authority ID = "NJPL" (NASA/JPL) - 4 bytes
 *  4     - Version ID = "2" - 1 byte = 
 *          indicates that the length attribute field in bytes 12 to 19 of the i
 *          SFDU label is formatted as a binary unsigned integer); restricted ASCII.
 *  5     - Class ID = "I" - 1 byte
 *          indicates that this SFDU is an application data object); restricted ASCII
 *  6,7   - Spare = "00" - 2 bytes
 *  8-11  - Data Description ID = "C220" - 4 bytes
 *          ADID - Authority and Description Identifier
 *          See 820-013_0171-telecom-VGR-NJPL-RevG-Appx_L5 for a listing of values for VGR
 *  12-19 - Length Attribute = (variable - length, in binary, of the remainder, 
 *          starting with byte 20 of this sfdu) - 8 bytes
 * ******************************************
 * Aggregation CHDO has 4 bytes
 * 0,1    - Type Attribute = 1 (Aggregation CHDO) - 2 bytes
 * 2,3    - Length Attribute = 92 (length of Primary and Secondary CHDOs) - 2 bytes
 * ******************************************
 * Primary CHDO has 8 bytes
 * 0,1   - Type Attribute = 2 (Primary CHDO) - 2 bytes
 * 2-9   - Length Attribute = 4 (4 16-bit words) - 8 bytes
 * 10    - Major Class = 1 - 1 byte
 * 11    - Minor Class = (variable 7-17) - 1 byte
 * 12    - Mission ID = (variable 0-255) - 1 byte
 * 13    - Format Code = 0  (DSS formatted SFDU) - 1 byte
 * ******************************************
 * Secondary CHDO has 84 bytes
 * 0,1   - Type Attribute = 78 (telemetry secondary SFDU) - 2 bytes
 * 2,3   - Length Attribute = 80 (bytes in this CDHO not including Type Attribute) - 2 bytes
 * 4     - Originator ID = 48 (within DSN) - 1 byte
 * 5     - Last Modifier ID = 48 (within DSN) -  byte
 * 6     - Reserved - 1 byte
 * 7     - Spacecraft ID (variable) - 1 byte  (23x or 35d =VGR-2)
 * 8,9   - Pass Number (0-9999) 2 bytes
 * 10    - Data Source ID (0-255) - 1 byte
 * 11    - Arrayed Stations (variable) - 1 byte
 * 12,13 - Flags (variable) - 2 bytes
 * 14-21 -       - Earth Received Time - 8 bytes
 * 14-19 -  SERT_DS1958 = days since 1 Jan 1958
 * 22,21 -  SERT_MSEC = ms of day
 * 22-25 - Record Sequence Number - 4 bytes
 * 26    - U/L Band - 1 byte
 * 27    - D/L Band - 1 byte
 * 28    - PDX mode - 1 byte 
 * 29    - D/L Source - 1 byte
 * 30    - Virtual Stream ID - 1 byte
 * 31    - Virtual Channel ID - 1 byte
 * 32,33 - Telemetry Lock Status - 2 bytes
 * 34-37 - Number of Bits = (Number of Bits contained in Telemetry Data CHDO below) - 4 bytes
 * 38-41 - Measured Bit Rate - 4 bytes
 * 42-45 - System Noise Temperature - 4 bytes
 * 46-49 - Symbol SNR - 4 bytes
 * 50-53 - Receiver Signal Level - 4 bytes
 * 54    - Acquisition BET - 1 byte
 * 55    - Maintenance BET - 1 byte
 * 56    - Verify Count - 1 byte
 * 57    - Flywheel Count - 1 byte
 * 58    - Frame Sync Mode Flags - 1 byte
 * 59    - Sync Status| Bit slip - 1 byte
 * 60    - FS Error Count - 1 byte
 * 61    - FS Buffer Size - 1 byte
 * 62    - RS Decoder Status - 1 byte
 * 63    - RS Symbol Error Count - 1 byte
 * 64    - Turbo Decoder Status - 1 byte
 * 65    - Processor number - 1 byte
 * 66    - Number of Iterations - 1 byte
 * 67    - Reserve - 1 byte
 * 68    - Code Rate Numerator - 1 byte
 * 69    - Code Rate Denominator - 1 byte
 * 70,71 - Frame Data Bits - 2 bytes
 * 72,73 - Decoder Confidence - 2 bytes
 * 74,75 - Telemetry Equipment ID - 2 bytes
 * 76,77 - Telemetry Software ID - 2 bytes
 * 78-83 - Reserved - 6 bytes
 ** **********************************************************
 * Telemetry Data CHDO has N-117 bytes
 * Type Attribute = 10 (CHDO contains binary data only) - 2 bytes 
 * Length Attribute = N-117 - 2 bytes
 * Received Telemetry Bits (Can contain telemetry from S/C or
 *           undecoded signals) - see either Turbo Decoder Status
 *           Byte 64 (Bit 8) of Secondary CHDO or
 *           Minor Data Class in Primary CHDO Byte 5 
 *           (7-11 non-turbo encoded), (12-16 turbo encoded data)
 *           (17 for no processing done) 
 */

/***************************************************************************/
#ifndef VGR_SFDU_H
#define VGR_SFDU_H

#include "vgr_def.H"
#include "vgr_sfdu.H"

#define  PRIM_AUTH_ID_LOC  0
#define  PRIM_DDP_ID_LOC   8

/* See 820-013_0171-telecom-VGR-NJPL-RevG-Appx_L5  
 *  D-17765 - Composition of Standard CHDO-structured Logical SFDUs.
 *
 * for a listing of ADID values for VGR 
 * Low Rate Science Frames:             
 * C220   3 128   0   0   15  50  53  10    CR-5A science, VGR 
 * C221   3 129   0   0   15  50  53  10    UV-5A science, VGR 
 * C222   3 130   0   0   15  50  53  10    CR-7A science, VGR 
 *
 *  High Rate Science Frames
 *
 *  C223   4 128 128   0   15  50  53  10    GS-08 science, VGR 
 *  C224   4 128 129   0   15  50  53  10    GS-10 science, VGR 
 *  C225   4 128 130   0   15  50  53  10    GS-10A science, VGR 
 */

DDP_DATA_MODE_STRUCT ddp_map[MAX_DDP] =

{
   {"C220",     3,      128,    0,      0,      CR_5A},      //ddp_id,  maj,  min,  fmt_id   msn_id  data_mode
   {"C221",     3,      129,    0,      0,      UV_5A},      // added 1-21-2022
   {"C222",     3,      130,    0,      0,      CR_7A},      //added 2-4-2022
   {"C223",     4,      128,    0,      0,      GS_8},     //added 2-4-2022
   {"C224",     4,      129,    0,      0,      GS_10},     //added 2-4-2022
   {"C225",     4,      130,    0,      0,      GS_10A}     //added 2-4-2022
};

MF_MAP  mf_map[MAX_INSTR_NUM] = 
{
 { UVS,  26,  6},             //instr_type,  start_bit  bit_len
 { MAG,  32, 51},
 { CRS,  83, 39},
 { PLS,  122,20},
 { LECP, 142,16},
 { PRA,  158, 7},
 { PWS,  165,11}
};

#define MAX_BIT_RATE_NO 42

U32 bit_rate_tbl[MAX_BIT_RATE_NO] =      //should it be float??
{ 
	99999,	10,	    20,	    40,	    80,	    160,	320,
	640,	1200,	1280,	2560,	7200,	19200,	21600,
	29866,	600,	67200,	89600,	115200,	33600,	57600,
	186,	120,	450,	1350,	2100,	3600,	2500,
	7500,	11,	    5600,	9600,	47, 	8400,	12800,
	14400,	4800,	4200,	16800,	10800,	22400,	25200
};


// EDR Standard Header Format - See         618-306 - Figure C Rev D Chg 2
//           and descriptions in Table C-1    pg C-5 ff
// EDRHDR_ITEMS_PART1 = 50
// Cross Ref of EDR item number with location in sfdu2 secondary (SEC_HDR_ID)
// and tertiary CHDO (TERT_HDR_ID)
//
// 820-013_0182-Telecomm-VGR Rev F Jun 23, 1011 
// SFDUs Generated from TIS for Voyager
//
// sfdu2 secondary header is 15 TIS Telemetry - 
// Secondary CHDO details:  000f of Secondary CHDO = 15 TIS Telemetry - Sec 4.2.1.2 (pg 4-17) 
// Tertiary CHDO details:   0032 of Tertiary CHDO = 50 Voyager Telemetry - Sec 4.2.1.3 (pg 4-22)

// listed items are copied from HDR IDs of sfdu to edrs
SFDU_EDR_HDR_MAP sfdu_edr_hdr_map[EDRHDR_ITEMS_PART1] =
{
 //item1 Project ID
 //item2 Rec ID
 {item_3_SCID,		SEC_HDR_ID,   6,      0,      8},       //Item  HeaderID   start_byt  start_bit  bit_len
 //item4 Physical Record ID
 //item5 S/C mode
 //item6 A=Engineering Flag
 //item7 B=S/C Playback flag
 {item_7_SCPB_FLG,	SEC_HDR_ID,   8,      0,      1}, 
 //item8 SPARE
 //item9  divided SERT_HR(9),SERT_SEC(10),SCRT_MSEC(11),YEAR(12) of current year 
 {item_9_SERT_DAY,	SEC_HDR_ID,	 10,      0,      16},
 {item_10_SERT_SEC,	SEC_HDR_ID,	 12,	  0,      32},
 //item11 SCET_MSEC
 //item12 YEAR
 //item13 C=Data Source
 //item14 (see Below)
 //item15 Segment No
 //item16 EERT_HR
 //item17 EERT_SEC
 //item18 EERT_MSEC
 //item19 YEAR 
 //item20 Software Version
 //item21 divided SCET_HDR(21),SCET_SEC(22),SCET_MSEC(23),YEAR(24)
 {item_21_SCET_DAY,	SEC_HDR_ID,	 16, 	  0, 	 16},
 {item_22_SCET_SEC,	SEC_HDR_ID,	 18,	  0,	 32},
 //item23 SCET_MSEC
 //item24 YEAR
 //item25 SCE FLAG
//      {8,     0,      16,     2,      15,     16      },
//      {9,     0,      16,     3,      15,     16      },
//      {19,    0,      8,      3,      15,     8       },
                                                            //instr -- fill
// sdr_to_time
//      {12,    0,      16,     6,      31,     16      },  //item21 SCET yr
//      {13,    0,      16,     6,      15,     16      },  //item22      hr
//      {14,    0,      16,     7,      31,     16      },  //item23     msc
// clk_to_sdr, calc_clk
// item14 - D=Golay Encoded Flag
 {item_14_GOLAY,	SEC_HDR_ID,	  24,	  0,	  16}, // reed & golay
//item26 FDSC flg
 {item_26_FDSC_FLG,	TERT_HDR_ID,   5,     0,      3 },     
//item27 M16 count (2^16)
 {item_27_M16,		TERT_HDR_ID,  10,     0,      16},
//item28 M60 count (2^60)
 {item_28_M60,		TERT_HDR_ID,  12,     0,      16},     
//item29 LN count (across word boundary)
 {item_29_LN_CNT,	TERT_HDR_ID,  14,     0,      16},
                                                        //item30,D/L TLM Rate
                                                        //item31,Effective Expt Bit Rate
//      {27,    8,      8,      9,      15,     8       },  //item33 efec.rate?
 //item32 FID,?eng
 {item_32_BIT_RATE,	SEC_HDR_ID,	  34,  	  0,	  32}, 
 {item_34_FMT_ID,	TERT_HDR_ID,   9,     0,       8},      
  //item35 BET
 {item_35_BET,		SEC_HDR_ID,	  22,	  0,	   8},
 {item_36_LOCK_CNF, SEC_HDR_ID,	  59,	  0,	   8}, 
 {item_37_AGC,		SEC_HDR_ID,	  48,	  0,	  32},
 //item38 DSN st#
 {item_38_DSN_ST,	SEC_HDR_ID,    7,     0,       8},      
                                                            //item39 spare
// add_bit_error_cnt  *** for every MF
  {item_40_EBEC,     SEC_HDR_ID,   30,     0,       8},
 //item41,SNR, ?
 {item_41_SNR,      SEC_HDR_ID,   44,     0,       32},
 {item_44_LOCK_ST,  SEC_HDR_ID,	  58,	  0,	    8},
 {item_48_PN_ERR,	SEC_HDR_ID,   27,	  0,	    8},
 {item_49_DATA_FLG,	SEC_HDR_ID,	   8,	  0,	    1},  // SHOULD BE {item_49_DATA_FLG, SEC_HDR_ID,    8,     1,        1}
 {item_50_GCF_ERR,	SEC_HDR_ID,	   9,	  1,	    3},
 {item_60_REPLY_F,	SEC_HDR_ID,	  62,	  0,	    8},
 {item_61_ORG_SRC,	SEC_HDR_ID,	   8,	  3,	    1},
//      {29,    1,      1,      13,     4,      1       },  //item45 PN out
//      {29,    0,      1,      13,     3,      1       },  //item46 PN in
//      {0,     4,      1,      13,     1,      1       },  //item47 valid
//      {29,    2,      1,      13,     0,      1       }   //item48 GCFerr
 {ITEM_99,		             0,	   0,	  0,	    0}
};
#endif
