/**************************************************************************
TITLE:  type_conv.H

AUTHOR:         Judy Yin
CREATED:        Aug 4, 1994
COMPILER:       CC
****************************************************************************/
#ifndef TYPE_CONV_H
#define TYPE_CONV_H


#ifndef U8 
#define U8 unsigned char
#endif
#ifndef I16
#define I16 short
#endif
#ifndef U16
#define U16 unsigned short
#endif
#ifndef I32
#define I32 long
#endif
#ifndef U32
#define U32 unsigned long
#endif


#ifndef U16_LEN
#define U16_LEN 2
#endif

#ifndef U32_LEN
#define U32_LEN 4
#endif

/* Convert a two byte char array containing GDR format data    */
/* to a signed 16 bit integer.  'cp' is of type 'char *'.      */

#define C16_TO_I16(cp)    ((I16) ((((cp)[0] & 0xFF) << 8) \
                                | (((cp)[1] & 0xFF))))


/* Convert a two byte char array containing GDR format data    */
/* to an unsigned 16 bit integer.  'cp' is of type 'char *'.   */

#define C16_TO_U16(cp)    ((U16) ((((cp)[0] & 0xFF) << 8) \
                                | (((cp)[1] & 0xFF))))


/* Convert a three byte char array containing GDR format data  */
/* to a signed 32 bit integer.  'cp' is of type 'char *'.      */

#define C24_TO_I32(cp)    ((I32) ((((cp)[0] & 0xFF) << 16) \
                                | (((cp)[1] & 0xFF) <<  8) \
                                | (((cp)[2] & 0xFF)      )))


/* Convert a three byte char array containing GDR format data  */
/* to an unsigned 32 bit integer.  'cp' is of type 'char *'.   */
 
#define C24_TO_U32(cp)    ((U32) ((((cp)[0] & 0xFF) << 16) \
                                | (((cp)[1] & 0xFF) <<  8) \
                                | (((cp)[2] & 0xFF)      )))


/* Convert a four byte char array containing GDR format data   */
/* to a signed 32 bit integer.  'cp' is of type 'char *'.      */

#define C32_TO_I32(cp)    ((I32) (((cp)[0] << 24) \
                               | (((cp)[1] & 0xFF) << 16) \
                               | (((cp)[2] & 0xFF) <<  8) \
                               | (((cp)[3] & 0xFF)      )))


/* Convert a four byte char array containing GDR format data   */
/* to an unsigned 32 bit integer.  'cp' is of type 'char *'.   */

#define C32_TO_U32(cp)    ((U32) (((cp)[0] << 24) \
                               | (((cp)[1] & 0xFF) << 16) \
                               | (((cp)[2] & 0xFF) <<  8) \
                               | (((cp)[3] & 0xFF)      )))


/* Convert a six byte char array of GDR data to an unsigned    */
/* 64 bit integer (that is, an array of two 32 bit unsigned    */
/* integers, with the MSW lowest in memory).  'cp' is of type  */
/* 'char *'.  i is an array of two 32 bit unsigned ints.       */

#define C48_TO_U64(cp, i)                                               \
{                                                                       \
        i[0] =  (U32)(((((cp)[0]) & 0xFF) << 8 ) |                      \
                (((cp)[1]) & 0xFF));                                    \
        i[1] =  (U32)(((((cp)[2]) & 0xFF) << 24 ) |                     \
                ((((cp)[3]) & 0xFF) << 16 ) |                           \
                ((((cp)[4]) & 0xFF) << 8 )  |                           \
                (((cp)[5]) & 0xFF));                                    \
}
/* Copy a signed 16 bit integer into a two byte character      */
/* array, in GDR format.  'cp' is of type 'char *', 'i' is a   */
/* 16 bit integer.                                             */

#define I16_TO_C16(cp, i) \
                        ((void) ((cp)[0] = (((i) >> 8) & 0xFF), \
                                 (cp)[1] = ((i) & 0xFF)))


/* Copy an unsigned 16 bit integer into a two byte character   */
/* array, in GDR format.  'cp' is of type 'char *', 'u' is a   */
/* 16 bit integer.                                             */

#define U16_TO_C16(cp, u) \
                        ((void) ((cp)[0] = (((u) >> 8) & 0xFF), \
                                 (cp)[1] = ((u) & 0xFF)))


/* Copy a signed 32 bit integer into a three byte character    */
/* array, in GDR format.  'cp' is of type 'char *', 'i' is a   */
/* 32 bit unsigned integer.                                    */

#define I32_TO_C24(cp, i) \
                       ((void) ((cp)[0] = (((i) >> 16) & 0xFF), \
                                (cp)[1] = (((i) >>  8) & 0xFF), \
                                (cp)[2] = ((i) & 0xFF)))


/* Copy an unsigned 32 bit integer into a three byte character */
/* array, in GDR format.  'cp' is of type 'char *', 'u' is a   */
/* 32 bit unsigned integer.                                    */

#define U32_TO_C24(cp, u) \
                       ((void) ((cp)[0] = (((u) >> 16) & 0xFF), \
                                (cp)[1] = (((u) >>  8) & 0xFF), \
                                (cp)[2] = ((u) & 0xFF)))
/* Copy a signed 32 bit integer into a four byte character     */
/* array, in GDR format.  'cp' is of type 'char *', 'i' is     */
/* a 32 bit integer.                                           */

#define I32_TO_C32(cp, i) \
                      ((void) ((cp)[0] = (((i) >> 24) & 0xFF), \
                               (cp)[1] = (((i) >> 16) & 0xFF), \
                               (cp)[2] = (((i) >>  8) & 0xFF), \
                               (cp)[3] = ((i) & 0xFF)))


/* Copy an unsigned 32 bit integer into a four byte character  */
/* array, in GDR format.  'cp' is of type 'char *', 'u' is a   */
/* 32 bit unsigned integer.                                    */

#define U32_TO_C32(cp, u) \
                       ((void) ((cp)[0] = (((u) >> 24) & 0xFF), \
                                (cp)[1] = (((u) >> 16) & 0xFF), \
                                (cp)[2] = (((u) >>  8) & 0xFF), \
                                (cp)[3] = ((u) & 0xFF)))

/* Convert a four byte char array containing GDR formatted     */
/* data to a 32 bit IEEE float.  'cp' is of type 'char *', 'f' */
/* is of type 'float'.  The value of the expression is the     */
/* float value placed in 'f'.                                  */

#define C32_TO_F32(cp, f) \
            (*((float *) memcpy((char *) &(f), (cp), 4)))
/* Copy a 32 bit IEEE float value into a four byte character   */
/* array, in GDR format.  'cp' is of type 'char *', 'f' is a   */
/* 32 bit IEEE float.  The value of the expression is 'f'.     */

#define F32_TO_C32(cp, f) \
            (memcpy((cp), (char *) &(f), 4), (f))


/* Convert an eight byte char array containing GDR formatted   */
/* data to a 64 bit IEEE float.  'cp' is of type 'char *', 'd' */
/* is of type 'double'.  The value of the expression is the    */
/* double value placed in 'd'.                                 */

#define C64_TO_F64(cp, d) \
            (*((double *) memcpy((char *) &(d), (cp), 8)))


/* Copy a 64 bit IEEE float value into an eight byte character */
/* array, in GDR format.  'cp' is of type 'char *', 'd' is a   */
/* 64 bit IEEE float.  The value of the expression is the      */
/* double value in 'd'.                                        */

#define F64_TO_C64(cp, d) \
            (memcpy((cp), (char *) &(d), 8), (d))



/* Create an unsigned 32 bit integer mask, given the starting  */
/* bit and length of the mask.  'bitoffset' and 'bitlen' are   */
/* integers.  A method is used which does not depend on the    */
/* the size of the intermediate register to truncate left      */
/* shifts.                                                     */

#define U32MASK(bitoffset, bitlen) \
    ((U32) ((U32) (0xFFFFFFFF & (0xFFFFFFFF << (bitoffset))) \
    >> (32 - (bitlen))) \
    << (32 - ((bitoffset) + (bitlen))))


/* Create an unsigned 16 bit integer mask, given the starting  */
/* bit and length of the mask.  'bitoffset' and 'bitlen' are   */
/* integers.  A method is used which does not depend on the    */
/* the size of the intermediate register to truncate left      */
/* shifts.                                                     */

#define U16MASK(bitoffset, bitlen) \
    ((U16) ((U16) (0xFFFF & (0xFFFF << (bitoffset))) \
    >> (16 - (bitlen))) \
    << (16 - ((bitoffset) + (bitlen))))

/* Create an unsigned 8 bit integer mask, given the starting  */
/* bit and length of the mask.  'bitoffset' and 'bitlen' are   */
/* integers.  A method is used which does not depend on the    */
/* the size of the intermediate register to truncate left      */
/* shifts.                                                     */
#define U8MASK(bitoffset, bitlen) \
    ((U8) ((U8) (0xFF & (0xFF << (bitoffset))) \
    >> (8 - (bitlen))) \
    << (8 - ((bitoffset) + (bitlen))))


/* Extract a bit field from a GDR character array, returning   */
/* an unsigned 32 bit integer containing the bit field right   */
/* justified.  'cp' is of type 'char *', 'bitoffset' and       */
/* 'bitlen' are integers.                                      */

#define CBITS_TO_U32(cp, bitoffset, bitlen) \
    ((U32) ((0xFFFFFFFF & (C32_TO_U32(cp) << (bitoffset))) \
    >> (32 - (bitlen))))


/* Extract a bit field from a GDR character array, returning   */
/* a signed 32 bit integer containing the bit field right      */
/* justified.  'cp' is of type 'char *', 'bitoffset' and       */
/* 'bitlen' are integers.                                      */

#define CBITS_TO_I32(cp, bitoffset, bitlen) \
    ((I32) CBITS_TO_U32(cp, bitoffset, bitlen))


/* Extract a bit field from a GDR character array, returning   */
/* an unsigned 16 bit integer containing the bit field right   */
/* justified.  'cp' is of type 'char *', 'bitoffset' and       */
/* 'bitlen' are integers.                                      */

#define CBITS_TO_U16(cp, bitoffset, bitlen) \
    ((U16) ((0xFFFF & (C16_TO_U16(cp) << (bitoffset))) \
    >> (16 - (bitlen))))
/* Extract a bit field from a GDR character array, returning   */
/* a signed 16 bit integer containing the bit field right      */
/* justified.  'cp' is of type 'char *', 'bitoffset' and       */
/* 'bitlen' are integers.                                      */

#define CBITS_TO_I16(cp, bitoffset, bitlen) \
    ((I16) CBITS_TO_U16(cp, bitoffset, bitlen))


#define CBITS_TO_U8(cp, bitoffset, bitlen) \
    ((U8) ((0xFF & (*(cp) << (bitoffset))) \
    >> (8 - (bitlen))))

/* The following three macros reposition the value      */
/* into its original position using the given           */
/* bit offset and bit length.                           */

#define U8_RESTORE_OFFSET(u, bitoffset, bitlen) \
    ((U8) ((0xFF & (((U8)(u)) << (8 - bitlen))) \
    >> (bitoffset)))

#define U16_RESTORE_OFFSET(u, bitoffset, bitlen) \
    ((U16) ((0xFFFF & (((U16)(u)) << (16 - bitlen))) \
    >> (bitoffset)))

#define U32_RESTORE_OFFSET(u, bitoffset, bitlen) \
    ((U32) ((0xFFFFFFFF & (((U32)(u)) << (32 - bitlen))) \
    >> (bitoffset)))



#endif

