#include <stdlib.h>
#include <math.h>
#include "UDFpltStr.h"
#include "UDFpltAnsi.h"
#include "gph_opind.h"

/*****************************************************************************/
/*  Note that in its default mode - this routine is set up to draw a 3D      */
/*  Earth in a plot in which phase is plotted along X and elevation along    */
/*  Y.  When axis are reversed then the window needs to be rotated by 90     */
/*  degrees and the flipped about the new X axis for realignment.            */
/*                                                                           */
/*  This routine assumes that the satellite spin axis is parallel to the     */
/*  elevation axis of the plot and orients the globe accordingly.  This is   */
/*  done by rotating the window in which the globe is drawn so that the spin */
/*  axis is parallel to the phase axis and then rotating the globe so that   */
/*  the projected geographic longitude of the spin axis is pointed toward    */
/*  maximum side of that axis.                                               */
/*****************************************************************************/

ByTe_1 EarthWin (ByTe_4 N, struct ToolInfo *Ti, ReaL_4 *Sp, ReaL_4 A, 
               ReaL_4 gLaT, ReaL_4 gLnG, ReaL_4 gLaTs, ReaL_4 gLnGs)
{
   extern struct UserDefs InFo;
   extern struct config parm;
   extern struct mapinfo map;

   struct PlotDef *Pd;
   struct AxisInfo *aI;

   ReaL_4 X[2], Y[2], RoT;
   ReaL_4 SubSat;
   ReaL_4 RToD;
   ReaL_4 Sx[2], Sy[2];
   ReaL_4 x[2], y[2];
   ByTe_4 I, J;
   ByTe_1 FlipX = 0, FlipY, Yaxis;

   Pd = (struct PlotDef *)InFo.PlotDefs + N;           /* plot info struct   */
   RToD = 180.0 / M_PI;                                /* radians to degrees */

   /**************************************************************************/
   /* Compute the out of plane angle of the satellite spin axis.  Compensate */
   /* for the out of plane angle by rotating window.                         */
   /**************************************************************************/

   RoT = -asin (Sp[2] / sqrt (Sp[0] * Sp[0] + Sp[1] * Sp[1] + Sp[2] * Sp[2]));
   RoT *= RToD;

   /**************************************************************************/
   /* If the x axis is elevation then we need to rotate the window by 90 deg */
   /**************************************************************************/

   if (Pd->AData[0] != 5) {
      RoT -= 90.0;
      FlipX = (FlipX + 1) % 2;
   }

   /**************************************************************************/
   /* determine the maximum and minumum values being plotted on each axis    */
   /**************************************************************************/

   x[0] = (Pd->Xb.Min < Pd->Xb.Max) ? Pd->Xb.Min : Pd->Xb.Max;
   x[1] = (Pd->Xb.Min < Pd->Xb.Max) ? Pd->Xb.Max : Pd->Xb.Min;
   if ((Pd->Yr.Min == Pd->Yl.Min) || (Pd->Yr.Min == -Pd->Yl.Min)) { 
       aI = &Pd->Yr;
       Yaxis = 6;
   } else {
       aI = &Pd->Yl;
       Yaxis = 4;
   }
   y[0] = (aI->Min < aI->Max) ? aI->Min : aI->Max;
   y[1] = (aI->Min < aI->Max) ? aI->Max : aI->Min;
   I = (Pd->Xb.Min < Pd->Xb.Max) ? 0 : 1;
   J = (aI->Min < aI->Max) ? 0 : 1;

   /**************************************************************************/
   /* set plot universe to azimuthal cartagraphic                            */
   /**************************************************************************/

   parm.univ = 15;

   /**************************************************************************/
   /* Flip Axes flags set if the max and min scaling on the axis is reversed */
   /* from the norm.                                                         */
   /*                                                                        */
   /* Also establish the window placement and scaling according to whether   */
   /* the angular radius of the earth will fit within the parent window or   */
   /* not.                                                                   */
   /**************************************************************************/

   FlipX = (FlipX + I) % 2;
   X[I] = (-A < x[0]) ? x[0] : -A;
   Sx[0] = (-A < x[0]) ? 180.0 * A / x[0] : -180.0;
   I = (I + 1) % 2;
   X[I] = (A > x[1]) ? x[1] : A;
   Sx[1] = (A > x[1]) ? 180.0 * A / x[1] : 180.0;
   
   FlipY = J;
   Y[J] = (-A < y[0]) ? y[0] : -A;
   Sy[0] = (-A < y[0]) ? 180.0 * A / x[0] : -180.0;
   J = (J + 1) % 2;
   Y[J] = (A > y[1]) ? y[1] : A;
   Sy[1] = (A > y[1]) ? 180.0 * A / y[1] : 180.0;
   
   /**************************************************************************/
   /* Establish the window                                                   */
   /**************************************************************************/

   set_axis (Pd->Pn, 0, Yaxis, 8);
   set_relwin (50, Pd->Wn, X[0], Y[0], 0., X[1], Y[1], 0., 
                                     Sx[0], Sy[0], .0, Sx[1], Sy[1], 0.);

   /**************************************************************************/
   /* Turn on clipping if the window scaling is not maxed out                */
   /**************************************************************************/

   if (Sx[0] != -180.0 || Sy[0] != -180.0 || Sx[1] != 180.0 || Sy[1] != 180.0)
      clip (50, 2);

   /**************************************************************************/
   /* Set up the the globe orientation                                       */
   /**************************************************************************/

   map.dlong = Ti->DeltaLong;
   map.rotlong = (gLnGs < 0.0) ? -270.0 - gLnGs : 90 - gLnGs;
   SubSat = gLnG + map.rotlong;
   if (SubSat < 0.0) SubSat += 360.0;
   map.center = (SubSat < 90.0 || SubSat > 270.0 ) ? -gLaT : -180 + gLaT;
   map.dlat = Ti->DeltaLat;
   map.Nlastlat  = Ti->LatEnds; 
   map.Slastlat  = -Ti->LatEnds;

   /**************************************************************************/
   /* Do any flipping of the window axes                                     */
   /**************************************************************************/

   rot_win (50, 'z', -RoT, 0);
   if (FlipX)
      inv_win (50, 'x', 1);
   if (FlipY)
      inv_win (50, 'y', 1);

   return (50);
}
