# TUtimeJulian converts an input time into modified Julian time.  This is 
# returned as element 0 in the output array.  In addition if Compute is set
# to 1 the routine will compute the tranformation matrices used in a number 
# of coordinate transformations.  These are handy in a compute once use many 
# times senario and are required when calling the TclUtil coordinate 
# transformation routine which converts between  GEI, GEO, MAG, SM, GSE.
#
# INPUTS
#
#  yR      - year
#  dY      - day of year
#  mS      - UT milliseconds of day
#  Compute - 0: don't compute tranformation matrices
#            1: compute the tranformation matrices
#
# OUTPUTS 
#   
#  OuT  - Element 0:      MjD
#       - Elements 1-9:   T1
#       - Elements 10-18: T2
#       - Elements 19-27: T3
#       - Elements 28-36: T4
#       - Elements 37-45: T5

package provide TclUtils 1.0

proc TUtimeJulian { yR dY mS OuT {Compute 0} } {
   global RtoD PI

   upvar $OuT A

   set fD [expr $mS / 3600000.0 / 24.0 ]

# MODIFIED Julian day number from 00UT Nov 17, 1858

   set MjD [expr 45.0 + ($yR - 1859) * 365.0 + int(($yR - 1861)/4.0)]  
   set MjD [expr $MjD + $dY - 1.0 + $fD]  
   set A(0) $MjD
   set MjD [expr ($MjD - 51544.5)/36525.0]

   if { $Compute == 1 } {

# SET the Univeral Time.

      set uT [expr $mS / 3600000.0 ]

# SET up tranformation matrix 1

      set Ang(0) [expr 100.461 + 36000.770 * $MjD + 15.04107 * $uT]
      set Ang(0) [expr fmod($Ang(0), 360.0) ]
      TUmatrixRot $Ang(0) 0.0 0.0 A Z 1
                                                                                
# SET up tranformation matrix 2

      set Ang(0) [expr 23.439 - 0.013 * $MjD]
      set Ang(0) [expr fmod($Ang(0), 360.0) ]
      set Ut [expr 0.04107 * $uT]
      set M [expr fmod((357.528 + 35999.050 * $MjD + $Ut), 360.0) / $RtoD]
      set L [expr fmod((280.460 + 36000.772 * $MjD + $Ut), 360.0)]
      set Ang(1) [expr $L + (1.915 - 0.0048 * $MjD) * sin($M) + \
                       0.02 * sin (2.0 * $M)]
      set Ang(1) [expr fmod($Ang(1), 360.0) ]
      TUmatrixRot $Ang(0) 0.0 0.0 M1 X 
      TUmatrixRot $Ang(1) 0.0 0.0 M2 Z 
      TUmatrixMath M2 * M1 A 3 3 3 3 0 0 10
            
# SET up tranformation matrix 3  (IGRF 2000)

      set G01 -29615.0
      set H11   5186.0 
      set G11  -1728.0
      set L [expr atan($H11 / $G11)]
      set P [expr $PI/2.0 - atan(($G11 * cos($L) + $H11 * sin($L)) / $G01)]
      set Q1(0) [expr cos($P) * cos($L)]
      set Q1(1) [expr cos($P) * sin($L)]
      set Q1(2) [expr sin($P)]
      TUmatrixMath A TRANSPOSE A M3 3 3 3 3 1 0 0
      TUmatrixMath A * M3 M4 3 3 3 3 10 0 0
      TUmatrixMath M4 * Q1 Q2 3 3 3 1 0 0 0
      set Ang(0) [expr -atan($Q2(1) / $Q2(2)) * $RtoD]
      TUmatrixRot $Ang(0) 0.0 0.0 A X 19
            
# SET up tranformation matrix 4

      set YZ [expr sqrt($Q2(1) * $Q2(1) + $Q2(2) * $Q2(2))]
      set Ang(0) [expr -atan($Q2(0) / $YZ) * $RtoD]
      TUmatrixRot $Ang(0) 0.0 0.0 A Y 28

# SET up tranformation matrix 5

      set Ang(0) [expr $P * $RtoD - 90.0]
      set Ang(1) [expr $L * $RtoD]
      TUmatrixRot $Ang(0) 0.0 0.0 M1 Y 
      TUmatrixRot $Ang(1) 0.0 0.0 M2 Z 
      TUmatrixMath M1 * M2 A 3 3 3 3 0 0 37
   }
}
