#  Routine adapted from NUMERICAL RECIPES IN C
#
#  A non-linear least squares fit routine using the Levenberg-Marquart
#  method of solution. The routine attempts to reduce the ChiSq of a fit 
#  between the actual and measured values for a known functional dependence. 
#  The measured values are allowed a standard deviation Sq.  The known
#  function is described by a number of coefficients Aa.
#
#  User needs to supply a routine FunC that evaluates the fitting function
#  returning the value through the call.  It also needs to return its
#  derivitives.  The function has the form
#
#                      Y = FunC nA dYdA Ops
#  
#  A are the coefficients,  dYdA are the derivitives and Y is the measured 
#  value based on the input data.  The input data should be globally
#  available to FunC 
#
#  Inputs
#     Vv - Measured result or the function output 
#     Sg - Weighting applied to each variable
#     nP - Number of points in Vv
#     Aa - best fit coefficients - needs initial guess
#     iA - coefficient flags
#          0:  do not fit this coefficient
#          1:  fit this coefficient
#     mA - total number of coefficients
#     Ops - Input/Output Options and scratch space.  This array is used to
#           pass information back and forth between the calling program
#           and the procedure and to provide static scratch space.  The 
#           elements are:
#             0:  Fitting Test - set < 0 for first call to let procedure do
#                 initializations and set to 0 for last call to get the
#                 covariance and curvature matrices. Procedure sets varaible
#                 during iterations  
#             1:  Current Chi Squared Value           
#             2:  Previous Chi Squared Value
#             3:  The number of coefficients being fit - computed during
#                 the first call from iA
#             4:  The current data set being evaluated
#             5:  Scratch space starts here.  The first mA values hold the
#                 the current guess of the coefficients being used by the
#                 procedure.  The next mA*mA coefficients hold the 
#                 Covarience Matrix which only has meaning if Ops(0) has
#                 been set to 0. The next mA*mA coefficients hold the 
#                 Curvature Matrix which only has meaning if Ops(0) has
#                 been set to 0.  After this is pure scratch space.

package provide TclUtils_C 1.0

proc TUdataNLinLSq { Vv Sg nP Aa Ac mA FunCNamE Ops } {

   upvar $Vv  V
   upvar $Sg  SiG
   upvar $Aa  A
   upvar $Ac  iA
   upvar $Ops oP

# Set up the offsets to the Covariance (cBeg), Curvature (aBeg), and the 
#   start of the Scratch Space (sBeg)

   set TmP { $mA dYdA oP }
   set TmP  "\[ $FunCNamE $TmP \]"
   set FunC "set FuncRet $TmP"

   set mA2 [expr $mA * $mA]
   set cBeg [expr 5 + $mA]
   set aBeg [expr $cBeg + $mA2]
   set sBeg [expr $aBeg + $mA2]
   set dBeg [expr $sBeg + $mA2]

# Initialize for new set of trials if oP(0), the step size is negative.  
# This consists of:
# 
# 1. Determining how many coefficients there are to fit.
# 2. Initializing oP(0)
# 3. Copying the coefficients into the temporary coefficient area
# 4. Computing the initial ChiSq value.
# 5. Initializing the previous chi squared value to current one.

   if { $oP(0) < 0.0 } {
      set oP(3) 0
      for { set J 0 } { $J < $mA } { incr J } { if { $iA($J) } { incr oP(3) }}

      set sV [expr abs($oP(0))]
      if { $sV > .9 } { set oP(0) .001 } else { set oP(0) $sV }

      set I 5
      for { set J 0 } { $J < $mA } { incr J ; incr I } { set oP($I) $A($J) }

      TUdataMqCoef V SiG $nP iA $mA $aBeg $sBeg $FunC oP
      set oP(2) $oP(1)
   }

# Set up a couple of  convenient loop ends

   set cEnD [expr $cBeg + $oP(3) * $oP(3)]
   set dEnD [expr $dBeg + $oP(3)]

#  Get the current covariance matrix

   for {set I $aBeg ; set J $cBeg} {$J < $cEnD} { incr I ; incr J } {
      set oP($J) $oP($I)
   }

# Set up the current approximation vector

   set J  $sBeg
   for {set I $dBeg} {$I < $dEnD} {incr I ; incr J } { 
      set oP($I) $oP($J) 
   }

# If there is a non-zero step size the we need to make an iteration run,
# otherwise just compute and size the covariance and curvature matrices

   if { $oP(0) > 0.0 } {
   
# Change the linearized fitting matrix by augmenting the diagonal elements
# of the covariance matrix by the step size, oP(0).

      set Inc [expr $oP(3) + 1]
      for {set I $cBeg} {$I < $cEnD} {incr I $Inc } {
         set oP($I) [expr $oP($I) * (1.0 + $oP(0))]
      }

#  Compute new covariance matrix

      TUmatrixInv  $oP(3) oP $cBeg 1 oP $dBeg

# Establish the new coefficients to test

      set J $dBeg
      for { set I 5 ; set L 0 } { $L < $mA } { incr L ; incr I } {
         if { $iA($L) } { 
            set oP($I) [expr $A($L) + $oP($J)]
            incr J
         }
      }

# Now try them out and get the resultant ChiSq

      TUdataMqCoef V SiG $nP iA $mA $cBeg $dBeg $FunC oP

# If the new value of ChiSq is less than the last one then we are moving
# in the right direction - keep going.  Update all values to the current
# ones.

      if { $oP(1) < $oP(2) } {
         set oP(0) [ expr $oP(0) * 0.1 ]
         set oP(2) $oP(1)

         for {set I $aBeg ; set J $cBeg} {$J < $cEnD} { incr I ; incr J } {
            set oP($I) $oP($J)
         }

         set J  $sBeg
         for {set I $dBeg} {$I < $dEnD} {incr I ; incr J } { 
            set oP($J) $oP($I) 
         }

         for { set I 5 ; set J 0 } { $I < $cBeg } { incr I ; incr J } {
            set A($J) $oP($I)
         }
      } else {
         set oP(0) [ expr $oP(0) * 10.0 ]
         set oP(1) $oP(2)
      }

   } else {
      TUmatrixInv  $oP(3) oP $cBeg

#  Expand the CoVariance and Curvature matrices to match the number of 
#  coefficients in the function.  This only needs to be done if all 
#  of the coefficients are not being fit (ie. some were being held constant)

      if { $mA !=  $oP(3) } {

         set InC [expr $mA - $oP(3)]

         set I $cBeg
         set J $sBeg
         set EnD [expr $I + $oP(3) * $oP(3)]
         for { } { $I < $EnD } { incr I ; incr J } { set oP($J) $oP($I) }

         set EnD [expr $cBeg + $mA2]
         for { set I $cBeg } { $I < $EnD } { incr I } { set oP($I) 0.0 }

         set I $cBeg
         set L $sBeg
         for { set J 0 } { $J < $oP(3) } { incr J } {
            for { set K 0 } { $K < $oP(3) } { incr K ; incr L ; incr I } {
               set oP($I) $oP($L) 
            }
            incr I $InC
         }

         set I $aBeg
         set J $sBeg
         set EnD [expr $I + $oP(3) * $oP(3)]
         for { } { $I < $EnD } { incr I ; incr J } { set oP($J) $oP($I) }

         set EnD [expr $aBeg + $mA2]
         for { set I $aBeg } { $I < $EnD } { incr I } { set oP($I) 0.0 }

         set I $aBeg
         set L $sBeg
         for { set J 0 } { $J < $oP(3) } { incr J } {
            for { set K 0 } { $K < $oP(3) } { incr K ; incr L ; incr I } {
               set oP($I) $oP($L) 
            }
            incr I $InC
         }
         
         set K [expr $oP(3) - 1]
         for { set J [expr $mA - 1] } { $J >= 0 } { incr J -1 } {
            if { $iA($J) } {
               set Pc1 [expr $cBeg + $K] 
               set Pa1 [expr $aBeg + $K]
               set Pc2 [expr $cBeg + $J] 
               set Pa2 [expr $aBeg + $J]
               for { set I 0 } { $I < $mA } { incr I } {
                  TUdataSwap oP($Pc1) oP($Pc2)
                  TUdataSwap oP($Pa1) oP($Pa2)
                  incr Pc1 $mA
                  incr Pc2 $mA
                  incr Pa1 $mA
                  incr Pa2 $mA
               }
               set Pc1 [expr $cBeg + $K * $mA]
               set Pc2 [expr $cBeg + $J * $mA]
               set Pa1 [expr $aBeg + $K * $mA]
               set Pa2 [expr $aBeg + $J * $mA]
               for { set I 0 } { $I < $mA } { incr I } {
                  TUdataSwap oP($Pc1) oP($Pc2)
                  TUdataSwap oP($Pa1) oP($Pa2)
                  incr Pc1
                  incr Pc2
                  incr Pa1
                  incr Pa2
               }
               incr K -1
            }
         }
      }    
   }
}
