#
#  Procedure
#    make a least squares fit to data with a polynomial curve of the form
#       v = a[0] + a[1]*x + a[2]*y + a[3]*z + a[4]*x**2 + a[5]*xy + a[6]*xz ...
#
#  usage
#      set nCoefs  [TUdataLSq3D X Y Z V nP nOrder Mode Coef Var SigY]
#
#  parameter description
#     X        array of data points for independent variable
#     Y        array of data points for independent variable
#     Z        array of data points for independent variable
#     V        array of data points for dependent variable
#     nP       number of pairs of data points
#     nOrder   order of polynomial (0, 1 ... )
#     Mode     determines the method of weighting least-squares fit
#                       +1 (instrumental) weight[i] = 1/sigmay[i]**2
#                        0 (no weighting) weight[i] = 1
#                       -1 (statistical)  weight[i] = 1/y[i]
#     Coef     array of coefficients of polynomial
#     Var      Compute and return variance
#                 0 - NO
#                 1 - Yes
#     SigZ     array of standard deviations for z data points
#              (not passed if not used)
#     nCoefs   Number of coefficients produced in the fit
#

package provide TclUtils 1.0

proc TUdataLSq3D { XX YY ZZ VV nP Mode cF icF oP CoVar FNamE { SiGY YY } } {

   upvar $XX X
   upvar $YY Y
   upvar $ZZ Z
   upvar $VV V
   upvar $oP P
   upvar $cF A
   upvar $icF iA
   upvar $CoVar cM

#  BUILD the coeffient function procedure call
                     
   set I 0
   if ![info exists P(NC)] { set P(NC) 0 }
   set TmP { 0.0 0.0 0.0 tC $P(OR) $P(NC) }
   set TmP "\[ $FNamE $TmP \]"
   set FunC "set nC $TmP"
   eval $FunC
   set P(NC) $nC
   set TmP { $X($I) $Y($I) $Z($I) tC $P(OR) $P(NC) }
   set FunC "$FNamE $TmP"

#  FIRST see how many of the coefficiencts need to be fit.  Should that be
#      0 then we have a problem.  Give error and leave. If iCoef is undefined
#      then we need to fit them all

   if { [info exists iA] == 0 } { 
      for { set I 0 } { $I < $nC } { incr I } { set iA($I) 1 }
      set mFit $nC
   } else {
      set mFit 0
      for { set I 0 } { $I < $nC } { incr I } { if $iA($I) { incr mFit } }
      if { $mFit == 0 } { 
         puts stderr "TUdataLSq1D:  ERROR - No coefficients to fit"
         return -1
      }
   }

# INITIALIZE the covariance matrix as well as the solution vector

   set mS [expr $mFit * $mFit]
   for {set I 0 } {$I < $mS} { incr I } { set cM($I) 0.0 }
   for {set I 0 } {$I < $mFit} { incr I } { set sV($I) 0.0 }

#  BUILD the weighting arrray a the weighting array

   switch -exact -- $Mode {
      0  {  for { set I 0 } { $I < $nP } { incr I } { set S($I) 1.0 } }
      1  {  upvar $SiGY S }
      default  {
         set Pw [expr -$Mode / 2.0 ]
         for { set I 0 } { $I < $N } { incr I } {
            set W [expr $X($I) * $X($I) + $Y($I) * $Y($I) + $Z($I) * $Z($I)]
            if { $W != 0.0 } {
               set S($I) [expr 1.0 / pow ($W, $Pw)]
            } else { set S($I) 1.0 }
         }
      }
   }

#  BUILD the coeffient function procedure call
                     
   set tW 0.0
   for { set I 0 } { $I < $nP } { incr I } {
      eval $FunC
      set yM $V($I)
      if { $mFit < $nC } {
         for { set J 0 } { $J < $nC } { incr J } {
            if { !$iA($J) } { set yM [expr $yM - $A($J) * $tC($J)] }
         }
      }

      set EnD 1
      set J 0
      set K 0
      set tW [expr $tW + $S($I)]
      for { set L 0 } { $L < $nC } { incr L ; incr EnD } {
         if { $iA($L) } {
	    set wT [expr $tC($L) * $S($I)]
            set pO [expr $J * $mFit]
            for { set M 0 } { $M < $EnD } { incr M } {
               if { $iA($M) } {
                  set cM($pO) [ expr $cM($pO) + $wT * $tC($M)]
                  incr pO
               }
            }
            set sV($K) [ expr $sV($K) + $yM * $wT]
            incr J
            incr K
         }
      }
   }
   set cM(0) $tW

# FILL in the covariance matrix

   for { set J 1 } { $J < $mFit } { incr J } {
       set pO [expr $J * $mFit]
       for { set K 0 } { $K < $J } { incr K ; incr pO } {
          set P1 [expr $K * $mFit + $J]
          set cM($P1) $cM($pO)
       }
   }

# DO the matrix solution

   TUmatrixInv $mFit cM 0 1 sV 0

# GET the coefficients

   for { set J 0; set I 0 } { $J < $nC } { incr J } {
      if { $iA($J) } { set A($J) $sV($I) ; incr I }
   }

# COMPUTE Chi Squared

   set ChiSq 0.0
   for { set I 0 } { $I < $nP } { incr I } {
      eval $FunC
      set Sum 0.0
      for { set J 0 } { $J < $nC } { incr J } { 
         set Sum [expr $Sum + $A($J) * $tC($J)]
      }
      set vR [expr ($V($I) - $Sum) / $S($I)]
      set ChiSq [expr $ChiSq + $vR * $vR] 
   }


#  EXPAND the CoVariance matrix to match the number of coefficients in 
#  the function.  This only needs to be done if all of the coefficients 
#  were not being fit (ie. some were being held constant)

   if { $nC !=  $mFit } {

# REBUILD the Covarience matrix

      set L 0
      for { set I 0 } { $I < $mFit } { incr I } {
         set K [expr $I * $nC]
         for { set J 0 } { $J < $mFit } { incr J ; incr K } {
             set cM($K) $cM($L)
             incr L
         }
      }

# ADD missing elements 

      for { set I $mFit } { $I < $nC } { incr I } {
         set P1 $I 
         set P2 [expr $I * $nC] 
         for { set J 0 } { $J <= $I } { incr J ; incr P1 $nC ; incr P2 } {
            set cM($P1) 0.0
            set cM($P2) 0.0
         }
      }

# SHUFFLE elements 

      set K [expr $mFit - 1]
      for { set J [expr $nC - 1] } { $J >= 0 } { incr J -1 } {
         if { $iA($J) } {
            set P1 $K
            set P2 $J
            for { set I 0 } { $I < $nC } { incr I } {
               TUdataSwap cM($P1) cM($P2)
               incr P1 $nC
               incr P2 $nC
            }
            set P1 [expr $K * $nC]
            set P2 [expr $J * $nC]
            for { set I 0 } { $I < $nC } { incr I ; incr P1 ; incr P2 } {
               TUdataSwap cM($P1) cM($P2)
            }
            incr K -1
         }
      }
   }

   return $ChiSq
}
