#
#  Procedure
#    make a least squares fit to data with an arbitrary function form of 
#       y = F($X($I), xV, nA)
#
#  usage
#      set ChiSq [TUdataLSq1D X Y nP Mode Coef iCoef nC FunC SigY]
#
#  parameter description
#     X        array of data points for independent variable
#     Y        array of data points for dependent variable
#     nP       number of pairs of data points
#     Mode     determines the method of weighting least-squares fit
#                       +1 (instrumental) weight[i] = 1/SigY[i]**2
#                        0 (no weighting) weight[i] = 1
#                       -1 (statistical)  weight[i] = 1/y[i]
#     Coef     array of coefficients of polynomial (OUTPUT)
#     iCoef    array of coefficient status (1 = SOLVE, 0 = SKIP)
#     nC       number of coefficients
#     CoVar    covarience matrix
#     FunC     the function which determines the X value for each coefficient
#     SigY     array of standard deviations for y data points
#              (not passed if not used)
#

package provide TclUtils_C 1.0

proc TUdataLSq1D { XX YY nP Mode Coef iCoef nC CoVar FunCNamE { SiGY YY } } {

   upvar $XX X
   upvar $YY Y
   upvar $Coef A
   upvar $iCoef iA
   upvar $CoVar cM

#  FIRST see how many of the coefficiencts need to be fit.  Should that be
#      0 then we have a problem.  Give error and leave. If iCoef is undefined
#      then we need to fit them all

   if { [info exists iA] == 0 } { 
      for { set I 0 } { $I < $nC } { incr I } { set iA($I) 1 }
      set mFit $nC
   } else {
      set mFit 0
      for { set I 0 } { $I < $nC } { incr I } { if $iA($I) { incr mFit } }
      if { $mFit == 0 } { 
         puts stderr "TUdataLSq1D:  ERROR - No coefficients to fit"
         return -1
      }
   }

# INITIALIZE the covariance matrix as well as the solution vector

   set mS [expr $mFit * $mFit]
   for {set I 0 } {$I < $mS} { incr I } { set cM($I) 0.0 }
   for {set I 0 } {$I < $mFit} { incr I } { set sV($I) 0.0 }

#  BUILD the weighting arrray a the weighting array

   switch -exact -- $Mode {
      -1  {  for { set I 0 } { $I < $nP } { incr I } {
                if { $Y($I) != 0.0 } { 
                    set W($I) [expr 1.0 / $Y($I)]  
                } else { set W($I) 1.0 }
             }
             upvar 0 W S
          }
       0  {  for { set I 0 } { $I < $nP } { incr I } { set W($I) 1.0 }   
             upvar 0 W S
          }
       1  {  upvar $SiGY S                                               
             for { set I 0 } { $I < $nP } { incr I } {
               set W($I) [expr 1.0 / ( $S($I) * $S($I) ) ]  
             }
          }
   }

#  BUILD the coeffient function procedure call
                     
   set TmP { $X($I) tC $nC }
   set FunC "$FunCNamE $TmP"

   for { set I 0 } { $I < $nP } { incr I } {
      eval $FunC
      set yM $Y($I)
      if { $mFit < $nC } {
         for { set J 0 } { $J < $nC } { incr J } {
            if { !$iA($J) } { set yM [expr $yM - $A($J) * $tC($J)] }
         }
      }
     
      set EnD 1
      set J 0
      set K 0
      for { set L 0 } { $L < $nC } { incr L ; incr EnD } {
         if { $iA($L) } {
            set wT [expr $tC($L) * $W($L)]
            set P [expr $J * $mFit]
            for { set M 0 } { $M < $EnD } { incr M } {
               if { $iA($M) } {
                  set cM($P) [ expr $cM($P) + $wT * $tC($M)]
                  incr P
               }
            }
            set sV($K) [ expr $sV($K) + $yM * $wT]
            incr J
            incr K
         }
      }
   }

# FILL in the covariance matrix

   for { set J 1 } { $J < $mFit } { incr J } {
       set P [expr $J * $mFit]
       for { set K 0 } { $K < $J } { incr K ; incr P } {
          set P1 [expr $K * $mFit + $J]
          set cM($P1) $cM($P)
       }
   }

# DO the matrix solution

   TUmatrixInv $mFit cM 0 1 sV 0

# GET the coefficients

   for { set J 0; set I 0 } { $J < $nC } { incr J } {
      if { $iA($J) } { set A($J) $sV($I) ; incr I }
   }

# COMPUTE Chi Squared

   set ChiSq 0.0
   for { set I 0 } { $I < $nP } { incr I } {
      eval $FunC
      set Sum 0.0
      for { set J 0 } { $J < $nC } { incr J } { 
         set Sum [expr $Sum + $A($J) * $tC($J)]
      }
      set V [expr ($Y($I) - $Sum) / $S($I)]
      set ChiSq [expr $ChiSq + $V * $V] 
   }


#  EXPAND the CoVariance matrix to match the number of coefficients in 
#  the function.  This only needs to be done if all of the coefficients 
#  were not being fit (ie. some were being held constant)

   if { $nC !=  $mFit } {

# REBUILD the Covarience matrix

      set L 0
      for { set I 0 } { $I < $mFit } { incr I } {
         set K [expr $I * $nC]
         for { set J 0 } { $J < $mFit } { incr J ; incr K } {
             set cM($K) $cM($L)
             incr L
         }
      }

# ADD missing elements 

      for { set I $mFit } { $I < $nC } { incr I } {
         set P1 $I 
         set P2 [expr $I * $nC] 
         for { set J 0 } { $J <= $I } { incr J ; incr P1 $nC ; incr P2 } {
            set cM($P1) 0.0
            set cM($P2) 0.0
         }
      }

# SHUFFLE elements 

      set K [expr $mFit - 1]
      for { set J [expr $nC - 1] } { $J >= 0 } { incr J -1 } {
         if { $iA($J) } {
            set P1 $K
            set P2 $J
            for { set I 0 } { $I < $nC } { incr I } {
               TUdataSwap cM($P1) cM($P2)
               incr P1 $nC
               incr P2 $nC
            }
            set P1 [expr $K * $nC]
            set P2 [expr $J * $nC]
            for { set I 0 } { $I < $nC } { incr I ; incr P1 ; incr P2 } {
               TUdataSwap cM($P1) cM($P2)
            }
            incr K -1
         }
      }
   }

   return $ChiSq
}
