/*  The C equivalent of TUmagGei,  It consists of a TCL interface over      */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  There are 4 OBJV elements.  These are:                                  */
/*     OBJV[0]:  Routine name                                               */
/*     OBJV[1]:  The MAG vector (MaG)                                       */
/*     OBJV[2]:  The transfer information array (xInfo)                     */
/*           Index 0: Mean Julian Day (MjD)                                 */
/*           Index 1: Greenwich mean Siderial Time (GsT)                    */
/*           Index 2: cos(GsT)                                              */
/*           Index 3: sin(GsT)                                              */
/*     OBJV[3]:  The GeI vector (GeI)                                       */

#include <tcl.h>
#include <stdlib.h>
#include <math.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int MagToGeiCmd (ClientData cD, Tcl_Interp *tIntp, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   void *memPtr = NULL; 
   ReaL_8 *GeI, *MaG, *xInfo;

   ByTe_4 Bytes, rV;

/* Make sure that all the IO parameters are present */

   if (objc != 4 ) {
     Tcl_WrongNumArgs(tIntp, 1, objv, "Usage: TUmagGEI MAG XInfo GEI");
     return TCL_ERROR;
   }

/* Now lets to all the necessary mallocs at once                           */

   Bytes = 10 * sizeof(ReaL_8);
   if ((memPtr = malloc (Bytes)) == NULL)
     return TCL_ERROR;

/* OK now lets assign all the pointers into that malloc                   */

   MaG = (ReaL_8 *)memPtr; 
   GeI = (ReaL_8 *)memPtr + 3; 
   xInfo = (ReaL_8 *)memPtr + 3; 

/* Transfer array data from the TCL array to the C array                   */

   if (TclArrayToC (tIntp, objv[1], (void *)MaG, 3, 0, 'D') == 0) {
      free (memPtr);
      return TCL_OK;
   }
   if (TclArrayToC (tIntp, objv[2], (void *)xInfo, 4, 0, 'D') == 0) {
      free (memPtr);
      return TCL_OK;
   }

/* Call the C routine                                                       */

   rV =  MagToGei_C (MaG, xInfo, GeI);

/* Reconvert to the Grid and Norm C Arrays to Tcl Arrays                    */

   CArrayToTcl (tIntp, objv[3], GeI, 3, 0, 'D'); 

/* Get rid of the temporary array and return                                */

   free (memPtr);
   return TCL_OK;
}


/* THE C ROUTINE                                                            */


ByTe_4 MagToGei_C (ReaL_8 *MaG, ReaL_8 *xInfo, ReaL_8 *GeI)
{

    register ReaL_8 *fPt;
    ReaL_8 tX, tY, tZ;
    ReaL_8 X, Y, Z;

    tX = xInfo[8] * MaG[0] - xInfo[9] * MaG[2];
    tY = MaG[2];
    tZ = xInfo[9] * MaG[0] + xInfo[8] * MaG[2];

    X = xInfo[5] * tX - xInfo[6] * tY;
    Y = xInfo[6] * tX + xInfo[5] * tY;
    Z = tZ;

    fPt = GeI;
    *fPt++ = xInfo[2] * X - xInfo[3] * Y;
    *fPt++ = xInfo[3] * X + xInfo[2] * Y;
    *fPt = Z;

    return 1;
}
