/*  The C equivalent of TUgridSmooth,  It consists of a TCL interface over  */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  There are 4 OBJV elements.  These are:                                  */
/*     OBJV[0]:  Routine name                                               */
/*     OBJV[1]:  The Grid array (Grid)                                      */
/*     OBJV[2]:  Smooth Info Array                                            */
/*                 0: Layers: 0 = No Smoothing
/*                 1: Distances beyond this value are ignored
/*                 2: Weighting Factor (RVal(1))                            */
/*                 3: Min X grid position                                   */
/*                 4: Max X grid position                                   */
/*                 5: Min Y grid position                                   */
/*                 6: Max Y grid position                                   */
/*     OBJV[3]:  Grid Info Array                                            */
/*                 0:  Beginning X grid position                            */
/*                 1:  Ending X grid position                               */
/*                 2:  Beginning Y grid position                            */
/*                 3:  Ending Y grid position                               */
/*                 4:  Number of grids along X                              */
/*                 5:  Number of grids along Y                              */
/*                 6:  Use POINT or BAND method to store X data             */
/*                 7:  Use POINT or BAND method to store Y data             */
/*                 8:  Storage Method ROW or COLUMN                         */
/*                 9:  IGNORE or KEEP Zeros when averaging                  */
/*                10:  NEW: initialize grid                                 */
/*                     ADD: add data to grid                                */
/*                     END: normalize grid                                  */
/*                     or any combination as ADDEND or NEWADDEND            */
/*                11:  Value to set unfilled grids to                       */
/*                12:  Bad grid value                                       */
/*                13:  Remove Data below this value                         */
/*                14:  Remove Data above this value                         */
/*                15:  X Data is Cyclic (YES, NO)                           */
/*                16:  Y Data is Cyclic (YES, NO)                           */
/*                                                                          */
/*  The 2D Least Squares uses only a subset of this info.  These are        */
/*  packaged into a single array for passage to the C routine. These are    */
/*                                                                          */
/*      gI(0)         0:  Beginning X grid position                         */
/*      gI(1)         1:  Ending X grid position                            */
/*      gI(2)         2:  Beginning Y grid position                         */
/*      gI(3)         3:  Ending Y grid position                            */
/*      gI(4)         4:  Number of grids along X                           */
/*      gI(5)         5:  Number of grids along Y                           */
/*      gI(6)        11:  Value to set unfilled grids to                    */
/*      gI(7)         8:  Storage Method                                    */
/*      gI(8)        15:  Cyclic in X                                       */
/*      gI(9)        16:  Cyclic in Y                                       */
/*                                                                          */

#include <tcl.h>
#include <stdlib.h>
#include <math.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int GridSmoothCmd (ClientData cD, Tcl_Interp *tIntp, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   void *memPtr1 = NULL;
   ReaL_8 RVal[2];
   ReaL_8 *Grid;
   ReaL_8 gI[11]; 
   int TmpI;
   register ByTe_4 I;
   ByTe_4 Layers;
   ByTe_4 TotGrids, Bytes, rV, EdgE[4];

   int Flg = TCL_LEAVE_ERR_MSG;

   char *aName, *Value, Index[10];

/* Make sure that all the IO parameters are present */

   if (objc != 4 ) {
     Tcl_WrongNumArgs(tIntp, 1, objv, "Usage: TUgridSmooth Grid sInfo gInfo");
     return TCL_ERROR;
   }

   aName = Tcl_GetString(objv[2]);
   for (I = 0; I <= 9; ++I) {
      sprintf (Index, "%d", I);                                            
      if ((Value = Tcl_GetVar2(tIntp, aName, Index, Flg)) != NULL) {
         switch (I) {
            case 0:
              Tcl_GetInt (tIntp, Value, &TmpI);
              Layers = TmpI;
            break;
            case 1:
              Tcl_GetDouble (tIntp, Value, &RVal[0]);
            break;
            case 2:
              Tcl_GetDouble (tIntp, Value, &RVal[1]);
            case 3:
              Tcl_GetInt (tIntp, Value, &TmpI);
              EdgE[0] = TmpI;
            break;
            case 4:
              Tcl_GetInt (tIntp, Value, &TmpI);
              EdgE[1] = TmpI;
            break;
            case 5:
              Tcl_GetInt (tIntp, Value, &TmpI);
              EdgE[2] = TmpI;
            break;
            case 6:
              Tcl_GetInt (tIntp, Value, &TmpI);
              EdgE[3] = TmpI;
            break;
         }
      } 
   }

/* Get the grid array inputs - must be handled individually since they are */
/* a mixture of values and strings                                         */

   aName = Tcl_GetString(objv[3]);
   for (I = 0; I <= 16; ++I) {
      sprintf (Index, "%d", I);                                            
      if ((Value = Tcl_GetVar2(tIntp, aName, Index, Flg)) != NULL) {
         switch (I) {
            case 0:
              Tcl_GetDouble (tIntp, Value, &gI[0]);
            break;
            case 1:
              Tcl_GetDouble (tIntp, Value, &gI[1]);
            break;
            case 2:
              Tcl_GetDouble (tIntp, Value, &gI[2]);
            break;
            case 3:
              Tcl_GetDouble (tIntp, Value, &gI[3]);
            break;
            case 4:
              Tcl_GetInt (tIntp, Value, &TmpI);
              gI[4] = TmpI;
            break;
            case 5:
              Tcl_GetInt (tIntp, Value, &TmpI);
              gI[5] = TmpI;
            break;
            case 8:
               gI[7] = Value[0];
            break;
            case 11:
              Tcl_GetDouble (tIntp, Value, &gI[6]);
            break;
            case 15:
               gI[8] = Value[0];
            break;
            case 16:
               gI[9] = Value[0];
            break;
         }
      } 
   }

/* Now get the input array                                                 */

   TotGrids = (ByTe_4)(gI[4] * gI[5]);
   Bytes = sizeof(ReaL_8) * TotGrids;
   if ((memPtr1 = malloc (Bytes)) == NULL)
     return TCL_ERROR;
   Grid = (ReaL_8 *)memPtr1; 

   if (TclArrayToC (tIntp, objv[1], (void *)Grid, TotGrids, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }

/* Call the C routine                                                       */

   rV = GridSmooth_C (Grid, Layers, RVal, EdgE, gI);

/* Reconvert to the Tcl Array                                               */

   CArrayToTcl (tIntp, objv[1], Grid, TotGrids, 0, 'D'); 

/* Get rid of the temporary array and return                                */

   free (memPtr1);
   return TCL_OK;

}


/* THE C ROUTINE                                                            */


ByTe_4 GridSmooth_C (ReaL_8 *G, Layers, ByTe_4 MaxL, ByTe_1 , 
                     ReaL_8 *RVal, ByTe_4 mQuad,  
                     ByTe_4 *sI, ReaL_8 *gI)
{
   void *memPtr1, *memPtr2, *memPtr3;
   register ReaL_8 *S;
   ReaL_8 Sum, Norm;
   ReaL_8 *Md, *Mx, *My, *Mv, *Mw, A[20], VarV;
   ReaL_8 Xd, Yd; 
   ReaL_8 *DataV, Ax, Ay;
   ReaL_8 MaxV = 1.0e30, MinV = -1.0e30;
   ByTe_4 *DataX, *DataY;
   ByTe_4 Quad[4], nQ;
   register ByTe_4 I, L;
   ByTe_4 J, gN, Q, qX, qY, sX, mQ, aN;
   ByTe_4 TotGrids, Bytes, lX, lY, RoW, CoL, nXMesh, nYMesh;
   ByTe_4 MaxB;
   ByTe_4 X, Y, Xc, Yc, MnI, MxI; 
   ByTe_4 bX, eX, bY, eY; 
   ByTe_4 rV, dN, bdN; 
   ByTe_1 xQ, cX, cY, sTor;

/* DO some integer conversions from the grid input array.  Also set         */
/*     variables to the absolute last grid positions                        */

   nXMesh = (ByTe_4)gI[4];
   nYMesh = (ByTe_4)gI[5];
   sTor = (ByTe_1)gI[7];
   cX = (ByTe_1)gI[8];
   cY = (ByTe_1)gI[9];
   lX = nXMesh - 1;
   lY = nYMesh - 1;
   EdgE[1] = lX - EdgE[1];
   EdgE[3] = lY - EdgE[3];

/* SET the grid edges which are used to determine when we need to change    */
/*     to requiring only two filled quadrants for a fit to be made          */

   sI[3] = lX - sI[3];
   sI[4] = lY - sI[4];

/* SET factors which will take an X,Y grid location to a linear location in */
/*     the grid array.  These depend on how the grid was laid down which    */
/*     could have been ROW by ROW or COLUMN by COLUMN.                      */

   if ( sTor == 'R' ) {
      Ax = 1;
      Ay = nXMesh;
   } else {
      Ax = nYMesh;
      Ay = 1;
   } 

/* INITIALIZE the grid flag array.                                          */

   TotGrids = nXMesh * nYMesh;
   Bytes = sizeof(ReaL_8) * TotGrids;
   if ((memPtr1 = malloc (Bytes)) == NULL) { return -1; }
   S = (ReaL_8 *)memPtr1;
   for (J = 0; J < TotGrids; ++J) { S[J] = gI[6]; }

/* LOOP through all the grids                                              */
/  for ( X = EdgE[0];  X < EdgE[1]; ++X ) {                                */

   for ( X = 0;  X < gI[4]; ++X ) { 
      sX = Ax * X;
      xQ = (((X < sI[0]) || ( X > sI[1])) && (cX == 'N')) ? 1 : 0;
      for ( Y = 0;  Y < gI[5]; ++Y ) {
         I = sX + Y * Ay;
         if ( N[I] > 0 ) { continue; } 

         if (xQ || (((Y < sI[2]) || (Y > sI[3])) && (cY == 'N')))  {
            mQ = (mQuad > 2) ? 2 : mQuad;
         } else { mQ = mQuad; }

         L = 1;
         dN = 0;
         nQ = 0;
         for ( J = 0; J < 4; ++J ) { Quad[J] = 0; }
         while (L <= MaxL) {
            Xc = X;
            Yc = Y;
            bdN = dN;
            bX = X - L;
            eX = X + L;
            if ( (eX >= nXMesh) && (cX == 'N')) { eX = lX; }
            if (bX < 0) {
               if (cX == 'Y') {
                 bX += nXMesh;
                 eX += nXMesh;
                 Xc += nXMesh;
               } else { bX = 0; }
            }
            bY = Y - L;
            eY = Y + L;
            if ( (eY >= nYMesh) && (cY == 'N')) { eY = lY; }
            if (bY < 0) {
               if (cY == 'Y') {
                 bY += nYMesh;
                 eY += nYMesh;
                 Yc += nYMesh;
               } else { bY = 0; }
            }
         
            RoW = (bY % nYMesh) * Ay; 
            for ( J = bX;  J < eX; ++J ) {
               CoL = J % nXMesh;
               gN = CoL * Ax + RoW;
               if (N[gN] == 1)  {
                  DataX[dN] = J;
                  DataY[dN] = bY;
                  DataV[dN++] = G[gN];
               }
            }
            CoL = (eX % nXMesh) * Ax; 
            for ( J = bY;  J < eY; ++J ) {
               RoW = J % nYMesh;
               gN = CoL + RoW * Ay;
               if (N[gN] == 1)  {
                  DataX[dN] = eX;
                  DataY[dN] = J;
                  DataV[dN++] = G[gN];
               }
            }

            RoW = (eY % nYMesh) * Ay; 
            for ( J = eX;  J > bX; --J ) {
               CoL = J % nXMesh;
               gN = CoL * Ax + RoW;
               if (N[gN] == 1)  {
                  DataX[dN] = J;
                  DataY[dN] = eY;
                  DataV[dN++] = G[gN];
               }
            }
            CoL = (bX % nXMesh) * Ax; 
            for ( J = eY;  J > bY; --J ) {
               RoW = J % nYMesh;
               gN = CoL + RoW * Ay;
               if (N[gN] == 1)  {
                  DataX[dN] = bX;
                  DataY[dN] = J;
                  DataV[dN++] = G[gN];
               }
            }

            for ( J = bdN; J < dN; ++J ) { 
               Xd = (DataX[J] - Xc) * dXConv;
               Yd = (DataY[J] - Yc) * dYConv;
               Md[J] = sqrt(Xd * Xd + Yd * Yd);
               Mx[J] = Xd;
               My[J] = Yd;
               Mv[J] = DataV[J];
               qX = ( Xd >= 0.0 ) ? 0 : 1;
               qY = ( Yd >= 0.0 ) ? 0 : 2;
               Q = qX + qY;
               Quad[Q] = 1;
            }

            nQ = 0;
            for ( J = 0; J < 4; ++J ) { nQ += Quad[J]; }

            if ((dN >= nN) && (nQ >= mQ)) {
              break;
            } else { ++L; }
         }

         if (( dN < aN ) || (nQ < mQ)) { 
             N[I] = 0;
             G[I] = gI[6];
         } else { 
            DataMxMn_C (Md, dN, '<', MinV, MaxV, 1, &MnI, &MxI);
            MinD = Md[MnI];
            if ( (MinD > RVal[0]) ) {
               N[I] = 0;
               G[I] = gI[6];
            } else {
               for ( J = 0; J < dN; ++J ) { 
                  Mw[J] = exp(RVal[1] * log(MinD / Md[J]));
               }
 
               if (CFlgs[3] == 'Y') {
                  rV = DataLSq2D_C (Mx, My, Mv, dN, 1, 0, A, 0, Mw, &VarV);
                  N[I] = 0;
                  G[I] = A[0];

                  if (CFlgs[2] == 'A') {
                    if (G[I] > Mv[MnI]) { G[I] = Mv[MnI] ; } 
                  } else {
                     if (CFlgs[2] == 'L') {
                        DataMxMn_C (Mv, dN, '>', MinV, MaxV, 1, &MnI, &MxI);
                        if (G[I] > Mv[MxI]) { G[I] = Mv[MxI] ; } 
                     }
                  }
               } else {
                   Sum = 0.0;
                   Norm = 0.0;
                   for ( J = 0; J < dN; ++J ) { 
                      Norm += Mw[J];
                      Sum += Mv[J] * Mw[J];
                   }
                  N[I] = 0;
                  G[I] = Sum / Norm;
               }
            }
         }
      }
   }

   free (memPtr1);
   free (memPtr2);
   free (memPtr3);
   return 1;
}
