/*  The C equivalent of TUgridFill2D,  It consists of a TCL interface over  */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  There are 4 OBJV elements.  These are:                                  */
/*     OBJV[0]:  Routine name                                               */
/*     OBJV[1]:  The Grid array (Grid)                                      */
/*     OBJV[2]:  The Grid Normalization Array                               */
/*     OBJV[3]:  Grid Info Array                                            */
/*                 0:  Beginning X grid position                            */
/*                 1:  Ending X grid position                               */
/*                 2:  Beginning Y grid position                            */
/*                 3:  Ending Y grid position                               */
/*                 4:  Number of grids along X                              */
/*                 5:  Number of grids along Y                              */
/*                 6:  Use POINT or BAND method to store X data             */
/*                 7:  Use POINT or BAND method to store Y data             */
/*                 8:  Storage Method ROW or COLUMN                         */
/*                 9:  IGNORE or KEEP Zeros when averaging                  */
/*                10:  NEW: initialize grid                                 */
/*                     ADD: add data to grid                                */
/*                     END: normalize grid                                  */
/*                     or any combination as ADDEND or NEWADDEND            */
/*                11:  Value to set unfilled grids to                       */
/*                12:  Bad grid value                                       */
/*                13:  Remove Data below this value                         */
/*                14:  Remove Data above this value                         */
/*                15:  X Data is Cyclic (YES, NO)                           */
/*                16:  Y Data is Cyclic (YES, NO)                           */
/*     OBJV[4]:  Fill Info Array                                            */
/*                 0: Minimum Nearest Neighbors to use per quadrand (nN)    */
/*                 1: Maximim Search Radius (MaxR)                          */
/*                 2: Minimum number of quadrants needed for fit (mQuad)    */
/*                 3: 1 = fit can't produce max above nearest value in fit  */
/*                    2 = fit can't produce max above any value in fit      */
/*                 4: Min X grid position                                   */
/*                 5: Max X grid position                                   */
/*                 6: Min Y grid position                                   */
/*                 7: Max Y grid position                                   */
/*                 8: Weighting factor (RVal(0))                            */
/*                 9: Fit Order                                             */
/*                                                                          */
/*  The 2D Least Squares uses only a subset of this info.  These are        */
/*  packaged into a single array for passage to the C routine. These are    */
/*                                                                          */
/*      gI(0)         0:  Beginning X grid position                         */
/*      gI(1)         1:  Ending X grid position                            */
/*      gI(2)         2:  Beginning Y grid position                         */
/*      gI(3)         3:  Ending Y grid position                            */
/*      gI(4)         4:  Number of grids along X                           */
/*      gI(5)         5:  Number of grids along Y                           */
/*      gI(6)        11:  Value to set unfilled grids to                    */
/*      gI(7)        12:  Bad grid value                                    */
/*      gI(8)         8:  Storage Method                                    */
/*      gI(9)        15:  Cyclic in X                                       */
/*      gI(10)       16:  Cyclic in Y                                       */
/*      gI(11)       13:  Min value in Grid                                 */
/*      gI(12)       14:  Max value in Grid                                 */
/*                                                                          */

#include <tcl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int GridFillCmd (ClientData cD, Tcl_Interp *tI, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   void *memPtr1 = NULL;
   ReaL_8 RVal[2];
   ReaL_8 *Grid, *Norm;
   ReaL_8 gI[13]; 
   int TmpI, Seed;
   register ByTe_4 I;
   ByTe_4 nN, MaxR, mQuad;
   ByTe_4 TotGrids, Bytes, rV, EdgE[5];
   ByTe_1 CFlgs[2];

   int Flg = TCL_LEAVE_ERR_MSG;
   Tcl_Obj *rVO = NULL;

   char *aName, *Value, Index[10];

/* Make sure that all the IO parameters are present */

   if (objc != 5 ) {
     Tcl_WrongNumArgs(tI, 1, objv, "Usage: TUgridFill2D Grd gI Nrm sI");
     return TCL_ERROR;
   }

   EdgE[4] = 1;
   aName = Tcl_GetStringFromObj(objv[4], (int *)0);
   for (I = 0; I <= 9; ++I) {
      sprintf (Index, "%d", I);                                            
      if ( (rVO = Tcl_GetVar2Ex(tI, aName, Index, Flg)) != NULL) {
         switch (I) {
            case 0:
              Tcl_GetIntFromObj (tI, rVO, &nN);
            break;
            case 1:
              Tcl_GetIntFromObj (tI, rVO, &MaxR);
            break;
            case 2:
              Tcl_GetIntFromObj (tI, rVO, &mQuad);
            break;
            case 3:
              Tcl_GetIntFromObj (tI, rVO, &TmpI);
              CFlgs[0] = (TmpI == 1) ? 'N' : (TmpI == 2) ? 'A' : 'O';
            break;
            case 4:
              Tcl_GetIntFromObj (tI, rVO, &EdgE[0]);
            break;
            case 5:
              Tcl_GetIntFromObj (tI, rVO, &EdgE[1]);
            break;
            case 6:
              Tcl_GetIntFromObj (tI, rVO, &EdgE[2]);
            break;
            case 7:
              Tcl_GetIntFromObj (tI, rVO, &EdgE[3]);
            break;
            case 8:
              Tcl_GetDoubleFromObj (tI, rVO, &RVal[0]);
            break;
            case 9:
              Tcl_GetIntFromObj (tI, rVO, &EdgE[4]);
            break;
         }
      } 
   }

/* Get the grid array inputs - must be handled individually since they are */
/* a mixture of values and strings                                         */

   aName = Tcl_GetStringFromObj(objv[2], (int *)0);
   for (I = 0; I <= 16; ++I) {
      sprintf (Index, "%d", I);                                            
      if ( (rVO = Tcl_GetVar2Ex(tI, aName, Index, Flg)) != NULL) {
         switch (I) {
            case 0:
               Tcl_GetDoubleFromObj (tI, rVO, &gI[0]);
            break;
            case 1:
               Tcl_GetDoubleFromObj (tI, rVO, &gI[1]);
            break;
            case 2:
               Tcl_GetDoubleFromObj (tI, rVO, &gI[2]);
            break;
            case 3:
               Tcl_GetDoubleFromObj (tI, rVO, &gI[3]);
            break;
            case 4:
               Tcl_GetDoubleFromObj (tI, rVO, &gI[4]);
            break;
            case 5:
               Tcl_GetDoubleFromObj (tI, rVO, &gI[5]);
            break;
            case 8:
               Value = Tcl_GetStringFromObj (rVO, (int *)0);
               gI[8] = Value[0];
            break;
            case 11:
               Tcl_GetDoubleFromObj (tI, rVO, &gI[6]);
            break;
            case 12:
               Tcl_GetDoubleFromObj (tI, rVO, &gI[7]);
            break;
            case 13:
               Tcl_GetDoubleFromObj (tI, rVO, &gI[11]);
            break;
            case 14:
               Tcl_GetDoubleFromObj (tI, rVO, &gI[12]);
            break;
            case 15:
               Value = Tcl_GetStringFromObj (rVO, (int *)0);
               gI[9] = Value[0];
            break;
            case 16:
               Value = Tcl_GetStringFromObj (rVO, (int *)0);
               gI[10] = Value[0];
            break;
         }
      } 
   }

/* Now get the input array                                                 */

   aName = Tcl_GetStringFromObj(objv[3], (int *)0);
   CFlgs[1] = (strlen(aName) == 0 ) ? 0 : 1;

   TotGrids = (ByTe_4)(gI[4] * gI[5]);
   Bytes = sizeof(ReaL_8) * 2 * TotGrids;
   if ((memPtr1 = malloc (Bytes)) == NULL)
     return TCL_ERROR;
   Grid = (ReaL_8 *)memPtr1; 
   Norm = (ReaL_8 *)memPtr1 + TotGrids; 

   if (TclArrayToC (tI, objv[1], (void *)Grid, TotGrids, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }

   if (CFlgs[1] == 1) {
      if (TclArrayToC (tI, objv[3], (void *)Norm, TotGrids, 0, 'D') == 0) {
         free (memPtr1);
         return TCL_OK;
      }
   }

/* GET the seed value */

   rV = Tcl_UpVar(tI, "#0", "Rnd1Seed", "rSeed", 0);
   if ( (rVO = Tcl_GetVar2Ex(tI, "rSeed", (char *)0, Flg)) == NULL) {
      Seed = 566378;
   } else { Tcl_GetIntFromObj (tI, rVO, &Seed); }

/* Call the C routine                                                       */

   rV = GridFill_C (tI, Grid, Norm, nN, MaxR, CFlgs, RVal, mQuad, 
                              EdgE, gI, &Seed);

/* Replace seed value */

   if (rVO == NULL) { rVO = Tcl_NewObj(); }
   rVO = Tcl_DuplicateObj(rVO);
   Tcl_SetIntObj (rVO, Seed);
   Tcl_SetVar2Ex(tI, "rSeed", (char *)0, rVO, Flg);

/* Reconvert to the Tcl Array                                               */

   CArrayToTcl (tI, objv[1], Grid, TotGrids, 0, 'D'); 

/* Get rid of the temporary array and return                                */

   free (memPtr1);
   return TCL_OK;

}

/* THE C ROUTINE                                                            */

ByTe_4 GridFill_C (Tcl_Interp *tI, ReaL_8 *G, ReaL_8 *Nm, ByTe_4 nN, 
                        ByTe_4 MaxR, ByTe_1 *CFlgs, ReaL_8 *RVal, 
			ByTe_4 mQuad,  ByTe_4 *sI, ReaL_8 *gI, int *Seed)
{
   void *memPtr1, *memPtr2, *memPtr3;
   ReaL_8 MinR, rMax, rndN[100];
   ReaL_8 *Mx, *My, *Mv, *Mw, A[40], cM[200];
   ReaL_8 *DataV, *DataR, *dR, Ax, Ay, Xbase, Ybase;
   register ByTe_4 I, J;
   ByTe_4 *DataX, *DataY;
   ByTe_4 Quad[4], nQ, rInc;
   ByTe_4 gN, Q, qX, qY, sX, mQ, gC, nT;
   ByTe_4 TotGrids, Bytes, lX, lY, RoW, CoL, gX, gY;
   ByTe_4 tN, nC;
   ByTe_4 X, Y, MnI, MxI, MxL[4], aN[4], nP, oS; 
   ByTe_4 xS, xE, xC, yS, yE, yC; 
   ByTe_4 rV, MnV, TmP; 
   ByTe_4 x, y, xD, yD, M, R; 
   ByTe_1 *N, xQ, cX, cY, sTor, iA[40], fN[100];

   *Seed = DataRnd1_C (rndN, 100, 2, .02, *Seed);

/* Figure out how many coefficients we are using in filling the holes in   */
/*   the grid and then set the compute coefficient array to 1 for each     */

   nC = 1;
   for ( I = 1; I <= sI[4]; ++I ) { nC += I + 1; }
   for ( I = 0; I < nC; ++I ) { iA[I] = 1; }

   sprintf (fN, "%s $vX $vY tC %d %d", "TUpoly2DFunc", sI[4], nC);

/* DO some interger conversions from the grid input array.  Also set       */
/*     variables to the absolute last grid positions                       */

   gX = (ByTe_4)gI[4];
   gY = (ByTe_4)gI[5];
   lX = gX - 1;
   lY = gY - 1;
   sTor = (ByTe_1)gI[8];
   cX = (ByTe_1)gI[9];
   cY = (ByTe_1)gI[10];

/* DETERMINE the minimum number of values neeeded for a fit                 */

   MnV = 1;
   for (I = 1; I <= sI[4]; ++I) { MnV += I + 1; }

/* SET the grid edges which are used to determine when we need to change    */
/*     to requiring only two filled quadrants for a fit to be made          */

   sI[1] = lX - sI[1];
   sI[3] = lY - sI[3];

   rMax = MaxR * MaxR;

   aN[0] = (nN  > MnV) ? nN : MnV;
   TmP = MnV / 2 + MnV % 2;
   aN[1] = (nN  > TmP) ? nN : TmP;
   TmP = MnV / 3;
   if ((MnV % 3) > 0 ) { ++TmP; }
   aN[2] = (nN  > TmP) ? nN : TmP;
   TmP = MnV / 4;
   if ((MnV % 4) > 0 ) { ++TmP; }
   aN[3] = (nN  > TmP) ? nN : TmP;

   tN = 4 * aN[0];

/* SET factors which will take an X,Y grid location to a linear location in */
/*     the grid array.  These depend on how the grid was laid down which    */
/*     could have been ROW by ROW or COLUMN by COLUMN.                      */

   if ( sTor == 'R' ) {
      Ax = 1;
      Ay = gX;
   } else {
      Ax = gY;
      Ay = 1;
   } 

/* INITIALIZE the grid flag array.  Each grid location is flagged as either */
/*    being processed (0) original (1), empty (2) or bad (-1)               */

   TotGrids = gX * gY;
   Bytes = sizeof(ByTe_1) * TotGrids;
   if ((memPtr1 = malloc (Bytes)) == NULL) { return -1; }
   N = (ByTe_1 *)memPtr1;
   if ( CFlgs[1] == 0 ) {
      for (J = 0; J < TotGrids; ++J) {
         N[J] = (G[J] == gI[6]) ? 0 : (G[J] == gI[7]) ? 2 : 1;
      }
   } else {
      for (J = 0; J < TotGrids; ++J) {
         if (G[J] == gI[7])  {
             N[J] = 2;
         } else { N[J] = (Nm[J] == 0.0) ? -1 : 1; }
      }
   }

/* SETUP the internal double arrays which can have varaible lengths        */

   Bytes = 6 * tN *  sizeof(ReaL_8);
   if ((memPtr2 = malloc (Bytes)) == NULL) { return -1; }
   DataV = (ReaL_8 *)memPtr2; 
   DataR = (ReaL_8 *)memPtr2 + tN; 
   Mx =    (ReaL_8 *)memPtr2 + 2 * tN; 
   My =    (ReaL_8 *)memPtr2 + 3 * tN; 
   Mv =    (ReaL_8 *)memPtr2 + 4 * tN; 
   Mw =    (ReaL_8 *)memPtr2 + 5 * tN; 

   Bytes = 2 * tN * sizeof(ByTe_4);
   if ((memPtr3 = malloc (Bytes)) == NULL) { return -1; }
   DataX = (ByTe_4 *)memPtr3; 
   DataY = (ByTe_4 *)memPtr3 + tN; 

/*  Loop through all the data and assemble the requested number of nearest  */
/*  neighbors to the contour mesh position being solved for.  At the same   */
/*  time keep a set of nearest neighbors in each of the 4 quadrants about   */
/*  the mesh point.  This is added to nearest neighbors in end if not       */
/*  already included.  This should be an option but is not at present. It   */
/*  ensures that contributions if available from all sides are included in  */
/*  the fit.                                                                */
   
   /*
   for ( X = sI[0];  X < sI[1]; ++X ) {
   for ( Y = sI[2];  Y < sI[3]; ++Y ) {
   */
   for ( X = 0;  X < gX; ++X ) {
      sX = Ax * X;
      xQ = (((X <= sI[0]) || ( X >= sI[1])) && (cX == 'N')) ? 1 : 0;
      xS = X - MaxR;
      xE = X + MaxR;
      xC = X;
      if ( cX == 'Y') {
         if ( xS < 0 ) {
            xS += gX;
            xE += gX;
            xC += gX;
         }
      } else {
         if ( xS < sI[0] ) { xS = sI[0]; }
         if ( xE > sI[1] ) { xE = sI[1]; }
      }

      for ( Y = 0;  Y < gY; ++Y ) {
         gC = sX + Y * Ay;
         if ( N[gC] > 0 ) { continue; } 

         if (xQ || (((Y <= sI[2]) || (Y >= sI[3])) && (cY == 'N')))  {
            mQ = (mQuad > 2) ? 2 : mQuad;
         } else { mQ = mQuad; }
         nT = mQ - 1;

         for (I = 0; I < tN; ++I) { DataR[I] = TotGrids; } 
         for (I = 0; I < 4; ++I) { 
            MxL[I] = I * aN[nT]; 
            Quad[I] = 0; 
         } 

         yS = Y - MaxR;
         yE = Y + MaxR;
         yC = Y;
         if ( cY == 'Y') {
            if ( yS < 0 ) {
               yS += gY;
               yE += gY;
               yC += gY;
            }
         } else {
            if ( yS < sI[2] ) { yS = sI[2]; }
            if ( yE > sI[3] ) { yE = sI[3]; }
         }

         for (x = xS; x <= xE; ++x) {
            CoL = (x % gX) * Ax;
            for (y = yS; y <= yE; ++y) {
               RoW = (y % gY) * Ay;
               gN = CoL + RoW;
               if ( N[gN] != 1 ) { continue; }
               xD = xC - x;
               yD = yC - y;
               R  = xD * xD + yD * yD;
               if ( R > rMax ) { continue; }
               if ( xD == 0 ) {
	           qX = ( yD >= 0 ) ? 0 : 1;
               } else { qX = ( xD > 0 ) ? 0 : 1; }

               if ( yD == 0 ) {
                  qY = ( xD > 0 ) ? 2 : 0;
               } else { qY = ( yD > 0 ) ? 0 : 2; }
           
               Q = qX + qY;

               M = MxL[Q];
               if ( R < DataR[M] ) {
                  DataX[M] = xD;
                  DataY[M] = yD;
                  DataV[M] = G[gN];
                  DataR[M] = R;
                  oS = Q * aN[nT];
                  dR = DataR + oS;
                  DataMxMn_C (dR, aN[nT], '>', gI[11], gI[12], 1, &MnI, &MxL[Q]);
                  MxL[Q] += oS;
                  Quad[Q] = 1;
               }
            }
         }

         nQ = Quad[0] + Quad[1] + Quad[2] + Quad[3];
         if (nQ < mQ) { 
             N[gC] = 0;
             G[gC] = gI[6];
         } else { 
            DataMxMn_C (DataR, tN, '<', gI[11], gI[12], 1, &MnI, &MxI);
            MinR = DataR[MnI];
            nP = 0;
            for ( J = 0; J < tN; ++J ) { 
               if ((ByTe_4)DataR[J] != TotGrids) {
                  Mx[nP] = DataX[J];
                  My[nP] = DataY[J];
                  Mv[nP] = DataV[J];
                  Mw[nP++] = exp(RVal[0] * log(MinR / DataR[J]));
               }
            }
     
            if (nP >= MnV) {
               if ( mQ < 3 ) {
                  rInc = 0;
		  Xbase = Mx[0];
	          for ( J = 1; J < nP; ++J) {
		     if ( Mx[J] != Xbase ) { break; }
                  }
		  if ( J == nP ) {
	             for ( J = 0; J < nP; ++J) { 
		        Mx[J] *= (1.0 + rndN[rInc++]);
                     }
                  }
		  Ybase = My[0];
	          for ( J = 1; J < nP; ++J) {
		     if ( My[J] != Ybase ) { break; }
                  }
		  if ( J == nP ) {
	             for ( J = 0; J < nP; ++J) { 
		        My[J] *= (1.0 + rndN[rInc++]);
                     }
                  }
               }

               rV = DataLSq2D_C (tI, Mx, My, Mv, nP, 1, A, iA, nC, cM, fN, Mw);

               N[gC] = 0;
               G[gC] = A[0];
               if (CFlgs[0] == 'N') {
                  DataMxMn_C (Mw, nP, '>', gI[11], gI[12], 1, &MnI, &MxI);
                  if (G[gC] > Mv[MxI]) { G[gC] = Mv[MxI] ; } 
               } else {
                  if (CFlgs[0] == 'A') {
                     DataMxMn_C (Mv, nP, '>', gI[11], gI[12], 1, &MnI, &MxI);
                     if (G[gC] > Mv[MxI]) { G[gC] = Mv[MxI] ; } 
                  }
               }
            } else {
               N[gC] = 0;
               G[gC] = gI[6];
            }
         }
      }
   }

   free (memPtr1);
   free (memPtr2);
   free (memPtr3);
   return 1;
}
