/*  The C equivalent of TUdataNLinLSq,  It consists of a TCL interface over */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  There are 9 OBJV elements.  These are:                                  */
/*     OBJV[0]:   Routine name                                              */
/*     OBJV[1]:   Measured Result (Function Output)              (V)        */
/*     OBJV[2]:   Array of standard deviations                   (SiG)      */
/*     OBJV[3]:   Elements In Data Arrays                        (nP)       */
/*     OBJV[4]:   Coefficient Array                              (A)        */
/*     OBJV[5]:   Coefficients to be Fitted Array                (iA)       */
/*     OBJV[6]:   Number of Coefficients                         (nA)       */ 
/*     OBJV[7]:   Function Name                                  (FunC)     */ 
/*     OBJV[8]:   Input/Output and Scratch                       (oP)       */
/*        Element 0:  Fitting Test - set < 0 for first call to let the      */
/*                    procedure do initializations,  set to 0 for last      */
/*                    call to get the covariance and curvature matrices.    */
/*                    Procedure sets varaible during iterations             */
/*        Element 1:  Current Chi Squared Value                             */
/*        Element 2:  Previous Chi Squared Value                            */
/*        Element 3:  The number of coefficients being fit - computed       */
/*                    during  the first call from iA                        */
/*        Element 4:  Input Element to process                              */
/*        Element 5:  Scratch space starts here.  The first mA values hold  */
/*                    the current guess of the coefficients being used by   */
/*                    the procedure.  The next mA*mA coefficients hold the  */
/*                    Covarience Matrix which only has meaning if Ops(0) =  */
/*                    0. The next mA*mA coefficients hold the Curvature     */
/*                    Matrix which only has meaning if Ops(0) = 0.  After   */
/*                    this pure scratch space                               */

#include <tcl.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int DataNLinLSqCmd (ClientData cD, Tcl_Interp *tI, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   void *memPtr1, *memPtr2;
   ReaL_8 *oP;
   ReaL_8 *V, *SiG, *A;
   ByTe_1 *iA;
   int nP, nA;

   ByTe_4 Bytes, rV, nOp, nASq;
   ByTe_1 *FunC, NFunC[100];
   

/* Make sure that the minimum number of IO parameters are present          */

   if (objc != 9 ) {
     Tcl_WrongNumArgs(tI, 1, objv, "Usage: TUdataNLinLSq V SiG nP ... ");
     return TCL_ERROR;
   }

/* Get the constant input parameters                                        */

   Tcl_GetIntFromObj (tI, objv[3], &nP);
   Tcl_GetIntFromObj (tI, objv[6], &nA);
   FunC = Tcl_GetStringFromObj (objv[7], (int *)0);

/* Make the function string                                                 */
   sprintf (NFunC, "set FuncRet [ %s %d dYdA %s ]\n", FunC, nA, objv[8]->bytes);

/* Get the constant input parameters                                        */
/* Establish the Memory for the needed arrays                               */

   nASq = nA * nA;
   nOp = 5 + 2 * nA + 4 * nASq;
   Bytes = sizeof(ReaL_8) *(2 * nP + nA +  nOp);
   if ((memPtr1 = malloc (Bytes)) == NULL)
      return TCL_ERROR;
   V = (ReaL_8 *)memPtr1;
   SiG = (ReaL_8 *)memPtr1 + nP;
   A = (ReaL_8 *)memPtr1 + 2 * nP;
   oP = (ReaL_8 *)memPtr1 + 2 * nP + nA;

   if (TclArrayToC (tI, objv[1], (void *)V, nP, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }
   if (TclArrayToC (tI, objv[2], (void *)SiG, nP, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }
   if (TclArrayToC (tI, objv[4], (void *)A, nA, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }
   if (TclArrayToC (tI, objv[8], (void *)oP, nOp, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }

   Bytes = sizeof(ByTe_1) * (nA);
   if ((memPtr2 = malloc (Bytes)) == NULL) {
      free (memPtr1);
      return TCL_ERROR;
   }
   iA = (ByTe_1 *)memPtr2;

   if (TclArrayToC (tI, objv[5], (void *)iA, nA, 0, 'C') == 0) {
      free (memPtr1);
      free (memPtr2);
      return TCL_OK;
   }

/* Call the  straight C array procedure                                   */

   rV = DataNLinLSq_C (tI, objv[8], V, SiG, (ByTe_4)nP, A, iA, nA, NFunC, oP); 

   if (CArrayToTcl (tI, objv[4], (void *)A, nA, 0, 'D') != 1) {
      return TCL_OK;
   }
   if (CArrayToTcl (tI, objv[8], (void *)oP, nOp, 0, 'D') != 1) {
      return TCL_OK;
   }

/* Gather output - free data array - and gone                             */

   free (memPtr1);
   free (memPtr2);

   return TCL_OK;

}

/*  This is a 2D Non Linear Least Squares Code - straight C                */

ByTe_4 DataNLinLSq_C (Tcl_Interp *tI, Tcl_Obj *oV, ReaL_8 *V, ReaL_8 *SiG, 
                       ByTe_4 nP, ReaL_8 *A, ByTe_1 *iA,  ByTe_4 nA, 
                      ByTe_1 *FunC, ReaL_8 *oP)
{
   ByTe_4 I, J, K, L;
   ByTe_4 Pc1, Pc2, Pa1, Pa2;
   ByTe_4 nASq, cBeg, aBeg, sBeg, dBeg, cEnD, dEnD,  EnD;
   ByTe_4 nC, Inc, XFer;
   ReaL_8 TmP;

   XFer = 5 + nA;

/* Set up the offsets to the Covariance, Curvature, and the start of the   */
/* Scratch Space                                                           */

   nASq = nA * nA;
   cBeg = 5 + nA;
   aBeg = cBeg + nASq;
   sBeg = aBeg + nASq;
   dBeg = sBeg + nASq;

/* Initialize for new set of trials if oP(0), the step size is negative.  */
/* This consists of:                                                      */
/*                                                                        */
/* 1. Determining how many coefficients there are to fit.                 */
/* 2. Initializing oP[0]                                                  */
/* 3. Copying the coefficients into the temporary coefficient area        */
/* 4. Computing the initial ChiSq value.                                  */
/* 5. Initializing the previous chi squared value to current one.         */

  if ( oP[0] < 0.0 ) {
      oP[3] = 0.0;
      for (J = 0; J < nA; ++J ) {
         if ( iA[J] == 1 ) { oP[3] += 1.0; }
      }

      oP[0] = (oP[0] < -0.9) ? .001 : -oP[0];
      I = 5;
      for (J = 0; J < nA; ++J, ++I ) { oP[I] = A[J]; }
 
      CArrayToTcl (tI, oV, (void *)oP, XFer, 0, 'D');
      DataMqCoef_C (tI, oV, V, SiG, nP, iA, nA, aBeg, sBeg, FunC, oP);
      oP[2] = oP[1];
   } 

/*  Set up a couple of convenient end loops                                 */

    nC = (ByTe_4)oP[3];
    cEnD = cBeg + nC * nC;
    dEnD = dBeg + nC;

/*  Get the current covariance matrix                                       */
 
   for (I = aBeg, J = cBeg; J < cEnD; ) { oP[J++] = oP[I++]; }
 
/* Set up the current approximation vector                                  */
 
   for (I = dBeg, J = sBeg; I < dEnD; ) { oP[I++] = oP[J++]; }

/* If there is a non-zero step size the we need to make an iteration run,   */
/* otherwise just compute and size the covariance and curvature matrices    */
 
   if ( oP[0] > 0.0 ) {
 
/* Change the linearized fitting matrix by augmenting the diagonal elements */
/* of the covariance matrix by the step size, oP(0).                        */
 
      Inc = nC + 1;
      for (I = cBeg; I < cEnD; I += Inc) { 
         oP[I] = oP[I] * (1.0 + oP[0]);
      }
 
/*  Compute new covariance matrix                                          */

      MatrixInv_C (nC, &oP[cBeg], 1, &oP[dBeg]);

/* Establish the new coefficients to test                                  */
 
      J = dBeg;
      for (I = 5, L = 0; L < nA; ++L, ++I) { 
         if ( iA[L] == 1 ) {
            oP[I] = A[L] + oP[J++];
         }
      }
 
/* Now try them out and get the resultant ChiSq                            */
 
      CArrayToTcl (tI, oV, (void *)oP, XFer, 0, 'D');
      DataMqCoef_C (tI, oV, V, SiG, nP, iA, nA, cBeg, dBeg, FunC, oP);
 
/* If the new value of ChiSq is less than the last one then we are moving  */
/* in the right direction - keep going.  Update all values to the current  */
/* ones.                                                                   */

      if ( oP[1] < oP[2] ) {
         oP[0] *= 0.1;
         oP[2] = oP[1];
 
         for (I = aBeg, J = cBeg; J < cEnD; ) { oP[I++] = oP[J++]; }
         for (I = dBeg, J = sBeg; I < dEnD; ) { oP[J++] = oP[I++]; }
 
         for (I = 5, J = 0; I < cBeg; ) { A[J++] = oP[I++]; }
      } else {
         oP[0] *= 10.0;
         oP[1] = oP[2];
      }
   } else {
      for (I = aBeg, J = cBeg; J < cEnD; ) { oP[J++] = oP[I++]; }
      MatrixInv_C (nC, &oP[cBeg], 0, &oP[cBeg]);

/*  Expand the CoVariance and Curvature matrices to match the number of    */
/*  coefficients in the function.  This only needs to be done if all of    */
/*  the coefficients are not being fit (ie. some were held constant).      */

      if ( nA !=  nC ) {
         Inc = nC - nA;
         
         EnD = cBeg + nA * nA;
         for ( I = cBeg, J = sBeg ; I < EnD; ) { oP[J++] = oP[I++]; }

         EnD = cBeg + nC * nC;
         for ( I = cBeg  ; I < EnD; ) { oP[I++] = 0.0; }

         for ( I = cBeg, L = sBeg, J = 0; J < nA; ++J ) {
            for ( K = 0; K < nA; ++K ) { oP[I++] = oP[L++]; }
            I += Inc;
         }
         
         EnD = aBeg + nA * nA;
         for ( I = aBeg, J = sBeg ; I < EnD; ) { oP[J++] = oP[I++]; }

         EnD = aBeg + nC * nC;
         for ( I = aBeg  ; I < EnD; ) { oP[I++] = 0.0; }

         for ( I = aBeg, L = sBeg, J = 0; J < nA; ++J ) {
            for ( K = 0; K < nA; ++K ) { oP[I++] = oP[L++]; }
            I += Inc;
         }

         K = nC - 1;
         for (J = nA - 1; J >= 0; --J) {
            if ( iA[J] ) {
               Pc1 = cBeg + K;
               Pc2 = cBeg + J;
               Pa1 = aBeg + K;
               Pa2 = aBeg + J;
               I = 0;
               for (I = 0; I < nA; ++I ) {
                  TmP = oP[Pc1];
                  oP[Pc1] = oP[Pc2];
                  oP[Pc2] = TmP;
                  Pc1 += nA; Pc2 += nA;
                  TmP = oP[Pa1];
                  oP[Pa1] = oP[Pa2];
                  oP[Pa2] = TmP;
                  Pa1 += nA; Pa2 += nA;
               }

               Pc1 = cBeg + K * nA;
               Pc2 = cBeg + J * nA;
               Pa1 = aBeg + K * nA;
               Pa2 = aBeg + J * nA;

               for (I = 0; I < nA; ++I, ++Pc1, ++Pc2, ++Pa1, ++ Pa2 ) {
                  TmP = oP[Pc1];
                  oP[Pc1] = oP[Pc2];
                  oP[Pc2] = TmP;
                  TmP = oP[Pa1];
                  oP[Pa1] = oP[Pa2];
                  oP[Pa2] = TmP;
               }
               --K;
            }
         }
      }
   }

   return 1;
}
