/*  The C equivalent of TUdataLSq3D,  It consists of a TCL interface over  */
/*  a straight C subroutine                                                */
/*                                                                         */
/*  There are 11 OBJV elements.  These are:                                */
/*     OBJV[0]:  Routine name                                              */
/*     OBJV[1]:  Array of X data (X)                                       */
/*     OBJV[2]:  Array of Y data (Y)                                       */
/*     OBJV[3]:  Array of Z data (Z)                                       */
/*     OBJV[4]:  Array of V data (V)                                       */
/*     OBJV[5]:  Length of Data Arrays (nP)                                */
/*     OBJV[6]:  Statistical Method to apply (Mode)                        */
/*     OBJV[7]:  Solution Coefficients (cF)                                */
/*     OBJV[8]:  Coefficients to be Fitted (icF)                           */
/*     OBJV[9]:  Function Options                                          */
/*     OBJV[10]:  Covariance Matrix (CoVar)                                */
/*     OBJV[11]:  The function which computes basis values                 */
/*     OBJV[12]: Variations of Z data (S)                                  */

#include <tcl.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int DataLSq3DCmd (ClientData cD, Tcl_Interp *tI, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   void *memPtr1 = NULL, *memPtr2 = NULL;
   ReaL_8 *X, *Y, *Z, *V, *S;
   ReaL_8 *cF, *CoVar, rV;
   int Mode;
   ByTe_4 nP, I, vC;
   ByTe_1 *icF, *FunC, NFunC[100];

   char *aName, Index[10];

   ByTe_4 Bytes;
   int nO, nC;
   int Flg = TCL_LEAVE_ERR_MSG;

   Tcl_Obj *objPtr, *rVO;

/* Make sure that the minimum number of IO parameters are present          */

   if (objc < 12 ) {
     Tcl_WrongNumArgs(tI, 1, objv, "Usage: TUdataLSq3D X Y nP  ... ");
     return TCL_ERROR;
   }

/* Get the constant input parameters                                        */

   Tcl_GetIntFromObj (tI, objv[5], &nP);
   Tcl_GetIntFromObj (tI, objv[6], &Mode);

/* Get the function input parameters                                        */

   aName = Tcl_GetString(objv[9]);
   sprintf (Index, "%s", "OR");                                            
   if ( (rVO = Tcl_GetVar2Ex(tI, aName, Index, Flg)) != NULL) {
      Tcl_GetIntFromObj (tI, rVO, &nO);
   } else { nO = 1; }

   sprintf (Index, "%s", "NC");                                            
   if ( (rVO = Tcl_GetVar2Ex(tI, aName, Index, Flg)) != NULL) {
      Tcl_GetIntFromObj (tI, rVO, &nC);
    } else { nC = 0; }

/* Get the function                                                         */

   FunC = Tcl_GetString (objv[11]); 

/* Make the function string for getting the number of coefficients          */
/*    and get them                                                          */

    sprintf (NFunC, "set nC [%s 0.0 0.0 0.0  tC %d %d]", FunC, nO, nC);

    Tcl_Eval(tI, NFunC);

    if ( (rVO = Tcl_GetVar2Ex(tI, "nC", (char *)0, Flg)) != NULL) {
          Tcl_GetIntFromObj (tI, rVO, &nC);
    } else { return -1; }

    rVO = Tcl_DuplicateObj(rVO);
    Tcl_SetIntObj (rVO, nC);
    Tcl_SetVar2Ex(tI, aName, Index, rVO, Flg);

     sprintf (NFunC, "%s $vX $vY $vZ tC %d %d", FunC, nO, nC);


/* Get the space for the real arrays: X and Y, cF, CoVar, S            */

   Bytes = sizeof(ReaL_8) * (5 * nP + nC + nC * nC);

   if ((memPtr1 = malloc (Bytes)) == NULL) { return TCL_ERROR; }

   X = (ReaL_8 *)memPtr1;
   Y = (ReaL_8 *)memPtr1 + nP;
   V = (ReaL_8 *)memPtr1 + 2 * nP;
   Z = (ReaL_8 *)memPtr1 + 3 * nP;
   S = (ReaL_8 *)memPtr1 + 4 * nP;
   cF = (ReaL_8 *)memPtr1 + 5 * nP;
   CoVar = (ReaL_8 *)memPtr1 + 5 * nP + nC;

   if (TclArrayToC (tI, objv[1], (void *)X, nP, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }
   if (TclArrayToC (tI, objv[2], (void *)Y, nP, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }
   if (TclArrayToC (tI, objv[3], (void *)Z, nP, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }
   if (TclArrayToC (tI, objv[4], (void *)V, nP, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }

/* If present get the S vectors                                     */

   if (Mode == 1) {
      if (TclArrayToC (tI, objv[12], (void *)S, nP, 0, 'D') == 0) {
         free (memPtr1);
         return TCL_OK;
      }
   }

/* GET the space for the 1 Byte array:  icF                               */

   Bytes = sizeof(ByTe_1) * nC;
   if ((memPtr2 = malloc (Bytes)) == NULL)
      return TCL_ERROR;
   icF = (ByTe_1 *)memPtr2;

/*  IF we don't get nC cF values back then set those we don't get back    */
/*    to 0                                                                */

   if ((vC = TclArrayToC (tI, objv[7], (void *)cF, nC, 0, 'D')) != nC) {
      for (I = vC; I < nC; ) { cF[I++] = 0.0; }
   }

/*  IF we don't get nC icF values back then set the whole array to 1      */
/*    and we will solve for all of the coefficients!                      */

   if (TclArrayToC (tI, objv[8], (void *)icF, nC, 0, 'C') != nC) {
      for (I = 0; I < nC; ) { icF[I++] = 1; }
   }

/* Call the  straight C array procedure                                   */

   rV = DataLSq3D_C (tI, X, Y, Z, V, nP, Mode, cF, icF, nC, CoVar, NFunC, S); 

   if (CArrayToTcl (tI, objv[7], (void *)cF, nC, 0, 'D') != 1) {
      if ( memPtr1 != NULL ) { free (memPtr1); }
      if ( memPtr2 != NULL ) { free (memPtr2); }
      return TCL_OK;
   }
   if (CArrayToTcl (tI, objv[8], (void *)icF, nC, 0, 'C') != 1) {
      if ( memPtr1 != NULL ) { free (memPtr1); }
      if ( memPtr2 != NULL ) { free (memPtr2); }
      return TCL_OK;
   }
   if (CArrayToTcl (tI, objv[10], (void *)CoVar, nC * nC, 0, 'D') != 1) {
      if ( memPtr1 != NULL ) { free (memPtr1); }
      if ( memPtr2 != NULL ) { free (memPtr2); }
      return TCL_OK;
   }

/* Gather output - free data array - and gone                             */

   free (memPtr1);
   free (memPtr2);

   objPtr = Tcl_GetObjResult(tI);
   Tcl_SetDoubleObj (objPtr, rV);
   return TCL_OK;

}

/*  This is a 3D Least Squares Code - straight C                           */

ReaL_8 DataLSq3D_C (Tcl_Interp *tI, ReaL_8 *X, ReaL_8 *Y, ReaL_8 * Z, 
                    ReaL_8 *V, ByTe_4 nP, ByTe_4 Mode, ReaL_8 *A, ByTe_1 *iA, 
		    ByTe_4 nC, ReaL_8 *cM, ByTe_1 *FunC, ReaL_8 *S)
{
   void *memPtr;
   ReaL_8 *sV, *W, tC[nC];
   ReaL_8 cS, ChiSq, Sum, wT, TmP, vM, pW, tW;
   ByTe_4 I, J, K, L, M, EnD; 
   ByTe_4 mFit = 0, mS, P1, P2; 
   ByTe_4 Bytes;
   int Flg = TCL_LEAVE_ERR_MSG;
   ByTe_1 TexT[50];

   Tcl_Obj *rVO, *rV1;

   rV1 = Tcl_NewObj();

   Bytes = sizeof(ReaL_8) * (nC + nP);
   if ((memPtr = malloc (Bytes)) == NULL)
      return TCL_ERROR;
   sV = (ReaL_8 *)memPtr;
   W  = (ReaL_8 *)memPtr + nC;

   for (J = 0; J < nC; ++J ) { if ( iA[J] == 1 ) { ++mFit; } }
   if ( mFit == 0 ) { 
      fprintf (stderr, "TUdataLSq3D:  ERROR - No coefficients to fit");
      return -1;
   }

   mS = nC * nC;
   for (J = 0; J < mS; ) { cM[J++] = 0.0; }
   for (J = 0; J < nC; ) { sV[J++] = 0.0; }

   switch (Mode) {
      case  0:  
         for (J = 0; J < nP; ++J ) { S[J] = 1.0; }
      break;
      case  1:  
      break;
      default: 
	 pW = (double)(-Mode / 2.0);
         for (J = 0; J < nP; ++J ) {
            tW = X[J] * X[J] + Y[J] * Y[J] + Z[J] * Z[J];
            if ( tW != 0.0 ) {
               S[J] = 1.0 / pow(tW, pW);
            } else { S[J]= 1.0; }
         }
      break;
   }

   tW = 0;
   for (I = 0;  I < nP; ++I) {
      rV1 = Tcl_DuplicateObj(rV1);
      Tcl_SetDoubleObj (rV1, X[I]);
      Tcl_SetVar2Ex(tI, "vX", (char *)0, rV1, Flg);
      rV1 = Tcl_DuplicateObj(rV1);
      Tcl_SetDoubleObj (rV1, Y[I]);
      Tcl_SetVar2Ex(tI, "vY", (char *)0, rV1, Flg);
      rV1 = Tcl_DuplicateObj(rV1);
      Tcl_SetDoubleObj (rV1, Z[I]);
      Tcl_SetVar2Ex(tI, "vZ", (char *)0, rV1, Flg);
      Tcl_Eval(tI, FunC);
      for (J = 0; J < nC; ++J ) {
          sprintf (TexT, "%d", J);
	  if ( (rVO = Tcl_GetVar2Ex(tI, "tC", TexT, Flg)) != NULL) {
	     Tcl_GetDoubleFromObj (tI, rVO, &tC[J]);
          } else { return -1; }
      }

      vM = V[I];
      if ( mFit < nC ) {
         for (J = 0;  J < nC; ++J) {
            if (!iA[J]) { vM -= A[J] * tC[J]; }
         }
      }

      EnD = 1;
      J = 0;
      K = 0;
      tW += S[I];
      for (L = 0;  L < nC; ++L, ++EnD) {
         if ( iA[L] ) {
            wT = tC[L] * S[I];
            P1 = J * mFit;
            for (M = 0;  M < EnD; ++M ) {
               if ( iA[M] ) { cM[P1++] += wT * tC[M]; }
            }
            sV[K] += vM * wT;
            ++J;
            ++K;
         }
      }
   }

   cM[0] = tW;
   for (J = 1;  J < mFit; ++J) {
      P1 = J * mFit;
      for (K = 0;  K < J; ++K, ++P1) {
         P2 = K * mFit + J;
         cM[P2] = cM[P1];
      }
   }

   MatrixInv_C (mFit, cM, 1, sV );

   for (J = 0, I = 0; J < nC; ++J) {
      if ( iA[J] ) { A[J] = sV[I++]; }
   }

   ChiSq = 0.0;
   for (I = 0; I < nP; ++I) {
      rV1 = Tcl_DuplicateObj(rV1);
      Tcl_SetDoubleObj (rV1, X[I]);
      Tcl_SetVar2Ex(tI, "vX", (char *)0, rV1, Flg);
      Tcl_Eval(tI, FunC);
      for (J = 0; J < nC; ++J ) {
          sprintf (TexT, "%d", J);
	  if ( (rVO = Tcl_GetVar2Ex(tI, "tC", TexT, Flg)) != NULL) {
	     Tcl_GetDoubleFromObj (tI, rVO, &tC[J]);
          } else { return -1; }
      }
      Sum = 0.0;
      for (J = 0; J < nC; ++J) { Sum += A[J] * tC[J]; }
      cS = (V[I] - Sum) / S[I];
      ChiSq += cS * cS;
   }

   if ( nC !=  mFit ) {

      for (I = 0, L = 0; I < mFit; ++I) { 
         K = I * nC;
         for (J = 0; J < mFit; ++J) { 
            cM[K++] = cM[L++];
         }
      }
      
      for (I = mFit; I < nC; ++I) { 
         P1 = I;
         P2 = I * nC;
         for (J = 0; J <= I; ++J, P1 += nC, ++P2) { 
            cM[P1] = 0.0;
            cM[P2] = 0.0;
         }
      }

      K = mFit - 1;
      for (J = nC - 1; J >= 0; --J) { 
         if ( iA[J] ) {
            P1 = K;
            P2 = J;
            for (I = 0; I < nC; ++I, P1 += nC, P2 += nC) { 
               TmP = cM[P1];
               cM[P1] = cM[P2];
               cM[P2] = TmP;
            }
            P1 = K * nC;
            P2 = J * nC;
            for (I = 0; I < nC; ++I, ++P1, ++P2 ) { 
               TmP = cM[P1];
               cM[P1] = cM[P2];
               cM[P2] = TmP;
            }
            --K;
         }
      }
   }

   free (memPtr);
   return (ChiSq);
}
