/*  The C equivalent of TUdataLSq1D,  It consists of a TCL interface over   */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  There are 9 OBJV elements.  These are:                                  */
/*     OBJV[0]:  Routine name                                               */
/*     OBJV[1]:  Array of X data (X)                                        */
/*     OBJV[2]:  Array of Y data (Y)                                        */
/*     OBJV[3]:  Length of Data Arrays (nP)                                 */
/*     OBJV[4]:  Statistical Method to apply (Mode)                         */
/*     OBJV[5]:  Solution Coefficients (Coef)                               */
/*     OBJV[6]:  Coefficients to be Fitted (iCoef)                          */
/*     OBJV[7]:  Number of Coefficients (nC)                                */
/*     OBJV[8]:  Covariance Matrix (CoVar)                                  */
/*     OBJV[9]:  The function which gives X for each A                      */
/*     OBJV[10]: Variations of Z data (SigZ)                                */

#include <tcl.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int DataLSq1DCmd (ClientData cD, Tcl_Interp *tI, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   void *memPtr1 = NULL, *memPtr2 = NULL;
   ReaL_8 *X, *Y, *SigZ;
   ReaL_8 *Coef, *CoVar, rV;
   ByTe_4 Mode;
   ByTe_4 nSigZ, nC, nP, I, vC;
   ByTe_1 *iCoef, *FunC, NFunC[100];

   ByTe_4 Bytes;

   Tcl_Obj *objPtr;

/* Make sure that the minimum number of IO parameters are present          */

   if (objc < 10 ) {
     Tcl_WrongNumArgs(tI, 1, objv, "Usage: TUdataLSq1D X Y nP  ... ");
     return TCL_ERROR;
   }

/* Get the constant input parameters                                        */

   Tcl_GetIntFromObj (tI, objv[3], &nP);
   Tcl_GetIntFromObj (tI, objv[4], &Mode);
   Tcl_GetIntFromObj (tI, objv[7], &nC);
   FunC = Tcl_GetString (objv[9]); 

/* Make the function string                                                 */
    sprintf (NFunC, "%s $vX tC %d", FunC, nC);

/* See if there is a SigZ array present                                     */

   if (Mode != 1) {
      nSigZ = 0;
      SigZ = NULL;
   } else { nSigZ = nP; }

/* Get the space for the real arrays: X and Y, Coef, CoVar, SigZ            */

   Bytes = sizeof(ReaL_8) * (2 * nP  + nC + nC * nC  + nSigZ);

   if ((memPtr1 = malloc (Bytes)) == NULL)
      return TCL_ERROR;
   X = (ReaL_8 *)memPtr1;
   Y = (ReaL_8 *)memPtr1 + nP;
   Coef = (ReaL_8 *)memPtr1 + 2 * nP;
   CoVar = (ReaL_8 *)memPtr1 + 2 * nP + nC;

   if (TclArrayToC (tI, objv[1], (void *)X, nP, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }
   if (TclArrayToC (tI, objv[2], (void *)Y, nP, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }

/* If present get the SigZ vectors                                     */


   if (nSigZ > 0) {
      SigZ = (ReaL_8 *)memPtr1 + 2 * nP + nC + nC * nC;
      if (TclArrayToC (tI, objv[10], (void *)SigZ, nP, 0, 'D') == 0) {
         free (memPtr1);
         return TCL_OK;
      }
   }

/* GET the space for the 1 Byte array:  iCoef                             */

   Bytes = sizeof(ByTe_1) * nC;
   if ((memPtr2 = malloc (Bytes)) == NULL)
      return TCL_ERROR;
   iCoef = (ByTe_1 *)memPtr2;

/*  IF we don't get nC Coef values back then set those we don't get back  */
/*    to 0                                                                */

   if ((vC = TclArrayToC (tI, objv[5], (void *)Coef, nC, 0, 'D')) != nC) {
      for (I = vC; I < nC; ) { Coef[I++] = 0.0; }
   }

/*  IF we don't get nC iCoef values back then set the whole array to 1    */
/*    and we will solve for all of the coefficients!                      */

   if (TclArrayToC (tI, objv[6], (void *)iCoef, nC, 0, 'C') != nC) {
      for (I = 0; I < nC; ) { iCoef[I++] = 1; }
   }

/* Call the  straight C array procedure                                   */

   rV = DataLSq1D_C (tI, X, Y, nP, Mode, Coef, iCoef, nC, CoVar, NFunC, SigZ); 

   if (CArrayToTcl (tI, objv[5], (void *)Coef, nC, 0, 'D') != 1) {
      if ( memPtr1 != NULL ) { free (memPtr1); }
      if ( memPtr2 != NULL ) { free (memPtr2); }
      return TCL_OK;
   }
   if (CArrayToTcl (tI, objv[6], (void *)iCoef, nC, 0, 'C') != 1) {
      if ( memPtr1 != NULL ) { free (memPtr1); }
      if ( memPtr2 != NULL ) { free (memPtr2); }
      return TCL_OK;
   }
   if (CArrayToTcl (tI, objv[8], (void *)CoVar, nC * nC, 0, 'D') != 1) {
      if ( memPtr1 != NULL ) { free (memPtr1); }
      if ( memPtr2 != NULL ) { free (memPtr2); }
      return TCL_OK;
   }

/* Gather output - free data array - and gone                             */

   free (memPtr1);
   free (memPtr2);

   objPtr = Tcl_GetObjResult(tI);
   Tcl_SetDoubleObj (objPtr, rV);
   return TCL_OK;

}

/*  This is a 1D Least Squares Code - straight C                           */

ReaL_8 DataLSq1D_C (Tcl_Interp *tI, ReaL_8 *X, ReaL_8 *Y, ByTe_4 nP,
                    ByTe_4 Mode, ReaL_8 *A, ByTe_1 *iA, ByTe_4 nC,
                    ReaL_8 *cM, ByTe_1 *FunC, ReaL_8 *S)
{
   void *memPtr;
   ReaL_8 *sV, *W, tC[nC], *sP;
   ReaL_8 V, ChiSq, yM, Sum, wT, TmP;
   ByTe_4 I, J, K, L, M, EnD; 
   ByTe_4 mFit = 0, mS, P1, P2; 
   ByTe_4 Bytes;
   int Flg = TCL_LEAVE_ERR_MSG;
   ByTe_1 TexT[50];

   Tcl_Obj *rVO, *rV1;

   rV1 = Tcl_NewObj();

   Bytes = sizeof(ReaL_8) * (nC + nP);
   if ((memPtr = malloc (Bytes)) == NULL)
      return TCL_ERROR;
   sV = (ReaL_8 *)memPtr;
   W  = (ReaL_8 *)memPtr + nC;

   for (J = 0; J < nC; ++J ) { if ( iA[J] == 1 ) { ++mFit; } }
   if ( mFit == 0 ) { 
      fprintf (stderr, "TUdataLSq1D:  ERROR - No coefficients to fit");
      return -1;
   }

   mS = nC * nC;
   for (J = 0; J < mS; ) { cM[J++] = 0.0; }
   for (J = 0; J < nC; ) { sV[J++] = 0.0; }

   switch (Mode) {
      case -1:  
         for (J = 0; J < nP; ++J ) { W[J] = (Y[J] != 0.0) ? 1.0 / Y[J] : 1.0; }
         sP = (ReaL_8 *)W;
      break;
      case  0:  
         for (J = 0; J < nP; ++J ) { W[J] = 1.0; }
         sP = (ReaL_8 *)W;
      break;
      case  1:  
         for (J = 0; J < nP; ++J ) { W[J] = 1.0 / (S[J] * S[J]); }
         sP = (ReaL_8 *)S;
      break;
   }

   for (I = 0;  I < nP; ++I) {
      rV1 = Tcl_DuplicateObj(rV1);
      Tcl_SetDoubleObj (rV1, X[I]);
      Tcl_SetVar2Ex(tI, "vX", (char *)0, rV1, Flg);
      Tcl_Eval(tI, FunC);
      for (J = 0; J < nC; ++J ) {
          sprintf (TexT, "%d", J);
	  if ( (rVO = Tcl_GetVar2Ex(tI, "tC", TexT, Flg)) != NULL) {
	     Tcl_GetDoubleFromObj (tI, rVO, &tC[J]);
          } else { return -1; }
      }

      yM = Y[I];
      if ( mFit < nC ) {
         for (J = 0;  J < nC; ++J) {
            if (!iA[J]) { yM -= A[J] * tC[J]; }
         }
      }

      EnD = 1;
      J = 0;
      K = 0;
      for (L = 0;  L < nC; ++L, ++EnD) {
         if ( iA[L] ) {
            wT = tC[L] * W[L];
            P1 = J * mFit;
            for (M = 0;  M < EnD; ++M ) {
               if ( iA[M] ) { cM[P1++] += wT * tC[M]; }
            }
            sV[K] += yM * wT;
            ++J;
            ++K;
         }
      }
   }

   for (J = 1;  J < mFit; ++J) {
      P1 = J * mFit;
      for (K = 0;  K < J; ++K, ++P1) {
         P2 = K * mFit + J;
         cM[P2] = cM[P1];
      }
   }

   MatrixInv_C (mFit, cM, 1, sV );

   for (J = 0, I = 0; J < nC; ++J) {
      if ( iA[J] ) { A[J] = sV[I++]; }
   }

   ChiSq = 0.0;
   for (I = 0; I < nP; ++I) {
      rV1 = Tcl_DuplicateObj(rV1);
      Tcl_SetDoubleObj (rV1, X[I]);
      Tcl_SetVar2Ex(tI, "vX", (char *)0, rV1, Flg);
      Tcl_Eval(tI, FunC);
      for (J = 0; J < nC; ++J ) {
          sprintf (TexT, "%d", J);
	  if ( (rVO = Tcl_GetVar2Ex(tI, "tC", TexT, Flg)) != NULL) {
	     Tcl_GetDoubleFromObj (tI, rVO, &tC[J]);
          } else { return -1; }
      }
      Sum = 0.0;
      for (J = 0; J < nC; ++J) { Sum += A[J] * tC[J]; }
      V = (Y[I] - Sum) / sP[I];
      ChiSq += V * V;
   }

   if ( nC !=  mFit ) {

      for (I = 0, L = 0; I < mFit; ++I) { 
         K = I * nC;
         for (J = 0; J < mFit; ++J) { 
            cM[K++] = cM[L++];
         }
      }
      
      for (I = mFit; I < nC; ++I) { 
         P1 = I;
         P2 = I * nC;
         for (J = 0; J <= I; ++J, P1 += nC, ++P2) { 
            cM[P1] = 0.0;
            cM[P2] = 0.0;
         }
      }

      K = mFit - 1;
      for (J = nC - 1; J >= 0; --J) { 
         if ( iA[J] ) {
            P1 = K;
            P2 = J;
            for (I = 0; I < nC; ++I, P1 += nC, P2 += nC) { 
               TmP = cM[P1];
               cM[P1] = cM[P2];
               cM[P2] = TmP;
            }
            P1 = K * nC;
            P2 = J * nC;
            for (I = 0; I < nC; ++I, ++P1, ++P2 ) { 
               TmP = cM[P1];
               cM[P1] = cM[P2];
               cM[P2] = TmP;
            }
            --K;
         }
      }
   }

   free (memPtr);
   return (ChiSq);
}
