#  THIS routine positions the file pointer for the UDF with the given KEY and
#    VERsion number at the input time.  How close to the input time the file
#    pointer is positioned depends on SType.  If set to COARSE or COARSE_C the 
#    file pointer is set to the beginning of the data record which either
#    contains or is closest to the input time respectively and if FINE it is
#    set to the actual measurement closest to the time.
#
#    The procedure is callable in two modes: INITIAL POSITIONING and RANDOM
#    ACCESS.  The first is represented by MustBeIn 0 the second by MustBeIn 1. 
#    INITIAL POSITIONING mode is generally used to do the initial time position
#    after the FileOpen call.  In this case if the specified time is prior
#    to the open file then the file pointer is placed at the start of the file.
#    In RANDOM ACCESS mode if the time is not in the file then there is no
#    positioning of the file pointer and a status flag is returned.
#
#   RETURNS:
#       1 - successful
#      -1 - time beyond last file
#      -2 - time not included in file list

package provide TclUDF 1.0

proc FilePos { Key Ver Yr Dy Hr Mn Ss { SType COARSE } { MustBeIn 0 } } {
   global SInfo ByTe_4 ByTe_2 ExDa

   set B $Key,$Ver
   set G $B,GN
   set S $B,SS

# SAVE how we position in the file.

   set SInfo($B,FPOS) $SType

# ESTABLISH the time of day in milliseconds and set up the time string for the
#    input time

   set Ms [expr int($Hr * 3600000 + $Mn * 60000 + $Ss * 1000)]
   set cTm [format "%4d%03d%08d" $Yr $Dy $Ms]

#  PICK out the file which contains the input time from the list of files set
#     up in the FileOpen call.  If the file is not already open then open it.
#     There are a number of different error conditions to check some of
#     which depend on MustBeIn.

   set rV [UDFfileFirst $B $Yr $Dy $Hr $Mn $Ss $MustBeIn]
   if { $rV == 3 } { return -1 }
   if { ($MustBeIn == 1) && ($rV != 1) } { return -2 }
   if { $rV == 0 } { 
       UDFreadDATA $B 0
       UDFreadHDR $B 0
       set SInfo($Key,$Ver,RSTATUS) -1 
       return 1 
   }

# SET local copies of the open header/data filedescriptors and configure them
#    so that they are reading binary data

   set fd $SInfo($B,DFD)
   set fh $SInfo($B,HFD)
   fconfigure $fd -translation binary
   fconfigure $fh -translation binary

#  FIGURE out how many data records there are in the data file

   set Fname [file join $SInfo($B,IHDPATH) \
                  [lindex [lindex $SInfo($B,HDLIST) $SInfo($B,HDNUM)] 0]D]
   set FLen [file size $Fname]
   set NumRec [expr int($FLen / $SInfo($G,DLEN))]

#  DO a quick binary search of the data file to establish which data record
#    the input lies in.

   set Low 0
   set High [expr $NumRec - 2]
   set FirstTime 1
   set Nsec 0
   while {$Low <= $High} {
      set Mid [expr int(($Low + $High)/2)]
      set Bytes [expr $SInfo($G,DLEN) * $Mid]
      UDFreadDATA $B $Bytes
      UDFreadHDR $B 0
      UDFretVar $B YRDY yD 1
      UDFretVar $B TIME Tm
      set fTm [format "%4d%03d%08d" $yD(0) $yD(1) $Tm]
      set tList [lsort -ascii -increasing [list $fTm $cTm]]
      set Pos [lsearch -exact $tList $cTm]
      if {$Pos == 0 } { 
         if ![string match $fTm $cTm] {
            set High [expr $Mid - 1] 
        } else { set High -10 }
      } else { set Low [expr $Mid + 1] } 
   }

#  NOW do a final adjustment to get to the start of the record.  This consists
#      of reading the next record and making sure that the requested time is
#      between the current and this record.  If not - search a bit forward or
#      backward.

   set Done 0
   set lTm [format "%4d%03d%08d" $yD(0) $yD(1) $Tm]
   while { $Done == 0 } {
      if { [UDFreadDATA $B -1] == 0 } {
         if { [UDFreadHDR $B 0] > 0 } {
            UDFretVar $B YRDY yD 1
            UDFretVar $B TIME Tm
            set nTm [format "%4d%03d%08d" $yD(0) $yD(1) $Tm]
            set tList [lsort -ascii -increasing [list $lTm $cTm $nTm]]
            set Pos [lsearch -exact $tList $cTm]
            switch -exact -- $Pos {
                0 {   set Cen [lindex $tList 1]
                      if ![string match $cTm $Cen] {
                         set Bytes [expr -3 * $SInfo($G,DLEN)]
                         if { [TUfileSeek $fd $Bytes current] >= 0 } {
                            UDFreadDATA $B -1
                            UDFreadHDR $B 0
                            UDFretVar $B YRDY yD 1
                            UDFretVar $B TIME Tm
                            set lTm [format "%4d%03d%08d" $yD(0) $yD(1) $Tm]
                            incr Mid -1
                         } else { set Done 1 }
                      } else { set Done 1 }
                   }
                1  {  if {[string match COARSE_C $SType] == 1 } {
                         set dT1 [expr $cTm - $lTm]
                         set dT2 [expr $nTm - $cTm]
                         if { $dT1 > $dT2 } { incr Mid }
                      }
                      set Done 1
                   }
                2  {  incr Mid
                      set lTm $nTm
                   }
            }
         }
      } else { set Done 1 }
   }

   set Bytes [expr $Mid * $SInfo($G,DLEN)]
   TUfileSeek $fd $Bytes start
   set rV [UDFreadDATA $B $Bytes]
   UDFreadHDR $B 0

#
#  Initialize some of the read variables here
#

   UDFvarInits $B 1
   UDFretVar $B YRDY yD 1
   UDFretVar $B TIME Tm

   if {[string match FINE $SType] == 1 } {
       set Done 0
       set N T
       while { $Done == 0 } {
          ReadUDF $Key $Ver $N -1 0 0 0 
          set lTm [format "%4d%03d%08d" $ExDa($N,BYR) $ExDa($N,BDY) \
                                        $ExDa($N,BMSEC) ]
          set nTm [format "%4d%03d%08d" $ExDa($N,EYR) $ExDa($N,EDY) \
                                        $ExDa($N,EMSEC) ]
          set tList [lsort -ascii -increasing [list $lTm $cTm $nTm]]
          set Pos [lsearch -exact $tList $cTm]
          if {$Pos == 2 } {
             set SInfo($Key,$Ver,RSTATUS) 1
          } else { set Done 1 }
       }
       UDFunsetExDa T 
   }
   set SInfo($Key,$Ver,RSTATUS) -1
   return 1
}
