# THIS procedure plots a colorbar and returns the color mapping information.
#    A colorbar may be identified with two different mappings.  The scaling
#    is identified as either primary or secondary.  Primary scaling is 
#    labeled to the right of a VERTICAL axis and bottom of HORIZONTAL axes.
#    Secondary scaling is the opposite.
#
#  INPUT Ops (array)
#    Element  pMIN:  Minimum for primary mapping
#    Element  pMAX:  Maximum for primary mapping 
#    Element  pSCA:  Scaling for primary mapping (LINEAR / LOG / NONE)
#    Element  WHITE: Inclusion of white in mapping (NO/YES)
#    Element  sMIN:  Minimum for secondary mapping 
#    Element  sMAX:  Maximum for secondary mapping 
#    Element  sSCA:  Scaling for secondary mapping (LINEAR / LOG / NONE)
#    Element  FMT:   Format (LHORIZONTAL / RHORIZONTAL / VERTICAL / WHEEL)
#    Element  POS:   Positioning in window (ABSOLUTE / RELATIVE)
#    Element  LEV:   Annotation Level (PRIMARY/SECONDARY) 
#    Element  NLEV:  Number Level (PRIMARY/SECONDARY) 

package provide TclPLT 1.0

proc ColorBar { wN X1 Y1 Z1 X2 Y2 Z2 InFo cMapA cMapB { Sep : } } {
   global WinInfo GphInfo DevInfo 

   upvar $InFo oP
   upvar $cMapA cA
   upvar $cMapB cB

#  GET an unused window for the colorbar to output into.

   set cW 50
   while { [info exists WinInfo($cW,Dimen)] } { incr cW }

# SET some of the option defaults if the user didn't

   set bC $GphInfo(bColor)
   if ![info exists oP(pSCA)] { set oP(pSCA) NONE }
   if ![info exists oP(sSCA)] { set oP(sSCA) NONE }
   if ![info exists oP(WHITE)] { set oP(WHITE) NO }
   if ![info exists oP(FMT)] { set oP(FMT) VERTICAL }
   if ![info exists oP(POS)] { set oP(POS) RELATIVE }
   if ![info exists oP(ANNOTATE)] { set oP(ANNOTATE) YES }
   if ![info exists oP(AXIS)] { set oP(AXIS) ON }
   if ![info exists oP(ACOLOR)] { set oP(ACOLOR) $bC }
   if ![info exists oP(PCOLORS)] { set op(PCOLORS) $bC:$bC:$bC:$bC }
   if ![info exists oP(SCOLORS)] { set op(SCOLORS) $bC:$bC:$bC:$bC }
   if ![info exists oP(LEV)] { 
       set pLeV 1.5
       set sLeV 1.5
   } else {
       set pLeV [lindex [split $oP(LEV) ":"] 0 ]
       set sLeV [lindex [split $oP(LEV) ":"] 1 ]
       if { [string length $pLeV] == 0 } { set pLeV 1.5 }
       if { [string length $sLeV] == 0 } { set sLeV 1.5 }
   }
   if ![info exists oP(NLEV)] { 
       set pnLeV 0.0
       set snLeV 0.0
   } else {
       set pnLeV [lindex [split $oP(NLEV) ":"] 0 ]
       set snLeV [lindex [split $oP(NLEV) ":"] 1 ]
       if { [string length $pnLeV] == 0 } { set pnLeV 0.0 }
       if { [string length $snLeV] == 0 } { set snLeV 0.0 }
   }

# IF a primary color scaling was defined set it up.

   set PriMap 0
   if [info exists oP(pSCA)] {
      if ![string match NONE $oP(pSCA)] {
         set nC [PLTcolorMap $oP(pMIN) $oP(pMAX) $oP(pSCA) $oP(WHITE) cA]
         set PriMap 1
      }
   }

# SAME for a secondary color scaling 

   set SecMap 0
   if [info exists oP(sSCA)] {
      if ![string match NONE $oP(sSCA)] {
         set nC [PLTcolorMap $oP(sMIN) $oP(sMAX) $oP(sSCA) $oP(WHITE) cB]
         set SecMap 1
      }
   }

# NOW set up the scaling max and min for both the primary and secondary
#   color maps.  If only one scaling was defined then duplicate it into
#   the second.

   if { $PriMap == 1 } {
      set pMin $oP(pMIN) ; set pMax $oP(pMAX) ; set pSca $oP(pSCA)
   } else {
      set pMin $oP(sMIN) ; set pMax $oP(sMAX) ; set pSca $oP(sSCA)
      for { set I 0 } { $I < 5 } { incr I } { set cA($I) $cB($I) }
   }

   if { $SecMap == 1 } {
      set sMin $oP(sMIN) ; set sMax $oP(sMAX) ; set sSca $oP(sSCA)
   } else {
      set sMin $oP(pMIN) ; set sMax $oP(pMAX) ; set sSca $oP(pSCA)
      for { set I 0 } { $I < 5 } { incr I } { set cB($I) $cA($I) }
   }
   
#  FOR output purposes need to know if the position of the colorbar was
#    given RELATIVE to the window scaling or in window ABSOLUTE coordinates
#    ( 0 to 1.0)

   if [string match RELATIVE $oP(POS)] {
       set WinInfo($cW,corFmt) 1
   } else { set WinInfo($cW,corFmt) 0 }

# NOW setup all the variables which will be needed to draw the colorbar.
#   This depends on the format.  We can use the primary axis for drawing
#   purposes. 

   switch -exact -- $oP(FMT) {
      LHORIZONTAL  { 
         set pX1 $pMax; set sX1 $sMax;  set pY1 -1.0 ; set sY1 -1.0
         set pX2 $pMin; set sX2 $sMin;  set pY2  1.0 ; set sY2 1.0
         set dX [expr ($pMax - $pMin) / double($nC)]
         set dY 2
         set IncX $dX
         set IncY 0
         set aP $cW,pa0
         set aS $cW,sa0
         set Xmin $pMin ; set Ymin -1.0
         set PAxis Xb ; set SAxis Xt 
         set Sq SqY
      }
      RHORIZONTAL  { 
         set pX1 $pMin; set sX1 $sMin;  set pY1 -1.0 ; set sY1 -1.0
         set pX2 $pMax; set sX2 $sMax;  set pY2  1.0 ; set sY2 1.0
         set dX [expr ($pMax - $pMin) / double($nC)]
         set dY 2
         set IncX $dX
         set IncY 0
         set aP $cW,pa0
         set aS $cW,sa0
         set Xmin $pMin ; set Ymin -1.0
         set PAxis Xb ; set SAxis Xt
         set Sq SqY
      }
      VERTICAL  {  
         set pX1 -1.0 ; set sX1 -1.0 ; set sY1 $pMin ; set pY1 $sMin 
         set pX2 1.0 ; set sX2 1.0 ;  set sY2 $pMax ; set pY2 $sMax
         set dX 2
         set dY [expr ($sMax - $sMin) / double($nC)]
         set IncX 0
         set IncY $dY
         set aP $cW,sa1
         set aS $cW,pa1
         set Xmin -1.0 ; set Ymin $sMin
         set PAxis Yt ; set SAxis Yb
         set Sq SqX
      }
      WHEEL  {  
         set pX1 -1.0 ; set sX1 -1.0 ; set pY1 -1.0 ; set sY1 -1.0
         set pX2 1.0 ; set sX2 1.0 ; set pY2 1.0 ; set sY2 1.0
         set dX [expr 360.0 / double($nC)]
         set dY 2
         set IncX $dX
         set IncY 0
         set Xmin 0.0 ; set Ymin 0.0
         set Sq SqX
         set PAxis XYP ; set SAxis XYS
         set aP $cW,pa0
         set aS $cW,sa0
      }
   }

#  SET up the window with the primary scaling.  Set up a square window
#    if we are to ouput a color wheel otherwise rectangular

   if [string match WHEEL $oP(FMT)] {
      RelSqWindow $cW $wN $X1 $Y1 $Z1 $X2 $pX1 $pY1 0.0 $pX2 $pY2 0.0
      for { set I 0 } { $I < $nC } { incr I } {
         PlotColor HOLD $I $I
         Arc $cW 0.0 0.0 0.0 1.0 $Xmin [expr $Xmin + $IncX] 1 1 0.5
         set Xmin [expr $Xmin + $IncX]
      }
      set WinInfo($aP) 0
      set WinInfo($aS) 0
   } else {
      RelWindow $cW $wN $X1 $Y1 $Z1 $X2 $Y2 $Z2 $pX1 $pY1 0.0 $pX2 $pY2 0.0
      set WinInfo($cW,sp0) $sX1
      set WinInfo($cW,sp1) $sY1
      set WinInfo($cW,sp3) $sX2
      set WinInfo($cW,sp4) $sY2
      for { set I 0 } { $I < $nC } { incr I } {
         PlotColor HOLD $I $I
         Box $cW $Xmin $Ymin 0 [expr $Xmin + $dX] [expr $Ymin + $dY] 0
         set Xmin [expr $Xmin + $IncX]
         set Ymin [expr $Ymin + $IncY]
      }
   
      if [string match LINEAR $pSca] {
          set WinInfo($aP) 0
      } else { set WinInfo($aP) 1 }

      if [string match LINEAR $sSca] {
         set WinInfo($aS) 0
      } else { set WinInfo($aS) 1 }
   }

   GPHwinScale $cW Xp Ys Zp 0

   if ![string match YES $oP(ANNOTATE)] { return $cW }

# AT this point annotate the colorbar if that has been setup

   DefinePlot $cW $cW 

# SET up all of the default annotation definitions.  You get this for
#   any annotation options not specified.

   set Outline 1
   set PMj  1    ; set SMj 1
   set PMn  1    ; set SMn 1 
   if { $PriMap } { set PNum 1 } else { set PNum 0 }
   if { $SecMap } { set SNum 1 } else { set SNum 0 }
   set PMjS .20  ; set SMjS .20 
   set PMnS .10  ; set SMnS .10
   set PNmS 12   ; set SNmS 12
   set PLbS 12   ; set SLbS 12
   set PMjF INSIDE ; set SMjF INSIDE
   set PMnF INSIDE ; set SMnF INSIDE
   set PNmF %.1f ; set SNmF %.1f
   set PMjN 10 ; set SMjN 10
   set PMnN 5  ; set SMnN 5
   set PLaB "" ; set SLaB ""
   set PNwgT OFF ; set PTwgT OFF
   set SNwgT OFF ; set STwgT OFF
   set PMjC #ffffff
   set PMnC #ffffff
   set PNmC #ffffff
   set PLbC #ffffff
   set SMjC #ffffff
   set SMnC #ffffff
   set SNmC #ffffff
   set SLbC #ffffff

   if [string match WHEEL $oP(FMT)] { set PMjF SPAN ; set SMjF SPAN }

# NOW loop over all the options and see what the user has specified

   set ItemList [list AXIS ACOLOR PRIMARY SECONDARY PSIZE SSIZE PBOLD SBOLD \
                      PFMTS SFMTS PNUMS SNUMS PCOLORS SCOLORS LABELS]
   set ILen [llength $ItemList]

   for { set I 0 } { $I < $ILen } { incr I } {
      set Item [lindex $ItemList $I]
      if ![info exists oP($Item)] { continue }
      set V [split $oP($Item) $Sep]
      switch -exact -- $Item {
         AXIS {  
            if [string match OFF [lindex $V 0]] { set Outline 0 } 
         } 
         ACOLOR {  
            if ![string match HOLD [lindex $V 0]] { set AxC [lindex $V 0] } 
         } 
         PRIMARY {  
            if [string match OFF [lindex $V 0]] { set PMj 0 } 
            if [string match OFF [lindex $V 1]] { set PMn 0 } 
            if [string match OFF [lindex $V 2]] { set PNum 0 } 
         }
         SECONDARY {  
            if [string match OFF [lindex $V 0]] { set SMj 0 } 
            if [string match OFF [lindex $V 1]] { set SMn 0 } 
            if [string match OFF [lindex $V 2]] { set SNum 0 } 
         }
         PSIZE {
            if ![string match HOLD [lindex $V 0]] { set PMjS [lindex $V 0] } 
            if ![string match HOLD [lindex $V 1]] { set PMnS [lindex $V 1] } 
            if ![string match HOLD [lindex $V 2]] { set PNmS [lindex $V 2] } 
            if ![string match HOLD [lindex $V 3]] { set PLbS [lindex $V 3] } 
         }
         SSIZE {
            if ![string match HOLD [lindex $V 0]] { set SMjS [lindex $V 0] } 
            if ![string match HOLD [lindex $V 1]] { set SMnS [lindex $V 1] } 
            if ![string match HOLD [lindex $V 2]] { set SNmS [lindex $V 2] } 
            if ![string match HOLD [lindex $V 3]] { set SLbS [lindex $V 3] } 
         }
         PBOLD {
            if ![string match HOLD [lindex $V 0]] { set PNwgT [lindex $V 0] } 
            if ![string match HOLD [lindex $V 1]] { set PTwgT [lindex $V 1] } 
         }
         SBOLD {
            if ![string match HOLD [lindex $V 0]] { set SNwgT [lindex $V 0] } 
            if ![string match HOLD [lindex $V 1]] { set STwgT [lindex $V 1] } 
         }
         PFMTS {
            if ![string match HOLD [lindex $V 0]] { set PMjF [lindex $V 0] } 
            if ![string match HOLD [lindex $V 1]] { set PMnF [lindex $V 1] } 
            if ![string match HOLD [lindex $V 2]] { set PNmF [lindex $V 2] } 
         }
         SFMTS {
            if ![string match HOLD [lindex $V 0]] { set SMjF [lindex $V 0] } 
            if ![string match HOLD [lindex $V 1]] { set SMnF [lindex $V 1] } 
            if ![string match HOLD [lindex $V 2]] { set SNmF [lindex $V 2] } 
         }
         PNUMS {
            if ![string match HOLD [lindex $V 0]] { set PMjN [lindex $V 0] } 
            if ![string match HOLD [lindex $V 1]] { set PMnN [lindex $V 1] } 
         }
         SNUMS {
            if ![string match HOLD [lindex $V 0]] { set SMjN [lindex $V 0] } 
            if ![string match HOLD [lindex $V 1]] { set SMnN [lindex $V 1] } 
         }
         PCOLORS {
            if ![string match HOLD [lindex $V 0]] { set PMjC [lindex $V 0] } 
            if ![string match HOLD [lindex $V 1]] { set PMnC [lindex $V 1] } 
            if ![string match HOLD [lindex $V 2]] { set PNmC [lindex $V 2] } 
            if ![string match HOLD [lindex $V 3]] { set PLbC [lindex $V 3] } 
         }
         SCOLORS {
            if ![string match HOLD [lindex $V 0]] { set SMjC [lindex $V 0] } 
            if ![string match HOLD [lindex $V 1]] { set SMnC [lindex $V 1] } 
            if ![string match HOLD [lindex $V 2]] { set SNmC [lindex $V 2] } 
            if ![string match HOLD [lindex $V 3]] { set SLbC [lindex $V 3] } 
         }
         LABELS {
            if ![string match HOLD [lindex $V 0]] { set PLaB [lindex $V 0] } 
            if ![string match HOLD [lindex $V 1]] { set SLaB [lindex $V 1] } 
         }
      }
   }

# ADJUST the tick mark lenght so that its absolute from 0 to 1.  This is
#    different for WHEEL than the others

   set sC Scr$GphInfo(CurScr)

   if [string match WHEEL $oP(FMT)] { 
      set PMjS [expr 2.0 * $PMjS ] 
      set SMjS [expr 2.0 * $SMjS ] 
      set PMnS [expr 2.0 * $PMnS ] 
      set SMnS [expr 2.0 * $SMnS ] 
   } else {
      set PMjS [expr $PMjS / $DevInfo($sC.$Sq)] 
      set SMjS [expr $SMjS / $DevInfo($sC.$Sq)] 
      set PMnS [expr $PMnS / $DevInfo($sC.$Sq)] 
      set SMnS [expr $SMnS / $DevInfo($sC.$Sq)] 
   }

# OK now set the annotation conditions

   PLTinfoChg $cW $PAxis TICKS MJLENGTH $PMjS
   PLTinfoChg $cW $PAxis TICKS MNLENGTH $PMnS
   PLTinfoChg $cW $PAxis TICKS MJPOSITION $PMjF
   PLTinfoChg $cW $PAxis TICKS MNPOSITION $PMnF
   PLTinfoChg $cW $PAxis TICKS MJCOLOR $PMjC
   PLTinfoChg $cW $PAxis TICKS MNCOLOR $PMnC
   PLTinfoChg $cW $PAxis TICKS MJNUMBER $PMjN
   PLTinfoChg $cW $PAxis TICKS MNNUMBER $PMnN
   PLTinfoChg $cW $PAxis NUMBERS NSIZE $PNmS
   PLTinfoChg $cW $PAxis NUMBERS NFORMAT $PNmF
   PLTinfoChg $cW $PAxis NUMBERS NCOLOR $PNmC
   PLTinfoChg $cW $SAxis NUMBERS TCOLOR $PLbC
   PLTinfoChg $cW $PAxis NUMBERS TSIZE $PLbS
   PLTinfoChg $cW $PAxis NUMBERS NBOLD $PNwgT
   PLTinfoChg $cW $PAxis NUMBERS TBOLD $PTwgT
   PLTinfoChg $cW $SAxis TICKS MJLENGTH $SMjS
   PLTinfoChg $cW $SAxis TICKS MNLENGTH $SMnS
   PLTinfoChg $cW $SAxis TICKS MJPOSITION $SMjF
   PLTinfoChg $cW $SAxis TICKS MNPOSITION $SMnF
   PLTinfoChg $cW $SAxis TICKS MJCOLOR $SMjC
   PLTinfoChg $cW $SAxis TICKS MNCOLOR $SMnC
   PLTinfoChg $cW $SAxis TICKS MJNUMBER $SMjN
   PLTinfoChg $cW $SAxis TICKS MNNUMBER $SMnN
   PLTinfoChg $cW $SAxis NUMBERS NSIZE $SNmS
   PLTinfoChg $cW $SAxis NUMBERS NFORMAT $SNmF
   PLTinfoChg $cW $SAxis NUMBERS NCOLOR $SNmC
   PLTinfoChg $cW $SAxis NUMBERS TSIZE $SLbS
   PLTinfoChg $cW $SAxis NUMBERS NBOLD $SNwgT
   PLTinfoChg $cW $SAxis NUMBERS TBOLD $STwgT
   PLTinfoChg $cW $SAxis NUMBERS TCOLOR $SLbC
   PLTinfoChg $cW $SAxis AXIS COLOR $AxC

# DETERMINE which Labels and major and minor tick marks are being output

   set pNuMs NONE
   set sNuMs NONE
   if { $PNum } { set pNuMs AUTO } 
   if { $SNum } { set sNuMs AUTO } 

   set pTicKs NONE
   set sTicKs NONE
   if { $PMj && $PMn } {
      set pTicKs BOTH 
   } elseif { $PMj } {
      set pTicKs ALLMAJOR
   } elseif { $PMn } { pTicKs ALLMINOR }

   if { $SMj && $SMn } {
      set sTicKs BOTH 
   } elseif { $SMj } {
      set sTicKs ALLMAJOR
   } elseif { $SMn } { sTicKs ALLMINOR }

   switch -exact -- $oP(FMT) {
      LHORIZONTAL - 
      RHORIZONTAL { 
         PLTaxisRec $cW Xt $sTicKs $sNuMs $pnLeV
         PLTaxisRec $cW Xb $pTicKs $pNuMs $snLeV
         if { [string length $SLaB] > 0 } {
            PLTaxisRec $cW Xt NONE AXIS $sLeV CENTER center $SLaB
         }
         if { [string length $PLaB] > 0 } {
            PLTaxisRec $cW Xb NONE AXIS $pLeV CENTER center $PLaB
         }
      }
      VERTICAL    {
         PLTaxisRec $cW Yt $pTicKs $pNuMs $pnLeV
         PLTaxisRec $cW Yb $sTicKs $sNuMs $snLeV
         if { [string length $SLaB] > 0 } {
            PLTaxisRec $cW Yb NONE AXIS $sLeV CENTER center $SLaB
         }
         if { [string length $PLaB] > 0 } {
            PLTaxisRec $cW Yt NONE AXIS $pLeV CENTER center $PLaB
         }
      }
      WHEEL       {
         PLTinfoChg $cW ALL AXIS FORMAT POLAR
         PLTinfoChg $cW Xb TICKS MJNUMBER 1
         PLTinfoChg $cW Yb TICKS MJLENGTH .2
         PLTinfoChg $cW Yb TICKS MNLENGTH .1
         PLTinfoChg $cW Yb TICKS MJSTART 0.0
         PLTinfoChg $cW Yb TICKS MJPOSITION SPAN
	 PLTwheelLabs $cW $pTicKs $pNuMs $pnLeV $oP(pMIN) $oP(pMAX) $oP(pSCA)
         if { [string length $PLaB] > 0 } {
            PLTaxisPolar $cW Xb NONE AXIS $pLeV CENTER center $PLaB
         }
      }
   }

# DRAW the plot outline if needed

   if $Outline { PLTaxisDraw $cW }

   return $cW
}
