package provide EUVSim_C 1.0

#  ZEROSOL provides a matrix of the same dimension as the solution matrix
#    with all locations which must have a zero value identified.  These
#    are determined by the fact that they are used in the column density
#    solution of a measured pixel with 0 intensity.
#
#    One of the problems is to make sure that all of the elements which
#    should be zeroed out are.  You can do this by making the step size
#    dH very small but then it takes forever to scan through all of the
#    lines of sight.  The way its done here is to look at successive 
#    points to fill in any gaps in the matrix. 
#
#  nP   -  total number of elements in the measured image grid.

proc ZeroSol { nP MaxZ dH zI } {
   global euvANS HaTm SaTm pStat gHDM cDm 

   upvar $zI zImG

#  INITIALIZE the zero element matrix

   set nG [expr $gHDM(4) * $gHDM(5)]
   for { set I 0 } { $I < $nG } { incr I } { set zImG($I) 1 }  
   
#  LOOP over the lines of sight 

   for { set I 0 } { $I < $nP } { incr I } {

# ONLY need to process if the line of sight is usable; doesn't hit the
#   Earth, has a zero intensity, and has a minimum L-Shell value less
#   than the maximum L-Shell value included in the solution matrix.

      if { ($pStat($I) <= 0) || ($cDm($I) > 0.0) } { continue }

      set J [expr 3 * $I]
      set K [expr $J + 1]
      set L [expr $K + 1]

      set A [expr $HaTm($J) / $HaTm($L)]
      set B [expr $HaTm($K) / $HaTm($L)]
      set C [expr -$A * $SaTm(2) + $SaTm(0)]
      set D [expr -$B * $SaTm(2) + $SaTm(1)]

# MOVE along the line of sight in steps of dH.  At each step compute both 
#   the current L-shell and the radial distance.  These are input into the 
#   zero solution matrix.  
 
      set Z 0.0
      if [string match PT $euvANS(SolGrid)] {
         while { $Z < $MaxZ } {
            set X [expr $A * $Z + $C]
            set Y [expr $B * $Z + $D]
            set XY [expr $X * $X + $Y * $Y]
            set T  [expr sqrt($XY + $Z * $Z)]
            set lS [expr $T * $T * $T / ($XY + 1.0e-9)]
            set gL [expr atan2($Y, $X)]
            TUgridInfo 0 gHDM Ret $gL $lS
            if { $Ret(0) >= 0 } { set zImG($Ret(0)) 0 }

            set X [expr -$A * $Z + $C]
            set Y [expr -$B * $Z + $D]
            set XY [expr $X * $X + $Y * $Y]
            set T  [expr sqrt($XY + $Z * $Z)]
            set lS [expr $T * $T * $T / ($XY + 1.0e-9)]
            set gL [expr atan2($Y, $X)]
            TUgridInfo 0 gHDM Ret $gL $lS
            if { $Ret(0) >= 0 } { set zImG($Ret(0)) 0 }

            set Z [expr $Z + $dH]
         }
      } else {
         while { $Z < $MaxZ } {
            set X [expr $A * $Z + $C]
            set Y [expr $B * $Z + $D]
            set XY [expr $X * $X + $Y * $Y + 1.0e-9]
            set T  [expr sqrt($XY + $Z * $Z)]
            set lS [expr $T * $T * $T / $XY]
            set gL [expr atan2($Y, $X)]
            set xSm [expr $lS * cos($gL)]
            set ySm [expr $lS * sin($gL)]
            TUgridInfo 0 gHDM Ret $xSm $ySm
            if { $Ret(0) >= 0 } { set zImG($Ret(0)) 0 }

            set X [expr -$A * $Z + $C]
            set Y [expr -$B * $Z + $D]
            set XY [expr $X * $X + $Y * $Y + 1.0e-9]
            set T  [expr sqrt($XY + $Z * $Z)]
            set lS [expr $T * $T * $T / $XY]
            set gL [expr atan2($Y, $X)]
            TUgridInfo 0 gHDM Ret $xSm $ySm
            if { $Ret(0) >= 0 } { set zImG($Ret(0)) 0 }

            set Z [expr $Z + $dH]
         }
      }

# IN an XY solution grid you need to physically mask out the Earth otherwise
#    it can get a non-zero solution in it.  This comes about due to the
#    filling of the solution grid which does not exclude the <= 1 region.
#    Masking it out makes the solution have that nice satisfying black
#    hole inside r = 1. 

      TUgridInfo 0 gHDM CorA -1.0 -1.0
      TUgridInfo 0 gHDM CorB  1.0  1.0
      if { $CorA(1) < $CorB(1) } {
         set cBeG $CorA(1)
         set cEnD $CorB(1)
      } else { set cBeG $CorB(1) ; set cEnD $CorA(1) }
      if { $CorA(2) < $CorB(2) } {
         set rBeG $CorA(2)
         set rEnD $CorB(2)
      } else { set rBeG $CorB(2) ; set rEnD $CorA(2) }

      for { set I $rBeG } { $I <= $rEnD } { incr I } {
         for { set J $cBeG } { $J <= $cEnD } { incr J } {
            TUgridInfo 1 gHDM rVa $J $I
            if { [TUgridInfo 20 gHDM rVb $rVa(0) ] > 0 } {
               set D [expr $rVb(2) * $rVb(2) + $rVb(3) * $rVb(3)]
               if { $D <= 1.0 } { set zImG($rVa(0)) 0 }
            }
         }
      }
   }
   return 0
}
