# Variable Definitions
#          volume enclosed by the points. Each location contains an (x,y,z).
#          There must be a minimum of 4 points.
#    _ex - The number of locations to add to the volume definition. These
#          locations are not taken at the same time as the base locations
#          and need to be further adjusted by the solar wind velocity.
#    tvS - The total number of locations used to define the volume.
#    tnV - The last set of locations.
#
 
package provide UDFAnalysisTh 1.0

proc APsolveSD { fD } {
   global apANS env Prefs

   APkeepTabs "STEP $fD : SPATIAL-DERIVATIVE"

# THIS is the text window for this function definition

   set W .apFDEF$fD.body.list
   if ![winfo exists $W] {
      set GuI [lindex $apANS($apANS($fD,Func)) 0]
      eval $GuI $fD 1
   }

# THIS defines the procedure(s) recognized by the threads
 
   set PgmA {
      source [ file join $env(TCLTOOLS_HOME) TclToolInits.tcl ]
      TclToolInits THREAD UTILS FFT
      lappend auto_path [file join $env(TCLTOOLS_HOME) TclAnalysisTh]
      package require UDFAnalysis
      package require UDFAnalysisTh

# THIS computes the spatial derivatives. The spacecraft position variables
#    must and the volume centers are set up outside this procedure and the
#    information is available in tsv space.
 
      proc SpaDrv { rN } {

# ESTABLISH some of the basic information that will be needed in the
#   computations.
 
         set _iT  [tsv::get sD _iT]
         set _ex  [tsv::get sD _ex]
         set _vD  [tsv::get sD _vD]
         set _vR  [tsv::get sD _vR$rN]
         set _fO  [tsv::get sD _fO$rN]
         set _vSD [tsv::get sD _vSD$rN]
         set _vI  [tsv::get sD _vI$rN]
         set tnV  [tsv::get sD tnV]
         set tvS  [tsv::get sD tvS]
         set vS   [tsv::get sD vS]
         set nE   [tsv::get sD nE]

# THIS is the time width of the data grid

         set cS [expr double([tsv::get apANS CellSz])]

# THIS is the value associated with bad data
 
         set Bad [tsv::get apANS BaD]

# Save the fit order for use in the Least-Sq algorithm

         set fOps(OR) $_fO

# WE are always solving a fit to a polynomial function.  This is the
#   procedure which will return the basis functions 

         set _fN TUpoly3DFunc

# TRANSFER the array gI and Dim variables to local AddOn
 
         THarrayXfer FROM AddOn _AddOn
 
# TRANSFER the input data, spacecraft position data, velocity data, and the
#   volume center data from tsv space to local space and delete the tsv 
#   versions.
 
         if [string match VECTOR $_iT] { 
            THarrayXfer FROM _XI _XI$rN YES
            THarrayXfer FROM _YI _YI$rN YES
            THarrayXfer FROM _ZI _ZI$rN YES
         } else { THarrayXfer FROM _XI _XI$rN YES }

# SET up the NoData array.  We are assuming that the data is all related.
#   So we only set up a single bad data array. 

         for { set J 0 } { $J < $tnV } { incr J } { 
	    set NoData($J) 0
	    set K [expr $J * $vS]
	    set End [expr $K + $tvS]
            for { } { $K < $End } { incr K } { 
               if { $_XI($K) < [tsv::get apANS BaDL] } { set NoData($J) 1 }
            }
         }

# PROCESS the input data. Both vector and scalar data can return one
#    scalar and one vector output.
# FOR scalars we compute the Gradient and Laplacian, however the Laplacian
#   is only computed if the fit order is > 1.  If its 1 the data only has
#   a linear dependence on position so the Laplacian is everywhere 0. If
#   there are at least 3 return varaibles then return gradient, if 4 then
#   gradient and Laplacian, and if 1 Laplacian.
# FOR vectors its the Cross and Dot products.

# SET up the returns and initialize all of the return arrays. sV are vector
#    returns, sS are scalar returns, and sR are raw returns

         set sV 0 ; set sS 0 ; set sR 0

# INITIALIZE the return variable list
 
	 set oVc 0
 
# INITIALIZE the dimensions and gridding information associated with each of
#   the output variables.
 
         set rL ""
         if ![string match "---"  $_vSD] {
            set tNames [lindex [APgetVNames $_vSD] 0]
            set nN [llength $tNames]
	    if { $nN >= 3 } { 
	       set sV 1 
               if [string match SCALAR $_iT] { 
		  lappend rL GRADIENT
               } else { lappend rL CROSS }

               set oX [lindex $tNames 0]
               set oY [lindex $tNames 1]
               set oZ [lindex $tNames 2]
               THarrayXfer TO AddOn $oX
               THarrayXfer TO AddOn $oY
               THarrayXfer TO AddOn $oZ

	       set oVc 3

               for { set J 0 } { $J < $nE } { incr J } {
                  tsv::set $oX $J $Bad 
		  tsv::set $oY $J $Bad  
		  tsv::set $oZ $J $Bad 
               }
            }
	    if { ($nN > 3) || !$sV } { 
               if [string match SCALAR $_iT] {
	          if { $_fO > 1 } { set sS 1 ; lappend rL LAPLACIAN }
               } else { set sS 1 ; lappend rL DOT }
            }

	    if $sS {
               set oS [lindex $tNames $oVc]
               THarrayXfer TO AddOn $oS
               for { set J 0 } { $J < $nE } { incr J } { 
                  tsv::set $oS $J $Bad 
               }
            } 
         }

         if ![string match "---"  $_vR] { 
	    set sR 1 
            set tNames [lindex [APgetVNames $_vR] 0]
            set rX [lindex $tNames 0]
            THarrayXfer TO AddOn $rX
            for { set J 0 } { $J < $nE } { incr J } {
               tsv::set $rX $J $Bad 
            }
            if [string match VECTOR $_iT] { 
               set rY [lindex $tNames 1]
               set rZ [lindex $tNames 2]
               THarrayXfer TO AddOn $rY
               THarrayXfer TO AddOn $rZ
               for { set J 0 } { $J < $nE } { incr J } { 
                  tsv::set $rY $J $Bad
                  tsv::set $rZ $J $Bad
	       } 
	    }
         } 
	 
# THIS line checks to make sure something is being returned.  If not then
#   there is nothing to do.
 
	 if {!$sV && !$sR && !$sS } { return } 

# FIT the data. Scalar and vector data are processed slightly differently

         if [string match SCALAR $_iT] {

            for { set J 0 } { $J < $tnV } { incr J } {

# IF the data is bad then there is no reason trying to fit it
 
               if $NoData($J) { continue }

# GET the spacecraft positions and the data used in this fit
 
	       set K [expr $J * $vS]
               for { set L 0 } { $L < $vS } { incr K ; incr L } { 
	          set _tX($L) [tsv::get _X $K]
	          set _tY($L) [tsv::get _Y $K]
	          set _tZ($L) [tsv::get _Z $K]
	          set _tD($L) $_XI($K)
               }

# IF we are using an expanded volume add the extra spacecraft positions here.
 
               for { set N 1 } { $N <= $_ex } { incr N } { 
                  set T [expr $N * $cS]
	          set End [expr $L + $vS]
                  for { } { $L < $End } { incr K ; incr L } { 
                     set _tX($L) \
		           [expr [tsv::get _X $K] + [tsv::get _XV $K] * $T]
                     set _tY($L) \
		           [expr [tsv::get _Y $K] + [tsv::get _YV $K] * $T]
                     set _tZ($L) \
		           [expr [tsv::get _Z $K] + [tsv::get _ZV $K] * $T]
	             set _tD($L) $_XI($K)
                  }
               }

# PURGE any old fit data that might be lying around

               if [info exists _iA] { unset _iA }
               if [info exists _cX] { unset _cX }
               if [info exists fOps(NC)] { unset fOps(NC) }

# FIT the 1D data

	       TUdataLSq3D _tX _tY _tZ _tD $tvS 0 _cX _iA fOps cV $_fN 
	       APpolyDeriv _cX _cX _cX $_fO $rL SCALAR \
		     [tsv::get _Xc $J] [tsv::get _Yc $J] [tsv::get _Zc $J] rV

# PUT the return data into the return variables
 
	       if $sS { tsv::set $oS $J $rV(0) }
	       if $sV  {
                  tsv::set $oX $J $rV(1) 
	          tsv::set $oY $J $rV(2) 
	          tsv::set $oZ $J $rV(3)
               }

# IF we are returning the Raw data, that is the value extapolated to the 
#    center of the volume do that here
#
	       if $sR {
	          TUpoly3DFunc [tsv::get _Xc $J] [tsv::get _Yc $J] \
		                        [tsv::get _Zc $J] _bF $fOps(NC) _fO
		  tsv::set $rX $J $_cX(0) 
		  for { set K 1 } { $K < $fOps(NC) } { incr K } {
		     tsv::set $rX $J \
		         [expr [tsv::get $rX $J] + $_bF($K) * $_cX($K)] 
		  }
               }
            } 
         } else {
            for { set J 0 } { $J < $tnV } { incr J } {
               if $NoData($J) { continue }
               
	       set K [expr $J * $vS]
               for { set L 0 } { $L < $vS } { incr K ; incr L } { 
	          set _tX($L) [tsv::get _X $K]
	          set _tY($L) [tsv::get _Y $K]
	          set _tZ($L) [tsv::get _Z $K]
	          set _tDx($L) $_XI($K)
	          set _tDy($L) $_YI($K)
	          set _tDz($L) $_ZI($K)
               }
 
               for { set N 1 } { $N <= $_ex } { incr N } { 
                  set T [expr $N * $cS]
	          set End [expr $L + $vS]
                  for { } { $L < $End } { incr K ; incr L } { 
                     set _tX($L) \
		           [expr [tsv::get _X $K] + [tsv::get _XV $K] * $T]
                     set _tY($L) \
		           [expr [tsv::get _Y $K] + [tsv::get _YV $K] * $T]
                     set _tZ($L) \
		           [expr [tsv::get _Z $K] + [tsv::get _ZV $K] * $T]
	             set _tDx($L) $_XI($K)
	             set _tDy($L) $_YI($K
	             set _tDz($L) $_ZI($K)
                  }
               }
               if [info exists _iAx] { unset _iAx }
               if [info exists _cX] { unset _cX }
               if [info exists _iAy] { unset _iAy }
               if [info exists _cY] { unset _cY }
               if [info exists _iAz] { unset _iAz }
               if [info exists _cZ] { unset _cZ }
               if [info exists fOps(NC)] { unset fOps(NC) }

	       TUdataLSq3D _tX _tY _tZ _tDx $tvS 0 _cX _iAx fOps cV $_fN 
	       TUdataLSq3D _tX _tY _tZ _tDy $tvS 0 _cY _iAy fOps cV $_fN 
	       TUdataLSq3D _tX _tY _tZ _tDz $tvS 0 _cZ _iAz fOps cV $_fN 

	       APpolyDeriv _cX _cY _cZ $_fO $rL VECTOR \
		     [tsv::get _Xc $J] [tsv::get _Yc $J] [tsv::get _Zc $J] rV

	       if $sS { tsv::set $oS $J $rV(0) }
	       if $sV {
                  tsv::set $oX $J $rV(1) 
	          tsv::set $oY $J $rV(2) 
	          tsv::set $oZ $J $rV(3)
               }

	       if $sR {
	          TUpoly3DFunc [tsv::get _Xc $J] [tsv::get _Yc $J] \
		                        [tsv::get _Zc $J] _bF $fOps(NC) _fO
		  tsv::set $rX $J $_cX(0) 
		  tsv::set $rY $J $_cY(0) 
		  tsv::set $rZ $J $_cZ(0)
		  for { set K 1 } { $K < $fOps(NC) } { incr K } {
		     tsv::set $rX $J \
		           [expr [tsv::get $rX $J] + $_bF($K) * $_cX($K)] 
		     tsv::set $rY $J \
		           [expr [tsv::get $rY $J] + $_bF($K) * $_cY($K)] 
		     tsv::set $rZ $J \
		           [expr [tsv::get $rZ $J] + $_bF($K) * $_cZ($K)] 
		  }
               }
            }
         }
      }
   }

# XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
# THE computations start here
# XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
 
# THIS is the number of definitions for this function

   set nF [$W index end]

# NO definitions then return

   if { $nF == 0 } { return }

# OPEN up the thread pool
 
   set tPoolID [THpoolOpen $PgmA]

# THIS is the time width of the data grid

   set cS [expr double($apANS(CellSz))]

# LOOP over the instances to see how to group definitions. A group uses a
#    common set of spacecraft position data which is present only for the
#    first definition in the group.  The group us assume to use the same
#    velocity definitions, the same _ex, and _iT definitions.

   set Starts ""
   for { set I 0 } { $I < $nF } { incr I } {
      set LiNe [$W get $I]
      scan $LiNe "%s" _vP
      if ![string match "---"  $_vP] { lappend Starts $I }
   }

# LOOP over the definition groups

   set nG [expr [llength $Starts] - 1 ]
   for { set G 0 } { $G <= $nG } { incr G } {

# SET the position of the first and last definition in the group
 
      if { $G != $nG } {
         set eG [lindex $Starts [expr $G + 1]]
      } else { set eG $nF }
      set bG [lindex $Starts $G]

# DO all the position computations associated with this group up-front. Get
#   the info from the first line in the definition

      set LiNe [$W get $bG]
      scan $LiNe "%s %s %s %s %s %s %s %s" _vP _vD _iT _vI _ex _fO _vSD _vR
      set _ex [expr $_ex - 1]

# THESE definitions are valid for the whole group so put them into tsv space 
#   now. The _ex definition is also but since it can be altered we'll put it
#   into tsv space later
 
      tsv::set sD _iT $_iT
      tsv::set sD _vD $_vD

# GET the spacecraft position variables. There should be three (x,y,z) for 
#   each location
 
      set vNames [lindex [APgetVNames $_vP] 0]
      set npV [llength $vNames]

# COMPUTE the number of positions defining the volume.  There needs to be
#   at least 4.

      set vS [expr $npV / 3]
      if { $vS < 4 } { return }

# SET up the position variable arrays. Leave the tsv versions in place.

      APsdVar X $vS $vNames _X YES
      APsdVar Y $vS $vNames _Y YES
      APsdVar Z $vS $vNames _Z YES

      set nE [lindex $_X(Dim) 0]

      APxferGInfo _X _AddOn
      set _AddOn(Dim) [list $nE 1]

      THarrayXfer TO _AddOn _AddOn 
      tsv::set sD nE $nE

# THIS is the real end of data.  It shortens if the volumes are being extended

      set tnV [expr $nE - $_ex]

# THIS is the total number of points which make up a volume

      set tvS [expr $vS * (1 + $_ex)]

# PUSH some of the information just set into tsv space.
 
      tsv::set sD tnV $tnV
      tsv::set sD tvS $tvS
      tsv::set sD vS $vS

# IF the volume is to be extended unpack the velocity array which must
#   be defined.  If there is no velocity data then set _ex to 0 (no
#   volume extension)

      if { $_ex > 0  } {
         if ![string match "---"  $_vD] { 
            set vNames [lindex [APgetVNames $_vI] 0]
            if { $npV != [llength $vNames] } { 
               APsdVar X $vS $vNames _XV YES
               APsdVar Y $vS $vNames _YV YES
               APsdVar Z $vS $vNames _ZV YES
               set nE [lindex $_XV(Dim) 0]
	    } else { set _ex 0 }
	 } else { set _ex 0 }
      }
      tsv::set sD _ex $_ex

# COMPUTE the volume center.  Need to add any of the extended volume to this. 

      for { set J 0 } { $J < $tnV } { incr J } { 
         set _Xc($J) 0.0 ; set _Yc($J) 0.0 ; set _Zc($J) 0.0
         set K [expr $J * $vS]
         for { set L 0 } { $L < $vS } { incr K ; incr L } { 
            set _Xc($J) [expr $_Xc($J) + $_X($K)]
            set _Yc($J) [expr $_Yc($J) + $_Y($K)]
            set _Zc($J) [expr $_Zc($J) + $_Z($K)]
         }

         for { set L 1 } { $L <= $_ex } { incr L } { 
            set T [expr $L * $cS]
            set End [expr $K + $vS]
            for {  } { $K < $End } { incr K } { 
               set _Xc($J) [expr $_Xc($J) + $_X($K) + $_XV($K) * $T]
	       set _Yc($J) [expr $_Yc($J) + $_Y($K) + $_YV($K) * $T]
	       set _Zc($J) [expr $_Zc($J) + $_Z($K) + $_ZV($K) * $T]
            }
         }

         set _Xc($J) [expr $_Xc($J) / double($tvS)]
         set _Yc($J) [expr $_Yc($J) / double($tvS)]
         set _Zc($J) [expr $_Zc($J) / double($tvS)]
      }

# TRANSFER the volume center positions to tsv space and remove the local
#   center position variables.
 
      THarrayXfer TO _Xc _Xc YES
      THarrayXfer TO _Yc _Yc YES
      THarrayXfer TO _Zc _Zc YES

# FREE the local spacecraft position arrays, the velocity arrays, and the
#   volume centers.  We don't need this information locally anymore.
 
      unset _X ; unset _Y ; unset _Z
      if { ![string match "---"  $_vD] && ($_ex > 0) } { 
         unset _XV ; unset _YV ; unset _ZV
      }

# LOOP over the definitions in each group. Each definition runs in its own
#   thread.

      set rN 0
      for { set I $bG } { $I < $eG } { incr I ; incr rN } {

# GET the information for this definition
   
         set LiNe [$W get $I]
         scan $LiNe "%s %s %s %s %s %s %s %s" _vP _vD _iT _vI _ex _fO _vSD _vR

# PUSH the options which can change between runs to tsv space
 
         tsv::set sD _fO$rN $_fO
         tsv::set sD _vSD$rN $_vSD
         tsv::set sD _vR$rN $_vR
         tsv::set sD _vI$rN $_vI

# GET input data variables and stick them in super varaibles then delete the
#   local versions of the data since we don't need them.

         set vNames [lindex [APgetVNames $_vI] 0]
         if [string match VECTOR $_iT] { 
            APsdVar X $vS $vNames _XI$rN YES
            APsdVar Y $vS $vNames _YI$rN YES
            APsdVar Z $vS $vNames _ZI$rN YES
	    unset _XI$rN ; unset _YI$rN ; unset _ZI$rN
         } else { APsdVar S $vS $vNames _XI$rN YES ; unset _XI$rN }

# SET up a list of return variables
 
         set sV 0 ; set sS 0 ; set sR 0
         if ![string match "---"  $_vSD] {
            set tNames [lindex [APgetVNames $_vSD] 0]
            set nN [llength $tNames]
            if { $nN >= 3 } { set sV 1 }
            if { $nN >  3 } { set sS 1 }
         }
         if ![string match "---"  $_vR] { set sR 1 }
 
         set rVaR($rN) ""
	 set oVc 0
         if $sV {
            set tNames [lindex [APgetVNames $_vSD] 0]
	    lappend rVaR($rN) [lindex $tNames 0]
	    lappend rVaR($rN) [lindex $tNames 1]
	    lappend rVaR($rN) [lindex $tNames 2]
	    set oVc 3
         }
         if $sS {
            set tNames [lindex [APgetVNames $_vSD] 0]
	    lappend rVaR($rN) [lindex $tNames $oVc]
         }
         if $sR {
            set tNames [lindex [APgetVNames $_vR] 0]
	    lappend rVaR($rN) [lindex $tNames 0]
	    if [string match VECTOR $_iT] {
	       lappend rVaR($rN) [lindex $tNames 1]
	       lappend rVaR($rN) [lindex $tNames 2]
            }
         }

# START a thread to compute the spatial derivatives for this set of input
#    definitions
 
	 lappend JoBs [THschdTask $tPoolID [list SpaDrv $rN]]
      }

# WAIT for all of the threads to complete
 
      THjobsDone $tPoolID $JoBs

# TRANSFER all the tsv output arrays into their respective global arrays.
 
      set rN 0
      for { set I $bG } { $I < $eG } { incr I ; incr rN } {
	 set nOut [llength $rVaR($rN)]
         for { set K 0 } { $K < $nOut } { incr K } {
	    set vR [lindex $rVaR($rN) $K]
	    global [set vR] ; upvar 0 [set vR] _fR
	    THarrayXfer FROM _fR $vR YES
         }
      }

      tsv::unset sD
      tsv::unset _X ; tsv::unset _Y ; tsv::unset _Z 
      if { ![string match "---"  $_vD] && ($_ex > 0) } {
         tsv::unset _XV ; tsv::unset _YV ; tsv::unset _ZV 
      }
      tsv::unset _Xc ; tsv::unset _Yc ; tsv::unset _Zc 
      tsv::unset _AddOn 
   }
}
