# THIS procedure solves for the cross helicity and the Elsasser variables.
#
# NOTES:  
#
# Conversion of B to Alfven units:
#             vA = B/sqrt(4*pi*ni*mi)
#                = 2.18e11(cm/s) * B/sqrt(mi * ni/mp)
#
#              B has units of gauss and n units of /cc
#
#  If B has units of nT and using proton mass then
#
#             vA (km/s) = 21.8 * B/sqrt(n)
#
#
#  To ensure tha + Elsasser variable represents outward propagation, Rotate
#      B field by 180 degrees whenever it points away from the Sun.  In GSE
#      assume Bx < 0 points away from the Sun.

package provide UDFAnalysis 1.0

proc APsolveXH { fD } {
   global apANS env Prefs

   APkeepTabs "STEP $fD : CROSS-HELICITY"

# THIS is the text window for this function definition

   set W .apFDEF$fD.body.list
   if ![winfo exists $W] {
      set GuI [lindex $apANS($apANS($fD,Func)) 0]
      eval $GuI $fD 1
   }

# THIS is the number of instances to run the function

   set nF [$W index end]

# NO instances then return

   if { $nF == 0 } { return }

# LOOP over the instances

   for { set I 0 } { $I < $nF } { incr I } {

# GET the first line
   
      set LiNe [$W get $I]

# BREAK it apart: 
#   _D   - Density 
#   _V   - Velocity
#   _B   - Magnetic Field
#   _Xp  - Return + (Elasser Variable) 
#   _Xn  - Return - (Elasser Variable) 
#   _Xh  - Return Cross Helicity

      scan $LiNe "%s %s %s %s %s %s" _D _V _B _Xp _Xn _Xh

# GET the input density variable - should only be one so take what you get

      set rV [APgetVNames $_D]
      set vR [lindex [lindex $rV 0] 0]
      global [set vR]
      upvar 0 [set vR] _N
      set nE [lindex $_N(Dim) 0]
      

# COMPONENT list

      set Comp [list x y z]

# GET the input velocity vector 
      
      set vNames [lindex [APgetVNames $_V] 0]
      for { set J 0 } { $J < 3 } { incr J } {
         set vR [lindex $vNames $J]
         global [set vR]
         upvar 0 [set vR] _V[lindex $Comp $J]
      }

# GET the input magnetic field vector 
      
      set vNames [lindex [APgetVNames $_B] 0]
      for { set J 0 } { $J < 3 } { incr J } {
         set vR [lindex $vNames $J]
         global [set vR]
         upvar 0 [set vR] _B[lindex $Comp $J]
      }

# SEE what's coming back

      set _rXp [string match $apANS(EmptyVar) $_Xp]
      set _rXn [string match $apANS(EmptyVar) $_Xn]
      set _rXh [string match $apANS(EmptyVar) $_Xh]

      if { ($_rXp && !$_rXn) || ($_rXn && !$_rXp) } {
         puts stderr "Only one Elasser variable defined!"
	 exit
      }

# GET the output + and -  Elasser vectors
      
      if { !$_rXp && !$_rXn } {
         set vNames [lindex [APgetVNames $_Xp] 0]
         for { set J 0 } { $J < 3 } { incr J } {
            set vR [lindex $vNames $J]
            global [set vR]
            upvar 0 [set vR] _XP[lindex $Comp $J]
	    upvar 0 _XP[lindex $Comp $J] _TmP_
            set _TmP_(Dim) [list $nE 1]
	    APxferGInfo _N _TmP_
         }

         set vNames [lindex [APgetVNames $_Xn] 0]
         for { set J 0 } { $J < 3 } { incr J } {
            set vR [lindex $vNames $J]
            global [set vR]
            upvar 0 [set vR] _XN[lindex $Comp $J]
	    upvar 0 _XN[lindex $Comp $J] _TmP_
            set _TmP_(Dim) [list $nE 1]
	    APxferGInfo _N _TmP_
         }
      }

# GET the output Cross Helicity
      
      if { !$_rXh } {
         set vNames [lindex [APgetVNames $_Xh] 0]
         set vR [lindex $vNames 0]
         global [set vR]
         upvar 0 [set vR] _XH
         set _XH(Dim) [list $nE 1]
         APxferGInfo _N _XH
      }

# Rectify magnetic field so that + Elsasser variable represent outward 
#    propogating  waves
 
      for { set J 0 } { $J < $nE } { incr J } {
         if { $_Bx($J) < 0.0 } { 
            set _bX($J) [expr -1.0 * $_Bx($J)]
            set _bY($J) [expr -1.0 * $_By($J)]
            set _bZ($J) [expr -1.0 * $_Bz($J)]
         } else { 
	    set _bX($J) $_Bx($J) ; set _bY($J) $_By($J) ; set _bZ($J) $_Bz($J)
         }
      }

      APbadGrid FLAG $nE _bX Good 
      APbadGrid AFLAG $nE _N Good 

# Compute the average Velocity and Magnetic component values for the 
#   time interval used.  These become the mean field and velocity.
 
      set aVx [AParrayAvg _Vx $nE AVG Good]
      set aVy [AParrayAvg _Vy $nE AVG Good]
      set aVz [AParrayAvg _Vz $nE AVG Good]
      set aBx [AParrayAvg _bX $nE AVG Good]
      set aBy [AParrayAvg _bY $nE AVG Good]
      set aBz [AParrayAvg _bZ $nE AVG Good]

# FORM the mean free values of V and B 
 
      for { set J 0 } { $J < $nE } { incr J } {
         if { $_N($J) > 0 } {
            set sD [expr sqrt($_N($J))]
            set _MfBx($J) [expr 21.8 * ($_bX($J) - $aBx) / $sD]
            set _MfBy($J) [expr 21.8 * ($_bY($J) - $aBy) / $sD]
            set _MfBz($J) [expr 21.8 * ($_bZ($J) - $aBz) / $sD]
            set _MfVx($J) [expr $_Vx($J) - $aVx]
            set _MfVy($J) [expr $_Vy($J) - $aVy]
            set _MfVz($J) [expr $_Vz($J) - $aVz]
         } else {
            set _MfBx($J) 0.0 ; set _MfBy($J) 0.0 ; set _MfBz($J) 0.0 
            set _MfVx($J) 0.0 ; set _MfVy($J) 0.0 ; set _MfVz($J) 0.0 
	 }
      } 

# Compute the positive and negative Elsasser values
 
      if { !$_rXp && !$_rXn } {
         for { set J 0 } { $J < $nE } { incr J } {
            set _XNx($J) [expr $_MfVx($J) - $_MfBx($J)] 
            set _XNy($J) [expr $_MfVy($J) - $_MfBy($J)] 
            set _XNz($J) [expr $_MfVz($J) - $_MfBz($J)] 
            set _XPx($J) [expr $_MfVx($J) + $_MfBx($J)] 
            set _XPy($J) [expr $_MfVy($J) + $_MfBy($J)] 
            set _XPz($J) [expr $_MfVz($J) + $_MfBz($J)] 
         }
         APbadGrid RESET $nE _XNx Good 0 $apANS(BaD) 
         APbadGrid RESET $nE _XNy Good 0 $apANS(BaD) 
         APbadGrid RESET $nE _XNz Good 0 $apANS(BaD) 
         APbadGrid RESET $nE _XPx Good 0 $apANS(BaD) 
         APbadGrid RESET $nE _XPy Good 0 $apANS(BaD) 
         APbadGrid RESET $nE _XPz Good 0 $apANS(BaD) 
       }

# Compute the Cross Helicity values
 
      if { !$_rXh } {
         for { set J 0 } { $J < $nE } { incr J } {
            set _XH($J) [expr $_MfVx($J) * $_MfBx($J)] 
            set _XH($J) [expr $_XH($J) + $_MfVy($J) * $_MfBy($J)] 
            set _XH($J) [expr $_XH($J) + $_MfVz($J) * $_MfBz($J)] 
            set _XH($J) [expr 0.5 * $_XH($J)] 
         }
         APbadGrid RESET $nE _XH  Good 0 $apANS(BaD) 
      }
   }
}
