#include <stdlib.h>
#include <string.h>
#include "OpSySInD.h"
#include "Server.h"
#include "dbf.h"

/******************************************************************************
 |  >FILE:  find_entry.c                                                      |
 |                                                                            |
 |  DESCRIPTION                                                               |
 |                                                                            |
 |  SYNOPSIS                                                                  |
 |     int find_entry(D, vname, btime, etime, mode)                           |
 |                                                                            |
 |  ARGUMENTS                                                                 |
 |     dbfRecord *D      pointer to database structure to apply the find      |
 |     char *vname       virtual instrument name                              |
 |     char mode         find closest or exact match only                     |
 |     long btime        beginning time of request                            |
 |     long etime        ending time of request                               |
 |                                                                            |
 |  NECESSARY SUBPROGRAMS                                                     |
 |     sprintf()             formatted output to a string                     |
 |     strcmp()              string compare                                   |
 |     _FieldRecord FIELD()  returns a pointer to field within a database     |
 |                           record                                           |
 |     int Find()            locate a key within a database                   |
 |     int GoTo()            read a specific database record                  |
 |     int Skip()            position and read another database record        |
 |                                                                            |
 |  EXTERNAL VARIABLES                                                        |
 |     None                                                                   |
 |                                                                            |
 |  INTERNAL VARIABLES                                                        |
 |     char key[30]        key to locate within the database                  |
 |     char r_vname[20]    the virtual instrument read from the databse file  |
 |     char str[20]        string used in reading the times from the database |
 |     int  rval           return value from subroutine                       |
 |     int  r_btime        begin time read from database file                 |
 |     int  r_etime        end time read from database file                   |
 |     int  rec            database record number holder                      |
 |                                                                            |
 |  SUBSYSTEM                                                                 |
 |     DATABASE (Utility)                                                     |
 |                                                                            |
 ******************************************************************************/

#define FG(a,b,c) if (FieldGetN(a, b, c) != SUCCESS) \
                     return 0

static ByTe_1 rcsid[] = "$Id: find_entry.c,v 1.1 1999/11/12 20:36:21 chris.gurgiolo.b2r Stab chrisg $";

int find_entry( int dbf, int ndx, char *vname, Time_t btime, 
                Time_t etime, int mode)
{
   char key[30], rvname[30], *r_vname;
   int   rec, irec;
   Time_t   r_btime, r_etime;


   /*
    | if there are no database records in the database, no need to continue
    */
   if (mNumRecs(dbf) == 0)
   {
      return 0;
   }

   /*
    | generate the key for the Find() routine
    */
   (void)sprintf(key, "%-8s%04d%3d%8d", vname, btime.yr, btime.day, btime.msec);

   /*
    | preform the find and act accordingly on the return value
    */
   switch (dbFind(dbf, ndx, key, mode))
   {
      case EXACT_MATCH:

         /*
          | no need to do anything, found the one we want, return a 1
          */
         return 1;

      case MATCH_LESS:
         /*
          | the one found is one entry less than the one we want, so look at
          | the one before it and see if it is the one we want
          */
         if (Skip(dbf, ndx, -1) != SUCCESS)
         {
            return 0;
         }

         FG(dbf, "V_INST", rvname);
         r_vname = rtrim(rvname);

         if (strcmp(r_vname, vname) != 0)
         {
            if (Skip(dbf, ndx, 1) != SUCCESS)
            {
               return 0;
            }

            FG(dbf, "V_INST", rvname);
            r_vname = rtrim(rvname);

            if (strcmp(r_vname, vname) != 0)
            {
               return 0;
            }
         }

         break;

      case MATCH_GREATER:
         /*
          | the one found is one entry greater than the one we want, so look at
          | the one after it and see if it is the one we want
          */
         FG(dbf, "V_INST", rvname);
         r_vname = rtrim(rvname);

         if (strcmp(r_vname, vname) != 0)
         {
            if (Skip(dbf, ndx, 1) != SUCCESS)
            {
               return 0;
            }

            FG(dbf, "V_INST", rvname);
            r_vname = rtrim(rvname);

            if (strcmp(r_vname, vname) != 0)
            {
               return 0;
            }
         }

         break;

      case NO_MATCH:
      case FIND_ERROR:
         return 0;
   }
         
   /*
    | get the times of this entry
    */
   FG(dbf, "B_YR",   &(r_btime.yr));
   FG(dbf, "B_DAY",  &(r_btime.day));
   FG(dbf, "B_MSEC", &(r_btime.msec));
   FG(dbf, "E_YR",   &(r_etime.yr));
   FG(dbf, "E_DAY",  &(r_etime.day));
   FG(dbf, "E_MSEC", &(r_etime.msec));

   /*
    | make sure the times requested still fall within this new record
    */
   if (cmpTime(btime, r_etime) <= 0 && cmpTime(etime, r_btime) >= 0)
   {
      irec = mIndexRec(dbf, ndx);
      rec = mCurRec(dbf);
      if (Skip(dbf, ndx, -1) != SUCCESS)
      {
         return 0;
      }

      FG(dbf, "V_INST", rvname);
      r_vname = rtrim(rvname);
      FG(dbf, "B_YR",   &(r_btime.yr));
      FG(dbf, "B_DAY",  &(r_btime.day));
      FG(dbf, "B_MSEC", &(r_btime.msec));
      FG(dbf, "E_YR",   &(r_etime.yr));
      FG(dbf, "E_DAY",  &(r_etime.day));
      FG(dbf, "E_MSEC", &(r_etime.msec));

      if (strcmp(r_vname, vname) == 0)
      {
         if (cmpTime(btime, r_etime) <= 0 &&
             cmpTime(etime, r_btime) >= 0)
         {
            /*
             | found one that falls within our range
             */
            return 1;
         }

         if (GoTo(dbf, rec) != SUCCESS)
         {
            return 0;
         }
         mIndexRec(dbf, ndx) = irec;
      }
      else
      {
         if (GoTo(dbf, rec) != SUCCESS)
         {
            return 0;
         }
         mIndexRec(dbf, ndx) = irec;
      }
      
      return 1;
   }
 
   if (Skip(dbf, ndx, 1) == SUCCESS)
   {
      FG(dbf, "V_INST", rvname);
      r_vname = rtrim(rvname);
      FG(dbf, "B_YR",   &(r_btime.yr));
      FG(dbf, "B_DAY",  &(r_btime.day));
      FG(dbf, "B_MSEC", &(r_btime.msec));
      FG(dbf, "E_YR",   &(r_etime.yr));
      FG(dbf, "E_DAY",  &(r_etime.day));
      FG(dbf, "E_MSEC", &(r_etime.msec));

      if (strcmp(r_vname, vname) == 0 &&
          cmpTime(btime, r_etime) <= 0 &&
          cmpTime(etime, r_btime) >= 0)
      {
         /*
          | found one that falls within our range
          */
         return 1;
      }
   }

   return 0;
}
