#include <stdio.h>
#include <string.h>
#include "libErrorMsgs.h"
#include "ErrorCodes.h"

/*****************************************************************************
 *                                                                           *
 *                          GET_ERROR_STR SUBROUTINE                         *
 *                                                                           *
 *  DESCRIPTION                                                              *
 *    This routine is called to compose the error string encountered from    *
 *  the database software, the idfs read routines or the PIDF software.      *
 *  If the error code is not identified as coming from one of these sources, *
 *  an error code is returned.                                               *
 *                                                                           *
 *  INPUT VARIABLES                                                          *
 *    short error_code          the error code to be processed               *
 *    char *err_str             ptr to area where string is to be returned   *
 *    long str_len              length of string to be returned              *
 *                                                                           *
 *  USAGE                                                                    *
 *    x = get_error_str (error_code, err_str, str_len)                       *
 *                                                                           *
 *  NECESSARY SUBPROGRAMS                                                    *
 *    strlen ()                 determines the length of a string            *
 *    strcpy()                  copies a string to another string variable   *
 *    sprintf()                 creates a string based upon arguments and    *
 *                              conversion characters                        * 
 *    db_msgs ()                returns the message associated with the error*
 *                              code from the database software              *
 *    ir_error_message_str ()   returns the message associated with the error*
 *                              code from the idfs read routines             *
 *                                                                           *
 *  EXTERNAL VARIABLES                                                       *
 *    None                                                                   *
 *                                                                           *
 *  INTERNAL VARIABLES                                                       *
 *    long length               length of the error string                   *
 *    char e_str[]              holds the error string to be outputted       *
 *    char *tmp                 pointer to error string returned by DB_MSGS()*
 *    char error_found          flag indicating error code was identified    *
 *                                                                           *
 *  SUBSYSTEM                                                                *
 *    Display Level                                                          *
 *                                                                           *
 *****************************************************************************/

static ByTe_1 rcsid[] = "$Id: err_str.c,v 1.1 1999/11/20 19:17:23 chris.gurgiolo.b2r Stab chrisg $";
    
ByTe_2 get_error_str (ByTe_2 error_code, ByTe_1 *err_str, ByTe_4 str_len)
{
   ByTe_4 length;
   ByTe_1 e_str[200], *tmp, error_found;

   /************************************************************************/
   /*  Database software error.                                            */
   /************************************************************************/

   error_found = 1;
   if (error_code <= DB_START_neg && error_code >= DB_STOP_neg)
    {
      tmp = db_msgs ((int) error_code);
      sprintf (e_str, "DB_ERROR: %s.", tmp);
    }

   /************************************************************************/
   /*  Error code generated by idfs read routines.                         */
   /************************************************************************/

   else if (error_code <= UTIL_START_CODE && error_code >= UTIL_STOP_CODE)
     ir_error_message_str (error_code, e_str);

   /************************************************************************/
   /*  Error code from some unknown source area.                           */
   /************************************************************************/

   else
    {
      error_found = 0;
      sprintf (e_str, "A MATCH FOR ERROR CODE %d WAS NOT FOUND.", error_code);
    }

   /************************************************************************/
   /*  Make sure error string will fit in return string.                   */
   /************************************************************************/

   length = strlen (e_str);
   if (length > str_len)
     e_str[str_len-1] = '\0';
   strcpy (err_str, e_str);
 
   if (!error_found)
     return (!ALL_OKAY);
   else
     return (ALL_OKAY);
}
