#include "libbase_udf.h"

/*****************************************************************************
 *                                                                           *
 *                           IR_TRANSFER_DATA SUBROUTINE                     *
 *                                                                           *
 *  DESCRIPTION                                                              *
 *    This routine is called to copy the raw data into the data array that   *
 *  will be returned to the user from the convert_to_units() call.  The data *
 *  is copied into this array before any table operations are performed.  In *
 *  the transfer to a floating point array, a check is made to determine if  *
 *  the raw telemetry is unsigned binary, signed binary, or floating point.  *
 *                                                                           *
 *  INPUT VARIABLES                                                          *
 *    ReaL_8 *DataOut           pointer to the array that holds the values   *
 *                              returned by the module                       *
 *    long num_data             the number of elements being processed       *
 *    char *inp_data            pointer to the array that holds the input    *
 *                              data values                                  *
 *    char dtype                the format of the data - uns int, int, float *
 *    long inp_start            index value indicating where to start in the *
 *                              input data array                             *
 *                                                                           *
 *  USAGE                                                                    *
 *    ir_transfer_data (&ret_data, inp_data, num_data, dtype, inp_start)     *
 *                                                                           *
 *  NECESSARY SUBPROGRAMS                                                    *
 *    ir_intfp2fp ()             reconstructs raw floating point data	     *
 *    ir_to_sign()               converts raw data into signed integer data  *
 *                                                                           *
 *  EXTERNAL VARIABLES                                                       *
 *    None                                                                   *
 *                                                                           *
 *  INTERNAL VARIABLES                                                       *
 *    register float *f1         pointer to float values                     *
 *    register long *l1          pointer to a signed long data value         *
 *    register long j            looping variable                            *
 *    register uns long *ul      pointer to an unsigned long data value      *
 *                                                                           *
 *  SUBSYSTEM                                                                *
 *    Display Level                                                          *
 *                                                                           *
 ****************************************************************************/

void ir_transfer_data (ReaL_8 *DataOut, ByTe_1 *inp_data, ByTe_4 num_data, 
                       ByTe_1 dtype, ByTe_4 inp_start)
{
   register ReaL_8 *d1;
   register ByTe_4 *l1, j;
   register u_ByTe_4 *ul;

   /**************************************************************************/
   /*  The format of the injested data is unsigned integer binary data.      */
   /**************************************************************************/

   d1 = DataOut;
   if (dtype == 0)
    {
      ul = (u_ByTe_4 *) inp_data + inp_start;
      for (j = 0; j < num_data; ++j)
        *d1++ = *ul++;
    }

   /**************************************************************************/
   /*  The format of the injested data is signed integer binary data.        */
   /**************************************************************************/

   else if (dtype == 1)
    {
      l1 = (ByTe_4 *) inp_data + inp_start;
      for (j = 0; j < num_data; ++j)
        *d1++ = *l1++;
    }

   /**************************************************************************/
   /*  The format of the injested data is floating point data.               */
   /**************************************************************************/

   else if (dtype >= 2)
    {
      ul = (u_ByTe_4 *) inp_data + inp_start;
      for (j = 0; j < num_data; ++j)
        *d1++ = ir_intfp2fp (*ul++, dtype);
    }
}
