#include "OpSySInD.h"
#include "util_str.h"
#include "ret_codes.h"
#include "user_defs.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                           START_IMAGE SUBROUTINE                            *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine will find the sensor set which contains the start of an     *
 *  image (scan line = 0).  This routine should be called AFTER file_pos has   *
 *  been called.  Some assumptions are made by this module the first of which  *
 *  assumes that calibration set zero is ALWAYS defined to hold the scan line  *
 *  number and this is a single quantity (one value, not an array of values).  *
 *  Second, instruments are sweeping instruments so the value of FULL_SWP does *
 *  not matter for read_drec.  Third, the scan line value increases from       *
 *  sensor set to sensor set, so we only need to check the value for the first *
 *  sensor set in the data record.                                             *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    unsigned long data_key   key which uniquely identifies the data set      *
 *                             being processed                                 *
 *    char *exten              the filename extension for the data to be used  *
 *    unsigned short vnum      version number to be associated with this       *
 *                             combination (allows for multiple opens)         *
 *    void *UDF                ptr to the memory location for the structure    *
 *                             that holds returned data values (read_drec)     *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = start_image (data_key, exten, vnum, UDF)                             *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    lseek()                  moves the file pointer to a location within     *
 *                             the file                                        *
 *    read()                   reads N bytes from the file associated with     *
 *                             the file descriptor given                       *
 *    ir_locate_ex()           determines if the requested combination has     *
 *                             already been processed and points to the        *
 *                             correct structure allocated for the combo       *
 *    ir_read_header()         reads data from the header file                 *
 *    read_drec ()             universal read routine that retrieves data for  *
 *                             the time sample being processed                 *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info      structure that holds information concerning     *
 *        ginfo                the experiment that is being processed          *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct idf_data          structure holding all of the currently          *
 *        *EXP_DATA            returned data values to be processed            *
 *    struct experiment_info   a pointer to the structure that holds specific  *
 *          *ex                experiment information                          *
 *    struct ptr_rec *ptr      a pointer to the structure which holds all      *
 *                             pointers to the header and data for the         *
 *                             experiment of interest                          *
 *    int num_bytes            the number of bytes requested from the file     *
 *    int ret_bytes            the number of bytes actually read by READ()     *
 *    long cal_value           scan line value                                 *
 *    long base_value          comparison scan value                           *
 *    long ret_val             value returned by called module                 *
 *    short rval               value returned by called module                 *
 *    short SeN                sensor value of interest                        *
 *    char reset_called        flag indicating if LOCATE_EX was called         *
 *    char find_zero           flag indicating if a search for scan line zero  *
 *                             is needed                                       *
 *    char past_start          flag indicating that lseek returned an error    *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 start_image (u_ByTe_4 data_key, ByTe_1 *exten, u_ByTe_2 vnum, void *UDF)
{
   extern struct general_info ginfo;

   struct idf_data *EXP_DATA;
   struct experiment_info *ex;
   struct ptr_rec *ptr;
   int num_bytes, ret_bytes;
   ByTe_4 cal_value, base_value, ret_val;
   ByTe_2 rval, SeN;
   ByTe_1 reset_called, find_zero, past_start;

   /**************************************************************************/
   /*  Check to see if the combination being processed has been processed    */
   /*  before.  If not, an error condition - probably didn't call FILE_OPEN. */
   /*  Since a 0 is passed for the last parameter, the only possible error is*/
   /*  that the requested combination was not found among processed combos.  */
   /**************************************************************************/
  
   if (!ginfo.called_locate)
    {
      rval = ir_locate_ex (data_key, exten, vnum, 0);
      if (rval != ALL_OKAY)
        return (IMAGE_NOT_FOUND);
 
      ginfo.called_locate = 1;
      reset_called = 1;
    }
   else
     reset_called = 0;

   /***************************************************************************/
   /*  Set a pointer to the structure which holds all pointers for header and */
   /*  data information for the experiment currently being processed.         */
   /***************************************************************************/

   ex = ginfo.expt;
   ptr = ex->info_ptr;
   EXP_DATA = (struct idf_data *) UDF;
   SeN = *ptr->SENSOR_INDEX;

   /***************************************************************************/
   /*  Call read_drec to get the sample associated with the FILE_POS location */
   /*  and do not advance since we only want to get the first value for       */
   /*  comparison purposes.                                                   */
   /***************************************************************************/

   rval = read_drec (ex->data_key, ex->exten, ex->version, UDF, SeN, 0, 0, 1);
   if (rval < 0 || rval % 3 == 0)
    {
      if (reset_called)
        ginfo.called_locate = 0;
      return (rval);
    }

   cal_value = *(EXP_DATA->cal_data + 0);
   base_value = cal_value;
   find_zero = (cal_value == 0) ? 0 : 1;
   past_start = 0;

   /**************************************************************************/
   /*  Find the data record that contains a scan line value of zero.         */
   /**************************************************************************/

   while (find_zero)
    {
      /********************************************************************/
      /*  Back up and read the previous data record.  -2 is used since fd */
      /*  is at the END of the current data record.                       */
      /********************************************************************/

      num_bytes = ptr->d_size;
      ret_val = lseek (ex->fdd, (ByTe_4) (-2 * num_bytes), 1);

      /********************************************************************/
      /*  If an error is returned by lseek, set to beginning of file.     */
      /********************************************************************/

      if (ret_val == -1)
       {
         ret_val = lseek (ex->fdd, 0L, 0);
         ret_bytes = read (ex->fdd, ex->DATA_MEM, num_bytes);
         if(ex->BswaP)
           ReOrderUdf (1);
         cal_value = 0;
         past_start = 1;
       }
      else
       {
         ret_bytes = read (ex->fdd, ex->DATA_MEM, num_bytes);
         if(ex->BswaP)
           ReOrderUdf (1);
         if (ret_bytes <= 0 || ret_bytes != num_bytes)
          {
            /***************************************************************/
            /*  An error was encountered when trying to read from the file.*/
            /***************************************************************/
       
            if (ret_bytes < 0)
             {
               if (reset_called)
                 ginfo.called_locate = 0;
               return (IMAGE_READ_ERROR);
             }
       
            /****************************************************************/
            /*  An EOF was encountered.  If a non-real time file is being   */
            /*  processed, an error was encountered.                        */
            /****************************************************************/
       
            else
             {
               if (ret_bytes != num_bytes)
                  ret_val = lseek (ex->fdd, (ByTe_4) (-1 * ret_bytes), 1);
               if (reset_called)
                 ginfo.called_locate = 0;
               if (ex->btime_sec >= 0)
                  return (IMAGE_READ_ERROR);
               else
                  return (EOF_STATUS);
             }
          }
       }

      /*****************************************************************/
      /*  Read the header associated with the data record.             */
      /*****************************************************************/

      ret_val = lseek (ex->fdh, (ByTe_4)*(ptr->HDR_OFF), 0);
      rval = ir_read_header (UDF);
      if (rval != ALL_OKAY)
       {
         if (reset_called)
           ginfo.called_locate = 0;

         if (rval == RHDR_READ_ERROR)
           return (IMAGE_READ_ERROR);
         else if (rval == RHDR_HDR_MALLOC)
           return (IMAGE_HDR_MALLOC);
         else if (rval == RHDR_HDR_REALLOC)
           return (IMAGE_HDR_REALLOC);
         else
           return (rval);
       }

      /***********************************************************************/
      /*  Initialize variables pertinent to the read_drec routine.           */
      /***********************************************************************/
 
      ex->accum_ss_sz = 0;
      ptr->time_row = 0;
      ptr->time_col = 0;
      ptr->cur_sen_set = 0;
      ptr->reset_hdr = 1;

      rval = read_drec (ex->data_key, ex->exten, ex->version, UDF,
                        SeN, 0, 0, 1);
      if (rval < 0 || rval % 3 == 0)
       {
         if (reset_called)
           ginfo.called_locate = 0;
         return (rval);
       }

      if (!past_start)
        cal_value = *(EXP_DATA->cal_data + 0);

      if (cal_value == 0 || cal_value > base_value)
        find_zero = 0;
      else 
        base_value = cal_value;
    }

   /***********************************************************************/
   /*  Reset variables pertinent to the read_drec routine.                */
   /***********************************************************************/
 
   ex->accum_ss_sz = 0;
   ptr->time_row = 0;
   ptr->time_col = 0;
   ptr->cur_sen_set = 0;
   ptr->reset_hdr = 1;
   find_zero = (cal_value == 0) ? 0 : 1;

   /**********************************************************************/
   /*  Find the sensor set that sets the scan line to zero.              */
   /**********************************************************************/

   while (find_zero)
    {
      /**********************************************************************/
      /*  Read the sweep. (use no advance to test cal value).               */
      /**********************************************************************/

      rval = read_drec (ex->data_key, ex->exten, ex->version, UDF, 
                        SeN, 0, 0, 1);
      if (rval < 0 || rval % 3 == 0)
       {
         if (reset_called)
           ginfo.called_locate = 0;
         return (rval);
       }

      cal_value = *(EXP_DATA->cal_data + 0);
      if (cal_value == 0)
        find_zero = 0;
      else
       {
         /********************************************************************/
         /*  Advance to the next sensor set.                                 */
         /********************************************************************/

         rval = read_drec (ex->data_key, ex->exten, ex->version, UDF,
                           SeN, 0, 1, 1);
         if (rval < 0 || rval % 3 == 0)
          {
            if (reset_called)
              ginfo.called_locate = 0;
            return (rval);
          }
       }
    }

   if (reset_called)
     ginfo.called_locate = 0;

/*  Added CAG 12/23/98 - moved mem check into start_image                    */

  rval = ir_check_idf_data_memory (ex->data_key, ex->exten, ex->version, UDF);
  if (rval != ALL_OKAY)
      return (rval);

   return (ALL_OKAY);
}
