#include <stdlib.h>
#include "util_str.h"
#include "libbase_udf.h"
#include "OpSySInD.h"

/******************************************************************************
 *                                                                            *
 *                             IR_SAMPLE_TIME SUBROUTINE                      *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to determine the time associated with the sample *
 *  being processed and returned to the calling program. The time is returned *
 *  in two parts, the time in milliseconds and for finer resolution, the      *
 *  number of nanoseconds left over after the time in milliseconds has been   *
 *  determined.  The algorithm used to determine the time is taken directly   *
 *  from the VIDF definition document.                                        *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    unsigned short max_ele     the number of data values to be returned     *
 *    void UDF                   the UDF read info structure                  *
 *    unsigned short time_row    the matrix row being processed               *
 *    short time_col             the matrix column being processed            *
 *    long *btime_ms             start time (msec) of the sample              *
 *    long *btime_ns             start time (nanoseconds) of the sample       *
 *    long *etime_ms             end time (msec) of the sample                *
 *    long *etime_ns             end time (nanoseconds) of the sample         *
 *    char *decrement_bday       flag indicating if start day needs to be     *
 *                               modified (if btime ends up negative)         *
 *    char *decrement_eday       flag indicating if end day needs to be       *
 *                               modified (if etime ends up negative)         *
 *                                                                            *
 *  USAGE                                                                     *
 *    ir_sample_time (max_ele, UDF, time_row, time_col, &btime_ms,            *
 *                    &btime_ns, &etime_ms, &etime_ns, &decrement_bday,       *
 *                    &decrement_eday)                                        *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    abs()                      returns the absolute value of a number       *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info        structure that holds information concerning  *
 *        ginfo                  the experiment that is being processed       *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info     a pointer to the structure that holds        *
 *          *ex                  specific experiment information              *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all   *
 *                               pointers to the header and data for the      *
 *                               experiment of interest                       *
 *    long add_ms                the number of milliseconds to add to the     *
 *                               millisecond time element                     *
 *    long time_off              offset time (in millisecond) for each sensor *
 *    unsigned short n_sample    number of samples returned for each sensor   *
 *    unsigned short last_row    value used to compute the end time of the    *
 *                               last element in the sweep                    *
 *    short swp_off              the last step returned for the sweep         *
 *    short base_swp_off         the first step returned for the sweep        *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

void ir_sample_time (u_ByTe_2 max_ele, void *UDF, u_ByTe_2 time_row, 
                     ByTe_2 time_col, ByTe_4 *btime_ms, ByTe_4 *btime_ns, 
                     ByTe_4 *etime_ms, ByTe_4 *etime_ns, 
                     ByTe_1 *decrement_bday, ByTe_1 *decrement_eday)
{
  extern struct general_info ginfo;

  struct experiment_info *ex;
  struct ptr_rec *ptr;
  struct idf_data *ExDa;
  ByTe_4 add_ms, time_off;
  u_ByTe_2 n_sample, last_row;
  ByTe_2 swp_off, base_swp_off;

  /**************************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header    */
  /*  and data information for the experiment currently being processed and */
  /*  point to the sensor structure for the sensor of interest.             */
  /**************************************************************************/

  ex  = ginfo.expt;
  ptr = ex->info_ptr;
  ExDa = (struct idf_data *) UDF;

  n_sample = *ptr->N_SAMPLE;

  /****************************************************************/
  /* If this is a matrix sensor, modify the time_col to reflect   */
  /* the current relative matrix column number                    */
  /****************************************************************/

  if (ex->smp_id == 3) 
      time_col += ExDa->column - *ptr->LEAD_COL;

  /**************************************************************************/
  /*  Determine the beginning time for the sample being processed using all */
  /*  timing elements.  PTR->TIME and TIME_OFF are not added to the         */
  /*  nanoseconds element since they are returned in milliseconds only.     */
  /**************************************************************************/

  *decrement_bday = 0;
  *decrement_eday = 0;
  time_off = *(ex->time_off + ExDa->sensor);
  *btime_ms = *ptr->TIME + time_off + ex->accum_ss_ms + 
              time_row * ex->time_col_ms + time_col * ex->time_row_ms;
  *btime_ns = ex->accum_ss_ns + time_row * ex->time_col_ns + 
              time_col * ex->time_row_ns;

  /**************************************************************************/
  /*  If the time ends up negative due to negative time_off value, modify   */
  /*  before computing end time and send back flag since day value is not   */
  /*  sent in.                                                              */
  /**************************************************************************/

  if (*btime_ms < 0)
   {
     *decrement_bday = 1;
     *btime_ms += 86400000;
   }

  /**************************************************************************/
  /*  If the nanosecond time element can be converted into milliseconds, do */
  /*  so and add this to the millisecond time element and keep the leftover */
  /*  nanoseconds.  1000000 nanoseconds = 1 millisecond.                    */ 
  /**************************************************************************/

  add_ms = *btime_ns / 1000000;
  *btime_ms += add_ms;
  *btime_ns = *btime_ns % 1000000;

  /**************************************************************************/
  /*  Determine the ending time for the sample being processed.             */
  /**************************************************************************/

  if (!ex->TCP)
   {
     *etime_ms = *btime_ms + ex->time_sm_ms;
     *etime_ns = *btime_ns + ex->time_sm_ns;
   }
  else if (ex->smp_id == 2)
   {
     last_row = (max_ele == 1) ? time_row + 1 : n_sample;
     *etime_ms = *ptr->TIME + time_off + ex->accum_ss_ms +
                 last_row * ex->time_col_ms + time_col * ex->time_row_ms;
     *etime_ns = ex->accum_ss_ns + last_row * ex->time_col_ns +
                 time_col * ex->time_row_ns;
   }
  else
   {
     if (ex->da_method == 3)
      {
        /***********************************************************/
        /*  Instrument could be in a mode where a single step is   */
        /*  being returned.                                        */
        /***********************************************************/

        if (n_sample == 1)
          last_row = 1;

        /*************************************************************/
        /*  Determine how many steps are missing and then multiply   */
        /*  by the number of samples returned to get the length of   */
        /*  the sweep.  The assumption is that the number of missing */
        /*  steps is the same from one scan_index value to the next, */
        /*  so just use 0 and 1.                                     */
        /*************************************************************/

        else
         {
           base_swp_off = *ptr->SAMP_INDEX;
           swp_off = *(ptr->SAMP_INDEX + 1);
           last_row = abs (swp_off - base_swp_off) * n_sample;
         }
        *etime_ms = *ptr->TIME + time_off + ex->accum_ss_ms +
                    last_row * ex->time_col_ms + time_col * ex->time_row_ms;
        *etime_ns = ex->accum_ss_ns + last_row * ex->time_col_ns +
                    time_col * ex->time_row_ns;
      }
     else
      {
        base_swp_off = *ptr->SAMP_INDEX;
        swp_off = *(ptr->SAMP_INDEX + (n_sample - 1));
        last_row = (ex->da_method == 0) ? n_sample : (ex->da_method == 1) ?
                   ex->swp_len : abs (swp_off - base_swp_off);
        *etime_ms = *ptr->TIME + time_off + ex->accum_ss_ms +
                    last_row * ex->time_col_ms + time_col * ex->time_row_ms;
        *etime_ns = ex->accum_ss_ns + last_row * ex->time_col_ns +
                    time_col * ex->time_row_ns;
      }
   }

  /***************************************************************************/
  /*  If the time ends up negative due to negative time_off value, modify    */
  /*  before computing end time and send back flag since day value is not    */
  /*  sent in.                                                               */
  /***************************************************************************/

  if (*etime_ms < 0)
   {
     *decrement_eday = 1;
     *etime_ms += 86400000;
   }

  /***************************************************************************/
  /*  If the nanosecond time element can be converted into milliseconds, do  */
  /*  so and add this to the millisecond time element and keep the leftover  */
  /*  nanoseconds.  1000000 nanoseconds = 1 millisecond.                     */ 
  /***************************************************************************/

  add_ms = *etime_ns / 1000000;
  *etime_ms += add_ms;
  *etime_ns = *etime_ns % 1000000;
}
