#include <stdio.h>
#include <stdlib.h>
#include "OpSySInD.h"
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                       RESET_EXPERIMENT_INFO SUBROUTINE                      *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called whenever the end of the current data file has     *
 *  been reached and the status flag indicates that more data is forthcoming.  *
 *  The current files must be closed and certain flags must be reset to        *
 *  simulate a fresh start.  Some memory arrays must be freed in anticipation  *
 *  of new allocated space for these arrays and the new data files must be     *
 *  opened.                                                                    *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    unsigned long data_key   key which uniquely identifies the data set      *
 *                             being processed                                 *
 *    char *exten              the filename extension for the data to be used  *
 *    unsigned short vnum      version number to be associated with this       *
 *                             combination (allows for multiple opens)         *
 *    short btime_yr           the start time requested (year component)       *
 *    short btime_day          the start time requested (day component)        *
 *    long btime_sec           the start time requested (seconds component)    *
 *    long btime_nsec          the start time requested (nanoseconds)          *
 *    short etime_yr           the stop time requested (year component)        *
 *    short etime_day          the stop time requested (day component)         *
 *    long etime_sec           the stop time requested (seconds component)     *
 *    long etime_nsec          the stop time requested (nanoseconds)           *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = reset_experiment_info (data_key, exten, vnum, btime_yr, btime_day,   *
 *                               btime_sec, btime_nsec, etime_yr, etime_day,   *
 *                               etime_sec, etime_nsec)                        *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    ir_locate_ex()           determines if the requested combination has     *
 *                             already been processed and points to the        *
 *                             correct structure allocated for the combo       *
 *    ir_free_vidf_structure ()    frees memory allocated for VIDF processing  *
 *    file_open()              opens the necessary header, data, and VIDF files*
 *                             for the requested data set                      *
 *    free ()                  frees previously allocated memory               *
 *    close()                  closes the file associated with the file        *
 *                             descriptor specified                            *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info      structure that holds information concerning     *
 *        ginfo                the experiment that is being processed          *
 *    char inside_reset_exp    flag indicating in the RESET_EXPERIMENT_INFO    *
 *                             module (needed for FILE_OPEN logic)             *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info   a pointer to the structure that holds specific  *
 *          *ex                experiment information                          *
 *    struct sensor_tables     a pointer to the structure which holds the      *
 *         *sptr               various information for the tables utilized by  *
 *                             sensor-table combination being processed        *
 *    struct mode_tables       a pointer to the structure which holds the      *
 *         *mode_ptr           mode-dependent table information                *
 *    struct crit_action_info  a pointer to the structure that holds info.     *
 *       *crit_act_ptr         pertinent to table(s) that are flagged as       *
 *                             critical action table(s)                        *
 *    short rV                 holds the value returned by the called routine  *
 *    short sen, i             looping variables                               *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 reset_experiment_info (u_ByTe_4 data_key, ByTe_1 *exten, u_ByTe_2 vnum, 
                              ByTe_2 btime_yr, ByTe_2 btime_day, 
                              ByTe_4 btime_sec, ByTe_4 btime_nsec, 
                              ByTe_2 etime_yr, ByTe_2 etime_day, 
                              ByTe_4 etime_sec, ByTe_4 etime_nsec)
{
   extern struct general_info ginfo;
   extern ByTe_1 inside_reset_exp;

   struct experiment_info *ex;
   struct sensor_tables *sptr;
   struct mode_tables *mode_ptr;
   struct crit_action_info *crit_act_ptr;
   ByTe_2 rV, sen, i;

   /***************************************************************************/
   /*  Check to see if the combination being processed has been processed     */
   /*  before.  Since ir_locate_ex() is called by more than one routine,      */
   /*  return an error code that indicates which calling routine resulted in  */
   /*  the error. Since we pass a 0 for the last parameter, the only possible */
   /*  error is that the requested combination was not found among processed  */
   /*  combos.                                                                */
   /***************************************************************************/
  
   if (!ginfo.called_locate) {
      rV = ir_locate_ex (data_key, exten, vnum, 0);
      if (rV != ALL_OKAY)
        return (RESET_NOT_FOUND);
    }

   /***************************************************************************/
   /*  Close the header, data, and VIDF file currently opened and reset the   */
   /*  file descriptors to indicate that no file has been opened.             */
   /***************************************************************************/

   ex = ginfo.expt;
   if (ex->fdh != -1)
     close (ex->fdh);
   if (ex->fdd != -1)
     close (ex->fdd);
   if (ex->fdi != -1)
     close (ex->fdi);
   ex->fdh = -1;
   ex->fdd = -1;
   ex->fdi = -1;
   ir_free_vidf_structure (ex);

   /***************************************************************************/
   /*  Reset the following variables to indicate a fresh start to the system. */
   /***************************************************************************/

   ex->tdw_len = -1;
   ex->info_ptr->h_size = -1;
   ex->fnext = 0;

   /*******************************************************************/
   /*  Free the tables for each sensor table combination processed.   */
   /*******************************************************************/

   for (sen = 0; sen < ex->num_combo; ++sen) {
      sptr = ex->sen_tbl_ptr + sen;

      /************************************************************/
      /*  Free information concerning the tables.                 */
      /************************************************************/

      if (sptr->base_misc_tbls != NO_MEMORY) {
         free (sptr->base_misc_tbls);
         sptr->base_misc_tbls = NO_MEMORY;
       }

      /************************************************************/
      /*  Free the table values.                                  */
      /************************************************************/

      if (sptr->base_tbls != NO_MEMORY) {
         free (sptr->base_tbls);
         sptr->base_tbls = NO_MEMORY;
       }

      /************************************************************/
      /*  Free the structures that holds the CRIT_ACTION values.  */
      /************************************************************/
   
      if (sptr->base_crit_action != NO_MEMORY) {
         for (i = 0; i < ex->crit_action; ++i) {
            crit_act_ptr = sptr->act_ptr + i;
            if (crit_act_ptr->base_action != NO_MEMORY) {
               free (crit_act_ptr->base_action);
               crit_act_ptr->base_action = NO_MEMORY;
               crit_act_ptr->crit_action = NO_MEMORY;
             }
          }

         free (sptr->base_crit_action);
         sptr->base_crit_action = NO_MEMORY;
         sptr->act_ptr = NO_MEMORY;
       }
    }

   /***************************************************************/
   /*  Free the array of sensor_tables structures.                */
   /***************************************************************/

   if (ex->bmem.base_sen_ptr != NO_MEMORY) {
      free (ex->bmem.base_sen_ptr);
      ex->bmem.base_sen_ptr = NO_MEMORY;
    }
   
   /*******************************************************************/
   /*  Free the tables for each mode processed.                       */
   /*******************************************************************/

   for (i = 0; i < ex->num_modes; ++i) {
      mode_ptr = ex->mode_tbl_ptr + i;

      /************************************************************/
      /*  Free information concerning the tables.                 */
      /************************************************************/

      if (mode_ptr->base_misc_tbls != NO_MEMORY) {
         free (mode_ptr->base_misc_tbls);
         mode_ptr->base_misc_tbls = NO_MEMORY;
       }

      /************************************************************/
      /*  Free the table values.                                  */
      /************************************************************/

      if (mode_ptr->base_tbls != NO_MEMORY) {
         free (mode_ptr->base_tbls);
         mode_ptr->base_tbls = NO_MEMORY;
       }
    }

   /***************************************************************/
   /*  Free the array of mode_tables structures.                  */
   /***************************************************************/

   if (ex->bmem.base_mode_ptr != NO_MEMORY) {
      free (ex->bmem.base_mode_ptr);
      ex->bmem.base_mode_ptr = NO_MEMORY;
    }
   
   /****************************************************************/
   /*  Free the indexes for sensor table combinations.             */
   /****************************************************************/

   if (ex->bmem.base_tbl_index != NO_MEMORY) {
      free (ex->bmem.base_tbl_index);
      ex->bmem.base_tbl_index = NO_MEMORY;
    }
   ex->num_combo = 0;

   /************************************************************************/
   /*  Open the new set of data files for the combination being processed  */
   /*  for the master ex structure only, not the mode ex structure since   */
   /*  file_open is recursive and will open the mode ex files also.        */
   /************************************************************************/

   inside_reset_exp = 1;
   rV = file_open (data_key, exten, vnum, btime_yr, btime_day, btime_sec,
                   btime_nsec, etime_yr, etime_day, etime_sec, etime_nsec, 0);
   inside_reset_exp = 0;
   if (rV != ALL_OKAY)
     return (rV);

  return (ALL_OKAY);
}
