#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "util_str.h"
#include "idf_defs.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                          IR_READ_IDF_ONCE SUBROUTINE                       *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to retrieve information from the VIDF file that  *
 *  will be utilized by various subroutines.  The pertinent information is    *
 *  stored in the appropriate experiment_info structure.                      *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    None                                                                    *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_read_idf_once ()                                                 *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sizeof ()                the size of the specified object in bytes      *
 *    read_idf()               reads information from the IDF file            *
 *    malloc()                 allocates memory                               *
 *    free ()                  frees previously allocated memory              *
 *    ir_once_rval ()          returns an error code unique to this module    *
 *                             for each possible READ_IDF() error code        *
 *    ir_assign_pointers()     assigns the pointers for the header, data,     *
 *                             and calibration data elements                  *
 *    ir_get_constants()       retrieves all constants from the VIDF file     *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info      structure that holds information concerning    *
 *        ginfo                the experiment that is being processed         *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info   a pointer to the structure that holds specific *
 *          *ex                experiment information                         *
 *    struct ptr_rec *ptr      a pointer to the structure which holds all     *
 *                             pointers to the header and data for the exper- *
 *                             iment of interest                              *
 *    register short i         looping variable                               *
 *    unsigned int bytes       the number of bytes to allocate                *
 *    long rV                  holds the value returned by the called routine *
 *    long offset              offset to index into allocated memory          *
 *    char num_sen_tbls        no. of tables associated with sensor-specific  *
 *                             data (excluding ASCII tables)                  *
 *    char last_sen_tbl        the last sensor-specific table defined         *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *  MODS                                                                      * 
 *    4/25/97  ChrisG - reduced number of mallocs                             * 
 ******************************************************************************/

ByTe_2 ir_read_idf_once (void)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   struct ptr_rec *ptr;
   register ByTe_2 i;
   register ByTe_1 *c1, *c2, *cEnD;
   size_t bytes;
   ByTe_4 rV, offset;
   ByTe_4 Cs, Ss, Ls;
   ByTe_2 rval;
   ByTe_2 CwdLen = -1;
   ByTe_1 num_sen_tbls, last_sen_tbl;
   ByTe_1 TmP;

  /**************************************************************************/
  /*  Retrieve pertinent information from the VIDF file that needs to be    */
  /*  stored in the experiment_info structure.                              */
  /**************************************************************************/

   ex = ginfo.expt;
   ptr = ex->info_ptr;
   Cs = sizeof (ByTe_1);
   Ss = sizeof (ByTe_2);
   Ls = sizeof (ByTe_4);

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->ds_year, _DsYeaR, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->ds_day, _DsDaY, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->ds_msec, _DsMseC, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->de_year, _DeYeaR, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->de_day, _DeDaY, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->de_msec, _DeMseC, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->max_nss, _MaxNsS, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->smp_id, _SmpID, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->cal_sets, _CalSetS, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->da_method, _DaMethoD, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->num_tbls, _NumTblS, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                      (ByTe_1 *) &ex->num_modes, _StatuS, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->pa_def, _PaDefineD, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->num_sensor, _SeN, 0, 0L, 1);
   if (rV < 0 || ex->num_sensor <= 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->swp_len, _SwpLeN, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ptr->d_size, _DataLeN, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

  /************************************************************************/
  /* Make one malloc with base memory at ex->bmem.base_data.  All other   */
  /* variables which need to be malloced here will delta off this.  Note: */
  /* you must have a non-zero data record size and number of sensors.     */
  /* This is checked when acquired.                                       */
  /*                                                                      */
  /* Steps:                                                               */
  /*    (1)  Determine the number of bytes to malloc                      */
  /*    (2)  Malloc the space                                             */
  /*    (3)  Assign Pointers                                              */
  /************************************************************************/
 
  /************************************************************************/
  /* This accounts for the size of the data record                        */
  /************************************************************************/

   bytes = 4 * (ptr->d_size / 4 + 1);

  /************************************************************************/
  /* This accounts for the three 1 byte VIDF fields to be read which      */
  /* have one element per defined sensor: the sensor type, status         */
  /* and word length in bits.                                             */
  /************************************************************************/

   bytes += 3 * ex->num_sensor * Cs;

  /************************************************************************/
  /*  And this is the space needed by the time offsets                    */
  /************************************************************************/

   bytes += ex->num_sensor * Ls;

  /************************************************************************/
  /* If there are ancillary definitions present then add room for the:    */
  /*   1. The usage definitions  (2-bytes)                                */
  /*   2. The number of elements (2-bytes). Computed not read in          */
  /*   3. The word lengths (1-byte)                                       */
  /*   4. The word location in the data record (1-byte)                   */
  /************************************************************************/

   if (ex->cal_sets > 0)
      bytes += ex->cal_sets * (Ss  + sizeof(u_ByTe_2) + 2 * Cs);

  /************************************************************************/
  /* If there are table definitions present then add room for the:        */
  /*   1. The table type definitions  (1-byte)                            */
  /*   2. The table variable definitions  (1-byte)                        */
  /*   3. The table expand flags (1-byte)                                 */
  /************************************************************************/

   if (ex->num_tbls > 0)
      bytes += 3 * ex->num_tbls * Cs;

  /************************************************************************/
  /* Malloc the memory and initialize the relevent pointers               */
  /************************************************************************/

   if (ex->bmem.base_data != NO_MEMORY) {
      free (ex->bmem.base_data);
      ex->bmem.base_data = NO_MEMORY;
      ex->DATA_MEM = NO_MEMORY;
      ex->CAL_USE = NO_MEMORY;
      ex->CSET_NUM = NO_MEMORY;
      ex->cal_target = NO_MEMORY;
      ex->cal_wlen = NO_MEMORY;
      ex->tbl_type = NO_MEMORY;
      ex->tbl_var = NO_MEMORY;
      ex->tbl_expand = NO_MEMORY;
      ex->d_type = NO_MEMORY;
      ex->sen_tdw_len = NO_MEMORY;
      ex->sen_status = NO_MEMORY;
   }

   if ((ex->bmem.base_data = malloc (bytes)) == NO_MEMORY)
     return (ONCE_DATA_MALLOC);

  /************************************************************************/
  /* Define the points which all point to areas in the malloced memory.   */
  /************************************************************************/

   ex->DATA_MEM = (ByTe_1 *)ex->bmem.base_data;
   bytes = 4 * (ptr->d_size / 4 + 1);
   ex->time_off = (ByTe_4 *)(ex->DATA_MEM + bytes);
   bytes += ex->num_sensor * Ls;

   if (ex->cal_sets > 0) {
      ex->CAL_USE = (ByTe_2*)(ex->DATA_MEM + bytes);
      offset = bytes + (ex->cal_sets * Ss);
      ex->CSET_NUM = (u_ByTe_2 *) (ex->DATA_MEM + offset);
      offset += ex->cal_sets * sizeof (u_ByTe_2);
      ex->cal_target = ex->DATA_MEM + offset;
      offset += ex->cal_sets * Cs;
      ex->cal_wlen =  ex->DATA_MEM + offset;
      bytes = offset + ex->cal_sets * Cs;
   }

   if (ex->num_tbls > 0) {
      ex->tbl_type = ex->DATA_MEM + bytes;
      offset = bytes + ex->num_tbls * Cs;
      ex->tbl_var =  ex->DATA_MEM + offset;
      offset += ex->num_tbls * Cs;
      ex->tbl_expand = ex->DATA_MEM + offset;
      bytes = offset + ex->num_tbls * Cs;
   }

   ex->d_type = ex->DATA_MEM + bytes;
   offset = bytes + ex->num_sensor * Cs;
   ex->sen_tdw_len =  ex->DATA_MEM + offset;
   offset += ex->num_sensor * Cs;
   ex->sen_status = ex->DATA_MEM + offset;
   
  /**************************************************************************/
  /*  If tables present get all the info for each defined table             */
  /**************************************************************************/
 
   if (ex->num_tbls != 0) {
      for (i = 0; i < ex->num_tbls; ++i) {
         rV = read_idf (ex->data_key, ex->exten, ex->version,
                           (ByTe_1 *) (ex->tbl_type + i), _TblTypE, i, 0L, 1);
         if (rV < 0) { return (ir_once_rval (rV)); }

         rV = read_idf (ex->data_key, ex->exten, ex->version,
                             (ByTe_1 *) (ex->tbl_var + i), _TblVaR, i, 0L, 1);
         if (rV < 0) { return (ir_once_rval (rV)); }

         rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) (ex->tbl_expand + i), _TblExpanD, i, 0L, 1);
         if (rV < 0) { return (ir_once_rval (rV)); }
      }

  /************************************************************************/
  /*  Make sure that all ascii and mode-dependent tables appear AFTER all */
  /*  other tables (use this scheme for efficiency reasons).              */
  /************************************************************************/

      num_sen_tbls = 0;
      last_sen_tbl = -1;
      for (i = 0; i < ex->num_tbls; ++i) {
         if (*(ex->tbl_type + i) == 1) { continue; }
         if (*(ex->tbl_var + i) == 4 || *(ex->tbl_var + i) == 5) { continue; }
         ++num_sen_tbls;
         last_sen_tbl = (ByTe_1) i;
      }

      if (num_sen_tbls != 0) {
         for (i = 0; i < ex->num_tbls; ++i) {
            if (*(ex->tbl_type + i) == 1 && i < last_sen_tbl)
              return (ASCII_AFTER_SENSOR);
            if ((*(ex->tbl_var + i) == 4 || *(ex->tbl_var + i) == 5) &&
                i < last_sen_tbl)
              return (ASCII_AFTER_SENSOR);
         }
      }
   }

  /**************************************************************************/
  /*  If needed get all the calibration target and cal the word lengths (in */
  /*  bits) for each cal set.  Check target for TYPE SwP and set all to SeN */
  /**************************************************************************/
   
   if (ex->cal_sets > 0) {
      rV = read_idf (ex->data_key, ex->exten, ex->version,
                          (ByTe_1 *) ex->cal_target, _CalTargeT, 0, 0L, -1);
      if (rV < 0) { return (ir_once_rval (rV)); }

      c1 = ex->cal_target;
      cEnD = c1 + ex->cal_sets;
      for ( ; c1 < cEnD; ++c1) { if (*c1 == 1) { *c1 = 0; } }

      rV = read_idf (ex->data_key, ex->exten, ex->version,
                          (ByTe_1 *) ex->cal_wlen, _CalWleN, 0, 0L, -1);
      if (rV < 0) { return (ir_once_rval (rV)); }

      rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) ex->CAL_USE, _CalUsE, 0, 0L, ex->cal_sets);
      if (rV < 0) { return (ir_once_rval (rV)); }
   }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->fill_flag, _FillFlaG, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   if (ex->fill_flag == 1) {
      rV = read_idf (ex->data_key, ex->exten, ex->version,
                          (ByTe_1 *) &ex->fill_value, _FilL, 0, 0L, 1);
      if (rV < 0) { return (ir_once_rval (rV)); }
   }

  /**************************************************************************/
  /*  The flags that are being set depend on the sen_mode value and need to */
  /*  be set only once for use in determining the time of the data sample.  */
  /**************************************************************************/
  
   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->sen_mode, _SenModE, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   ex->TRC = 1 - ex->sen_mode / 4;
   ex->TRP = 1 - ((ex->sen_mode % 4) / 2);
   ex->TCP = 1 - (ex->sen_mode % 2);

  /**************************************************************************/
  /*  Read in the time offsets for each sensor.                             */
  /**************************************************************************/

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                                 (ByTe_1 *)ex->time_off, _TimeOfF, 0, 0L, -1);
   if (rV < 0) { return (ir_once_rval (rV)); }

  /**************************************************************************/
  /*  Read in the sensor status flags for each sensor.                      */
  /**************************************************************************/

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) ex->sen_status, _SenStatuS, 0, 0L, -1);
   if (rV < 0) { return (ir_once_rval (rV)); }

  /**************************************************************************/
  /*  Read in data format for each sensor.                                  */
  /**************************************************************************/

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) ex->d_type, _DataTypE, 0, 0L, -1);
   if (rV < 0) { return (ir_once_rval (rV)); }

  /**************************************************************************/
  /*  Read in the word length for all sensors and find the largest word     */
  /*  size which was used to pack the data in the data array.               */
  /**************************************************************************/

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) ex->sen_tdw_len, _TdwLeN, 0, 0L, -1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   ex->tdw_len = 0;
   for (i = 0; i < ex->num_sensor; ++i)
     if (*(ex->sen_tdw_len + i) > ex->tdw_len)
       ex->tdw_len = *(ex->sen_tdw_len + i); 
   
   if (ex->cal_sets > 0) {
      c1 = ex->cal_wlen;
      c2 = ex->cal_target;
      cEnD = c1 + ex->cal_sets;
      for ( ; c1 < cEnD; ++c1) {
         TmP = (*c1 < 0) ? -*c1 : *c1;
         if (TmP > CwdLen) { CwdLen = TmP; }
         switch (*c2++) {
             case 0:
               if (TmP > ex->AncSBits) { ex->AncSBits = TmP; }
             break;
             case 2:
               if (TmP > ex->AncGBytes) { ex->AncGBytes = TmP; }
             break;
             case 3:
               if (TmP > ex->AncPBytes) { ex->AncPBytes = TmP; }
             break;
         }
      }
   }

   if (ex->VIDFver < 0.0)  {
      if (ex->tdw_len < CwdLen) { ex->tdw_len = CwdLen; }
      ex->AncPBytes = ex->tdw_len;
      ex->AncGBytes = ex->tdw_len;
      ex->AncSBits = ex->tdw_len;
   } else {
      if (fabs(ex->VIDFver) < 2.199) {
         if (ex->tdw_len < ex->AncSBits) { ex->tdw_len = ex->AncSBits; }
         ex->AncSBits = ex->tdw_len;
         if (ex->smp_id == 3) { return (ILLEGAL_SMP_ID); }
      }
   }

   ex->AncPBytes = (ex->AncPBytes < 9) ? Cs : (ex->AncPBytes < 17) ? Ss : Ls;
   ex->AncGBytes = (ex->AncGBytes < 9) ? Cs : (ex->AncGBytes < 17) ? Ss : Ls;
   ex->AncSBytes = (ex->AncSBits < 9) ? Cs : (ex->AncSBits < 17) ? Ss : Ls;
   if (ex->AncSBits < 8)
       ex->AncSBits = (ex->AncSBits >= 5) ? 8 : 
                                  (ex->AncSBits == 3) ? 4 : ex->AncSBits;
   else
       ex->AncSBits = (ex->AncSBits < 9) ? 8 : (ex->AncSBits < 17) ? 16 : 32;

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &ex->num_consts, _NumConstS, 0, 0L, 1);
   if (rV < 0) { return (ir_once_rval (rV)); }

   if (ex->num_consts > 0) {
      rval = ir_get_constants ();
      if (rval != ALL_OKAY) { return (rval); }
   }

  /**************************************************************************/
  /*  The last zero represents a null pointer which is valid to set since   */
  /*  that pointer is past on to a module ONLY when setting header pointers */
  /*  which is NOT indicated by this calling sequence (only data ptrs).     */
  /**************************************************************************/

   rval = ir_assign_pointers (0, 1, 0);
   if (rval != ALL_OKAY) { return (rval); }

   return (ALL_OKAY);
}

/******************************************************************************
 *                                                                            *
 *                             IR_ONCE_RVAL SUBROUTINE                        *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine returns an error code unique to the module READ_IDF_ONCE() *
 *  based upon the error code returned by the call to READ_IDF().             *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    long rV                    the error code returned by READ_IDF()        *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_once_rval (rV)                                                   *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    None                                                                    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    None                                                                    *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    None                                                                    *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_once_rval (ByTe_4 rV)
{
   switch (rV) {
       case IDF_NOT_FOUND:  return (ONCE_IDF_NOT_FOUND);
       case IDF_MANY_BYTES: return (ONCE_IDF_MANY_BYTES);
       case IDF_TBL_NUM:    return (ONCE_IDF_TBL_NUM);
       case IDF_CON_NUM:    return (ONCE_IDF_CON_NUM);
       case IDF_NO_ENTRY:   return (ONCE_IDF_NO_ENTRY);
       default:             return ((ByTe_2) rV);
    }
}
