#include <stdlib.h>
#include "ret_codes.h"
#include "libbase_udf.h"
#include "OpSySInD.h"

/****************************************************************************
 *                                                                          *
 *                   IR_READ_DREC_TIME_ADVANCE SUBROUTINE                   *
 *                                                                          *
 *  DESCRIPTION                                                             *
 *    This routine is called to advance the file pointers to the next time  *
 *  sample for the data set being processed.                                *
 *                                                                          *
 *  INPUT VARIABLES                                                         *
 *    void *UDF                  memory location ptr for the structure that *
 *                               holds returned data values (read_drec)     *
 *    char full_swp              request 1 or all values in sensor set be   *
 *                               returned (for scalar parameters only)      *
 *    char found_sensor          flag indicating if the requested sensor is *
 *                               in the sensors returned and is associated  *
 *                               with the time period being processed       *
 *                                                                          *
 *  USAGE                                                                   *
 *    x = ir_read_drec_time_advance (UDF, full_swp, found_sensor)           *
 *                                                                          *
 *  NECESSARY SUBPROGRAMS                                                   *
 *    abs()                      returns the absolute value of a number     *
 *    ir_advance_time_pointers() adjusts two time pointers, time_row and    *
 *                               time_col, so that the NEXT time sample is  *
 *                               pointed to                                 *
 *    ir_reset_sen_set_time()    adjusts cumulative sensor set time values  *
 *    ir_new_data_record ()      retrieves the next data record             *
 *    ir_new_header_record ()    retrieves the next header record           *
 *                                                                          *
 *  EXTERNAL VARIABLES                                                      *
 *    struct general_info        structure holding information concerning   *
 *        ginfo                  the experiment that is being processed     *
 *                                                                          *
 *  INTERNAL VARIABLES                                                      *
 *    struct experiment_info     a pointer to the structure that holds      *
 *          *ex                  specific experiment information            *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all *
 *                               pointers to the header and data for the    *
 *                               experiment of interest                     *
 *    long num_nss               the number of sensor sets being processed  *
 *    long old_accum_ss_ms,      last values computed for the accumulation  *
 *         old_accum_ss_ns       over all sensor sets processed             *
 *    short rval                 holds called routine status flags          *
 *                                                                          *
 *  SUBSYSTEM                                                               *
 *    Display Level                                                         *
 *                                                                          *
 ***************************************************************************/

ByTe_2 ir_read_drec_time_advance (void *UDF,  
                                  ByTe_1 full_swp, ByTe_1 found_sensor)
{
  extern struct general_info ginfo;

  struct experiment_info *ex;
  struct ptr_rec *ptr;
  ByTe_4 num_nss, old_accum_ss_ms, old_accum_ss_ns;
  ByTe_2 rval;

  /******************************************************************/
  /*  Advance the pointers to the next time sample.                 */
  /******************************************************************/

  ex = ginfo.expt;
  ptr = ex->info_ptr;
  ptr->chg_sen_set = 0;
  ir_advance_time_pointers (full_swp);

  if (ptr->chg_sen_set) {
     ir_reset_sen_set_time ();
     ex->DswaP = 1;
  }

  /********************************************************************/
  /*  If we have processed all sensor sets, read another data record. */
  /*  Save off the old sensor set accumulation times before resetting */
  /*  them to zero since the routine next_file_start_time() needs     */
  /*  these values to compute times for mode values.                  */
  /********************************************************************/

  num_nss = abs (*ptr->NSS); 
  if (ptr->cur_sen_set >= num_nss)
   {
     ptr->cur_sen_set = 0;
     old_accum_ss_ms = ex->accum_ss_ms;
     old_accum_ss_ns = ex->accum_ss_ns;
     ex->accum_ss_ms = 0;
     ex->accum_ss_ns = 0;
     ex->accum_ss_sz = 0;
     if ((rval = ir_new_data_record (found_sensor)) != ALL_OKAY)
       return (rval);
   }
  else
   {
     old_accum_ss_ms = -1;
     old_accum_ss_ns = -1;
   }

  /********************************************************************/
  /*  Read the header associated with the data.                       */
  /********************************************************************/

  if ((rval = ir_new_header_record (found_sensor, UDF)) != ALL_OKAY)
   {
     /********************************************************************/
     /*  Put back the accumulation time for the sensor set in case the   */
     /*  end of file had been reached and next file is to be retrieved.  */
     /********************************************************************/

     if (old_accum_ss_ms != -1)
      {
        ex->accum_ss_ms = old_accum_ss_ms;
        ex->accum_ss_ns = old_accum_ss_ns;
      }

     return (rval);
   }

   return (ALL_OKAY);
}
