#include "util_str.h"
#include "libbase_udf.h"

/****************************************************************************
 *                                                                          *
 *                    IR_PITCH_ANGLE_SWEEP_TIMES SUBROUTINE                 *
 *                                                                          *
 *  DESCRIPTION                                                             *
 *    This routine is called in order to compute the time range for each    *
 *  element of the sweep for the parent data set which requests pitch angle *
 *  information.                                                            *
 *                                                                          *
 *  INPUT VARIABLES                                                         *
 *    struct experiment_info  a pointer to the structure that holds         *
 *          *ex               specific experiment information               *
 *    struct ptr_rec *ptr     a pointer to the structure which holds all    *
 *                            pointers to the header and data for the       *
 *                            experiment of interest                        *
 *    struct pitch_info       a pointer to the structure that holds pitch   *
 *        *pa_ptr             angle information                             *
 *    unsigned short max_ele  the number of elements in the sweep           *
 *    void *UDF               the UDF read info structure                   *
 *    char *use_end_yr_day    flag indicating if end year/day values are to *
 *                            be used                                       *
 *    char *same_times        flag indicating if the same time range is     *
 *                            applicable to all steps of the vector         *
 *                                                                          *
 *  USAGE                                                                   *
 *    ir_pitch_angle_sweep_times (ex, ptr, &pa_ptr, max_ele, UDF,           *
 *                                &use_end_yr_day, &same_times)             *
 *                                                                          *
 *  NECESSARY SUBPROGRAMS                                                   *
 *    ir_sample_time ()       returns the time associated with a single     *
 *                            data sample (row, column)                     *
 *                                                                          *
 *  EXTERNAL VARIABLES                                                      *
 *    None                                                                  *
 *                                                                          *
 *  INTERNAL VARIABLES                                                      *
 *    reg long *btime_ptr,    pointers to the start time / end time memory  *
 *       *etime_ptr           locations                                     *
 *    reg long *time_end      loop termination variable                     *
 *    register float *s1      fast short pointer looper                     *
 *    float time_accum        acquisition time of a single measurement      *
 *    long btime_ms           start time of the measurement in milliseconds *
 *    long etime_ms           end time of the measurement in milliseconds   *
 *    long dummy_ns           dummy placeholder for nanoseconds element     *
 *    long base_time          time correction constant                      *
 *    unsigned short time_row the matrix row being processed                *
 *    short base_swp_off      sweep step offset for first element in column *
 *    char decrement_bday     flag indicating if start day needs to be      *
 *                            modified (if btime ends up negative)          *
 *    char decrement_eday     flag indicating if end day needs to be        *
 *                            modified (if etime ends up negative)          *
 *                                                                          *
 *  SUBSYSTEM                                                               *
 *    Display Level                                                         *
 *                                                                          *
 ***************************************************************************/

void ir_pitch_angle_sweep_times (struct experiment_info *ex, 
                                 struct ptr_rec *ptr, 
                                 struct pitch_info *pa_ptr, u_ByTe_2 max_ele, 
                                 void *UDF, ByTe_1 *use_end_yr_day, 
                                 ByTe_1 *same_times)
{
  register ByTe_4 *btime_ptr, *etime_ptr, *time_end;
  register ByTe_2 *s1;
  ReaL_4 time_accum;
  ByTe_4 btime_ms, etime_ms, dummy_ns, base_time;
  u_ByTe_2 time_row;
  ByTe_2 base_swp_off;
  ByTe_1 decrement_bday, decrement_eday;

  /************************************************************************/
  /*  We use time_row since we need to pick up the rest of the elements   */
  /*  in that column being processed (for both SEN_MODE = 0 and 4) and    */
  /*  time_row tells us how many have already been processed for that     */
  /*  sensor.  When computing the angles, take into account fractions of  */
  /*  milliseconds (nanoseconds components).                              */
  /************************************************************************/

  time_row = (u_ByTe_2) ptr->time_row;
  time_accum = ex->accum_ms + ex->accum_ns * 1.0e-6 + 
               ex->lat_ms + ex->lat_ns * 1.0e-6;

  /*******************************************************************/
  /* If time advances down the column and the column timing is       */
  /* sequential, use DA_METHOD to determine start time of sample.    */
  /* In all cases, only the millisecond time component is of used.   */
  /*******************************************************************/
  
  base_time = ex->btime_ms % 86400000;
  btime_ptr = pa_ptr->btime_ms;
  time_end = pa_ptr->btime_ms + max_ele;
  if (ex->sen_mode == 0 || ex->sen_mode == 2)
   {
     switch (ex->da_method)
      {
        case 0:
          btime_ms = (ByTe_4) (time_row * time_accum);
          for (; btime_ptr < time_end;)
           {
             *btime_ptr++ = btime_ms + base_time;
             btime_ms += (ByTe_4) time_accum;
           }
          break;
  
        case 1:
          s1 = ptr->SAMP_INDEX + time_row;
          for (; btime_ptr < time_end; ++s1)
           {
             btime_ms = (ByTe_4) (*s1 * time_accum);
             *btime_ptr++ = btime_ms + base_time;
           }
          break;
 
        case 2:
        case 3:
          s1 = ptr->SAMP_INDEX + time_row;
          base_swp_off = *ptr->SAMP_INDEX;
          if (*ptr->SAMP_INDEX > *(ptr->SAMP_INDEX + 1))
           {
             for (; btime_ptr < time_end; ++s1)
              {
                btime_ms = (ByTe_4) ((base_swp_off - *s1) * time_accum);
                *btime_ptr++ = btime_ms + base_time;
              }
           }
          else
           {
             for (; btime_ptr < time_end; ++s1)
              {
                btime_ms = (ByTe_4) ((*s1 - base_swp_off) * time_accum);
                *btime_ptr++ = btime_ms + base_time;
              }
           }
          break;
      }

     /*****************************************************************/
     /*  The end time of the sample is equal to the start time of the */
     /*  next sample, except for the last sample.                     */
     /*****************************************************************/

     etime_ptr = pa_ptr->etime_ms;
     btime_ptr = pa_ptr->btime_ms + 1;
     time_end = pa_ptr->btime_ms + max_ele;
     for (; btime_ptr < time_end; ++btime_ptr, ++etime_ptr)
       *etime_ptr = *btime_ptr;
     *etime_ptr = ex->etime_ms % 86400000;
     *use_end_yr_day = 0;
   }
  else
   {
     /****************************************************************/
     /*  If time advances down the column and the column timing is   */
     /*  parallel, use start time of first element being returned    */
     /****************************************************************/

     etime_ptr = pa_ptr->etime_ms;
     *use_end_yr_day = 1;
     if (ex->sen_mode == 1 || ex->sen_mode == 3)
      {
        for (; btime_ptr < time_end;)
         {
           *btime_ptr++ = base_time;
           *etime_ptr++ = ex->etime_ms % 86400000;
         }
      }
     else
      {
        /****************************************************************/
        /*  Determine the time for the sample being processed.  Day     */
        /*  value is not used so no need to check decrement_day flag.   */
        /****************************************************************/

        for (; btime_ptr < time_end;)
         {
            ir_sample_time (max_ele, UDF, time_row, ptr->time_col, 
                            &btime_ms, &dummy_ns, &etime_ms, &dummy_ns,
                            &decrement_bday, &decrement_eday);
            *btime_ptr++ = btime_ms;
            *etime_ptr++ = etime_ms;
            ++time_row;
         }
      }
   }

  /************************************************************************/
  /*  Send back a flag to indicate if the times of each step are the same.*/
  /************************************************************************/

  *same_times = 1;
  btime_ptr = pa_ptr->btime_ms;
  time_end = pa_ptr->btime_ms + max_ele;

  btime_ms = *btime_ptr++;
  for (; btime_ptr < time_end; ++btime_ptr)
   {
     if (*btime_ptr != btime_ms)
      {
        *same_times = 0;
        break;
      }
   }
}
