#include "util_str.h"
#include "ret_codes.h"
#include "DbDefs.h"
#include "gen_defs.h"
#include "user_defs.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                     PITCH_ANGLE_SOURCE_STATUS SUBROUTINE                    *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine should be called to determine the state of the IDFS data    *
 *  set to be used for pitch angle calculations for the data_key specified.    *
 *  This routine was written since it was decided that FILE_OPEN and RESET_    *
 *  EXPERIMENT_INFO would return ALL_OKAY if there was an error pertinent to   *
 *  the processing of any pitch angle source so that the main data set could   *
 *  continue to be plotted (in many cases, plotting the main data set, not     *
 *  the pitch angles for the main data set).  If it is determined that pitch   *
 *  angle data is needed, the status of the pitch angle IDFS source can be     *
 *  determined AFTER FILE_OPEN and RESET_EXPERIMENT_INFO are called.           *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    unsigned long data_key     key which uniquely identifies the data set    *
 *                               being processed                               *
 *    char *exten                filename extension for the data to be used    *
 *    unsigned short vnum        version number to be associated with this     *
 *                               combination (allows for multiple opens)       *
 *    short *pa_status           the status of the pitch angle IDFS source     *
 *                               defined for the specified data set            *
 *    unsigned long *pa_data_key key which uniquely identifies the data set    *
 *                               to be used for the pitch angle calculations   *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = pitch_angle_source_status (data_key, exten, vnum, &pa_status,        *
 *                                   &pa_data_key)                             *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    ir_locate_ex()             determines if requested combination has       *
 *                               already been pand points to the               *
 *                               correct structure allocated for the combo     *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info        structure holding information concerning      *
 *        ginfo                  the experiment that is being processed        *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info     a pointer to the structure that holds         *
 *          *ex                  specific experiment information               *
 *    struct pitch_info          a pointer to the structure that holds pitch   *
 *        *pa_ptr                angle information                             *
 *    short ret_val              holds called routine status flags             *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 pitch_angle_source_status (u_ByTe_4 data_key, ByTe_1 *exten, 
                                  u_ByTe_2 vnum, ByTe_2 *pa_status, 
                                  u_ByTe_4 *pa_data_key)
{
  extern struct general_info ginfo;

  struct experiment_info *ex;
  struct pitch_info *pa_ptr;
  ByTe_2 ret_val;

  /**********************************************************************/
  /*  Get the ex structure for the specified data set.                  */
  /**********************************************************************/

  *pa_data_key = 0;
  ret_val = ir_locate_ex (data_key, exten, vnum, 0);
  if (ret_val != ALL_OKAY)
    return (PA_MAIN_DATA_MISSING);

  /**********************************************************************/
  /*  Is pitch angle defined for this data set?                         */
  /**********************************************************************/

  ex  = ginfo.expt;
  if (!ex->pa_def)
    *pa_status = PA_NOT_APPLICABLE;
  else
   {
     /**********************************************************************/
     /*  Check for non-null memory location in main data source in case    */
     /*  pitch angle source also has pa_def set in the VIDF file.          */
     /**********************************************************************/

     if (ex->bmem.base_pitch_info != NO_MEMORY)
      {
        pa_ptr = ex->pitch_angles;
        *pa_data_key = pa_ptr->data_key;

        /*******************************************************************/
        /*  Data files are opened and ready to go.                         */
        /*******************************************************************/
    
        if (pa_ptr->file_status == ALL_OKAY)
          *pa_status = PA_READY;

        /*******************************************************************/
        /*  Database errors for designed pitch angle IDFS source.          */
        /*******************************************************************/
    
        else if (pa_ptr->file_status <= DB_START_neg && 
                 pa_ptr->file_status >= DB_STOP_neg)
          *pa_status = PA_DB_ERROR;

        /*******************************************************************/
        /*  Data not online for designed pitch angle IDFS source.          */
        /*******************************************************************/
    
        else if (pa_ptr->file_status >= DB_START_plus)
          *pa_status = PA_DATA_MISSING;

        /*******************************************************************/
        /*  IDFS routine returned error after datafiles were opened.       */
        /*******************************************************************/
    
        else
          *pa_status = PA_IR_ERROR;
      }

     /*********************************************************************/
     /*  No pitch angle structure so no pitch angle IDFS data set.        */
     /*********************************************************************/

     else
       *pa_status = PA_NOT_APPLICABLE;
   }

  return (ALL_OKAY);
}
