#include <math.h>
#include "util_str.h"
#include "user_defs.h"
#include "libbase_udf.h"

/****************************************************************************
 *                                                                          *
 *                           IR_IDF_EX_FUNC SUBROUTINE                      *
 *                                                                          *
 *  DESCRIPTION                                                             *
 *    This routine applies any higher order mathematical operations defined *
 *  for the tables contained in the VIDF file.  These operations are always *
 *  applied after the base operation has been made and always replace the   *
 *  target data.                                                            *
 *                                                                          *
 *  INPUT VARIABLES                                                         *
 *    short oper_type            the higher order operation to be performed *
 *    float *DataOut             pointer to the array that holds the values *
 *                               returned by the module                     *
 *    long num_data              the number of elements being returned      *
 *    void *idf_data_ptr         ptr to memory location for the structure   *
 *                               that holds returned data values (read_drec)*
 *                                                                          *
 *  USAGE                                                                   *
 *    ir_idf_ex_func (oper_type, &DataOut, num_data, idf_data_ptr)          *
 *                                                                          *
 *  NECESSARY SUBPROGRAMS                                                   *
 *    exp()                      returns double-precision exponential       *
 *                               function of the argument                   *
 *    log()                      returns double-precision logarithm base e  *
 *                               of the argument                            *
 *    pow()                      returns double-precision power function of *
 *                               the arguments                              *
 *    log10()                    returns double-precision base 10 logarithm *
 *                               of the argument                            *
 *    sqrt()                     returns double-precision square root of    *
 *                               the argument                               *
 *    cos()                      returns double-precision cosine function   *
 *                               of the argument                            *
 *    sin()                      returns double-precision sine function of  *
 *                               the argument                               *
 *    tan()                      returns double-precision tangent function  *
 *                               of the argument                            *
 *    acos()                     returns double-precision arccos function   *
 *                               of the argument                            *
 *    asin()                     returns double-precision arcsin function   *
 *                               of the argument                            *
 *    atan()                     returns double-precision arctan function   *
 *                               of the argument                            *
 *                                                                          *
 *  EXTERNAL VARIABLES                                                      *
 *    struct general_info        structure that holds information concerning*
 *         ginfo                 the experiment that is being processed     *
 *                                                                          *
 *  INTERNAL VARIABLES                                                      *
 *    struct idf_data *ExDa      structure that holds all of the currently  *
 *                               returned data values to be processed       *
 *    register float *d1         pointer to the data being modified         *
 *    register float *f1, *f2    pointer to the azimuthal angles            *
 *    register float *dEnD       loop termination variable                  *
 *    register float dT          accumulation time                          *
 *    float Tmp                  temporary variable                         *
 *                                                                          *
 *  SUBSYSTEM                                                               *
 *    Display Level                                                         *
 *                                                                          *
 ***************************************************************************/

void ir_idf_ex_func (ByTe_2 oper_type, ReaL_8 *DataOut, ByTe_4 num_data, 
                     void *idf_data_ptr)
{
   extern struct general_info ginfo;

   struct idf_data *ExDa;
   register ReaL_8 *d1, *dEnD, dT;
   register ReaL_4 *f1, *f2;
   ReaL_8 int_part, frac_part;      
   ReaL_4 Tmp;
   ByTe_4 V, S;

#ifdef WIN32
   ReaL_8 C[6];
   ByTe_4 I;
   ReaL_8 Y, X;
   ReaL_8 TmP, SeR, AnS;
   ReaL_8 Ap, DeL, SuM;
   ReaL_8 Ba, Ca, Da, Ea;
#endif

   /********************************************************************/
   /*  Initialize variables.                                           */
   /********************************************************************/

   ExDa = (struct idf_data *) idf_data_ptr;
   d1 = DataOut;
   dEnD = d1 + num_data;

   switch (oper_type)
   {
      case 0:
      break;
      case 1:
         for ( ; d1 < dEnD; ++d1) { *d1 = exp (*d1); }
      break;
      case 2:
         for ( ; d1 < dEnD; ++d1)
             *d1 = (*d1 > 0.0) ? log (*d1) : OUTSIDE_MIN;
      break;
      case 3:
         for ( ; d1 < dEnD; ++d1) { *d1 = pow (10.0, *d1); }
      break;
      case 4:
         for ( ; d1 < dEnD; ++d1)
             *d1 = (*d1 > 0.0) ? log10 (*d1) : OUTSIDE_MIN;
      break;
      case 5:
         for ( ; d1 < dEnD; ++d1) { *d1 = pow (2.0, *d1); }
      break;
      case 6:
         for ( ; d1 < dEnD; ++d1)
             *d1 = (*d1 < 0.0) ? OUTSIDE_MIN : sqrt (*d1);
      break;
      case 7:
         for ( ; d1 < dEnD; ++d1) { *d1 = cos (TORAD * *d1); }
      break;
      case 8:
         for ( ; d1 < dEnD; ++d1) { *d1 = sin (TORAD * *d1); }
      break;
      case 9:
         for ( ; d1 < dEnD; ++d1) { *d1 = tan (TORAD * *d1); }
      break;
      case 10:
         for ( ; d1 < dEnD; ++d1) { *d1 = acos (*d1); }
      break;
      case 11:
         for ( ; d1 < dEnD; ++d1) { *d1 = asin (*d1); }
      break;
      case 12:
         for ( ; d1 < dEnD; ++d1) { *d1 = atan (*d1); }
      break;
      case 13:
         for ( ; d1 < dEnD; ++d1)
             *d1 = (*d1 != 0.0) ? 1.0 / *d1 : OUTSIDE_MIN;
      break;
      case 14:
         dT = ginfo.expt->accum_ms * 1.0e-3 + ginfo.expt->accum_ns * 1.0e-9;
         for ( ; d1 < dEnD; ++d1) { *d1 *= dT; }
      break;
      case 15:
         dT = ginfo.expt->accum_ms * 1.0e-3 + ginfo.expt->accum_ns * 1.0e-9;
         for ( ; d1 < dEnD; ++d1) { *d1 /= dT; }
      break;
      case 16:
         for ( ; d1 < dEnD; ++d1) { *d1 = -(*d1); }
      break;
      case 17:
         for ( ; d1 < dEnD; ++d1) { *d1 = *d1 * *d1; }
      break;
      case 18:
         f1 = ExDa->start_az;
         f2 = ExDa->stop_az;
         for ( ; d1 < dEnD; ++d1) {
             dT = (*f2++ + *f1) / 2.0;
             Tmp = dT - *f1++;
             if (Tmp > 60.0 || Tmp < -60.0)
                dT += 180.0;
             *d1 = *d1 - dT;
             if (*d1 < 0.0)
                *d1 += 360.0;
         }
      break;
      case 19:
         for ( ; d1 < dEnD; ++d1) { *d1 = (*d1 >= 0.0) ? *d1 : -*d1; }
      break;
      case 20:
         f1 = ExDa->start_az;
         for ( ; d1 < dEnD; ++d1) {
            *d1 += *f1++;
            if (*d1 < 0.0) {
               *d1 += 360.0;
            } else {
               if (*d1 >= 360.0)
                  *d1 -= 360.0;
            }
         }
      break;
      case 21:
         for ( ; d1 < dEnD; ++d1) { *d1 = (ByTe_4) *d1; }
      break;
      case 22:                                         /* 8 bit signed    */
         for ( ; d1 < dEnD; ) {
            V = (ByTe_4) *d1;
            S = (V >> 7) & 1;
            *d1++ = (S == 0) ? V : -256 + V; 
         }
      break;
      case 23:                                         /* 16 bit signed    */
         for ( ; d1 < dEnD; ) {
            V = (ByTe_4) *d1;
            S = (V >> 15) & 1;
            *d1++ = (S == 0) ? V : -65536 + V; 
         }
      break;
      case 24:                                         /* 12 bit signed    */
         for ( ; d1 < dEnD; ) {
            V = (ByTe_4) *d1;
            S = (V >> 11) & 1;
            *d1++ = (S == 0) ? V : -4096 + V; 
         }
      break;
      case 25:                                         /* 24 bit signed    */
         for ( ; d1 < dEnD; ) {
            V = (ByTe_4) *d1;
            S = (V >> 23) & 1;
            *d1++ = (S == 0) ? V : -16777216 + V; 
         }
      break;
      case 26:         
         for (; d1 < dEnD; ++d1) {            
            frac_part = modf (*d1, &int_part);
            *d1 = int_part;
         }
      break;     
      case 27:        
          for (; d1 < dEnD; ++d1) {
             frac_part = modf (*d1, &int_part);
             *d1 = frac_part;
          }
      break;
      case 28:
         for (; d1 < dEnD; ++d1) { *d1 = ceil (*d1); }
      break;
      case 29:
         for (; d1 < dEnD; ++d1) { *d1 = floor (*d1); }
      break;
      case 30:                                         /* error function   */
#ifndef WIN32
         for ( ; d1 < dEnD; ++d1) { *d1 = erf(*d1); }
#else
         C[0] =  76.18009172947146;
         C[1] = -86.50532032941677;
         C[2] =  24.01409824083091;
         C[3] =  -1.231739572450155;
         C[4] =   0.1208650973866179e-2;
         C[5] =  -0.5395239384953e-5;

         Y = 0.5;
         TmP = 6.0;
         TmP -= log(TmP);
         SeR = 1.000000000190015;
         for ( I = 0; I < 6; ++I ) { SeR += C[I]/++Y; }
         AnS = -TmP + log(2.5066282746310005 * SeR / 0.5);

         for ( ; d1 < dEnD; ++d1) { 
            if ( (*d1 > 0.0) ) { 
               X = *d1 * *d1;
               if ( X < 1.5 ) {
                  Ap = 0.5;
                  DeL = 2.0;
                  SuM = DeL;
                  for ( I = 0; I < 100; ++I ) { 
                     ++Ap;
                     DeL *= X / Ap;
	             SuM += DeL;
	             if ( fabs(DeL) < fabs(SuM)*3.0e-7 ) { break; }
                  }
                  *d1 = SuM * exp(-X + 0.5 * log(X) - AnS);
               } else {
                  Ba = X + 0.5;
                  Ca = 1.0e30;
                  Da = 1.0 / Ba;
                  Ea = Da;
                  for ( I = 0; I < 100; ++I ) { 
                     Ap = -I * (I - 0.5);
                     Ba += 2.0;
	             Da = Ap * Da + Ba;
	             if (fabs(Da) < 1.0e-30) { Da = 1.0e-30; }
	             Ca = Ba + Ap / Ca;
	             if (fabs(Ca) < 1.0e-30) { Ca = 1.0e-30; }
                     Da = 1.0 / Da;
	             DeL = Da * Ca;
	             Ea *= DeL;
	             if (fabs(DeL-1.0) < 3.0e-7) { break ; }
                  }
                  *d1 = 1.0 - Ea * exp(-X + 0.5 * log(X) - AnS);
                }
                if (*d1 < 0.0) { *d1 = -*d1; } 
             } else { *d1 = 0.0; }
          }
#endif
      break;
   }
}
