#include <stdlib.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "idf_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                           IR_GET_CRITICAL_INFO SUBROUTINE                  *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to retrieve the number of the lookup table that  *
 *  is to be reloaded and the array of critical actions for each sensor       *
 *  combination that utilizes a critical status byte.                         *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    short sen_combo         index into the array of sensor_table structures *
 *    short sen               the sensor being processed                      *
 *    char crit_tbl           the table flagged as the critical status table  *
 *    short act_index         index into the array of structures that hold    *
 *                            critical action information                     *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_get_critical_info (sen_combo, sen, crit_tbl, act_index)          *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sizeof ()               the size of the specified object in bytes       *
 *    free ()                 frees allocated memory                          *
 *    read_idf()              reads information from the IDF file             *
 *    malloc()                allocates memory                                *
 *    ir_get_crit_rval ()     returns an error code unique to this module     *
 *                            for each possible READ_IDF() error code         *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info     structure that holds information concerning     *
 *        ginfo               the experiment that is being processed          *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info  a pointer to the structure that holds specific  *
 *          *ex               experiment information                          *
 *    struct sensor_tables    a pointer to the structure which holds the      *
 *          *sptr             various information for the tables utilized by  *
 *                            sensor-table combination being processed        *
 *    struct crit_action_info a pointer to the structure that holds info.     *
 *       *crit_act_ptr        pertinent to table(s) that are flagged as       *
 *                            critical action table(s)                        *
 *    size_t B                the number of bytes to allocate                 *
 *    long rV                 holds the value returned by the called routine  *
 *    short offset            offset into the CRIT_ACTION array               *
 *    short num_states        the number of states defined for the critical   *
 *                            status byte                                     *
 *    char t_crst             the critical status byte for the table/sensor   *
 *                            in question                                     *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_get_critical_info (ByTe_2 sen_combo, ByTe_2 sen, ByTe_1 crit_tbl, 
                             ByTe_2 act_index)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   struct sensor_tables *sptr;
   struct crit_action_info *crit_act_ptr;
   size_t B;
   ByTe_4 rV;
   ByTe_2 offset, num_states;
   ByTe_1 t_crst;

  /**************************************************************************/
  /*  Point to the correct sensor_tables structure.                         */
  /**************************************************************************/

   ex = ginfo.expt;
   sptr = ex->sen_tbl_ptr + sen_combo;
   crit_act_ptr = sptr->act_ptr + act_index;

  /**************************************************************************/
  /*  Determine the number of states associated with this status.           */
  /*                                                                        */
  /*  Changed how this is done since now each status has its length defined */
  /*  and the critical length field gone. Get the critical status byte for  */
  /*  this table and sensor (which may exist somewhere already) then get    */
  /*  its length                                                            */
  /**************************************************************************/

   rV = read_idf (ex->data_key, ex->exten, ex->version, (ByTe_1 *) &t_crst,
                      _CritStatuS, crit_tbl, (ByTe_4) sen, 1);
   if (rV < 0) { return (ir_get_crit_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version, 
                     (ByTe_1 *) &num_states, _StateS, 0, (ByTe_4) t_crst, 1);
   if (rV < 0) { return (ir_get_crit_rval (rV)); }

  /**************************************************************************/
  /*  Malloc the space to hold the actions for this critical status.        */
  /**************************************************************************/

   if (crit_act_ptr->base_action != NO_MEMORY) {
      free (crit_act_ptr->base_action);
      crit_act_ptr->base_action = NO_MEMORY;
   }
   B = sizeof (ByTe_4) * num_states;
   if (B <= 0) { return (GET_ACTION_MALLOC); }

   if ((crit_act_ptr->base_action = malloc (B)) == NO_MEMORY)
      return (GET_ACTION_MALLOC);
   crit_act_ptr->crit_action = (ByTe_4 *) crit_act_ptr->base_action;

  /**************************************************************************/
  /*  Retrieve the critical offset and critical action table values.        */
  /**************************************************************************/

   rV = read_idf (ex->data_key, ex->exten, ex->version, (ByTe_1 *) &offset,
                      _CritOfF, crit_tbl, (ByTe_4) sen, 1);
   if (rV < 0) { return (ir_get_crit_rval (rV)); }

   rV = read_idf (ex->data_key, ex->exten, ex->version,
                     (ByTe_1 *) crit_act_ptr->crit_action, _CritActioN,
                     crit_tbl, (ByTe_4) offset, num_states);
   if (rV < 0) { return (ir_get_crit_rval (rV)); }

   return (ALL_OKAY);
}

/*******************************************************************************
 *                                                                             *
 *                           IR_GET_CRIT_RVAL SUBROUTINE                       *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine returns an error code unique to the GET_CRITICAL_INFO()     *
 *  module based upon the error code returned by the call to READ_IDF().       *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    long rV                     the error code returned by READ_IDF()        *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_get_crit_rval (rV)                                                *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    None                                                                     *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_get_crit_rval (ByTe_4 rV)
{
  switch (rV)
   {
      case IDF_NOT_FOUND:  return (CRIT_IDF_NOT_FOUND);
      case IDF_MANY_BYTES: return (CRIT_IDF_MANY_BYTES);
      case IDF_TBL_NUM:    return (CRIT_IDF_TBL_NUM);
      case IDF_CON_NUM:    return (CRIT_IDF_CON_NUM);
      case IDF_NO_ENTRY:   return (CRIT_IDF_NO_ENTRY);
      default:             return ((ByTe_2) rV);
   }
}
