#include "util_str.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                        IR_FILL_CAL_ARRAY_BYTE SUBROUTINE                   *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to retrieve the calibration data for each cali-  *
 *  bration set for the sensor being processed if the virtual instrument has  *
 *  a word length >= 8.  Each sensor has its own sensor calibration data and  *
 *  sample sequence calibration data, if any exists, with all cal. data laid  *
 *  down sequentially.                                                        *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    long SenIndex              index value that tells the position of the   *
 *                               sensor within the sensors returned           *
 *    void *UDF                  ptr to the memory location for the structure *
 *                               that holds returned data values (read_drec)  *
 *                                                                            *
 *  USAGE                                                                     *
 *    ir_fill_cal_array_byte (SenIndex, UDF)                                  *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    None                                                                    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info        structure that holds information concerning  *
 *        ginfo                  the experiment that is being processed       *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct idf_data *ExDa      structure that holds all of the currently    *
 *                               returned data values to be processed         *
 *    struct experiment_info     a pointer to the structure that holds        *
 *          *ex                  specific experiment information              *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all   *
 *                               pointers to the header and data for the      *
 *                               experiment of interest                       *
 *    register long *Ld          ptr to the array where the data is returned  *
 *    register uns long *lswp    the cal. data for the sample sequence is     *
 *                               interpreted as an array of longs             *
 *    register uns long *lsen    the cal. data for the sensor data is         *
 *                               interpreted as an array of longs             *
 *    register uns short *sswp   the cal. data for the sample sequence is     *
 *                               interpreted as an array of shorts            *
 *    register uns short *ssen   the cal. data for the sensor data is         *
 *                               interpreted as an array of shorts            *
 *    register uns char *uswp    the cal. data for the sample sequence is     *
 *                               interpreted as an array of characters        *
 *    register uns char *usen    the cal. data for the sensor data is         *
 *                               interpreted as an array of characters        *
 *    register uns short j       looping variable                             *
 *    register short i           looping variable                             *
 *    long sen_sen_off           no. of bytes (or bits) to bypass in order    *
 *                               to get to the data for the requested sensor  *
 *    long block_offset          no. of bytes to bypass in order to get to    *
 *                               the correct sensor set                       *
 *    long num_of_bits           number of bits to bypass the sensor set(s)   *
 *    long bytes                 number of bytes needed for 1 cal value       *
 *    long num_bytes             number of bytes to bypass the sensor set(s)  *
 *    long num_per_sen           the number of calibration values per sensor  *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

void ir_fill_cal_array_byte (ByTe_4 SenIndex, void *UDF)
{
  extern struct general_info ginfo;

  struct idf_data *ExDa;
  struct experiment_info *ex;
  struct ptr_rec *ptr;
  register ByTe_4 *Ld;
  register u_ByTe_4 *lsen, *lgen, *lpre;
  register u_ByTe_2 *ssen, *sgen, *spre;
  register u_ByTe_1 *usen, *ugen, *upre;
  register ByTe_1 *Cw, *Ct;
  register u_ByTe_2 j;
  ByTe_4 *Tpt, CurCol, NSeN;
  ByTe_4 sen_sen_off, block_offset, num_of_bits, bytes, num_bytes;
  ByTe_4 num_per_sen;
  u_ByTe_2 *uS, *EnD, *uS1;

  /***********************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header */
  /*  and data information for the experiment currently being processed. */
  /***********************************************************************/

  ex  = ginfo.expt;
  ptr = ex->info_ptr;
  ExDa = (struct idf_data *) UDF;

  /***************************************************************************/
  /*  Copy the number of elements in each cal. set into ExDa structure.      */
  /*  This addition was necessary since convert_to_units was using           */
  /*  ex->CSET_NUM, which may or may not be correct since, with fwd set to 1 */
  /*  on read_drec(), new header info. is read into this field.              */
  /***************************************************************************/

  EnD = ex->CSET_NUM + ex->cal_sets;
  for (uS = ex->CSET_NUM, uS1 = ExDa->cset_num; uS < EnD;)
     *uS1++ = *uS++;

  /********************************************************************/
  /*  Determine the number of bytes per sensor data element.          */
  /*  The premise at this time is that all data is <= 32 bits in      */
  /*  length.  Data is laid down in 1, 2, or 4 bytes.                 */
  /********************************************************************/

  bytes = (ex->tdw_len < 9) ? 1 : (ex->tdw_len < 17) ? 2 : 4;

  /********************************************************************/
  /*  The sensor ancillary data lies together.  (All sensor N , then  */
  /*  N + 1, etc.                                                     */
  /********************************************************************/ 

  num_of_bits = ex->accum_ss_sz;
  num_bytes = num_of_bits / 8;
  if (num_of_bits % 8 != 0) 
    ++num_bytes;

  block_offset = num_bytes;
  if (ex->smp_id != 3)
  {
     NSeN = *ptr->N_SEN;
     CurCol = SenIndex;
  }
  else
  {
     NSeN = *ptr->N_SEN * *ptr->N_COLS;
     CurCol = SenIndex * *ptr->N_COLS + (ExDa->column - *ptr->LEAD_COL);
  }

  num_per_sen = ptr->NumSenAnc / NSeN;

  sen_sen_off = num_per_sen * ex->AncSBytes * CurCol;

  /********************************************************************/
  /*  Retrieve the data associated with each calibration set for the  */
  /*  sensor being processed.  This is done by data size              */
  /********************************************************************/ 

  ExDa->cal_len = 0;
  Ld = ExDa->cal_data;
  Cw = ex->cal_wlen;
  Ct = ex->cal_target;
  EnD = ex->CSET_NUM + ex->cal_sets;

  usen = (u_ByTe_1 *)(ptr->CAL_SENSOR + block_offset + sen_sen_off);
  ugen = (u_ByTe_1 *)(ptr->CAL_GENERAL + block_offset);
  upre = (u_ByTe_1 *)ptr->CAL_PREFACE;
  ssen = (u_ByTe_2 *)(ptr->CAL_SENSOR + block_offset + sen_sen_off);
  sgen = (u_ByTe_2 *)(ptr->CAL_GENERAL + block_offset);
  spre = (u_ByTe_2 *)ptr->CAL_PREFACE;
  lsen = (u_ByTe_4 *)(ptr->CAL_SENSOR + block_offset + sen_sen_off);
  lgen = (u_ByTe_4 *)(ptr->CAL_GENERAL + block_offset);
  lpre = (u_ByTe_4 *)ptr->CAL_PREFACE;

  for (uS = ex->CSET_NUM; uS < EnD; ++uS, ++Cw)
  {
      Tpt = Ld;
      switch (*Ct++)
      {
         case 0:
            switch (ex->AncSBytes)
            {
                case 1:
                for (j = 0; j < *uS; ++j)
                   *Ld++ = *usen++;
                break;
                case 2:
                for (j = 0; j < *uS; ++j)
                   *Ld++ = *ssen++;
                break;
                case 4:
                for (j = 0; j < *uS; ++j)
                   *Ld++ = *lsen++;
                break;
            }
         break;
         case 2:
            switch (ex->AncGBytes)
            {
                case 1:
                for (j = 0; j < *uS; ++j)
                   *Ld++ = *ugen++;
                break;
                case 2:
                for (j = 0; j < *uS; ++j)
                   *Ld++ = *sgen++;
                break;
                case 4:
                for (j = 0; j < *uS; ++j)
                   *Ld++ = *lgen++;
                break;
            }
         break;
         case 3:
            switch (ex->AncPBytes)
            {
                case 1:
                   *Ld++ = *upre++;
                break;
                case 2:
                   *Ld++ = *spre++;
                break;
                case 4:
                   *Ld++ = *lpre++;
                break;
            }
         break;
      }   

      ExDa->cal_len += *uS;
      if (*Cw < 0)
         ir_to_sign (Tpt, *uS, -(*Cw));
  }
}
