#include "util_str.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                        IR_FILL_CAL_ARRAY_BIT SUBROUTINE                    *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to retrieve the calibration data for each cali-  *
 *  bration set for the sensor being processed if the virtual instrument has  *
 *  a bit word length (< 8 bits).  Each sensor has its own sensor calibration *
 *  data and sample sequence calibration data, if any exists, with all cal.   *
 *  data laid down sequentially.  (S0S0S0, S1S1S1, etc.)                      *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    long SenIndex              index value that tells the position of the   *
 *                               sensor within the sensors returned           *
 *    void *UDF                  the memory location ptr for the structure    *
 *                               that holds returned data values (read_drec)  *
 *                                                                            *
 *  USAGE                                                                     *
 *    ir_fill_cal_array_bit (SenIndex, UDF)                                   *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    None                                                                    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info        structure that holds information concerning  *
 *        ginfo                  the experiment that is being processed       *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct idf_data *ExDa  structure that holds all of the currently        *
 *                               returned data values to be processed         *
 *    struct experiment_info     a pointer to the structure that holds        *
 *          *ex                  specific experiment information              *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all   *
 *                               pointers to the header and data for the      *
 *                               experiment of interest                       *
 *    register long *lpt         ptr to the array where the data is returned  *
 *    register uns char *uswp    the cal. data for the sample sequence is     *
 *                               interpreted as an array of characters        *
 *    register uns char *usen    the cal. data for the sensor data is         *
 *                               interpreted as an array of characters        *
 *    register unsigned short j  looping variable                             *
 *    register short i           looping variable                             *
 *    long sen_offset            no. of bytes (or bits) to bypass in order    *
 *                               to get to the requested sensor               *
 *    long block_offset          no. of bytes to bypass in order to get to    *
 *                               the correct sensor set                       *
 *    long swp_cal_offset        the current bit pointed to within the sensor *
 *                               set for the sample sequence calibration data *
 *    long sen_cal_offset        the current bit pointed to within the sensor *
 *                               set for the sensor calibration data          *
 *    long num_per_sen           the number of calibration values per sensor  *
 *    short which_byte           indicates byte to point to for bit data      *
 *    short which_bit            indicates bit to point to for bit data       *
 *    short result               intermediate value used in bit manipulations *
 *    char byte_size             how many bits are used per data element      *
 *                               within each byte (necessary due to padding)  *
 *    stat short one_bit_mask[]  value to AND with to get the bit of interest *
 *    stat char two_bit_shift[]  the no. of bits to shift to get the two bits *
 *                               of interest to the last 2 bits in the byte   *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

void ir_fill_cal_array_bit (ByTe_4 SenIndex, void *UDF)
{
  extern struct general_info ginfo;

  struct idf_data *ExDa;
  struct experiment_info *ex;
  struct ptr_rec *ptr;
  register ByTe_4 *lpt;
  register u_ByTe_1 *usen;
  register u_ByTe_2 j;
  register ByTe_2 i;
  ByTe_4 *Tpt;
  ByTe_4 sen_offset, block_offset, sen_cal_offset, num_per_sen;
  u_ByTe_2 clen;
  u_ByTe_2 *uS, *EnD, *uS1;
  ByTe_2 which_byte, which_bit, result;
  ByTe_1 byte_size, wlen;
  static ByTe_2 one_bit_mask[8] = {128, 64, 32, 16, 8, 4, 2, 1};
  static ByTe_1 two_bit_shift[4] = {6, 4, 2, 0};

  /***************************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header and */
  /*  data information for the experiment currently being processed.         */
  /***************************************************************************/

  ex  = ginfo.expt;
  ptr = ex->info_ptr;
  ExDa = (struct idf_data *) UDF;

  /****************************************************************************/
  /*  Copy the number of elements in each cal. set into ExDa structure.   */
  /*  This addition was necessary since convert_to_units was using            */
  /*  ex->CSET_NUM, which may or may not be correct since, with fwd set to 1  */
  /*  on read_drec(), new header info. is read into this field.               */
  /****************************************************************************/

  EnD = ex->CSET_NUM + ex->cal_sets;
  for (uS = ex->CSET_NUM, uS1 = ExDa->cset_num; uS < EnD;)
     *uS1++ = *uS++;

  /****************************************************************************/
  /*  Byte_size indicates how many bits are used per data element within each */
  /*  byte.  For 5, 6 and 7 bit data, there are pad bits so that there is     */
  /*  only one sample per byte (takes 8 bits).  For 3 bit data, use 4 since   */
  /*  the data is laid down as a 4-bit value with the most significant bit    */
  /*  set to zero.  The rest of the cases divide evenly into the byte.        */
  /****************************************************************************/

  byte_size = (ex->tdw_len >= 5) ? 8 : (ex->tdw_len == 3) ? 4 : ex->tdw_len;

  /**************************************************************************/
  /*  The no. of bits needed to bypass in order to get to the current       */
  /*  sensor set.  Convert that no. to bytes.                               */
  /**************************************************************************/

  block_offset = ex->accum_ss_sz / 8;
  if (ex->accum_ss_sz % 8 != 0) 
    ++block_offset;

  /************************************************************************/
  /*  To point to the correct calibration data, we must be at the correct */
  /*  sensor set and the correct sensor.                                  */
  /************************************************************************/ 

  num_per_sen = ptr->NumSenAnc / *ptr->N_SEN;
  sen_offset = num_per_sen * byte_size * SenIndex;  /* in bits */
  sen_cal_offset = sen_offset;

  /********************************************************************/
  /*  Retrieve the data associated with each calibration set for the  */
  /*  sensor being processed.                                         */
  /********************************************************************/ 

  ExDa->cal_len = 0;
  lpt = ExDa->cal_data;
  for (i = 0; i < ex->cal_sets; ++i)
  {
     /******************************************************************/
     /*  Pick up sweep calibration data for this cal. set.             */
     /******************************************************************/

     Tpt = lpt;
     wlen = *(ex->cal_wlen + i);
     clen = *(ex->CSET_NUM + i);

     for (j = 0; j < *(ex->CSET_NUM + i); ++lpt, ++j)
     {
         which_bit = sen_cal_offset % 8;
         which_byte = sen_cal_offset / 8;
         usen =(u_ByTe_1 *)(ptr->CAL_SENSOR + block_offset + which_byte);

         /****************************************************************/
         /*  For 1-bit data, see if the bit of interest is 1 or 0.       */
         /****************************************************************/

         if (ex->tdw_len == 1)
         {
            result = *(usen) & one_bit_mask[which_bit];
            *(lpt) = (result != 0) ? 1 : result;
         }

         /****************************************************************/
         /*  Shift the bits of interest down to LSB bits and mask the 2. */
         /****************************************************************/

         else if (ex->tdw_len == 2)
         {
            result = *(usen) >> two_bit_shift[which_bit/2];
            *(lpt) = result & 3;
          }
   
         /****************************************************************/
         /*  Shift the 4 bits of interest down to LSB bits and mask off  */
         /*  the real 3 bits of data.  Which_bit should be either 0 or 4.*/
         /*  No need to shift if 2nd sample is retrieved from the byte.  */
         /****************************************************************/

         else if (ex->tdw_len == 3)
         {
            result = (which_bit == 0) ? *(usen) >> 4 : *(usen);
            *(lpt) = result & 7;
         }

         /****************************************************************/
         /*  Shift the bits of interest down to LSB bits and mask the 4. */
         /*  Which_bit should be either 0 or 4.  No need to shift if the */
         /*  second sample is being retrieved from the byte.             */
         /****************************************************************/

         else if (ex->tdw_len == 4)
         {
            result = (which_bit == 0) ? *(usen) >> 4 : *(usen);
            *(lpt) = result & 15;
         }

         /***************************************************************/
         /*  The samples require a whole byte.                          */
         /***************************************************************/

         else
           *(lpt) = *(usen);

         sen_cal_offset += byte_size;
      }

      ExDa->cal_len += *(ex->CSET_NUM + i);

      if (wlen < 0)
         ir_to_sign (Tpt, clen, -wlen);
   }
}
