#include <stdlib.h>
#include "util_str.h"
#include "gen_defs.h"
#include "ret_codes.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                        CREATE_IDF_DATA_STRUCTURE SUBROUTINE                *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is creates or allocates a data structure that holds the    *
 *  udf data once it is read from the files.  The address of the newly        *
 *  allocated data structure is passed back to the calling module. An array   *
 *  of pointers is maintained so that when FREE_EXPERIMENT_INFO is called the *
 *  memory for all allocated data structures is freed. There is no need to    *
 *  pass in the data key of interest; all items that depend upon sizes from   *
 *  the data set will be allocated/reallocated when new header records are    *
 *  read; thus items end up being maxed out by the virtual instrument with    *
 *  the largest size for the item in question if the data structure is used   *
 *  by more than one data set.                                                *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    void **idf_data_ptr          ptr to the memory location that holds the  *
 *                                 address of the structure holding returned  *
 *                                 data values (read_drec)                    *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = create_idf_data_structure (&idf_data_ptr)                           *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sizeof ()                    the size of the specified object in bytes  *
 *    malloc()                     allocates memory                           *
 *    realloc()                    reallocates previously allocated memory    * 
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info ginfo    structure holding information concerning   *
 *                                 the experiment that is being processed     *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct idf_data *EXP_DATA    structure that holds all of the currently  *
 *                                 returned data values to be processed       *
 *    unsigned int B               the number of bytes to allocate            *
 *    int index                    index into the array of pointers           *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 create_idf_data_structure (void **idf_data_ptr)
{
   extern struct general_info ginfo;

   struct idf_data *EXP_DATA;
   size_t B;
   int index;
   ByTe_4 I;

  /**************************************************************************/
  /*  If there is no array in place to hold the data structures, establish  */
  /*  it now.  Leave room for 25 instances of the structure.                */
  /**************************************************************************/

   if (ginfo.idf_data_ptr == NO_MEMORY) {
      B = 25 * sizeof (void *);
      if ((ginfo.idf_data_ptr = malloc (B)) == NO_MEMORY)
         return (CREATE_DATA_ALL_MALLOC);
 
      ginfo.num_idf_data_ptr = 25;
      for ( I = 0; I < ginfo.num_idf_data_ptr; ++I) {
         *(ginfo.idf_data_ptr + I) = NO_MEMORY;
     }
  }

  /**************************************************************************/
  /* Look for an empty slot                                                 */
  /**************************************************************************/

   index = -1;
   for ( I = 0; I < ginfo.num_idf_data_ptr; ++I) {
      if (*(ginfo.idf_data_ptr + I) == NO_MEMORY ) {
         index = I;
         break;
      }
   }

  /**************************************************************************/
  /*  If there is no empty slot then expand the array by 25 elements        */
  /**************************************************************************/

   if ( index < 0 ) {
      B = (ginfo.num_idf_data_ptr + 25) * sizeof (void *);
      if ((ginfo.idf_data_ptr = realloc(ginfo.idf_data_ptr,B)) == NO_MEMORY)
         return (CREATE_DATA_ALL_REALLOC);

      index = ginfo.num_idf_data_ptr;
      ginfo.num_idf_data_ptr += 25;
      for ( I = index; I < ginfo.num_idf_data_ptr; ++I) {
         *(ginfo.idf_data_ptr + I) = NO_MEMORY;
      }
   }

  /*************************************************************************/
  /*  Allocate the data structure.                                         */
  /*************************************************************************/

   B = sizeof (struct idf_data);
   if ((*(ginfo.idf_data_ptr + index) = malloc (B)) == NO_MEMORY)
      return (CREATE_DATA_MALLOC);
   *idf_data_ptr = *(ginfo.idf_data_ptr + index);

  /************************************************************************/
  /*  Initialize the data structure that is passed back to the user for   */
  /*  usage with the generic read utility program.                        */
  /************************************************************************/

   EXP_DATA = (struct idf_data *) *(ginfo.idf_data_ptr + index);
   EXP_DATA->data_key = 0;
   EXP_DATA->sensor = 0;
   EXP_DATA->byear = 0;
   EXP_DATA->bday = 0;
   EXP_DATA->bmilli = 0;
   EXP_DATA->bnano = 0;
   EXP_DATA->eyear = 0;
   EXP_DATA->eday = 0;
   EXP_DATA->emilli = 0;
   EXP_DATA->enano = 0;
   EXP_DATA->data_accum_ms = 0;
   EXP_DATA->data_accum_ns = 0;
   EXP_DATA->data_lat_ms = 0;
   EXP_DATA->data_lat_ns = 0;
   EXP_DATA->swp_reset_ms = 0;
   EXP_DATA->swp_reset_ns = 0;
   EXP_DATA->sen_reset_ms = 0;
   EXP_DATA->sen_reset_ns = 0;
   EXP_DATA->num_swp_steps = 0;
   EXP_DATA->num_sample = 0;
   EXP_DATA->num_angle = 0;
   EXP_DATA->num_pitch = 0;
   EXP_DATA->mode_len = 0;
   EXP_DATA->cal_len = 0;
   EXP_DATA->sun_sen = 0;
   EXP_DATA->spin_rate = 0;
   EXP_DATA->cal_size = 0;
   EXP_DATA->data_size = 0;
   EXP_DATA->swp_size = 0;
   EXP_DATA->angle_size = 0;
   EXP_DATA->pitch_size = 0;
   EXP_DATA->mode_size = 0;
   EXP_DATA->hdr_change = 0;
   EXP_DATA->d_qual = 0;

  /************************************************************************/
  /*  These elements are dependent upon information in the actual header  */
  /*  file and are allocated as the information becomes available.        */
  /************************************************************************/

   EXP_DATA->cal_data = NO_MEMORY;
   EXP_DATA->base_cal = NO_MEMORY;
 
   EXP_DATA->sen_data = NO_MEMORY;
   EXP_DATA->base_data = NO_MEMORY;

   EXP_DATA->swp_data = NO_MEMORY;
   EXP_DATA->base_swp = NO_MEMORY;

   EXP_DATA->start_az = NO_MEMORY;
   EXP_DATA->stop_az = NO_MEMORY;
   EXP_DATA->base_angle = NO_MEMORY;

   EXP_DATA->base_mode = NO_MEMORY;
   EXP_DATA->mode = NO_MEMORY;

   EXP_DATA->pitch_angles = NO_MEMORY;
   EXP_DATA->base_pitch = NO_MEMORY;

   EXP_DATA->base_cset = NO_MEMORY;
   EXP_DATA->cset_num = NO_MEMORY;

   return (ALL_OKAY);
}
