#include "util_str.h"
#include "ret_codes.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                           IR_CHECK_CRITICAL_STATUS SUBROUTINE               *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called whenever a header change has occurred to check    *
 *  the critical status bytes for each utilized sensor to see if a change in   *
 *  status has occurred.  If there are no critical bytes to check, the routine *
 *  simply returns; otherwise, the array of status flags are checked to see    *
 *  which of the utilized sensors has a critical status flag.  A value of -1   *
 *  indicates that the sensor does not utilize critical status flags.  Any     *
 *  other value corresponds to the status byte of interest.  If the contents   *
 *  of that byte changed from the previous header, take the appropriate        *
 *  action, if any.                                                            *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    None                                                                     *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_check_critical_status ()                                          *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    ir_get_new_tbl()             reads the requested table for the sensor    *
 *                                 from the VIDF file                          *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info ginfo    structure that holds information concerning *
 *                                 the experiment that is being processed      *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info *ex   a pointer to the structure that holds       *
 *                                 specific experiment information             *
 *    struct sensor_tables *sptr   a pointer to the structure which holds the  *
 *                                 various information for the tables utilized *
 *                                 by sensor-table combination being processed *
 *    struct crit_action_info      a pointer to the structure that holds info. *
 *       *crit_act_ptr             pertinent to table(s) that are flagged as   *
 *                                 critical action table(s)                    *
 *    register short i             looping variable                            *
 *    register short act_index     index into the array of structures that hold*
 *                                 critical action information                 *
 *    float *tbl_address           pointer to the table being processed        *
 *    long offset                  offset into the table of interest           *
 *    long next_tbl                index used to get to the table of interest  *
 *    short rval                   the value returned by the called routine    *
 *    short sen                    looping variable                            *
 *    unsigned char cvalue         the MODE_INDEX value for the critical status*
 *                                 byte for the sensor being processed         *
 *    char found_tbl               indicates if the table requested was found  *
 *    char reset_called            flag indicating if LOCATE_EX was called     *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_check_critical_status (void)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   register struct sensor_tables *sptr;
   register struct crit_action_info *crit_act_ptr;
   register short i, act_index;
   ReaL_4 *tbl_address;
   ByTe_4 offset, next_tbl;
   ByTe_2 rval, sen;
   u_ByTe_1 cvalue;
   ByTe_1 found_tbl, reset_called;

   /**********************************************************************/
   /*  Check for the existence of a critical status array.               */
   /**********************************************************************/

   ex = ginfo.expt;
   if (ex->crit_action == 0)
     return (ALL_OKAY);

   /**************************************************************************/
   /*  Determine the action, if any, that needs to be taken for each sensor. */
   /**************************************************************************/

   else
    {
      if (!ginfo.called_locate)
       {
         ginfo.called_locate = 1;
         reset_called = 1;
       }
      else
        reset_called = 0;
      for (sen = 0; sen < ex->num_sensor; ++sen)
       {
         /******************************************************************/
         /*  A value of -1 means that this sensor is not being utilized    */
         /*  (was not a requested sensor), so go on to the next sensor.    */
         /******************************************************************/

         if (*(ex->index_sen_tbl + sen) == -1)
           continue;

         /******************************************************************/
         /*  Loop over all tables that utilize critical action.            */
         /******************************************************************/

         sptr = ex->sen_tbl_ptr + *(ex->index_sen_tbl + sen);
         for (act_index = 0; act_index < ex->crit_action; ++act_index)
          {
            /*****************************************************************/
            /*  If crit_status for this sensor is -1, no action is necessary.*/
            /*****************************************************************/

            crit_act_ptr = sptr->act_ptr + act_index;
            if (crit_act_ptr->crit_status == -1)
              continue;

            /******************************************************************/
            /*  Initially, crit_flag is set to -1 so that the first time      */
            /*  a header is read, the correct action will be taken; otherwise,*/
            /*  it holds the contents of the critical status byte.  This field*/
            /*  is now a short to keep the -1 setting but to accomodate the   */
            /*  fact that MODE_INDEX is now unsigned char.                    */
            /******************************************************************/

            cvalue = *(ex->info_ptr->MODE_INDEX + crit_act_ptr->crit_status);
            if (cvalue != crit_act_ptr->crit_flag)
             {
               /***************************************************************/
               /*  CRIT_ACTION holds offsets into the appropriate table for   */
               /*  the current instrument mode and MODE_INDEX value is the    */
               /*  index into the CRIT_ACTION array.                          */
               /***************************************************************/
   
               offset = *(crit_act_ptr->crit_action + cvalue);
               if (offset == crit_act_ptr->crit_offset)
                 continue;

               crit_act_ptr->crit_offset = offset;
               crit_act_ptr->crit_flag = cvalue;

               /***************************************************************/
               /*  Loop over all tables.                                      */
               /***************************************************************/

               found_tbl = 0;
               next_tbl = 0;
               for (i = 0; i < sptr->num_tbls; ++i)
                {
                  /***********************************************************/
                  /*  Retrieve new table values.  The address of the table   */
                  /*  is calculated based upon table sizes because pointer   */
                  /*  to the table may be altered to the midpoint of table   */
                  /*  to handle signed binary input.                         */
                  /***********************************************************/

                  if (i == crit_act_ptr->crit_tbl)
                   {
                     if (*(sptr->tbl_fmt + i) >= 0)
                      {
                        tbl_address = sptr->tbls + next_tbl;
                        rval = ir_get_new_tbl (sen, crit_act_ptr->crit_tbl, 
                                  *(sptr->tbl_expand + i), *(sptr->tbl_fmt + i),
                                  tbl_address, *(sptr->tbl_size + i), offset);
                        if (rval != ALL_OKAY)
	                 {
                           if (reset_called)
                             ginfo.called_locate = 0;
                           return (rval);
                         }
                      }
                     found_tbl = 1;
                     break;
                   }
                  next_tbl += *(sptr->tbl_size + i);
                }

               /***************************************************************/
               /*  The table requested was not found - error condition.       */
               /***************************************************************/

               if (!found_tbl)
                {
                  if (reset_called)
                    ginfo.called_locate = 0;
                  return (CRIT_TBL_NOT_FOUND);
	        }
            }
         }
      }

     if (reset_called)
       ginfo.called_locate = 0;
   }

  return (ALL_OKAY);
}
