#include <stdlib.h>
#include "util_str.h"
#include "gen_defs.h"
#include "libbase_udf.h"
#include "OpSySInD.h"

/******************************************************************************
 *                                                                            *
 *                           IR_TBL_SIZE  SUBROUTINE                          *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called in order to determine the total number of bytes  *
 *  that are needed to hold all integer table values for all tables for the   *
 *  sensor in question.  No bytes are included for ascii tables since these   *
 *  tables are kept in a different structure from the integer table values.   *
 *  In the sizeof function, use float instead of long since, in the calling   *
 *  module, the malloced space is set to a floating point pointer.            *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    short sIndx           index into the SPTR structure to point to         *
 *                                 the correct sensor combination structure   *
 *    unsigned int *bytes          the number of bytes to allocate            *
 *                                                                            *
 *  USAGE                                                                     *
 *    ir_tbl_size (sIndx, &bytes)                                             *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sizeof ()                    the size of the specified object in bytes  *
 *    abs()                        returns the absolute value of a number     *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info ginfo    structure holding information concerning   *
 *                                 the experiment that is being processed     *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info *ex   a pointer to the structure that holds      *
 *                                 specific experiment information            *
 *    struct sensor_tables *sptr   a pointer to the structure holding the     *
 *                                 information for the tables utilized by     *
 *                                 by sensor-table combos being processed     *
 *    float **tP                   pointer to sptr->tbl_ptr                   *
 *    long  *tS                    pointer to sptr->tbl_size                  *
 *    short TLen                   the maximum number of elements that can be *
 *                                 associated with a given data word length   *
 *    char  TdwLen                 the data word length in bits               *
 *    char  CalLen                 the word length in bits of the cal. data   *
 *    char  CalSet                 the calibration set being processed        *
 *    char  *tF                    pointer to sptr->tbl_fmt                   *
 *    char  *tE                    pointer to sptr->tbl_expand                *
 *    char  *tV                    pointer to ex->tbl_var                     *
 *    char  *tT                    pointer to ex->tbl_type                    *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *                             CHANGE LOG                                     *
 *                                                                            *
 *    DATE           BY                         CHANGE                        *
 *  --------   -------------  -------------------------------------------     *
 *  3/25/98     C. Gurgiolo    Added ability to define lookup table for       *
 *                             processed data                                 *
 *****************************************************************************/

void ir_tbl_size (ByTe_2 sIndx, size_t *bytes)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   struct sensor_tables *sptr;

   register ByTe_4 *tS, *EnD;
  
   ReaL_4 **tP;

   u_ByTe_4 TLen = 1;
   ByTe_4 sF;
   ByTe_1 TdwLen, CalLen, CalSet;

   ByTe_1 *tE, *tT, *tV, *tF;


  sF = sizeof(ReaL_4);

  /*************************************************************************/
  /*  Point to the correct sensor_tables structure.                        */
  /*************************************************************************/

   ex = ginfo.expt;
   sptr = ex->sen_tbl_ptr + sIndx;

  /*********************************************************************/
  /*  The data word length is used to determine how many elements are  */
  /*  in an expanded table.(ex. 8 bits means 256 possible values)      */
  /*********************************************************************/

   TdwLen = sptr->max_tdw_len;
   while (TdwLen-- > 0) { TLen *= 2; }

  /**********************************************************************/
  /*  Loop over all defined tables for this virtual instrument in order */
  /*  to determine how much space to allocate for INTEGER table values. */
  /*  Ascii strings are saved in a different structure.                 */
  /**********************************************************************/

   *bytes = 0;                                     /* allocation size to 0 */
   tS  = sptr->tbl_size;                           /* table sizes          */
   tE  = sptr->tbl_expand;                         /* table expand flags   */
   tF  = sptr->tbl_fmt;                            /* table formats        */
   tP  = sptr->tbl_ptr;                            /* table pointers       */
   tV  = ex->tbl_var;                              /* table variables      */
   tT  = ex->tbl_type;                             /* table types          */
   EnD = tS + ex->num_tbls;                        /* end of loop          */ 
   for ( ; tS < EnD; ++tV, ++tT, ++tS, ++tE, ++tP, ++tF)  {  

  /**********************************************************************/
  /* Don't include mode or ascii tables here                            */
  /**********************************************************************/

      if (*tV == 4 || *tV == 5 || *tT == 1) 
        continue;

  /**********************************************************************/
  /* Size needed to hold table based on table variable                  */
  /**********************************************************************/

      if (*tF == 0 || (*tF > 0 && *tE == 1)) {
         *tS = 0;
         switch (*tV) {
            case 0:                                /* RAW SEN DATA         */
               *tS = TLen;
            break;
            case 1:                                /* PROCESSED DATA       */
               TdwLen = *tE;
               *tS = 1;
               while (TdwLen-- > 0) { *tS *= 2; }
            break;
            case 2:                                /* RAW SCAN DATA        */
               *tS = ex->swp_len;
            break;
            default:                               /* CAL DATA             */
               CalSet = abs(*tV) - 1;
               CalLen = *(ex->cal_wlen + CalSet);
               if (CalLen < 0) CalLen = -CalLen;
               *tS = 1;
               while (CalLen-- > 0) { *tS *= 2; }
            break;
         }
      } else {                                    

  /**********************************************************************/
  /* Size of polynomial tables                                          */
  /**********************************************************************/

         if (*tF > 0) {
            *tS = *tF;
         } else {

  /**********************************************************************/
  /* Table is undefined for this sensor                                 */
  /**********************************************************************/

            if (*tF < 0) { 
               *tS = 0;
               *tP = NO_MEMORY;
            }
         }
      }

  /**********************************************************************/
  /*  Memeory needed dependent on the table type                        */
  /**********************************************************************/

      switch (*tT) {
         case 2:                                     /* SCAN ELEMENT       */
            *tS *= ex->swp_len;
         break;
         case 3:                                     /* TIME BASED         */
            *tS += 5 * sF;
         break;
         case 4:                                     /* TIME & SWP TBL     */
            *tS = *tS * ex->swp_len + 5 * sF;
         break;
      }

  /**********************************************************************/
  /*  Compute the bytes needed                                          */
  /**********************************************************************/

      *bytes += sF * *tS;
   }
}
