#include "OpSySInD.h"
#include "idf_defs.h"
#include "local_defs.h"
#include "ret_codes.h"
#include "util_str.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                        IR_READ_IDF_DISK SUBROUTINE                          *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to retrieve information from the VIDF definition  *
 *  file, which is on the disk, for the data set specified.  This routine      *
 *  will hit the disk for every access of information from the VIDF file.      *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    char *var                   output value(s) associated with selected     *
 *                                field                                        *
 *    short quan                  specified field in VIDF file                 *
 *    short num                   the table or constant definition from        *
 *                                which the required field is to be retrieved  *
 *    short len                   if the selected field is an array, this is   *
 *                                the number of elements to be returned        *
 *    long start                  if the selected field is an array, this is   *
 *                                the starting position in the array           *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_read_idf_disk (&var, quan, num, start, len)                       *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    lseek()                     moves the file pointer to a location within  *
 *                                the file                                     *
 *    read()                      reads N bytes from the file associated with  *
 *                                the file descriptor given                    *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info         structure that holds information concerning  *
 *         ginfo                  the experiment that is being processed       *
 *    short SiZe[]                the size of each field (1byte, 2bytes, etc)  *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info      a pointer to the structure that holds        *
 *            *ex                 specific experiment information              *
 *    register long *l1           pointer to offset values                     *
 *    long bytes                  the number of bytes to read for the field    *
 *    long avail_bytes            the number of bytes left to retrieve         *
 *    long upper                  number of bytes to get to the next field     *
 *    long to_data                the number of bytes to bypass to get to the  *
 *                                requested element(s)                         *
 *    long Telem                  offset to get to table elements              *
 *    long lsize                  the size of the element being requested      *
 *    char type                   the type of data being requested             *
 *    char tmp                    temporary variable                           *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_4 ir_read_idf_disk (ByTe_1 *var, ByTe_2 quan, ByTe_2 num, ByTe_4 start, 
                         ByTe_2 len)
{
   extern struct general_info ginfo;
   extern ByTe_2 SiZe[];

   struct experiment_info *ex;
   register ByTe_4 *l1;
   ByTe_4 bytes, avail_bytes, upper, to_data, Telem, lsize;
   ByTe_4 qN, qS;
   ByTe_1 type, tmp;

   /**************************************************************************/
   /*  The ex structure of interest has been found by READ_IDF already.      */ 
   /**************************************************************************/

   ex = ginfo.expt;
   Telem = _ConstID - _TblScaSZ;
   type = (quan < _TblScaSZ) ? 0 : (quan < _ConstID) ? 1 : 2;

   qS = (quan == _PaBxbybZ) ? 2 : SiZe[quan];
   switch (type)
   {
      case 0:
         upper = ex->VIDF_INFO.offsets[quan+1];
         avail_bytes = upper - ex->VIDF_INFO.offsets[quan];
         if (len > 0)
         {
            bytes = SiZe[quan] * len;
            if (avail_bytes < bytes)
               return (IDF_MANY_BYTES);
         }
         else
         {
            bytes = avail_bytes;
            if (bytes == 0)
               return (IDF_NO_ENTRY);
         }

         to_data = ex->VIDF_INFO.offsets[quan] + SiZe[quan] * start + 
                   ex->VIDF_INFO.info_off;
         lseek (ex->fdi, to_data, 0);
         read (ex->fdi, var, bytes);
      break;

      case 1:
         if (num > ex->VIDF_INFO.TABLES - 1)
            return (IDF_TBL_NUM);

         if (quan == _TbL)
         {
            l1 = (ByTe_4 *)ex->VIDF_INFO.tbl_offs + Telem * num + _TblTypE - _TblScaSZ;
            bytes = SiZe[_TblTypE];
            to_data = *l1 + ex->VIDF_INFO.info_off;
            lseek (ex->fdi, to_data, 0);
            read (ex->fdi, &tmp, bytes);
            lsize = (tmp == 1) ? 21 : SiZe[quan];
         }
         else
            lsize = SiZe[quan]; 

         l1 = (ByTe_4 *)ex->VIDF_INFO.tbl_offs + Telem * num + quan - _TblScaSZ;
         upper = *(l1 + 1);
         avail_bytes = upper - *l1;
         if (len > 0)
         {
            bytes = lsize * len;
            if (avail_bytes < bytes)
               return (IDF_MANY_BYTES);
         }
         else
         {
            bytes = avail_bytes;
            if (bytes == 0)
               return (IDF_NO_ENTRY);
         }

         to_data = *l1 + lsize * start + ex->VIDF_INFO.info_off;
         lseek (ex->fdi, to_data, 0);
         read (ex->fdi, var, bytes);
      break;

      case 2:
         if (num > ex->VIDF_INFO.CONSTS - 1)
            return (IDF_CON_NUM);

         l1 = (ByTe_4 *)ex->VIDF_INFO.const_offs + 5 * num + quan - _ConstID;
         upper = *(l1 + 1);
         avail_bytes = upper - *l1;
         if (len > 0)
         {
            bytes = SiZe[quan] * len;
            if (avail_bytes < bytes)
               return (IDF_MANY_BYTES);
         }
         else
         {
            bytes = avail_bytes;
            if (bytes == 0)
               return (IDF_NO_ENTRY);
         }

         to_data = *l1 + SiZe[quan] * start + ex->VIDF_INFO.info_off;
         lseek (ex->fdi, to_data, 0);
         read (ex->fdi, var, bytes);
      break;
   }

   if (ex->BswaP && (qS ==2 || qS == 4) ) {
       qN = bytes / qS;
       ReOrderBytes (qS, (u_ByTe_1 *)var, qN); 
   }

   return (ALL_OKAY);
}
