#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/****************************************************************************
 *                                                                          *
 *                    IR_READ_DREC_DATA_VALUES SUBROUTINE                   *
 *                                                                          *
 *  DESCRIPTION                                                             *
 *    This routine is called when the requested sensor has been found among *
 *  the sensors returned for the current data record.  The data values and  *
 *  the calibration values for the sensor are returned, along with any      *
 *  angular information.                                                    *
 *                                                                          *
 *  INPUT VARIABLES                                                         *
 *    void *UDF                  memory location ptr for the structure that *
 *                               holds returned data values (read_drec)     *
 *    char fwd                   flag that indicates when to advance to the *
 *                               time sample                                *
 *    char full_swp              request 1 or all values in sensor set be   *
 *                               returned (for scalar parameters only)      *
 *    long SenIndex              index value that tells the position of the *
 *                               sensor within the sensors returned         *
 *                                                                          *
 *  USAGE                                                                   *
 *    x = ir_read_drec_data_values (UDF, fwd, full_swp, SenIndex)           *
 *                                                                          *
 *  NECESSARY SUBPROGRAMS                                                   *
 *    ir_fill_data_array ()      fills the data structure with the data for *
 *                               the requested sensor                       *
 *    ir_fill_cal_array_byte ()  fills in the data structure with the cal.  *
 *                               data for the requested sensor for virtual  *
 *                               instruments with TDW_LEN >= 8              *
 *    ir_fill_cal_array_bit ()   fills in the data structure with the cal.  *
 *                               data for the requested sensor for virtual  *
 *                               instruments with TDW_LEN < 8               *
 *    ir_fill_angle_array ()     fills the data structure with the starting *
 *                               and ending azimuthal angles for requested  *
 *                               sensor                                     *
 *    ir_fill_pitch_angles ()    fills the data structure with the computed *
 *                               pitch angles                               *
 *                                                                          *
 *  EXTERNAL VARIABLES                                                      *
 *    struct general_info        structure holding information concerning   *
 *        ginfo                  the experiment that is being processed     *
 *                                                                          *
 *  INTERNAL VARIABLES                                                      *
 *    struct idf_data *ExDa      structure that holds all of the currently  *
 *                               returned data values to be processed       *
 *    struct experiment_info     a pointer to the structure that holds      *
 *          *ex                  specific experiment information            *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all *
 *                               pointers to the header and data for the    *
 *                               experiment of interest                     *
 *    short rval                 holds called routine status flags          *
 *                                                                          *
 *  SUBSYSTEM                                                               *
 *    Display Level                                                         *
 *                                                                          *
 ***************************************************************************/

ByTe_2 ir_read_drec_data_values (void *UDF, ByTe_1 fwd, 
                                 ByTe_1 full_swp, ByTe_4 SenIndex)
{
  extern struct general_info ginfo;
  extern ByTe_1 BswaP;

  struct idf_data *ExDa;
  struct experiment_info *ex;
  struct ptr_rec *ptr;
  ByTe_2 rval;

  /*************************************************************************/
  /*  Retrieve the sensor data and any calibration data for the requested  */
  /*  sensor.                                                              */
  /*************************************************************************/
 
  ExDa = (struct idf_data *) UDF;
  ex = ginfo.expt;
  ptr = ex->info_ptr;

  if (BswaP && ex->DswaP)
     ReOrderUdf(2);

  ExDa->filled_data = 1;
  ir_fill_data_array (SenIndex, full_swp, UDF);

  /*************************************************************************/
  /*  NOTE:  the sensor tdw_len can be used here since we do not support   */
  /*  sensor wordlengths less than 8 bits in V2.2.  Anything less will be  */
  /*  be rounded up.                                                       */
  /*************************************************************************/

  if (ptr->NumSenAnc != 0 || ptr->NumPreAnc != 0 || ptr->NumGenAnc != 0) 
   {
     if (ex->tdw_len >= 8)
        ir_fill_cal_array_byte (SenIndex, UDF);
     else
        ir_fill_cal_array_bit (SenIndex, UDF);
   }

  ex->num_sample = ExDa->num_sample;

  /*************************************************************************/
  /*  Retrieve angular information.                                        */
  /*************************************************************************/

  ir_fill_angle_array (full_swp, UDF);
  ThetaConing (UDF);

  /**********************************************************************/
  /*  Calculate pitch angles.  Check for non-null memory location in    */
  /*  case pitch angle source also has pa_def set in the VIDF file.     */
  /**********************************************************************/

  if (ex->pa_def && ex->bmem.base_pitch_info != NO_MEMORY)
   {
     rval = ir_fill_pitch_angles (fwd, full_swp, UDF);
     if (rval != ALL_OKAY)
       return (rval);
   }

  return (ALL_OKAY);
}
